
#include "YGV642_SPI_Flash_GD25Q127C.h"

/*------------------------------------------------------------------------------*/
/* Function name    : PM_S25FL256S_ID_Check                                     */
/* Contents         : Pattern Memory JEDEC ID check                             */
/* Argument         : void                                                      */
/* Return value     : 1 - Pass 0 - Fail                                         */
/*------------------------------------------------------------------------------*/
uint8_t PM_GD25Q127C_ID_Check(void)
{
  uint8_t   Identifier[3];
  uint8_t   i;
  
  /* (1), (2) Write Pattern Memory Control register */
	PM_GD25Q127C_Write_Cmd(PM_GD25Q127C_CMD_READ_JEDEC_ID);

	/* (3) Write R#27h: AIPM = 1 */
	YGV642_Write_Reg(REG_R27H, 0x07);

	/* (4) Write P#2: CSNEG = 1 */
	YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_GD25Q127C_READACS_CSNEG);

	/* (5) Write R#2C-R#2F: PMA[25:0] */
	PM_GD25Q127C_Set_Pattern_Memory_Addr(0x00000000);
  
  for (i = 0; i < 3; i++)
  {
    /* Read request */
    YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_GD25Q127C_READACS_PMRREQ);	/* PMRREQ=1, CSNEG=1 */

    /* Waiting for ready to read */
    PM_GD25Q127C_Wait_Clear_Flag(PM_GD25Q127C_READACS_PMRREQ | PM_GD25Q127C_READACS_PWBUSY);

    /* Read register value */
    Identifier[i] = YGV642_Read_Port(YVC1_PORT_PTNMEM_DATA);
  }
  
  if ((Identifier[0] == PM_GD25Q127C_MANUFACTURER_ID) &&\
      (Identifier[1] == PM_GD25Q127C_DEVICE_ID_MSB  ) &&\
      (Identifier[2] == PM_GD25Q127C_DEVICE_ID_LSB  ))
    return 1;
    
  return 0;
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_GD25Q127C_Init                                         */
/* Contents         : Initialize Pattern Memory                                 */
/* Argument         : void                                                      */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/
void PM_GD25Q127C_Init(void)
{
  uint8_t ReadData;
  
	/*--------------------------------------*/
	/* Write Enable                         */
	/*--------------------------------------*/
  
	/* Write Pattern Memory Control Register */
  PM_GD25Q127C_Write_Cmd(PM_GD25Q127C_CMD_WRITE_ENABLE);
  
	/* Write R#2Ch-#2Fh: PMA[25:0] = 26h'000_0000 */
  PM_GD25Q127C_Set_Pattern_Memory_Addr(0x00000000);
  
	/* Write P#2: CSNEG = 1 */
	YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_GD25Q127C_READACS_CSNEG);
  
  /* Write P#3: MD[7:0] = 8'h00 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_DATA, 0x00);
  
  /* Polling P#2: PWBUSY */
  PM_GD25Q127C_Wait_Clear_Flag(PM_GD25Q127C_READACS_PWBUSY);
  
	/*--------------------------------------*/
	/* Write Status Regsiter-1              */
	/*--------------------------------------*/
  
	/* Write Status Register command */
  PM_GD25Q127C_Write_Cmd(PM_GD25Q127C_CMD_WRITE_STATUS1);
  
	/* Write R#2Ch-#2Fh: PMA[25:0] = 26h'000_0000 */
  PM_GD25Q127C_Set_Pattern_Memory_Addr(0x00000000);
  
  /* Write P#2: CSNEG = 1 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_GD25Q127C_READACS_CSNEG);

  /* Write P#3: MD[7:0] = 8'h02 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_DATA, 0x02);

  /* Polling P#2: PWBUSY */
  PM_GD25Q127C_Wait_Clear_Flag(PM_GD25Q127C_READACS_PWBUSY);

  
	/*--------------------------------------*/
	/* Read Status Register                 */
	/*--------------------------------------*/
  
	/* Write Pattern Memory Control register */
  PM_GD25Q127C_Write_Cmd(PM_GD25Q127C_CMD_READ_STATUS);
  
	/* Write R#27h: AIPM = 0 */
  YGV642_Write_Reg(REG_R27H, 0x00);
  
	/* Write R#2C-R#2F: PMA[25:0] */
	PM_GD25Q127C_Set_Pattern_Memory_Addr(0x00000000);
  
	/* Write P#2: CSNEG = 1 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_GD25Q127C_READACS_CSNEG);
  
	do
	{
    /* Write P#2: PMRREQ */
    ReadData  = YGV642_Read_Port(YVC1_PORT_PTNMEM_READACS);
    ReadData |= PM_GD25Q127C_READACS_PMRREQ;
    YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, ReadData);
        
    /* PMRREQ Polling */
    PM_GD25Q127C_Wait_Clear_Flag(PM_GD25Q127C_READACS_PMRREQ);

    /* Read P#3: MD[7:0] */
    ReadData  = YGV642_Read_Port(YVC1_PORT_PTNMEM_DATA);
	}while (ReadData &0x03);
  
	/*--------------------------------------*/
	/* Quad I/O Fast Read                   */
	/*--------------------------------------*/
  PM_GD25Q127C_Write_Cmd(PM_GD25Q127C_CMD_Q_IO_READ);
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_GD25Q127C_Set_Pattern_Memory_Addr                      */
/* Contents         : Set Pattern Memory Address register                       */
/* Argument         : uint32_t   Addr     (W) Pattern Memory Address            */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/  
void PM_GD25Q127C_Set_Pattern_Memory_Addr(uint32_t Addr)
{
	uint8_t MemAddr[4];
	
	/*---	Pattern memory address set	---*/
	MemAddr[0] = (uint8_t)(( Addr >> SHIFT_24BIT ) & MASK_03H);
	MemAddr[1] = (uint8_t)(( Addr >> SHIFT_16BIT ) & MASK_FFH);
	MemAddr[2] = (uint8_t)(( Addr >> SHIFT_08BIT ) & MASK_FFH);
	MemAddr[3] = (uint8_t)(( Addr                ) & MASK_FFH);

  YGV642_Write_Regs(REG_R2CH, MemAddr, 4);
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_GD25Q127C_Write_Cmd                                    */
/* Contents         : Write Pattern Memory Control registers                    */
/* Argument         : uint8_t   CmdNo    (W) Command No                         */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/
void PM_GD25Q127C_Write_Cmd(uint8_t CmdNo)
{
	/*	Access Port	
	 *
	 *	 No.   Description 		 						  bit7   bit6   bit5   bit4   bit3   bit2   bit1   bit0 
	 *	+---+-------------------------------------------+------+------+------+------+------+------+------+------+
	 *	| 2 | Pattern Memory Read Access Port / Status 	|PMRREQ|PWBUSY|INIEND|CSNEG | "0"  | "0"  | "0"  | "0"  |
	 *	+---+-------------------------------------------+------+------+------+------+------+------+------+------+
	 *	| 3 | Pattern Memory Data Port 				        	|                        MD[7:0]                        |
	 *	+---+-------------------------------------------+------+------+------+------+------+------+------+------+
	 */

	/* Waiting for ready to write */
	PM_GD25Q127C_Wait_Clear_Flag(PM_GD25Q127C_READACS_PMRREQ | PM_GD25Q127C_READACS_PWBUSY);

	/* Set command to R#55h-#5Bh */
	YGV642_Write_Regs(0x55, PMGD25Q127CCmdData[CmdNo], 7);
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_GD25Q127C_Wait_Clear_Flag                              */
/* Contents         : Wait to clear flags                                       */
/* Argument         : uint8_t   Mask    (W) Mask value                          */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/ 
void PM_GD25Q127C_Wait_Clear_Flag(uint8_t Mask)
{
	uint8_t ReadData;

	do
	{
		ReadData = YGV642_Read_Port(YVC1_PORT_PTNMEM_READACS);
	} while ((ReadData & Mask) != 0);
}

