

#ifndef _NON_VOLATILE_MEMORYS_H_
#define _NON_VOLATILE_MEMORYS_H_

#include "Const.h"
#include "EEPROM_24Cxx_Remap.h"

/******************************************************************************
非易失性数据存储器配置
******************************************************************************/
#define   NVM_USE_EEPROM                    1           //0：断开EEPROM,强制使用数据默认值
//1：使用EEPROM保存非易失性数据

extern uint8_t NVMEEPROMFault;
#define   NVM_EEPROM_FAULT                  NVMEEPROMFault

/******************************************************************************
初始化标识

数据组名称： NVM_INIT_DATA
  信号成员： -- NVM_INIT_FLAG
******************************************************************************/
/*** 结构声明 ***/
typedef union
{
    struct
    {
        uint8_t Flag[4];
    } Sig;

    uint8_t Data[4];
} NVMInitUnion;

/*** 信号定义 ***/
extern NVMInitUnion                         NVMInit;

#define   NVM_INIT_FLAG                     NVMInit.Sig.Flag

/*** 信号默认值 ***/
static const NVMInitUnion NVMDefaultInit =
{ 
 
 #if JEELYVERSION ==0
          /* Flag[]                       = */    "BAC",
 #endif
 
 #if JEELYVERSION ==1
          /* Flag[]                       = */    "ACC",
 #endif
 
 #if JEELYVERSION ==2
         /* Flag[]                       = */    "ELE",
 #endif 
 
 
 #if JEELYVERSION ==3
         /* Flag[]                       = */    "OIL",
 #endif
 
 #if JEELYVERSION ==4
         /* Flag[]                       = */    "ELO",
 #endif
 
 #if JEELYVERSION ==5
         /* Flag[]                       = */    "DEN",
 #endif
 
};

/******************************************************************************
里程总计

数据组名称： NVM_ODO_DATA
  信号成员： -- NVM_ODO_VALUE
******************************************************************************/
/*** 结构声明 ***/
typedef union
{
    struct
    {
        uint32_t  Value;
        uint8_t   ClrCnt;
        uint8_t   Rsvd;
        uint16_t  Cleared;
    } Sig;

    uint8_t Data[8];
} NVMODOUnion;

/*** 信号定义 ***/
extern NVMODOUnion                          NVMODO;

#define   NVM_ODO_VALUE                     NVMODO.Sig.Value
#define   NVM_ODO_CLEAR_COUNT               NVMODO.Sig.ClrCnt
#define   NVM_ODO_CLEARED                   NVMODO.Sig.Cleared

/*** 信号默认值 ***/
static const NVMODOUnion NVMDefaultODO =
{
    /* Value                        = */      0,
    /* ClrCnt                       = */      3,
    /* Rsvd                         = */      0,
    /* Cleared                      = */      0,
};

/******************************************************************************
用户设置

数据组名称： NVM_USER_SETTINGS
  信号成员： |- NVM_USER_SETTINGS_LANGUAGE               //语言
             |- NVM_USER_SETTINGS_THEME                  //主题
             |- NVM_USER_SETTINGS_ALARM_VOLUME           //报警声音调节
             |- NVM_USER_SETTINGS_BACK_LIGHT_LEVEL       //背光灯亮度调节
             |- NVM_USER_SETTINGS_TIME_FORMAT            //时间格式
             |- NVM_USER_SETTINGS_RELAX_REMIND           //疲劳驾驶提醒
             |- NVM_USER_SETTINGS_FOLLOW_ME_HOME         //伴我回家
             |- NVM_USER_SETTINGS_AUTO_LOCK              //自动落锁
             |- NVM_USER_SETTINGS_OVER_SPEED_ALARM       //超速报警状态
             |- NVM_USER_SETTINGS_SPEED_LIMIT            //超速报警值
             |- NVM_USER_SETTINGS_SERVICE                //保养服务
             |- NVM_USER_SETTINGS_AVG_FUEL_CONS_EN       //平均油耗
             |- NVM_USER_SETTINGS_INST_FUEL_CONS_EN      //瞬时油耗
             |- NVM_USER_SETTINGS_DTE_EN                 //可续航里程
             |- NVM_USER_SETTINGS_DRV_TIME_EN            //行驶时间
             |- NVM_USER_SETTINGS_AVG_SPEED_EN           //平均车速
             |- NVM_USER_SETTINGS_TIRE_STAT_EN           //轮胎状态
             |- NVM_USER_SETTINGS_TEMP_UNIT              //温度单位
             |- NVM_USER_SETTINGS_PRESSURE_UNIT          //压力单位
             |- NVM_USER_SETTINGS_FUEL_CONS_UNIT         //油耗单位
             |- NVM_USER_SETTINGS_DIST_UNIT              //距离单位
             -- NVM_USER_SETTINGS_FUNCTION               //版本功能
******************************************************************************/
/*** 结构声明 ***/
typedef union
{
    struct
    {
        uint8_t   Language            :  5;
        uint8_t   Theme               :  3;
        
        uint8_t   AlarmVolume         :  7;
        uint8_t   Time                :  1;
        
        uint8_t   BackLightLevel      :  7;
        uint8_t   TimeFormat          :  1;
        
        uint8_t   RelaxRemind         :  4;
        uint8_t   FollowMeHome        :  4;
        
        uint8_t   AutoLock            :  4;
        uint8_t   OverSpeedAlarm      :  4;
        uint8_t   SpeedLimit          :  8;
        uint32_t  Service             : 32;
        
        uint8_t   AvgFuelConsEn       :  1;
        uint8_t   InstFuelConsEn      :  1;
        uint8_t   DTEEn               :  1;
        uint8_t   DrvTimeEn           :  1;
        uint8_t   AvgSpeedEn          :  1;
        uint8_t   TireStatEn          :  1;
        uint8_t   TripEn              :  1;
        uint8_t   CurrentSpeedEn      :  1;
        
        uint8_t   TempUnit            :  2;
        uint8_t   PressureUnit        :  3;
        uint8_t   FuelConsUnit        :  3;
        
        uint8_t   DistUnit            :  2;
        uint8_t   SocCourse           :  2;
        uint8_t   Function            :  4;
        uint8_t   FlashChk            :  2;
        uint8_t                       :  2;
        uint8_t                       :  2;
        uint8_t                       :  2;
    } Sig;

    uint8_t Data[14];
} NVMUserSettingsUnion;

/*** 信号定义 ***/

extern NVMUserSettingsUnion                       NVMUserSettings;

#define   NVM_USER_SETTINGS_LANGUAGE              NVMUserSettings.Sig.Language
//#define   NVM_USER_SETTINGS_THEME                 NVMUserSettings.Sig.Theme
#define   NVM_USER_SETTINGS_THEME                 0
#define   NVM_USER_SETTINGS_ALARM_VOLUME          NVMUserSettings.Sig.AlarmVolume
#define   NVM_USER_SETTINGS_TIME                  NVMUserSettings.Sig.Time        //0 收动设置  1 自动GPS同步
#define   NVM_USER_SETTINGS_BACK_LIGHT_LEVEL      NVMUserSettings.Sig.BackLightLevel
#define   NVM_USER_SETTINGS_TIME_FORMAT           NVMUserSettings.Sig.TimeFormat
#define   NVM_USER_SETTINGS_RELAX_REMIND          NVMUserSettings.Sig.RelaxRemind
#define   NVM_USER_SETTINGS_FOLLOW_ME_HOME        NVMUserSettings.Sig.FollowMeHome
#define   NVM_USER_SETTINGS_AUTO_LOCK             NVMUserSettings.Sig.AutoLock
#define   NVM_USER_SETTINGS_OVER_SPEED_ALARM      NVMUserSettings.Sig.OverSpeedAlarm
#define   NVM_USER_SETTINGS_SPEED_LIMIT           NVMUserSettings.Sig.SpeedLimit
#define   NVM_USER_SETTINGS_SERVICE               NVMUserSettings.Sig.Service
#define   NVM_USER_SETTINGS_AVG_FUEL_CONS_EN      NVMUserSettings.Sig.AvgFuelConsEn
#define   NVM_USER_SETTINGS_INST_FUEL_CONS_EN     NVMUserSettings.Sig.InstFuelConsEn
#define   NVM_USER_SETTINGS_DTE_EN                NVMUserSettings.Sig.DTEEn
#define   NVM_USER_SETTINGS_DRV_TIME_EN           NVMUserSettings.Sig.DrvTimeEn
#define   NVM_USER_SETTINGS_AVG_SPEED_EN          NVMUserSettings.Sig.AvgSpeedEn
#define   NVM_USER_SETTINGS_TIRE_STAT_EN          NVMUserSettings.Sig.TireStatEn
#define   NVM_USER_SETTINGS_TRIP_EN               NVMUserSettings.Sig.TripEn
#define   NVM_USER_SETTINGS_CURRENT_SPEED_EN      NVMUserSettings.Sig.CurrentSpeedEn
#define   NVM_USER_SETTINGS_TEMP_UNIT             NVMUserSettings.Sig.TempUnit
#define   NVM_USER_SETTINGS_PRESSURE_UNIT         NVMUserSettings.Sig.PressureUnit
#define   NVM_USER_SETTINGS_FUEL_CONS_UNIT        NVMUserSettings.Sig.FuelConsUnit
#define   NVM_USER_SETTINGS_DIST_UNIT             NVMUserSettings.Sig.DistUnit
#define   NVM_USER_SETTINGS_SOCCOURSE             NVMUserSettings.Sig.SocCourse
//#define   NVM_USER_SETTINGS_FUNCTION              NVMUserSettings.Sig.Function
#define   NVM_USER_SETTINGS_FLASH_CHK_CNT         NVMUserSettings.Sig.FlashChk

/*** 信号值定义 ***/

//语言,缩写参考ISO 639-1
#define   NVM_LANGUAGE_ZH                   0x00  //中文
#define   NVM_LANGUAGE_EN                   0x01  //英语
#define   NVM_LANGUAGE_RU                   0x02  //俄语
#define   NVM_LANGUAGE_ES                   0x03  //西班牙语
#define   NVM_LANGUAGE_PT                   0x04  //葡萄牙语 
#define   NVM_LANGUAGE_AR                   0x05  //阿拉伯语

//主题
#define   NVM_THEME_1                       0x00  //主题1 传统
#define   NVM_THEME_2                       0x01  //主题2 科技
#define   NVM_THEME_3                       0x02  //主题3 运动

//报警音量
#define   NVM_ALARM_VOLUME_HI               0x00  //音量高
#define   NVM_ALARM_VOLUME_MID              0x01  //音量中 
#define   NVM_ALARM_VOLUME_LO               0x02  //音量低

//时间格式
#define   NVM_TIME_FORMAT_24H               0x00  //24小时制
#define   NVM_TIME_FORMAT_12H               0x01  //12小时制

//疲劳驾驶提醒
#define   NVM_RELAX_REMIND_3H               0x00  //3小时
#define   NVM_RELAX_REMIND_4H               0x01  //4小时
#define   NVM_RELAX_REMIND_OFF              0x02  //关

//超速报警
#define   NVM_OVER_SPEED_ALARM_OFF          0x00  //关闭
#define   NVM_OVER_SPEED_ALARM_DEFAULT      0x01  //默认
#define   NVM_OVER_SPEED_ALARM_USER_DEF     0x02  //自定义

//功能开关
#define   NVM_FUNC_DISABLE                  0x00  //禁止功能
#define   NVM_FUNC_ENABLE                   0x01  //使能功能

//温度单位
#define   NVM_TEMP_UNIT_C                   0x00
#define   NVM_TEMP_UNIT_F                   0x01

//压力单位
#define   NVM_PRESSURE_UNIT_KPA             0x00
#define   NVM_PRESSURE_UNIT_BAR             0x01
#define   NVM_PRESSURE_UNIT_PSI             0x02

//油耗单位
#define   NVM_FUEL_CONS_UNIT_L_100KM        0x00
#define   NVM_FUEL_CONS_UNIT_KM_L           0x01
#define   NVM_FUEL_CONS_UNIT_MPG_US         0x02
#define   NVM_FUEL_CONS_UNIT_MPG_UK         0x03

//距离单位
#define   NVM_DIST_UNIT_KM                  0x00
#define   NVM_DIST_UNIT_MILE                0x01

//SOC续航里程
#define   NVM_SOC_COURSE_TWOZEROZERO        0x00
#define   NVM_SOC_COURSE_ONEEIGHTZERO       0x01

/*** 信号默认值 ***/

static const NVMUserSettingsUnion NVMDefaultUserSettings =
{
    /* Language                     = */      NVM_LANGUAGE_ZH,
    /* Theme                        = */      NVM_THEME_1,
    /* AlarmVolume                  = */      NVM_ALARM_VOLUME_HI,
    /* Time/GPS                     = */      1,
    /* BackLightLevel               = */      0,
    /* TimeFormat                   = */      NVM_TIME_FORMAT_24H,
    /* RelaxRemind                  = */      NVM_RELAX_REMIND_4H,
    /* FollowMeHome                 = */      0,
    /* AutoLock                     = */      0,
    /* OverSpeedAlarm               = */      NVM_OVER_SPEED_ALARM_DEFAULT,
    /* SpeedLimit                   = */      100,           //出厂限速设置100
    /* Service                      = */      30000,
    /* AvgFuelConsEn                = */      NVM_FUNC_ENABLE,
    /* InstFuelConsEn               = */      NVM_FUNC_ENABLE,
    /* DTEEn                        = */      NVM_FUNC_ENABLE,
    /* DrvTimeEn                    = */      NVM_FUNC_ENABLE,
    /* AvgSpeedEn                   = */      NVM_FUNC_ENABLE,
    /* TireStatEn                   = */      NVM_FUNC_ENABLE,
    /* TripEn                       = */      NVM_FUNC_ENABLE,
    /* CurrentSpeedEn               = */      NVM_FUNC_ENABLE,
    /* TempUnit                     = */      NVM_TEMP_UNIT_C,
    /* PressureUnit                 = */      NVM_PRESSURE_UNIT_KPA,
    /* FuelConsUnit                 = */      NVM_FUEL_CONS_UNIT_L_100KM,
    /* DistUnit                     = */      NVM_DIST_UNIT_KM,
    /* SocCourse                    = */      0,
    /* Function                     = */      0,
    /* FlashChk                     = */      3,
};

/******************************************************************************
下线配置 F110

数据组名称： NVM_CONFIG_F110
  信号成员： |- NVM_CONFIG_EMS
             |- NVM_CONFIG_IPK
             |- NVM_CONFIG_BCM
             |- NVM_CONFIG_TCU
             |- NVM_CONFIG_ACU
             |- NVM_CONFIG_ESC
             |- NVM_CONFIG_PEPS
             |- NVM_CONFIG_EPS
             |- NVM_CONFIG_SAS
             |- NVM_CONFIG_FRS
             |- NVM_CONFIG_AC
             |- NVM_CONFIG_MMI
             |- NVM_CONFIG_PAS
             |- NVM_CONFIG_TPMS
             |- NVM_CONFIG_T_BOX
             |- NVM_CONFIG_ESCL
             |- NVM_CONFIG_EPB
             |- NVM_CONFIG_FCS
             |- NVM_CONFIG_PTG
             |- NVM_CONFIG_DSCU
             |- NVM_CONFIG_AFS
             |- NVM_CONFIG_ABS
             |- NVM_CONFIG_IMMO
             |- NVM_CONFIG_4WD
             |- NVM_CONFIG_PTC
             |- NVM_CONFIG_EGSM
             |- NVM_CONFIG_BSG
             |- NVM_CONFIG_BMS
             |- NVM_CONFIG_DCDC
             |- NVM_CONFIG_VCU
             |- NVM_CONFIG_OBC
             |- NVM_CONFIG_IPU
             |- NVM_CONFIG_ACCM
             |- NVM_CONFIG_DVR
             |- NVM_CONFIG_HVSM
             -- NVM_CONFIG_DMS
******************************************************************************/
/*** 结构声明 ***/
typedef union
{
    struct
    {
        uint8_t   EMS                 :  1;
        uint8_t   IPK                 :  1;
        uint8_t   BCM                 :  1;
        uint8_t   TCU                 :  1;
        uint8_t   ACU                 :  1;
        uint8_t   ESC                 :  1;
        uint8_t   PEPS                :  1;
        uint8_t   EPS                 :  1;

        uint8_t   SAS                 :  1;
        uint8_t   FRS                 :  1;
        uint8_t   AC                  :  1;
        uint8_t   MMI                 :  1;
        uint8_t   PAS                 :  1;
        uint8_t   TPMS                :  1;
        uint8_t   T_BOX               :  1;
        uint8_t   ESCL                :  1;

        uint8_t   EPB                 :  1;
        uint8_t   FCS                 :  1;
        uint8_t   PTG                 :  1;
        uint8_t   DSCU                :  1;
        uint8_t   AFS                 :  1;
        uint8_t   ABS                 :  1;
        uint8_t   IMMO                :  1;
        uint8_t   _4WD                :  1;

        uint8_t   PTC                 :  1;
        uint8_t   EGSM                :  1;
        uint8_t   BSG                 :  1;
        uint8_t   BMS                 :  1;
        uint8_t   DCDC                :  1;
        uint8_t   VCU                 :  1;
        uint8_t   OBC                 :  1;
        uint8_t   IPU                 :  1;

        uint8_t   ACCM                :  1;
        uint8_t                       :  2;
        uint8_t   DVR                 :  1;
        uint8_t   HVSM                :  1;
        uint8_t   DMS                 :  1;
        uint8_t                       :  2;

    } Sig;

    uint8_t Data[5];
} NVMConfigF110Union;

/*** 信号定义 ***/
extern NVMConfigF110Union                         NVMConfigF110;
/*
#define   NVM_CONFIG_EMS                          NVMConfigF110.Sig.EMS
#define   NVM_CONFIG_IPK                          NVMConfigF110.Sig.IPK
#define   NVM_CONFIG_BCM                          NVMConfigF110.Sig.BCM
#define   NVM_CONFIG_TCU                          NVMConfigF110.Sig.TCU
#define   NVM_CONFIG_ACU                          NVMConfigF110.Sig.ACU
#define   NVM_CONFIG_ESC                          NVMConfigF110.Sig.ESC
#define   NVM_CONFIG_PEPS                         NVMConfigF110.Sig.PEPS
#define   NVM_CONFIG_EPS                          NVMConfigF110.Sig.EPS
#define   NVM_CONFIG_SAS                          NVMConfigF110.Sig.SAS
#define   NVM_CONFIG_FRS                          NVMConfigF110.Sig.FRS
#define   NVM_CONFIG_AC                           NVMConfigF110.Sig.AC
#define   NVM_CONFIG_MMI                          NVMConfigF110.Sig.MMI
#define   NVM_CONFIG_PAS                          NVMConfigF110.Sig.PAS
#define   NVM_CONFIG_TPMS1                         NVMConfigF110.Sig.TPMS
#define   NVM_CONFIG_T_BOX                        NVMConfigF110.Sig.T_BOX
#define   NVM_CONFIG_ESCL                         NVMConfigF110.Sig.ESCL
#define   NVM_CONFIG_EPB                          NVMConfigF110.Sig.EPB
#define   NVM_CONFIG_FCS                          NVMConfigF110.Sig.FCS
#define   NVM_CONFIG_PTG                          NVMConfigF110.Sig.PTG
#define   NVM_CONFIG_DSCU                         NVMConfigF110.Sig.DSCU
#define   NVM_CONFIG_AFS                          NVMConfigF110.Sig.AFS
#define   NVM_CONFIG_ABS                          NVMConfigF110.Sig.ABS
#define   NVM_CONFIG_IMMO                         NVMConfigF110.Sig.IMMO
#define   NVM_CONFIG_4WD                          NVMConfigF110.Sig._4WD
#define   NVM_CONFIG_PTC                          NVMConfigF110.Sig.PTC
#define   NVM_CONFIG_EGSM                         NVMConfigF110.Sig.EGSM
#define   NVM_CONFIG_BSG                          NVMConfigF110.Sig.BSG
#define   NVM_CONFIG_BMS                          NVMConfigF110.Sig.BMS
#define   NVM_CONFIG_DCDC                         NVMConfigF110.Sig.DCDC
#define   NVM_CONFIG_VCU                          NVMConfigF110.Sig.VCU
#define   NVM_CONFIG_OBC                          NVMConfigF110.Sig.OBC
#define   NVM_CONFIG_IPU                          NVMConfigF110.Sig.IPU
#define   NVM_CONFIG_ACCM                         NVMConfigF110.Sig.ACCM
#define   NVM_CONFIG_DVR                          NVMConfigF110.Sig.DVR
#define   NVM_CONFIG_HVSM                         NVMConfigF110.Sig.HVSM
#define   NVM_CONFIG_DMS                          NVMConfigF110.Sig.DMS
*/
/*** 信号值定义 ***/

//功能开关
//#define NVM_FUNC_DISABLE                  0x00  //禁止功能
//#define NVM_FUNC_ENABLE                   0x01  //使能功能

/*** 信号默认值 ***/
static const NVMConfigF110Union NVMDefaultConfigF110 =
{
    /* EMS                          = */      NVM_FUNC_ENABLE,
    /* IPK                          = */      NVM_FUNC_ENABLE,
    /* BCM                          = */      NVM_FUNC_ENABLE,
    /* TCU                          = */      NVM_FUNC_ENABLE,
    /* ACU                          = */      NVM_FUNC_ENABLE,
    /* ESC                          = */      NVM_FUNC_ENABLE,
    /* PEPS                         = */      NVM_FUNC_ENABLE,
    /* EPS                          = */      NVM_FUNC_ENABLE,

    /* SAS                          = */      NVM_FUNC_ENABLE,
    /* FRS                          = */      NVM_FUNC_ENABLE,
    /* AC                           = */      NVM_FUNC_ENABLE,
    /* MMI                          = */      NVM_FUNC_ENABLE,
    /* PAS                          = */      NVM_FUNC_ENABLE,
    /* TPMS                         = */      NVM_FUNC_ENABLE,
    /* T_BOX                        = */      NVM_FUNC_ENABLE,
    /* ESCL                         = */      NVM_FUNC_ENABLE,

    /* EPB                          = */      NVM_FUNC_DISABLE,
    /* FCS                          = */      NVM_FUNC_ENABLE,
    /* PTG                          = */      NVM_FUNC_ENABLE,
    /* DSCU                         = */      NVM_FUNC_ENABLE,
    /* AFS                          = */      NVM_FUNC_ENABLE,
    /* ABS                          = */      NVM_FUNC_ENABLE,
    /* IMMO                         = */      NVM_FUNC_ENABLE,
    /* _4WD                         = */      NVM_FUNC_ENABLE,

    /* PTC                          = */      NVM_FUNC_ENABLE,
    /* EGSM                         = */      NVM_FUNC_ENABLE,
    /* BSG                          = */      NVM_FUNC_ENABLE,
    /* BMS                          = */      NVM_FUNC_ENABLE,
    /* DCDC                         = */      NVM_FUNC_ENABLE,
    /* VCU                          = */      NVM_FUNC_ENABLE,
    /* OBC                          = */      NVM_FUNC_ENABLE,
    /* IPU                          = */      NVM_FUNC_ENABLE,

    /* ACCM                         = */      NVM_FUNC_ENABLE,
    /* DVR                          = */      NVM_FUNC_ENABLE,
    /* HVSM                         = */      NVM_FUNC_ENABLE,
    /* DMS                          = */      NVM_FUNC_ENABLE,
};
/******************************************************************************
下线配置 F101     //NVM_FUNC_DISABLE

数据组名称： NVM_CONFIG_F101
  信号成员： |- NVM_CONFIG_FUNCTION
  信号成员： |- NVM_CONFIG_ACTIVE

******************************************************************************/
/*** 结构声明 ***/
typedef union
{
    struct
    {

        uint8_t     Function  :  6;    // 00甲醇重卡(默认)    01acc重卡    02混合动力    03柴油重卡
                                       // 04 柴油混动
        uint8_t     Tpms      :  2;    // 00 无胎压  01 有胎压

        uint8_t     Active    :  8;     //仪表是否失效 0失效 1有效

        uint8_t     SpeedRate :  8;     //速比  0x00 4.111 0x01  3.7  0x02 6.73   x03 2.846
        uint8_t               :  8;
        uint8_t               :  8;
        uint8_t               :  8;

    } Sig;

    uint8_t Data[6];
} NVMConfigF101Union;

/*** 信号定义 ***/

extern NVMConfigF101Union                         NVMConfigF101;
 /*
#define   NVM_CONFIG_FUNCTION         NVMConfigF101.Sig.Function
#define   NVM_CONFIG_TPMS             NVMConfigF101.Sig.Tpms
#define   NVM_CONFIG_ACTIVE           NVMConfigF101.Sig.Active
#define   NVM_CONFIG_SPEEDRATE        NVMConfigF101.Sig.SpeedRate       */


/*** 信号值定义 ***/



//续航里程计算
#define NVM_TYPE_BASE                      0x00  //甲醇重卡 基础版
#define NVM_TYPE_ACC                       0x01  //ACC 版重卡
#define NVM_TYPE_POW                       0x02  //混合动力
#define NVM_TYPE_DIESEL                    0x03  //柴油重卡 
 
             

       
  

/*** 信号默认值 ***/
static const NVMConfigF101Union NVMDefaultConfigF101 =
{
    JEELYVERSION,DEFAULT_TPMS,1, DEFAUTL_RATE,    //默认值 3.77
};

/******************************************************************************
下线配置 3300     //NVM_FUNC_DISABLE

数据组名称： NVM_CONFIG_3300
******************************************************************************/
/*** 结构声明 ***/
typedef union
{
    struct
    {

        uint8_t     S1        :  1;     // 00 :12V   01:24V  (默认)
        uint8_t               :  7;     //0 

        uint8_t     S2        :  3;     //潍柴发动机 0失效 1：WP2.3NQ130E61。 2：WP3NQ160E61。3： WP4.1NQ190E61   4: WP4.6NQ220E61
        uint8_t               :  5;     //1
        
        uint8_t     S3        :  2;     //康明斯发动机 0失效 1：ISF2.8S6154T 2：ISF3.8S6190
        uint8_t               :  6;     //2
        
        uint8_t     S4        :  2;     //玉柴发动机 0失效 1：YCS04180-60 2：YCY30165-60
        uint8_t               :  6;     //3

        uint8_t     S5        :  2;     //大柴发动机 0失效 1：CA4DD2-18E6  2�CCA4DH1-18E6
        uint8_t               :  6;     //4
        
        uint8_t     S25       :  2;     //锡柴发动机 0失效 1：锡柴发动机  
        uint8_t               :  6;     //5
        
        uint8_t     S26       :  2;     //云内发动机 0失效 1：云内发动机  
        uint8_t               :  6;     //6

        uint8_t     S6        :  3;     //万里扬变速箱 0失效 1：6G40（MT）2:变速器总成6G55（MT）3:变速器总成6G70（MT）4:变速器总成9TS90（MT）5:变速器总成6AG40（AMT）
        uint8_t               :  5;     //7

        uint8_t     S7        :  2;     // 星瑞变速箱   0:未使用1:变速器总成ML640（MT）2:变速器总成LC6T450B（MT）3:变速器总成ML650A（MT）
        uint8_t               :  6;     //8
        
        uint8_t     S8        :  3;     //法士特变速箱 0:未使用1:变速器总成C6J55T（MT）2:变速器总成C6J76T（MT）3:变速器总成875S（MT）4:变速器总成（AMT）
        uint8_t               :  5;     //9 

        uint8_t     S9        :  1;     //变速箱类型 0:MT  1:AMT
        uint8_t               :  7;     //10 
                
        uint8_t     S10       :  2;     //油箱类型    1:200L塑料 2:200L合金  3:120L
        uint8_t               :  6;     //11 

        uint8_t     S11       :  1;     //制动类型
        uint8_t               :  7;     //12 

        uint8_t     S13       :  1;     //挡位传感器                
        uint8_t     S14       :  1;     //驻车类型
        uint8_t               :  6;     //13
                                                
        uint8_t     S15       :  1;       //  ASR
        uint8_t     S17       :  1;       //  ESC
        uint8_t     S18       :  1;       //  PTO
        uint8_t     S19       :  1;       //  EBS
        uint8_t     S20       :  1;       //  AEB
        uint8_t     S21       :  1;       //  EPB
        uint8_t     S22       :  1;       //  TPMS
        uint8_t               :  1;       //  

        uint8_t               :  1;       //  
        uint8_t               :  1;       // 15 
        uint8_t               :  6;       
               
    } Sig;

    uint8_t Data[16];
} NVMConfig3300Union;

/*** 信号定义 ***/

extern NVMConfig3300Union                         NVMConfig3300;

#define   NVM_CONFIG_POWER           NVMConfig3300.Sig.S1      // 00 :12V   01:24V  (默认)
#define   NVM_CONFIG_WEI             NVMConfig3300.Sig.S2      //潍柴发动机 0失效 1：WP2.3NQ130E61。 2：WP3NQ160E61。3： WP4.1NQ190E61 4:发动机总成潍柴WP4.6NQ220E61
#define   NVM_CONFIG_KANG            NVMConfig3300.Sig.S3      //康明斯发动机 0失效 1：F2.8NS6B156 2：F3.8NS6B190
#define   NVM_CONFIG_YU              NVMConfig3300.Sig.S4      //玉柴发动机 0失效 1：YCS04180-60 2：YCY30165-60

#define   NVM_CONFIG_DA              NVMConfig3300.Sig.S5     //大柴发动机 0失效 1：发动机总成CA4DK1-20E6  2�C发动机总成CA4DK1-22E6  3:发动机总成CA4DD2-18E6
#define   NVM_CONFIG_WANLIYANG       NVMConfig3300.Sig.S6     //万里扬变速箱 
#define   NVM_CONFIG_XINGRUI         NVMConfig3300.Sig.S7     // 星瑞变速箱
#define   NVM_CONFIG_FASHITE         NVMConfig3300.Sig.S8

#define   NVM_CONFIG_TBOX            NVMConfig3300.Sig.S9          //手动自动，线下配置完成
#define   NVM_CONFIG_FUELBOX         NVMConfig3300.Sig.S10         //油箱，线下配置完成
#define   NVM_CONFIG_BREAK           NVMConfig3300.Sig.S11

#define   NVM_CONFIG_GEAR            NVMConfig3300.Sig.S13
#define   NVM_CONFIG_PARK            NVMConfig3300.Sig.S14
#define   NVM_CONFIG_ASR             NVMConfig3300.Sig.S15         //线下配置完成
#define   NVM_CONFIG_ESC             NVMConfig3300.Sig.S17         //线下配置完成

#define   NVM_CONFIG_PTO             NVMConfig3300.Sig.S18          //线下配置完成
#define   NVM_CONFIG_EBS             NVMConfig3300.Sig.S19          //线下配置完成
#define   NVM_CONFIG_AEB             NVMConfig3300.Sig.S20          //线下配置完成
#define   NVM_CONFIG_EPB             NVMConfig3300.Sig.S21          //线下配置完成

//#define   NVM_CONFIG_EPS             NVMConfig3300.Sig.S22          //线下配置完成
#define   NVM_CONFIG_TPMS            NVMConfig3300.Sig.S22          //线下配置完成
//#define   NVM_CONFIG_PEPS            NVMConfig3300.Sig.S24          //线下配置完成
//#define   NVM_CONFIG_ADAS            NVMConfig3300.Sig.S24          //线下配置完成

#define   NVM_CONFIG_XI              NVMConfig3300.Sig.S25     //  锡柴发动机 
#define   NVM_CONFIG_YUN             NVMConfig3300.Sig.S26     // 云内发动机

/*** 信号值定义 ***/

#define NVM_TYPE_BASE                      0x00  //甲醇重卡 基础版
#define NVM_TYPE_ACC                       0x01  //ACC 版重卡
#define NVM_TYPE_POW                       0x02  //混合动力
#define NVM_TYPE_DIESEL                    0x03  //柴油重卡 

/*** 信号默认值 ***/            
static const NVMConfig3300Union NVMDefault3300 =
{
    0x00, // 00 :12V   01:24V  (默认)
    0x00, //潍柴发动机 0失效 1：WP2.3NQ130E61。 2：WP3NQ160E61。3： WP4.1NQ190E61
    0x01, //康明斯发动机 0失效 1：F2.8NS6B156 2：F3.8NS6B190
    0x00, //玉柴发动机 0失效 1：YCS04180-60 2：YCY30165-60
    0x00, //大柴发动机 0失效 1：CA4DD2-18E6  2�CCA4DH1-18E6
    0x00, //  锡柴发动机 
    0x00, // 云内发动机
    0x00,  //万里扬变速箱 
    0x00,  // 星瑞变速箱
    0x00,  //变速箱类型 0:MT  1:AMT
    0x00,  //油箱类型    1:200L塑料 2:200L合金  3:120L
    0x01,  //制动类型
    0x00,  //挡位传感器  
    0x00,  //驻车类型
    0x20,
    0x00,  
}; 

/******************************************************************************
故障码存储
******************************************************************************/
/*** 存储长度 ***/
#define   NVM_DTC_LEN                       7

/*** 结构声明 ***/


typedef union
{
    uint8_t Data[NVM_DTC_LEN];
} NVMDTCUnion;

/*** 信号定义 ***/

extern NVMDTCUnion                                NVMDTC;

#define   NVM_DTC_STATUS                          NVMDTC.Data

/*** 信号默认值 ***/ 

static const NVMDTCUnion NVMDefaultDTC =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 
};


/******************************************************************************

     F199

******************************************************************************/

typedef union
{
    struct
    {

        uint8_t     BCDYearH  :  8;     //年0x20
        uint8_t     BCDYearL  :  8;     //年0x19
        uint8_t     BCDMonth  :  8;     //月0x01
        uint8_t     DevNum1   :  8;     //
        uint8_t     DevNum2   :  8;     //
        uint8_t     DevNum3   :  8;     //
        uint8_t     DevNum4   :  8;     //
    } Sig;

    uint8_t Data[7];
} NVMConfigF184Union;

extern NVMConfigF184Union                         NVMConfigF184;

#define   NVM_F184_BCDYEARH         NVMConfigF184.Sig.BCDYearH
#define   NVM_F184_BCDYEARL         NVMConfigF184.Sig.BCDYearL
#define   NVM_F184_BCDMONTH         NVMConfigF184.Sig.BCDMonth
#define   NVM_F184_BCDDAY           NVMConfigF184.Sig.BCDDay

static const NVMConfigF184Union NVMDefaultF184 =
{
    0x00, 0x01, 0x01, 0x97, 0x97,0x97,0x97,
};


/******************************************************************************

  F1A2

******************************************************************************/

typedef union
{
    struct
    {
        uint8_t     S1        :  5;     //    1
        uint8_t     S2          :  3;     //    1
        
        uint8_t     BCDYearL  :  8;     //年0x19     2
        uint8_t     BCDMonth  :  8;     //月0x01     3
        uint8_t     BCDDay    :  8;     //日0x01     4
        uint8_t     S3          :  8;     //     5
        uint8_t     S4          :  8;     //     6
        uint8_t     S5          :  8;     //     7
        uint8_t     S6          :  8;     //     8
        uint8_t     S7        :  8;     //     9
        uint8_t     S8        :  8;     //     10
        uint8_t     S9        :  8;     //     11
        uint8_t     S10        :  8;     //     12
        uint8_t     S11      :  8;     //     13
        uint8_t     S12      :  8;     //     14
        uint8_t     S13       :  8;     //     15
        uint8_t     S14        :  8;     //     16
        uint8_t     S15       :  8;     //     17
        uint8_t     S19       :  8;     //     18
        uint8_t     S17       :  8;     //     19
        uint8_t     S18          :  8;     //     20
    } Sig;

    uint8_t Data[20];
} NVMConfigF1A8Union;

extern NVMConfigF1A8Union                         NVMConfigF1A8;

#define   NVM_F1A8_S1               NVMConfigF1A8.Sig.S1
#define   NVM_F1A8_BCDYEARL         NVMConfigF1A8.Sig.BCDYearL
#define   NVM_F1A8_BCDMONTH         NVMConfigF1A8.Sig.BCDMonth
#define   NVM_F1A8_BCDDAY           NVMConfigF1A8.Sig.BCDDay

static const NVMConfigF1A8Union NVMDefaultF1A8 =
{
   0x00, 0x00, 0x00, 0x01, 0x01, 0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
};

/******************************************************************************

  F212

******************************************************************************/

/*typedef union
{
    struct
    {

        uint8_t     VehicleName  :  8;     
        //0x00:EV_BUS_10.5M  0x01:EV_BUS_12M  0x02:EX_BUS_12M   0x10:EV_TRUCK_8T 0x11:EV_TRUCK_6T
        //  0x12:EX_TRUCK_8T 0x13:EX_TRUCK_6T  0x14:EX_TRUCK_6T_N   0x15:M_TRUCK_8T（甲醇重卡产业化）  others：reserve

    } Sig;

    uint8_t Data[1];
} NVMConfigF212Union;

extern NVMConfigF212Union                         NVMConfigF212;

#define   NVM_F212_VEHICLE_NAME         NVMConfigF212Union.Sig.VehicleName

static const NVMConfigF212Union NVMDefaultF212 =
{
    0x15, 
};
 */
/******************************************************************************

  F190

******************************************************************************/

typedef union
{
    struct
    {
      uint8_t    S0    :  8;
      uint8_t    S1    :  8; 
      uint8_t    S2    :  8; 
      uint8_t    S3    :  8; 
      uint8_t    S4    :  8; 
      uint8_t    S5    :  8; 
      uint8_t    S6    :  8; 
      uint8_t    S7    :  8; 
      uint8_t    S8    :  8; 
      uint8_t    S9    :  8; 
      uint8_t    S10    :  8; 
      uint8_t    S11    :  8; 
      uint8_t    S12    :  8; 
      uint8_t    S13    :  8; 
      uint8_t    S14    :  8; 
      uint8_t    S15    :  8; 
      uint8_t    S17    :  8; 
      
    } Sig;

    uint8_t Data[17];
} NVMConfigF190Union;

extern NVMConfigF190Union                         NVMConfigF190;

static const NVMConfigF190Union NVMDefaultF190 =
{
    0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
};
 /******************************************************************************

  NVM_RESET_ODO

******************************************************************************/

typedef union
{
    struct
    {
      uint8_t    S0    :  1;
      uint8_t            :  7;
    } Sig;

    uint8_t Data[1];
} NVMRESETODOUnion;

extern NVMRESETODOUnion                         NVMRESETODO;

static const NVMRESETODOUnion NVMDefaultRESETODO =
{
    0x00,
};

#define   NVM_RESET_ODO_FLAG        NVMRESETODO.Sig.S0
/******************************************************************************
存储数据属性表
******************************************************************************/
/*** 结构声明 ***/
typedef struct
{
    uint8_t   *pData;
    uint8_t   *pDefaultValue;
    uint8_t   Len;
    uint8_t   DataType;
    uint16_t  Addr;
} NVMDataAttributeTableStruct;

/*** 存储数据列表 ***/
#define   NVM_INIT_DATA                      0
#define   NVM_ODO_DATA                       1
//#define   NVM_USER_SETTINGS                  2
//#define   NVM_CONFIG_F110                    3
//#define   NVM_CONFIG_F101                    4
#define   NVM_DTC_DATA                       2
#define   NVM_CONFIG_F184                    3
#define   NVM_CONFIG_F1A8                    4
#define   NVM_CONFIG_F190                    5
//#define   NVM_CONFIG_F212                    9 
#define   NVM_CONFIG_3300                    6    
#define   NVM_RESET_ODO                    7              
/*** 存储数据属性表 ***/
static const NVMDataAttributeTableStruct NVMDataAttributeTable[] =
{
    /*===================================================================================================================================================================
                              |            RAM中           |                                    |                        |        非易失数据        | 非易失数据存储器
             数据名称         |        数据存储位置        |               默认值               |        数据长度        |         存储类型         |    中存储地址
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------*/
    {/* NVM_INIT_DATA        */                NVMInit.Data,                 NVMDefaultInit.Data,         sizeof ( NVMInit ),    EEPROM_HIREL_DATA_TYPE,           0x0000 },
    {/* NVM_ODO_DATA         */                 NVMODO.Data,                  NVMDefaultODO.Data,          sizeof ( NVMODO ),    EEPROM_HIREL_DATA_TYPE,           0x0001 },
    //{/* NVM_USER_SETTINGS    */        NVMUserSettings.Data,         NVMDefaultUserSettings.Data,  sizeof ( NVMUserSettings ),   EEPROM_NORMAL_DATA_TYPE,          0x0000 },
   // {/* NVM_CONFIG_F110      */          NVMConfigF110.Data,           NVMDefaultConfigF110.Data,   sizeof ( NVMConfigF110 ),   EEPROM_NORMAL_DATA_TYPE,           0x0010 },
   // {/* NVM_CONFIG_F101      */          NVMConfigF101.Data,           NVMDefaultConfigF101.Data,   sizeof ( NVMConfigF101 ),   EEPROM_NORMAL_DATA_TYPE,           0x0018 },
    {/* NVM_DTC_DATA         */                 NVMDTC.Data,                  NVMDefaultDTC.Data,          sizeof ( NVMDTC ),   EEPROM_NORMAL_DATA_TYPE,           0x0000 },
    {/* NVM_CONFIG_F184      */          NVMConfigF184.Data,                  NVMDefaultF184.Data,   sizeof ( NVMConfigF184 ),   EEPROM_NORMAL_DATA_TYPE,          0x0000+sizeof ( NVMDTC ) },
    {/* NVM_CONFIG_F1A8      */          NVMConfigF1A8.Data,                  NVMDefaultF1A8.Data,   sizeof ( NVMConfigF1A8 ),   EEPROM_NORMAL_DATA_TYPE,          0x0000+sizeof ( NVMDTC )+sizeof ( NVMConfigF184 ) },
    {/* NVM_CONFIG_F190      */          NVMConfigF190.Data,                  NVMDefaultF190.Data,   sizeof ( NVMConfigF190 ),   EEPROM_NORMAL_DATA_TYPE,          0x0000+sizeof ( NVMDTC )+sizeof ( NVMConfigF184 ) +sizeof ( NVMConfigF1A8 ) },
   // {/* NVM_CONFIG_F212      */          NVMConfigF212.Data,                  NVMDefaultF212.Data,   sizeof ( NVMConfigF212 ),   EEPROM_NORMAL_DATA_TYPE,          0x0049 },
    {/* NVMConfig3300      */          NVMConfig3300.Data,                  NVMDefault3300.Data,   sizeof ( NVMConfig3300 ),   EEPROM_NORMAL_DATA_TYPE,          0x0000+sizeof ( NVMDTC )+sizeof ( NVMConfigF184 ) +sizeof ( NVMConfigF1A8 )+sizeof ( NVMConfigF190 ) },
    {/* NVM_RESET_ODO      */          NVMRESETODO.Data,                  NVMDefaultRESETODO.Data,   sizeof ( NVMRESETODO ),   EEPROM_NORMAL_DATA_TYPE,          0x0000+sizeof ( NVMDTC )+sizeof ( NVMConfigF184 ) +sizeof ( NVMConfigF1A8 )+sizeof ( NVMConfigF190 ) +sizeof ( NVMConfig3300 )},
    /*=================================================================================================================================================================*/
};

/*** 存储数据总数 ***/
#define   NVM_TOTAL_DATA_NUM                (sizeof(NVMDataAttributeTable) / sizeof(NVMDataAttributeTableStruct))

/******************************************************************************
数据更新控制结构
******************************************************************************/
typedef struct
{
    uint8_t Enable;
    uint8_t Req[NVM_TOTAL_DATA_NUM];
    uint8_t Current;
} NVMUpdateCtrlStruct;

/******************************************************************************
函数声明
******************************************************************************/
void NVM_Init ( void );
void NVM_Data_Update ( uint8_t DataName );
void NVM_Data_Update_Service ( void );

#endif