
#include "YGV642_SPI_Flash_S25FL256S.h"

/*------------------------------------------------------------------------------*/
/* Function name    : PM_S25FL256S_ID_Check                                     */
/* Contents         : Pattern Memory JEDEC ID check                             */
/* Argument         : void                                                      */
/* Return value     : 1 - Pass 0 - Fail                                         */
/*------------------------------------------------------------------------------*/
uint8_t PM_S25FL256S_ID_Check(void)
{
  uint8_t   ReadData;
  uint8_t   Identifier[3];
  uint8_t   i;
  
  /* (1), (2) Write Pattern Memory Control register */
	PM_S25FL256S_Write_Cmd(PM_S25FL256S_CMD_READ_JEDEC_ID);

	/* (3) Write R#27h: AIPM = 1 */
	YGV642_Write_Reg(REG_R27H, 0x07);

	/* (4) Write P#2: CSNEG = 1 */
	YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_S25FL256S_READACS_CSNEG);

	/* (5) Write R#2C-R#2F: PMA[25:0] */
	PM_S25FL256S_Set_Pattern_Memory_Addr(0x00000000UL);
  
  for (i = 0; i < 3; i++)
  {
    /* (6) to (8) */
		PM_S25FL256S_Read_Pattern_Memory_Data_Port(&ReadData);
    Identifier[i] = ReadData;
  }
  
  if ((Identifier[0] == PM_S25FL256S_MANUFACTURER_ID) &&\
      (Identifier[1] == PM_S25FL256S_DEVICE_ID_MSB  ) &&\
      (Identifier[2] == PM_S25FL256S_DEVICE_ID_LSB  ))
    return 1;
    
  return 0;
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_S25FL256S_Init                                         */
/* Contents         : Initialize Pattern Memory                                 */
/* Argument         : void                                                      */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/
void PM_S25FL256S_Init(void)
{
	/*--------------------------------------*/
	/* Write Enable                         */
	/*--------------------------------------*/
	PM_S25FL256S_Write_Enable();
	
	/*--------------------------------------*/
	/* Write Status Regsiter                */
	/*--------------------------------------*/
	/* (2-1) (2-2) Write Status Register command */
	PM_S25FL256S_Write_Cmd(PM_S25FL256S_CMD_WRITE_STATUS);

	/* (2-3) Write P#2: CSNEG = 0 */
	YGV642_Write_Port(Y642_PORT_PTNMEM_READACS, 0x00);

	/* (2-4) Write R#2Ch-#2Fh: PMA[25:0] = 26'h000_0000 */
	PM_S25FL256S_Set_Pattern_Memory_Addr(0x0000000);

	/* (2-5) Write P#3: MD[7:0] = 8'h00 */
	YGV642_Write_Port(Y642_PORT_PTNMEM_DATA, 0x00);

	/* (2-6) Polling P#2: PWBUSY */
	PM_S25FL256S_Wait_Clear_Flag(PM_S25FL256S_READACS_PWBUSY);

	/* (2-7) Write P#2: CSNEG = 1 */
	YGV642_Write_Port(Y642_PORT_PTNMEM_READACS, PM_S25FL256S_READACS_CSNEG);

	/* (2-8) Write R#2Ch-#2Fh: PMA[25:0] = 26h'000_0001 */
	PM_S25FL256S_Set_Pattern_Memory_Addr(0x00000001);

	/* (2-9) Write P#3: MD[7:0] = 8'h02 */
	YGV642_Write_Port(Y642_PORT_PTNMEM_DATA, 0x02);
	
	/* (2-10) Polling P#2: PWBUSY */
	PM_S25FL256S_Wait_Clear_Flag(PM_S25FL256S_READACS_PWBUSY);

	/*--------------------------------------*/
	/* Read Status Register                 */
	/*--------------------------------------*/
	PM_S25FL256S_Check_Pattern_Memory_Status_Reg(0x00000000UL);
	
	/*--------------------------------------*/
	/* 4亊I/O Read Mode                     */
	/*--------------------------------------*/
	PM_S25FL256S_Write_Cmd(PM_S25FL256S_CMD_Q_IO_READ);
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_S25FL256S_Write_Enable                                 */
/* Contents         : Write Enable Command                                      */
/* Argument         : void                                                      */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/
void PM_S25FL256S_Write_Enable(void)
{
	/* (1-1),(1-2) Write Pattern Memory Control Register */
	PM_S25FL256S_Write_Cmd(PM_S25FL256S_CMD_WRITE_ENABLE);

	/* (1-3) Write P#2: CSNEG = 1 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_S25FL256S_READACS_CSNEG);

	/* (1-4) Write R#2Ch-#2Fh: PMA[25:0] = 26h'000_0000 */
	PM_S25FL256S_Set_Pattern_Memory_Addr(0x0000000);

	/* (1-5) Write P#3: MD[7:0] = 8'h00 */
	YGV642_Write_Port(YVC1_PORT_PTNMEM_DATA, 0x00);

	/* (1-6) Polling P#2: PWBUSY */
	PM_S25FL256S_Wait_Clear_Flag(PM_S25FL256S_READACS_PWBUSY);
}

/*------------------------------------------------------------------------------*/
/* Function name    : YGV642_Read_Status_Reg                                    */
/* Contents         : Check Pattern Memory Status Register                      */
/* Arguments        : uint32_t Addr    (W) Flash Memory address                 */
/* Returned value   : void                                                      */
/*------------------------------------------------------------------------------*/
void PM_S25FL256S_Check_Pattern_Memory_Status_Reg(uint32_t Addr)
{
  uint8_t   ReadData;
  uint32_t  Cnt;
  
  /* (1), (2) Write Pattern Memory Control register */
	PM_S25FL256S_Write_Cmd(PM_S25FL256S_CMD_READ_STATUS);

	/* (3) Write R#27h: AIPM = 0 */
	YGV642_Write_Reg(REG_R27H, 0x00);

	/* (4) Write P#2: CSNEG = 1 */
	YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_S25FL256S_READACS_CSNEG);

	/* (5) Write R#2C-R#2F: PMA[25:0] */
	PM_S25FL256S_Set_Pattern_Memory_Addr(Addr);

  Cnt = 0;
  
  do
	{
		/* (6) to (8) */
		PM_S25FL256S_Read_Pattern_Memory_Data_Port(&ReadData);
    Cnt++;
	} while (((ReadData & PM_S25FL256S_READ_REGISTER_MASK) != 0) && (Cnt < PM_S25FL256S_POLLING_COUNT_MAX));
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_S25FL256S_Set_Pattern_Memory_Addr                      */
/* Contents         : Set Pattern Memory Address register                       */
/* Argument         : uint32_t   Addr     (W) Pattern Memory Address            */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/  
void PM_S25FL256S_Set_Pattern_Memory_Addr(uint32_t Addr)
{
	uint8_t MemAddr[4];
	
	/*---	Pattern memory address set	---*/
	MemAddr[0] = (uint8_t)(( Addr >> SHIFT_24BIT ) & MASK_03H);
	MemAddr[1] = (uint8_t)(( Addr >> SHIFT_16BIT ) & MASK_FFH);
	MemAddr[2] = (uint8_t)(( Addr >> SHIFT_08BIT ) & MASK_FFH);
	MemAddr[3] = (uint8_t)(( Addr                ) & MASK_FFH);

  YGV642_Write_Regs(REG_R2CH, MemAddr, 4);
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_S25FL256S_Write_Cmd                                    */
/* Contents         : Write Pattern Memory Control registers                    */
/* Argument         : uint8_t   CmdNo    (W) Command No                         */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/
void PM_S25FL256S_Write_Cmd(uint8_t CmdNo)
{
  /*	Access Port	
	 *
	 *	 No.   Description 		 						  bit7   bit6   bit5   bit4   bit3   bit2   bit1   bit0 
	 *	+---+-------------------------------------------+------+------+------+------+------+------+------+------+
	 *	| 2 | Pattern Memory Read Access Port / Status 	|PMRREQ|PWBUSY|INIEND|CSNEG | "0"  | "0"  | "0"  | "0"  |
	 *	+---+-------------------------------------------+------+------+------+------+------+------+------+------+
	 *	| 3 | Pattern Memory Data Port 					|                        MD[7:0]                        |
	 *	+---+-------------------------------------------+------+------+------+------+------+------+------+------+
	 */
   
	/* Waiting for ready to write */
	PM_S25FL256S_Wait_Clear_Flag(PM_S25FL256S_READACS_PMRREQ | PM_S25FL256S_READACS_PWBUSY);

	/* Set command to R#55h-#5Bh */
  YGV642_Write_Regs(0x55, PMS25FL256SCmdData[CmdNo], 7);
}

/*------------------------------------------------------------------------------*/
/* Function name    : YGV642_Read_Pattern_Memory_Data_Port                      */
/* Contents         : Read Pattern Memory Data Port                             */
/*                  : uint8_t *Data    (R) Data pointer                         */
/* Argument         : void                                                      */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/
void PM_S25FL256S_Read_Pattern_Memory_Data_Port(uint8_t *Data)
{
	/* (6) Write P#2: PMRREQ */
	YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_S25FL256S_READACS_PMRREQ);

	/* (7) PMRREQ Polling */
	PM_S25FL256S_Wait_Clear_Flag(PM_S25FL256S_READACS_PMRREQ | PM_S25FL256S_READACS_PWBUSY);

	/* (8) Read P#3: MD[7:0] */
	*Data = YGV642_Read_Port(YVC1_PORT_PTNMEM_DATA);
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_S25FL256S_Wait_Clear_Flag                              */
/* Contents         : Wait to clear flags                                       */
/* Argument         : uint8_t   Mask    (W) Mask value                          */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/ 
void PM_S25FL256S_Wait_Clear_Flag(uint8_t Mask)
{
  uint8_t   ReadData;
  uint32_t  Cnt;
  
  Cnt = 0;
  do
  {
    ReadData = YGV642_Read_Port(YVC1_PORT_PTNMEM_READACS);
    Cnt++;
  }
  while (((ReadData & Mask) != 0) && (Cnt < PM_S25FL256S_POLLING_COUNT_MAX));
}
