
#include "YGV642_SPI_Flash_W25Q256FV.h"

/*------------------------------------------------------------------------------*/
/* Function name    : PM_S25FL256S_ID_Check                                     */
/* Contents         : Pattern Memory JEDEC ID check                             */
/* Argument         : void                                                      */
/* Return value     : 1 - Pass 0 - Fail                                         */
/*------------------------------------------------------------------------------*/
uint8_t PM_W25Q256FV_ID_Check(void)
{
  uint8_t   Identifier[3];
  uint8_t   i;
  
  /* (1), (2) Write Pattern Memory Control register */
	PM_W25Q256FV_Write_Cmd(PM_W25Q256FV_CMD_READ_JEDEC_ID, 7, 0);

	/* (3) Write R#27h: AIPM = 1 */
	YGV642_Write_Reg(REG_R27H, 0x07);

	/* (4) Write P#2: CSNEG = 1 */
	YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_W25Q256FV_READACS_CSNEG);

	/* (5) Write R#2C-R#2F: PMA[25:0] */
	PM_W25Q256FV_Set_Pattern_Memory_Addr(0x00000000UL);
  
  for (i = 0; i < 3; i++)
  {
    /* Read request */
    YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_W25Q256FV_READACS_PMRREQ);	/* PMRREQ=1, CSNEG=1 */

    /* Waiting for ready to read */
    PM_W25Q256FV_Wait_Clear_Flag(PM_W25Q256FV_READACS_PMRREQ | PM_W25Q256FV_READACS_PWBUSY);

    /* Read register value */
    Identifier[i] = YGV642_Read_Port(YVC1_PORT_PTNMEM_DATA);
  }
  
  if ((Identifier[0] == PM_W25Q256FV_MANUFACTURER_ID) &&\
      (Identifier[1] == PM_W25Q256FV_DEVICE_ID_MSB  ) &&\
      (Identifier[2] == PM_W25Q256FV_DEVICE_ID_LSB  ))
    return 1;
    
  return 0;
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_W25Q256FV_Init                                         */
/* Contents         : Initialize Pattern Memory                                 */
/* Argument         : void                                                      */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/
void PM_W25Q256FV_Init(void)
{
  uint8_t ReadData;
  
  /*--------------------------------------*/
	/* Write Enable                         */
	/*--------------------------------------*/
	/* (1-1) (1-2) Write Enable command */
	PM_W25Q256FV_Write_Cmd(PM_W25Q256FV_CMD_WRITE_ENABLE, 7, 0);

	/* (1-3) Write P#2: CSNEG = 1 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_W25Q256FV_READACS_CSNEG);
  
	/* (1-4) Write R#2Ch-#2Fh: PMA[25:0] = 26h'000_0000 */
	PM_W25Q256FV_Set_Pattern_Memory_Addr(0x0000000);
  
	/* (1-5) Write P#3: MD[7:0] = 8乫h00 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_DATA, 0x00);
  
	/* (1-6) Polling P#2: PWBUSY */
  PM_W25Q256FV_Wait_Clear_Flag(PM_W25Q256FV_READACS_PWBUSY);

  /*--------------------------------------*/
	/* Enter 4-Byte Address Mode            */
	/*--------------------------------------*/
	/* (2-1) (2-2) Write Enable command */
	PM_W25Q256FV_Write_Cmd(PM_W25Q256FV_CMD_4_BYTE_ADDR_MODE, 7, 0);

	/* (2-3) Write P#2: CSNEG = 1 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_W25Q256FV_READACS_CSNEG);
  
	/* (2-4) Write R#2Ch-#2Fh: PMA[25:0] = 26h'000_0000 */
	PM_W25Q256FV_Set_Pattern_Memory_Addr(0x0000000);
  
	/* (2-5) Write P#3: MD[7:0] = 8乫h00 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_DATA, 0x00);
  
	/* (2-6) Polling P#2: PWBUSY */
  PM_W25Q256FV_Wait_Clear_Flag(PM_W25Q256FV_READACS_PWBUSY);
  
  /*--------------------------------------*/
	/* Read 4-Byte Address Mode             */
	/*--------------------------------------*/
	/* (3-3) Write R#27h: AIPM = 0 */
  YGV642_Write_Reg(REG_R27H, 0x00);
	
	/* (3-4) Write P#2: CSNEG = 1 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_W25Q256FV_READACS_CSNEG);
	
	/* (3-5) Write PMA[25:0] = 26h'000_0000  */
  PM_W25Q256FV_Set_Pattern_Memory_Addr(0x0000000);

	/* (3-1) to (3-7) Read enhanced volatile configuration register */
	do
  {
		ReadData = PM_W25Q256FV_Read_Reg(PM_W25Q256FV_CMD_READ_STATUS3, 7);
	}while((ReadData & 0x01) == 0x00);
	
	/*--------------------------------------*/
	/* Write Status Register-2              */
	/*--------------------------------------*/
	/* (4-1) (4-2) Write Enable command */
	PM_W25Q256FV_Write_Cmd(PM_W25Q256FV_CMD_WRITE_STATUS2, 7, 0);

	/* (4-3) Write P#2: CSNEG = 1 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_W25Q256FV_READACS_CSNEG);

	/* (4-4) Write R#2Ch-#2Fh: PMA[25:0] = 26h'000_0000 */
	PM_W25Q256FV_Set_Pattern_Memory_Addr(0x0000000);
	
	/* (4-5) Write P#3: MD[7:0] = 8乫h02 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_DATA, 0x02);
	
	/* (4-6) Polling P#2: PWBUSY */
  PM_W25Q256FV_Wait_Clear_Flag(PM_W25Q256FV_READACS_PWBUSY);
  
	/*--------------------------------------*/
	/* Read Status Register                 */
	/*--------------------------------------*/
	/* (5-3) Write R#27h: AIPM = 0 */
  YGV642_Write_Reg(REG_R27H, 0x00);
	
	/* (5-4) Write P#2: CSNEG = 1 */
  YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_W25Q256FV_READACS_CSNEG);
	
	/* (5-5) Write PMA[25:0] = 26h'000_0000  */
  PM_W25Q256FV_Set_Pattern_Memory_Addr(0x0000000);

	/* (5-1) to (5-7) Read enhanced volatile configuration register */
	do
  {
		ReadData = PM_W25Q256FV_Read_Reg(PM_W25Q256FV_CMD_READ_STATUS1, 7);
	}while(ReadData != 0x00);
	
	/*--------------------------------------*/
	/* Fast Read QuadI/O                    */
	/*--------------------------------------*/
	PM_W25Q256FV_Write_Cmd(PM_W25Q256FV_CMD_Q_IO_READ, 7, 0);
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_W25Q256FV_Set_Pattern_Memory_Addr                      */
/* Contents         : Set Pattern Memory Address register                       */
/* Argument         : uint32_t   Addr     (W) Pattern Memory Address            */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/  
void PM_W25Q256FV_Set_Pattern_Memory_Addr(uint32_t Addr)
{
	uint8_t MemAddr[4];
	
	/*---	Pattern memory address set	---*/
	MemAddr[0] = (uint8_t)(( Addr >> SHIFT_24BIT ) & MASK_03H);
	MemAddr[1] = (uint8_t)(( Addr >> SHIFT_16BIT ) & MASK_FFH);
	MemAddr[2] = (uint8_t)(( Addr >> SHIFT_08BIT ) & MASK_FFH);
	MemAddr[3] = (uint8_t)(( Addr                ) & MASK_FFH);

  YGV642_Write_Regs(REG_R2CH, MemAddr, 4);
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_W25Q256FV_Write_Cmd                                    */
/* Contents         : Write Pattern Memory Control registers                    */
/* Argument         : uint8_t   CmdNo    (W) Command No                         */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/
void PM_W25Q256FV_Write_Cmd(uint8_t CmdNo, uint8_t NumCmd, uint8_t Send)
{
	/*	Access Port	
	 *
	 *	 No.   Description 		 						  bit7   bit6   bit5   bit4   bit3   bit2   bit1   bit0 
	 *	+---+-------------------------------------------+------+------+------+------+------+------+------+------+
	 *	| 2 | Pattern Memory Read Access Port / Status 	|PMRREQ|PWBUSY|INIEND|CSNEG | "0"  | "0"  | "0"  | "0"  |
	 *	+---+-------------------------------------------+------+------+------+------+------+------+------+------+
	 *	| 3 | Pattern Memory Data Port 					|                        MD[7:0]                        |
	 *	+---+-------------------------------------------+------+------+------+------+------+------+------+------+
	 */

	/* Waiting for ready to write */
	PM_W25Q256FV_Wait_Clear_Flag(PM_W25Q256FV_READACS_PMRREQ | PM_W25Q256FV_READACS_PWBUSY);

	/* Set command to R#55h-#5Bh */
	YGV642_Write_Regs(0x55, PMW25Q256FVCmdData[CmdNo], NumCmd);

  /* Send command */
	if (Send)
		YGV642_Write_Port(YVC1_PORT_PTNMEM_DATA, PMW25Q256FVCmdData[CmdNo][7]);
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_W25Q256FV_Read_Reg                                     */
/* Contents         : Read Pattern Memory Data Port                             */
/*                  : uint8_t *Data    (R) Data pointer                         */
/* Argument         : void                                                      */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/
uint8_t PM_W25Q256FV_Read_Reg(uint8_t CmdNo, uint8_t NumCmd)
{
	uint8_t ReadData;

	/* Set Command */
	PM_W25Q256FV_Write_Cmd(CmdNo, NumCmd, 0);

	/* Read request */
	YGV642_Write_Port(YVC1_PORT_PTNMEM_READACS, PM_W25Q256FV_READACS_PMRREQ | PM_W25Q256FV_READACS_CSNEG);	/* PMRREQ=1, CSNEG=1 */

	/* Waiting for ready to read */
	PM_W25Q256FV_Wait_Clear_Flag(PM_W25Q256FV_READACS_PMRREQ | PM_W25Q256FV_READACS_PWBUSY);

	/* Read register value */
	ReadData = YGV642_Read_Port(YVC1_PORT_PTNMEM_DATA);
	
	return ReadData;
}

/*------------------------------------------------------------------------------*/
/* Function name    : PM_W25Q256FV_Wait_Clear_Flag                              */
/* Contents         : Wait to clear flags                                       */
/* Argument         : uint8_t   Mask    (W) Mask value                          */
/* Return value     : void                                                      */
/*------------------------------------------------------------------------------*/ 
void PM_W25Q256FV_Wait_Clear_Flag(uint8_t Mask)
{
	uint8_t ReadData;

	do
	{
		ReadData = YGV642_Read_Port(YVC1_PORT_PTNMEM_READACS);
	} while ((ReadData & Mask) != 0);
}

