/******************************************************************************
 * $Revision: 425 $
 * $Date:: 2017-05-19 10:28:09 +0900#$
 *****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************/
/** \file adc.h
 **
 ** Header file for adc module.
 **
 ** History:
 **   - 2014-11-27  0.01  HS  Initial version for Traveo
 *****************************************************************************/

#ifndef __ADC12B_H__
#define __ADC12B_H__

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "common_include.h"
#include "base_types.h"

/* C binding of definitions if building with C++ compiler */
#ifdef __cplusplus
extern "C"
{
#endif

/**
 *****************************************************************************
 ** \defgroup Adc12bGroup 12/10/8bit Analog to Digital Converter (ADC12B)
 **
 ** \brief This section describes the interface for the Analog to 
 **        Digital Converter.
 **
 ** Provided functions of ADC12B module:
 **
 ** - Adc12b_Init()
 ** - Adc12b_SetChannelConfig()
 ** - Adc12b_SetComparisonTime()
 ** - Adc12b_SetResumptionTime()
 ** - Adc12b_SetSamplingTime()
 ** - Adc12b_SetOffsetCompensation()
 ** - Adc12b_SetGainCompensation()
 ** - Adc12b_GetConversionInterruptFlag()
 ** - Adc12b_ClearConversionInterruptFlag()
 ** - Adc12b_GetTriggerOverrunStatus()
 ** - Adc12b_ClearTriggerOverrunStatus()
 ** - Adc12b_GetConversionData()
 ** - Adc12b_TriggerConversion()
 ** - Adc12b_ClearTrigger()
 ** - Adc12b_ForceStop()
 ** - Adc12b_SetRangeCompConfig()
 ** - Adc12b_SetFullRangeCompThresholds()
 ** - Adc12b_SetRangeCompThresholds()
 ** - Adc12b_GetRangeCompInterruptFlag()
 ** - Adc12b_ClearRangeCompInterruptFlag()
 ** - Adc12b_SetPulseDetectionConfig()
 ** - Adc12b_GetPulseDetectionInterruptFlag()
 ** - Adc12b_ClearPulseDetectionInterruptFlag()
 ** - Adc12b_GetGroupIntInterruptFlag()
 ** - Adc12b_ClearGroupIntInterruptFlag()
 ** - Adc12b_SetMultipleConversion()
 ** - Adc12b_SetDmaConfig()
 **
 ** Adc12b_Init() initializes an ADC instance according the configuration
 ** of the structure type #stc_adc12b_config_t.
 **
 ** Adc12b_SetChannelConfig() can be used for channel configuration.
 ** For more information of configuration, see #stc_adc12b_channel_config_t.
 **
 ** Adc12b_SetComparisonTime(), Adc12b_SetResumptionTime(), Adc12b_SetSamplingTime()
 ** Adc12b_SetOffsetCompensation(), Adc12b_SetGainCompensation()
 ** are used to set global settings.
 **
 ** Adc12b_TriggerConversion() sets software trigger.
 ** Adc12b_ClearTrigger() clear trigger status.
 ** Adc12b_ForceStop() request forced stop of A/D conversion.
 **
 ** Adc12b_SetPulseDetectionConfig() configures pulse detection function.
 ** Adc12b_SetMultipleConversion() configures multiple conversion function.
 ** Adc12b_SetDmaConfig() configures settings for DMA.
 **
 ** Following functions can be used to set range comparator setting.
 ** -Adc12b_SetRangeCompConfig()
 ** -Adc12b_SetFullRangeCompThresholds()
 ** -Adc12b_SetRangeCompThresholds()
 **
 ** Other functions can be used to get / clear statuses.
 **
 *****************************************************************************/
/*! @{ */

/*****************************************************************************/
/* Global pre-processor symbols/macros ('#define')                           */
/*****************************************************************************/
/** Max. value of range comparator threshold */
#define ADC12B_FULL_RANGE_COMP_THRESHOLD_MAX	(0xFFF)

/** Max. value of channel */
#define ADC12B_MAX_CHANNEL_COUNT                (64)
#define ADC12B_MAX_CHANNEL_PER_REG              (32)

/** Maximum Gain Comparison Value */
#define ADC12B_MAX_GAIN_COMP_VALUE              (0x1F)

/** value of Range comparator threshold registers */
#define ADC12B_RANGE_COMP_REGISTER_NUM          (8)

/** value of sampling time setting registers */
#define ADC12B_SAMPLING_TIME_REGISTER_NUM       (4)

/** value of DMA setting registers */
#define ADC12B_DMA_REGISTER_NUM                 (4)

/** value of multiple conversion channel */
#define ADC12B_MULTIPLE_CONVERSION_NUM          (4)

/** Maximum value of multiple conversion count */
#define ADC12B_MAX_MULTIPLE_CONVERSION_COUNT    (16)

/** Minimum resumption time */
#define ADC12B_MIN_RESUMPTION_TIME              (0 + 1)

/** Minimum comparison time */
#define ADC12B_MIN_COMPARISON_TIME              (0 + 1)

/** Minimum sampling time */
#define ADC12B_MIN_SAMPLING_TIME                (6 + 1)

/*****************************************************************************
 * Global type definitions
 *****************************************************************************/

/**
 *****************************************************************************
 ** \brief ADC Channel list
 *****************************************************************************/
typedef union stc_adc12b_channel_list_per_reg {
    uint32_t u32reg ;
    struct {
        unsigned AD_CH_0 : 1 ;
        unsigned AD_CH_1 : 1 ;
        unsigned AD_CH_2 : 1 ;
        unsigned AD_CH_3 : 1 ;
        unsigned AD_CH_4 : 1 ;
        unsigned AD_CH_5 : 1 ;
        unsigned AD_CH_6 : 1 ;
        unsigned AD_CH_7 : 1 ;
        unsigned AD_CH_8 : 1 ;
        unsigned AD_CH_9 : 1 ;
        unsigned AD_CH_10 : 1 ;
        unsigned AD_CH_11 : 1 ;
        unsigned AD_CH_12 : 1 ;
        unsigned AD_CH_13 : 1 ;
        unsigned AD_CH_14 : 1 ;
        unsigned AD_CH_15 : 1 ;
        unsigned AD_CH_16 : 1 ;
        unsigned AD_CH_17 : 1 ;
        unsigned AD_CH_18 : 1 ;
        unsigned AD_CH_19 : 1 ;
        unsigned AD_CH_20 : 1 ;
        unsigned AD_CH_21 : 1 ;
        unsigned AD_CH_22 : 1 ;
        unsigned AD_CH_23 : 1 ;
        unsigned AD_CH_24 : 1 ;
        unsigned AD_CH_25 : 1 ;
        unsigned AD_CH_26 : 1 ;
        unsigned AD_CH_27 : 1 ;
        unsigned AD_CH_28 : 1 ;
        unsigned AD_CH_29 : 1 ;
        unsigned AD_CH_30 : 1 ;
        unsigned AD_CH_31 : 1 ;
    } stc_ch;
} stc_adc12b_channel_list_per_reg_t ;

typedef struct stc_adc12b_channel_list {
    stc_adc12b_channel_list_per_reg_t stcChannel[2];
}stc_adc12b_channel_list_t;


/**
 *****************************************************************************
 ** \brief ADC trigger type selection
 *****************************************************************************/
typedef enum en_adc12b_trigger_type
{
    /** Trigger by software */
    Adc12bSoftTrigger                   = 0,
    /** Trigger by software and hardware*/
    Adc12bSoftAndHardTrigger            = 1,
    /** Trigger by conversion completion and updating of conversion
     * data register ADC12Bn_CD of the preceding channel. */
    Adc12bPrecedingChannelCompletion    = 2,
    /** Idle trigger, the channel trigger status is set if there is no channel having
     * trigger status flag set and inactive data protection function. */
    Adc12bIdleTrigger                   = 3
} en_adc12b_trigger_type_t;


/**
 *****************************************************************************
 ** \brief Full Range Comparator mode selection
 *****************************************************************************/
typedef enum en_adc12b_range_comparator_mode
{
    /** 8bit mode */
    Adc12bRangeComparatorMode8bit = 0,
    /** 12bit mode */
    Adc12bRangeComparatorMode12bit = 1
} en_adc12b_range_comparator_mode_t;

/**
 *****************************************************************************
 ** \brief ACH register mode selection
 *****************************************************************************/
typedef enum en_adc12b_ach_mode
{
    /** Direct ACH register mode. Show the number of currently converted logical channel */
    Adc12bAchModeDirect = 0,
    /** Latched ACH register mode. Show the number of the logical channel whose conversion was finished last. */
    Adc12bAchModeLatched = 1
} en_adc12b_ach_mode_t;

/**
 *****************************************************************************
 ** \brief Resume/Restart/Stop Selection
 *****************************************************************************/
typedef enum en_adc12b_reumemode
{
    /** Stop the group processing until next group start channel conversion request is issued. */
    Adc12bResumeModeStop = 0,
    /** Resume. If the group is interrupted just before the conversion of this channel
     *  is started, resume the group processing with this channel. */
    Adc12bResumeModeResume = 1,
    /** Restart. After the group is interrupted, restart with the start channel
     *  or the last converted channel configured as "resume" channel. */
    Adc12bResumeModeRestart = 2,
} en_adc12b_reumemode_t;

/**
 *****************************************************************************
 ** \brief Resolution selection
 *****************************************************************************/
typedef enum en_adc12b_resolution
{
    /** 12bit resolution mode */
    Adc12bResolution12bit = 0,
    /** 10bit resolution mode */
    Adc12bResolution10bit = 1,
    /** 8bit resolution mode */
    Adc12bResolution8bit = 3,
} en_adc12b_resolution_t;

/**
 *****************************************************************************
 ** \brief Callback function type for end of conversion callback.
 **
 ** \param [in] stcDetectList   List of conversion end channels.
 **                             Each bits are set to 1 when corresponding
 **                             channel interrupt is enabled and conversion has completed.
 *****************************************************************************/
typedef void (*adc12b_eof_conversion_func_ptr_t)(stc_adc12b_channel_list_t stcDetectList);

/**
 *****************************************************************************
 ** \brief Callback function type for group interrupted interrupt callback.
 **
 ** \param [in] stcDetectList   List of interrupt detected channels.
 *****************************************************************************/
typedef void (*adc12b_grp_interrupt_func_ptr_t)(stc_adc12b_channel_list_t stcDetectList);

/**
 *****************************************************************************
 ** \brief Callback function type for range comparator callback.
 **
 ** \param [in] stcDetectList   List of range comparator error detected channels.
 **                             Each bits are set to 1 when corresponding
 **                             channel interrupt is enabled and range comparison
 **                             error has detected.
 ** \param [in] stcExcessFlags  List of range comparison threshold excess flags.
 **                             If a bit = 1, corresponding channel detected over threshold.
 **                             If a bit = 0, corresponding channel detected under threshold.
 **                             This value is effective only when stc_adc12b_range_comparator_config_t
 **                             bInsideRangeComparison = false.
 *****************************************************************************/
typedef void (*adc12b_range_comparator_func_ptr_t)(stc_adc12b_channel_list_t stcDetectList,
                                                   stc_adc12b_channel_list_t stcExcessFlags);

/**
 *****************************************************************************
 ** \brief Callback function type for pulse detection interrupt callback.
 **
 ** \param [in] stcDetectList   List of interrupt detected channels.
 *****************************************************************************/
typedef void (*adc12b_pulse_detection_func_ptr_t)(stc_adc12b_channel_list_t stcDetectList);


/**
 *****************************************************************************
 ** \brief ADC configuration
 **        This settings are common to a unit.
 *****************************************************************************/
typedef struct stc_adc12b_config
{
    /** Power-down disable mode selection.
     *  If TRUE is set, A/D converter does not go idle state.
     */
    boolean_t 						bPowerDownDisable;
    /** Range Comparator mode selection. see #en_adc12b_range_comparator_mode_t */
    en_adc12b_range_comparator_mode_t	enRangeComparatorMode;
    /** Forced stop mode selection */
    boolean_t						bEnableForcedStop;
    /** ACH register mode selection. see #en_adc12b_ach_mode_t */
    en_adc12b_ach_mode_t			enAchMode;
    /** Debug mode selection.
     *  When TRUE is set and the processor is in debug state, the A/D Converter completes
     *  the current conversion, but further conversion is stopped when.
     */
    boolean_t						bEnableDebugMode;
    /** Resolution selection */
    en_adc12b_resolution_t			enResolution;
    /** End of conversion interrupt callback (can be NULL). */
    adc12b_eof_conversion_func_ptr_t   pfnEofConversionCallback;
    /** Group interrupted interrupt call back(can be NULL). */
    adc12b_grp_interrupt_func_ptr_t     pfnGprIntCallback;
    /** Pulse detection interrupt call back(can be NULL). */
    adc12b_pulse_detection_func_ptr_t   pfnPulseDetectionCallback;
    /** Range comparison interrupt call back(can be NULL). */
    adc12b_range_comparator_func_ptr_t pfnRangeComparatorCallback;
} stc_adc12b_config_t;


/**
 *****************************************************************************
 ** \brief ADC channel configuration
 *****************************************************************************/
typedef struct stc_adc12b_channel_config
{
    /** Physical Analog channel selection  */
    uint8_t                             u8PhysicalCh;
    /** Priority setting (0=Highest priority) */
    uint8_t                             u8Priority;
    /** Resume/Restart/Stop Selection See #en_adc12b_reumemode_t */
    en_adc12b_reumemode_t               enResumeMode;
    /** If TRUE, Protection enabled. If FALSE, Protection disabled.*/
    boolean_t                           bDataProtection;
    /** Sampling Time Selection */
    uint8_t                             u8SamplingTimeRegNum;
    /** TRUE = Use End of Conversion interrupt. */
    boolean_t                           bUseConversionDoneInterrupt;
    /** TRUE = Use Group interrupted interrupt */
    boolean_t                           bUseGroupInterruptedInterrupt;
    /** Activation trigger selection. See #en_adc12b_trigger_type_t */
    en_adc12b_trigger_type_t            enTriggerType;
} stc_adc12b_channel_config_t;


/**
 *****************************************************************************
 ** \brief ADC multiple conversion configuration
 *****************************************************************************/
typedef struct stc_adc12b_multi_conversion_config
{
    /** Number of conversions (Actual conversion = this value + 1) */
    uint8_t                             u8NumOfConversion;
    /** If TRUE, Multiple conversion logical channel cannot be interrupted between single conversions. */
    boolean_t                           bDisableIntraChannelInterrupt;
    /** If TRUE, AVRH voltage is selected for A/D conversion */
    boolean_t                           bConvertAvrhVoltage;
    /** If TRUE, AVRL voltage is selected for A/D conversion */
    boolean_t                           bConvertAvrlVoltage;
}stc_adc12b_multi_conversion_config_t;

/**
 *****************************************************************************
 ** \brief ADC range comparator configuration
 *****************************************************************************/
typedef struct stc_adc12b_range_comparator_config
{
    /** Enable analog channel for selected range comparator. */
    boolean_t                           bEnable;
    /** Range comparator selection. */
    uint8_t                             u8RangeComparator;
    /** If TRUE, Inside range comparison. If FALSE, Outside */
    boolean_t                           bInsideRangeComparison;
    /** TRUE = Use interrupt. */
    boolean_t                           bUseInterrupt;
} stc_adc12b_range_comparator_config_t;


/**
 *****************************************************************************
 ** \brief Pulse detection configuration
 *****************************************************************************/
typedef struct stc_adc12b_pulse_detection_config
{
    /** TRUE = Use interrupt. */
    boolean_t                           bUseInterrupt;
    /** Positive counter reload value */
    uint8_t                             u8PositiveCounterReloadValue;
    /** Negative counter reload value */
    uint8_t                             u8NegativeCounterReloadValue;
} stc_adc12b_pulse_detection_config_t;

/**
 *****************************************************************************
 ** \brief ADC setting for DMA transfer
 *****************************************************************************/
typedef struct stc_adc12b_dma_config
{
    /** If TRUE, Interrupt for triggering DMA request on conversion done interrupt flag
     * of the logical channel defined by u8TriggeringLogcalChNum is enabled. */
    boolean_t                       bEnableDmaDidicatedInterrupt;
    /** Number of the logical channel selected for conversion done
     * interrupt for triggering a DMA request */
    uint8_t                         u8TriggeringLogcalChNum;
}stc_adc12b_dma_config_t;

/**
 *****************************************************************************
 ** \brief Data type for holding internal data needed for ADC
 **
 ** This struct is used to store the interrupt callback functions for the ADC.
 *****************************************************************************/
typedef struct stc_adc_intern_data {
    adc12b_eof_conversion_func_ptr_t    pfnEofConversionCallback;           /*!< Callback for pulse detection interrupt. */
    stc_adc12b_channel_list_t           stcEofConvIntEnableChList;          /*!< List of end of conversion interrupt enable channels */
    adc12b_grp_interrupt_func_ptr_t     pfnGrpIntCallback;                  /*!< Callback for Group interrupted interrupt. */
    stc_adc12b_channel_list_t           stcGrpIntEnableChList;              /*!< List of group interrupted interrupt enable channels */
    adc12b_pulse_detection_func_ptr_t   pfnPulseDetectionCallback;          /*!< Callback for pulse detection interrupt. */
    stc_adc12b_channel_list_t           stcPulseDetectionIntEnableChList;   /*!< List of pulse detection interrupt enable channels */
    adc12b_range_comparator_func_ptr_t  pfnRangeComparatorCallback;         /*!< Callback for range comparator interrupt. */
    stc_adc12b_channel_list_t           stcRangeCompIntEnableChList;        /*!< List of end of range comparator interrupt enable channels */
} stc_adc12b_intern_data_t;

/** Enumeration to define an index for each enabled ADC instance */
typedef enum en_adc12b_instance_index
{
    #if (PDL_PERIPHERAL_ENABLE_ADC12B0 == PDL_ON)
    Adc12bInstanceIndexAdc0,
    #endif
    Adc12bInstanceIndexMax
} en_adc12b_instance_index_t;


/** ADC instance data type */
typedef struct stc_adc12b_instance_data
{
    volatile stc_adc12b0_t* pstcInstance;       /*!< pointer to registers of an instance */
    stc_adc12b_intern_data_t stcInternData;     /*!< module internal data of instance */
    uint8_t u8UnitIndex;                        /*!< unit index */
} stc_adc12b_instance_data_t;

/*****************************************************************************/
/* Global variable declarations ('extern', definition in C source)           */
/*****************************************************************************/

/*****************************************************************************/
/* Global function prototypes ('extern', definition in C source)             */
/*****************************************************************************/

extern en_result_t Adc12b_Init(volatile stc_adc12b0_t* pstcAdc,
                               const stc_adc12b_config_t* pstcConfig);
extern en_result_t Adc12b_SetChannelConfig(volatile stc_adc12b0_t* pstcAdc,
                                           uint8_t u8Channel,
                                           const stc_adc12b_channel_config_t* pstcConfig);
extern en_result_t Adc12b_SetComparisonTime(volatile stc_adc12b0_t* pstcAdc,
                                            uint16_t u16ComparisonTime);
extern en_result_t Adc12b_SetResumptionTime(volatile stc_adc12b0_t* pstcAdc,
                                            uint8_t u8ResumptionTime);
extern en_result_t Adc12b_SetSamplingTime(volatile stc_adc12b0_t* pstcAdc,
                                          uint8_t u8RegNum,
                                          uint16_t u16SamplingTime);
extern en_result_t Adc12b_SetOffsetCompensation(volatile stc_adc12b0_t* pstcAdc,
                                                uint8_t u8OffsetCompensationValue);
extern en_result_t Adc12b_SetGainCompensation(volatile stc_adc12b0_t* pstcAdc,
                                              uint8_t u8GainCompensationValue);
extern en_result_t Adc12b_GetConversionInterruptFlag(volatile stc_adc12b0_t* pstcAdc,
                                              stc_adc12b_channel_list_t *pstcChannelStatusList);
extern en_result_t Adc12b_ClearConversionInterruptFlag(volatile stc_adc12b0_t* pstcAdc,
                                                stc_adc12b_channel_list_t stcClearChList);
extern en_result_t Adc12b_GetTriggerOverrunStatus(volatile stc_adc12b0_t* pstcAdc,
                                           stc_adc12b_channel_list_t *pstcChannelStatusList);
extern en_result_t Adc12b_ClearTriggerOverrunStatus(volatile stc_adc12b0_t* pstcAdc,
                                             stc_adc12b_channel_list_t stcClearChList);
extern en_result_t Adc12b_GetConversionData(volatile stc_adc12b0_t* pstcAdc,
                                     uint8_t u8Channel,
                                     uint16_t *pu16Result);
extern en_result_t Adc12b_TriggerConversion(volatile stc_adc12b0_t* pstcAdc,
                                  uint8_t u8Channel);
extern en_result_t Adc12b_ClearTrigger(volatile stc_adc12b0_t* pstcAdc,
                                stc_adc12b_channel_list_t stcClearChList);
extern en_result_t Adc12b_ForceStop(volatile stc_adc12b0_t* pstcAdc);
extern en_result_t Adc12b_SetRangeCompConfig(volatile stc_adc12b0_t* pstcAdc,
        uint8_t u8Channel, const stc_adc12b_range_comparator_config_t* pstcConfig);
extern en_result_t Adc12b_SetFullRangeCompThresholds(volatile stc_adc12b0_t* pstcAdc,
                                              uint8_t  u8RegNum,
                                              uint16_t u16UpperThreshold,
                                              uint16_t u16LowerThreshold);
extern en_result_t Adc12b_SetRangeCompThresholds(volatile stc_adc12b0_t* pstcAdc,
                                          uint8_t u8RegNum,
                                          uint8_t u8UpperThreshold,
                                          uint8_t u8LowerThreshold);
extern en_result_t Adc12b_GetRangeCompInterruptFlag(volatile stc_adc12b0_t* pstcAdc,
                                             stc_adc12b_channel_list_t* pstcDetectChList,
                                             stc_adc12b_channel_list_t* pstcExcessFlags);
extern en_result_t Adc12b_ClearRangeCompInterruptFlag(volatile stc_adc12b0_t* pstcAdc,
                                               stc_adc12b_channel_list_t stcClearChList);
extern en_result_t Adc12b_SetPulseDetectionConfig(volatile stc_adc12b0_t* pstcAdc,
                                           uint8_t u8Channel,
                                           const stc_adc12b_pulse_detection_config_t* pstcConfig);
extern en_result_t Adc12b_GetPulseDetectionInterruptFlag(volatile stc_adc12b0_t* pstcAdc,
                                                  stc_adc12b_channel_list_t* pstcDetectChList);
extern en_result_t Adc12b_ClearPulseDetectionInterruptFlag(volatile stc_adc12b0_t* pstcAdc,
                                                    stc_adc12b_channel_list_t stcClearChList);
extern en_result_t Adc12b_GetGroupIntInterruptFlag(volatile stc_adc12b0_t* pstcAdc,
                                            stc_adc12b_channel_list_t* pstcDetectChList);
extern en_result_t Adc12b_ClearGroupIntInterruptFlag(volatile stc_adc12b0_t* pstcAdc,
                                              stc_adc12b_channel_list_t stcClearChList);
extern en_result_t Adc12b_SetMultipleConversion(volatile stc_adc12b0_t* pstcAdc,
                                         uint8_t u8Channel,
                                         const stc_adc12b_multi_conversion_config_t* pstcConfig);
extern en_result_t Adc12b_SetDmaConfig(volatile stc_adc12b0_t* pstcAdc,
                                uint8_t u8RegNum,
                                const stc_adc12b_dma_config_t* pstcConfig);

extern stc_adc12b_intern_data_t* Adc12bGetInternDataPtr(volatile stc_adc12b0_t* pstcAdc);
extern void Adc12bIrqHandlerEndOfConversion(volatile stc_adc12b0_t* pstcAdc,
                                            stc_adc12b_intern_data_t* pstcInternData);
extern void Adc12bIrqHandlerPulseDetection(volatile stc_adc12b0_t* pstcAdc,
                                           stc_adc12b_intern_data_t* pstcInternData);
extern void Adc12bIrqHandlerGroupInt(volatile stc_adc12b0_t* pstcAdc,
                                     stc_adc12b_intern_data_t* pstcInternData);
extern void Adc12bIrqHandlerRangeComparator(volatile stc_adc12b0_t* pstcAdc,
                                            stc_adc12b_intern_data_t* pstcInternData);


extern volatile uint16_t AdcValList[64];

extern void InitADC(void);
extern void InitADCch(uint8_t ch);
/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* __ADC_H__ */
/*****************************************************************************/
/* EOF (not truncated)                                                       */
/*****************************************************************************/
