/******************************************************************************
 * $Revision: 423 $
 * $Date:: 2017-04-07 16:03:30 +0900#$
 *****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************
 ** \file mfs_csio.h
 **
 ** Headerfile for MFS CSIO driver functions
 **
 ** History:
 ** - 2014-12-18  0.01  HSh  First version
 *****************************************************************************/

#ifndef _MFS_CSIO_H_
#define _MFS_CSIO_H_

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "pdl.h"

/**
 *****************************************************************************
 ** \defgroup CsioGroup Clock Synchronous Serial Interface (CSIO)
 **
 ** \brief This section describes the interface for Clock Synchronous Serial Interface.
 **
 ** Provided functions of CSIO module:
 ** - Mfs_Csio_DisableIrq();
 ** - Mfs_Csio_EnableIrq();
 ** - Mfs_Csio_SetBaudRate();
 ** - Mfs_Csio_Init ();
 ** - Mfs_Csio_DeInit();
 ** - Mfs_Csio_SetTimerCompareValue();
 ** - Mfs_Csio_SetCsHoldStatus();
 ** - Mfs_Csio_GetStatus();
 ** - Mfs_Csio_DisableFunc();
 ** - Mfs_Csio_EnableFunc();
 ** - Mfs_Csio_ClrStatus();
 ** - Mfs_Csio_ResetFifo ();
 ** - Mfs_Csio_SetFifoCount();
 ** - Mfs_Csio_GetFifoCount();
 ** - Mfs_Csio_FifoEnable ();
 ** - Mfs_Csio_SetTimerTransferByteCount();
 ** - Mfs_Csio_HalfDuplexComm();
 ** - Mfs_Csio_FullDuplexComm();
 **
 ** Mfs_Csio_Init() is used to initialize an MFS instance to CSIO mode with
 ** parameter pstcConfig of type #stc_mfs_csio_config_t.
 ** This function is only set basically CSIO configuration.
 ** Mfs_Csio_DeInit() is used to reset all MFS CSIO related register.
 ** So this function is used after initialization by Mfs_Csio_Init().
 **
 *****************************************************************************/
/** @{ */

/*****************************************************************************/
/* Global pre-processor symbols/macros ('#define')                           */
/*****************************************************************************/
/** CSIO ch0 */
#define CSIO0_Type   ((pstc_csio_type_t)&CPG_MFS00_CSIO)
#define CSIO0_TIE    CPG_MFS00_CSIO_SCR_TIE
#define CSIO0_RIE    CPG_MFS00_CSIO_SCR_RIE
/** CSIO ch1 */
#define CSIO1_Type   ((pstc_csio_type_t)&CPG_MFS01_CSIO)
#define CSIO1_TIE    CPG_MFS01_CSIO_SCR_TIE
#define CSIO1_RIE    CPG_MFS01_CSIO_SCR_RIE
/** CSIO ch2 */
#define CSIO2_Type   ((pstc_csio_type_t)&CPG_MFS02_CSIO)
#define CSIO2_TIE    CPG_MFS02_CSIO_SCR_TIE
#define CSIO2_RIE    CPG_MFS02_CSIO_SCR_RIE
/** CSIO ch3 */
#define CSIO3_Type   ((pstc_csio_type_t)&CPG_MFS03_CSIO)
#define CSIO3_TIE    CPG_MFS03_CSIO_SCR_TIE
#define CSIO3_RIE    CPG_MFS03_CSIO_SCR_RIE
/** CSIO ch4 */
#define CSIO4_Type   ((pstc_csio_type_t)&CPG_MFS04_CSIO)
#define CSIO4_TIE    CPG_MFS04_CSIO_SCR_TIE
#define CSIO4_RIE    CPG_MFS04_CSIO_SCR_RIE
/** CSIO ch5 */
#define CSIO5_Type   ((pstc_csio_type_t)&CPG_MFS05_CSIO)
#define CSIO5_TIE    CPG_MFS05_CSIO_SCR_TIE
#define CSIO5_RIE    CPG_MFS05_CSIO_SCR_RIE
/** CSIO ch6 */
#define CSIO6_Type   ((pstc_csio_type_t)&CPG_MFS06_CSIO)
#define CSIO6_TIE    CPG_MFS06_CSIO_SCR_TIE
#define CSIO6_RIE    CPG_MFS06_CSIO_SCR_RIE
/** CSIO ch7 */
#define CSIO7_Type   ((pstc_csio_type_t)&CPG_MFS07_CSIO)
#define CSIO7_TIE    CPG_MFS07_CSIO_SCR_TIE
#define CSIO7_RIE    CPG_MFS07_CSIO_SCR_RIE
/** CSIO ch8 */
#define CSIO8_Type   ((pstc_csio_type_t)&CPG_MFS08_CSIO)
#define CSIO8_TIE    CPG_MFS08_CSIO_SCR_TIE
#define CSIO8_RIE    CPG_MFS08_CSIO_SCR_RIE
/** CSIO ch9 */
#define CSIO9_Type   ((pstc_csio_type_t)&CPG_MFS09_CSIO)
#define CSIO9_TIE    CPG_MFS09_CSIO_SCR_TIE
#define CSIO9_RIE    CPG_MFS09_CSIO_SCR_RIE
/** CSIO ch10 */
#define CSIO10_Type   ((pstc_csio_type_t)&CPG_MFS10_CSIO)
#define CSIO10_TIE    CPG_MFS10_CSIO_SCR_TIE
#define CSIO10_RIE    CPG_MFS10_CSIO_SCR_RIE
/** CSIO ch11 */
#define CSIO11_Type   ((pstc_csio_type_t)&CPG_MFS11_CSIO)
#define CSIO11_TIE    CPG_MFS11_CSIO_SCR_TIE
#define CSIO11_RIE    CPG_MFS11_CSIO_SCR_RIE

/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/
/**
 *****************************************************************************
 ** \brief CSIO Unit Type
 *****************************************************************************/
typedef  volatile stc_cpg_mfsn_csio_t*  pstc_csio_type_t;

/**
 *****************************************************************************
 ** \brief Csio ring buffer.
 ** 
 ** Contains all parameter for ring buffer handling. 
 *****************************************************************************/
//typedef struct stc_csio_buffer
//{
//    uint16_t* pu16Buffer;     /*!< Pointer to character buffer. */
//    uint16_t  u16BufferSize;  /*!< Size of buffer. */
//    uint16_t  u16InIndex;     /*!< Index of next element to store on buffer. */
//    uint16_t  u16OutIndex;    /*!< Index of next element to read from buffer. */
//    uint16_t  u16FillCount;   /*!< Indicates if elements are available in buffer. */
//} stc_csio_buffer_t;

/**
 *****************************************************************************
 ** \brief CSIO configuration
 ** 
 ** Contains parameter required for MFS CSIO configuration. 
 *****************************************************************************/

/**
 ******************************************************************************
 ** \brief Mfs FIFO Selection
 ******************************************************************************/
typedef enum en_mfs_fifo_sel
{
    MfsTxFifo1RxFifo2 = 0u,          ///< Transmit FIFO:FIFO1, Received FIFO:FIFO2
    MfsTxFifo2RxFifo1 = 1u,          ///< Transmit FIFO:FIFO2, Received FIFO:FIFO1
} en_mfs_fifo_sel_t;

/**
 ******************************************************************************
 ** \brief Mfs FIFO configuration.
 ******************************************************************************/
typedef struct stc_mfs_fifo_config
{
    en_mfs_fifo_sel_t enFifoSel;    ///< FIFO selection, see #en_mfs_fifo_select_t for details
    uint8_t u8ByteCount1;           ///< Transfer data count for FIFO1
    uint8_t u8ByteCount2;           ///< Transfer data count for FIFO2
} stc_mfs_fifo_config_t;

/**
 ******************************************************************************
 ** \brief Mfs FIFO Number
 ******************************************************************************/
typedef enum en_mfs_fifo
{
    MfsFifo1 = 0u,                   ///< FIFO No.1
    MfsFifo2 = 1u,                   ///< FIFO No.2
} en_mfs_fifo_t;  
  
/**
 ******************************************************************************
 ** \brief CSIO chip selection clock division
 ******************************************************************************/
typedef enum en_cs_timing_clk
{
    CsClkNoDiv = 0,             /*!< No division of Bus clock */
    CsClkDiv2  = 1,             /*!< Division of Bus clock by 2 */
    CsClkDiv4  = 2,             /*!< Division of Bus clock by 4 */
    CsClkDiv8  = 3,             /*!< Division of Bus clock by 8 */
    CsClkDiv16 = 4,             /*!< Division of Bus clock by 16 */
    CsClkDiv32 = 5,             /*!< Division of Bus clock by 32 */
    CsClkDiv64 = 6              /*!< Division of Bus clock by 64 */
} en_cs_timing_clk_t;

/**
 ******************************************************************************
 ** \brief CSIO chip selection pin level
 ******************************************************************************/
typedef enum en_cs_pin_level                    // different from the MMIC 
{
    CsHighActive = 0,          /*!< Set the inactive level to "L" */
    CsLowActive  = 1           /*!< Set the inactive level to "H" */
} en_cs_pin_level_t;

/**
 ******************************************************************************
 ** \brief CSIO serial timer clock division
 ******************************************************************************/
typedef enum en_csio_timer_clk
{
    CsioTimerNoDiv  = 0,        /*!< No division of Bus clock */
    CsioTimerDiv2   = 1,        /*!< Division of Bus clock by 2 */
    CsioTimerDiv4   = 2,        /*!< Division of Bus clock by 4 */
    CsioTimerDiv8   = 3,        /*!< Division of Bus clock by 8 */
    CsioTimerDiv16  = 4,        /*!< Division of Bus clock by 16 */
    CsioTimerDiv32  = 5,        /*!< Division of Bus clock by 32 */
    CsioTimerDiv64  = 6,        /*!< Division of Bus clock by 64 */
    CsioTimerDiv128 = 7,        /*!< Division of Bus clock by 128 */
    CsioTimerDiv256 = 8         /*!< Division of Bus clock by 256 */
} en_csio_timer_clk_t;

/**
 ******************************************************************************
 ** \brief CSIO mode (Master/Slave)
 ******************************************************************************/

typedef enum en_csio_ms_mode
{
    CsioMaster = 0,     /*!< Master mode */
    CsioSlave  = 1      /*!< Slave mode */
} en_csio_ms_mode_t;

/**
 ******************************************************************************
 ** \brief CSIO active mode (Normal/SPI)
 ******************************************************************************/
typedef enum en_csio_act_mode
{
    CsioActNormalMode = 0,     /*!< Normal Synchronous Transfer */
    CsioActSpiMode    = 1      /*!< SPI Support */
} en_csio_act_mode_t;

/**
 ******************************************************************************
 ** \brief CSIO synchronous wait time
 ******************************************************************************/
typedef enum en_csio_sync_wait_time
{
    CsioSyncWaitZero  = 0,     /*!< Output SCK continuously */
    CsioSyncWaitOne   = 1,     /*!< Output SCK after a 1-bit time wait */
    CsioSyncWaitTwo   = 2,     /*!< Output SCK after a 2-bit time wait */
    CsioSyncWaitThree = 3      /*!< Output SCK after a 3-bit time wait */
} en_csio_sync_wait_time_t;

/**
 ******************************************************************************
 ** \brief CSIO bit direction
 ******************************************************************************/
typedef enum en_csio_data_dir
{
    CsioDataLsbFirst = 0,       /*!< Output LSB first */
    CsioDataMsbFirst = 1        /*!< Output Msb first */
} en_csio_data_dir_t;

/**
 ******************************************************************************
 ** \brief CSIO data length
 ******************************************************************************/
typedef enum en_csio_data_len
{
    CsioFiveBits        = 0,     /*!< 5-bit length */
    CsioSixBits         = 1,     /*!< 6-bit length */
    CsioSevenBits       = 2,     /*!< 7-bit length */
    CsioEightBits       = 3,     /*!< 8-bit length */    
    CsioNineBits        = 4,     /*!< 9-bit length */
    CsioTenBits         = 5,     /*!< 10-bit length */
    CsioElevenBits      = 6,     /*!< 11-bit length */
    CsioTwelveBits      = 7,     /*!< 12-bit length */
    CsioThirteenBits    = 8,     /*!< 13-bit length */
    CsioFourteenBits    = 9,     /*!< 14-bit length */
    CsioFifteenBits     = 10,    /*!< 15-bit length */
    CsioSixteenBits     = 11,    /*!< 16-bit length */
    CsioTwentyBits      = 12,    /*!< 20-bit length */
    CsioTwentyfourBits  = 13,    /*!< 24-bit length */
    CsioThirtytwoBits   = 14     /*!< 32-bit length */
} en_csio_data_len_t;

/**
 ******************************************************************************
 ** \brief CSIO Access width control bit
 ******************************************************************************/
typedef enum en_csio_access_width
{
    Csio16BitAccess = 0,        /*!< RDR or TDR register accessed as 16 bit     */  
    Csio32BitAccess = 1         /*!< RDR or TDR register accessed as 32 bit     */  
} en_csio_access_width_t;

/**
 ******************************************************************************
 ** \brief CSIO chip selection pin
 ******************************************************************************/
typedef enum en_cs_pin_sel
{
    CsPinScs0 = 0u,    ///< Use SCS0 as chip selection pin
    CsPinScs1 = 1u,    ///< Use SCS1 as chip selection pin
    CsPinScs2 = 2u,    ///< Use SCS2 as chip selection pin
    CsPinScs3 = 3u,    ///< Use SCS3 as chip selection pin  
}en_cs_pin_sel_t;

/**
 ******************************************************************************
 ** \brief CSIO serial timer configuration
 ******************************************************************************/
typedef struct stc_csio_serial_timer
{
    en_csio_timer_clk_t enClkDiv;           ///< CSIO Clock division. See #en_csio_timer_clk_t.
    uint8_t             u8TransferByteCnt;  ///< CSIO Transfer byte count
    uint16_t            u16CompareVal;      ///< CSIO Serial timer comparison
} stc_csio_serial_timer_t;

/**
 ******************************************************************************
 ** \brief CSIO function
 ******************************************************************************/
typedef enum en_csio_func
{
    CsioTx          = 0u,    ///< CSIO TX
    CsioRx          = 1u,    ///< CSIO RX 
    CsioSerialTimer = 2u,    ///< CSIO Serial TImer 
    CsioCsErrOccur  = 3u,    ///< CSIO chip selection error occurrance
} en_csio_func_t;

/**
 ******************************************************************************
 ** \brief CSIO interrupt selection
 ******************************************************************************/
typedef enum stc_csio_irq_sel
{
    CsioTxIrq           = 0u,           ///< CSIO TX interrupt
    CsioRxIrq           = 1u,           ///< CSIO RX interrupt
    CsioTxIdleIrq       = 2u,           ///< CSIO TX idle interrupt
    CsioTxFifoIrq       = 3u,           ///< CSIO TX FIFO interrupt
    CsioCsErrIrq        = 4u,           ///< CSIO chip selection interrupt
    CsioSerialTimerIrq  = 5u,           ///< CSIO serial timer interrupt
} en_csio_irq_sel_t;

/**
 ******************************************************************************
 ** \brief CSIO interrupt enable structure
 ******************************************************************************/
typedef struct stc_csio_irq_en
{
    boolean_t bTxIrq;           ///< CSIO TX interrupt
    boolean_t bRxIrq;           ///< CSIO RX interrupt
    boolean_t bTxIdleIrq;       ///< CSIO TX idle interrupt
    boolean_t bTxFifoIrq;       ///< CSIO TX FIFO interrupt
    boolean_t bCsErrIrq;        ///< CSIO chip selection interrupt
    boolean_t bSerialTimerIrq;  ///< CSIO serial timer interrupt
}stc_csio_irq_en_t;

/**
 ******************************************************************************
 ** \brief CSIO interrupt callback function
 ******************************************************************************/
typedef struct stc_csio_irq_cb
{
    func_ptr_t pfnTxIrqCb;                       ///< CSIO TX interrupt callback function
    func_ptr_t pfnRxIrqCb;                       ///< CSIO RX interrupt callback function
    func_ptr_t pfnTxIdleCb;                      ///< CSIO TX idle interrupt callback function
    func_ptr_t pfnTxFifoIrqCb;                   ///< CSIO TX FIFO interrupt callback function
    func_ptr_t pfnCsErrIrqCb;                    ///< CSIO chip selection interrupt callback function
    func_ptr_t pfnSerialTimerIrqCb;              ///< CSIO serial timer interrupt callback function
}stc_csio_irq_cb_t;

/**
 ******************************************************************************
 ** \brief CSIO status types
 ******************************************************************************/
typedef enum en_csio_status
{
    CsioOverrunError    = 0u,             ///< CSIO overrun error
    CsioRxFull          = 1u,             ///< CSIO RX completion 
    CsioTxEmpty         = 2u,             ///< CSIO TX buffer empty
    CsioTxIdle          = 3u,             ///< CSIO TX idle 
    CsioTxFifoRequest   = 4u,             ///< CSIO TX FIFO request
    CsioCsErrIntFlag    = 5u,             ///< CSIO chip selection error occurrance
    CsioTimerIntFlag    = 6u,             ///< CSIO serial timer interrupt 
}en_csio_status_t;

/**
 ******************************************************************************
 ** \brief CSIO chip selection configuration
 ******************************************************************************/
typedef struct stc_csio_cs
{
    en_cs_pin_sel_t     enCsStartPin;                   ///< Start CS config
    en_cs_pin_sel_t     enCsEndPin;                     ///< End CS config
    en_cs_pin_level_t   enLevel;                        ///< Set Active level of CS
    boolean_t           bActiveHold;                    ///< Hold Active level or not    
    en_cs_timing_clk_t  enClkDiv;                       ///< Clock division configuration for CS
    uint8_t             u8CsSetupDelayTime;             ///< CS setup time
    uint8_t             u8CsHoldDelayTime;              ///< CS Hold time
    uint16_t            u16CsDeselectTime;              ///< CS Deselct time
    uint8_t             u8Scs0TransferByteCnt;          ///< transfer byte count for CS 0
    uint8_t             u8Scs1TransferByteCnt;          ///< transfer byte count for CS 1
    uint8_t             u8Scs2TransferByteCnt;          ///< transfer byte count for CS 2
    uint8_t             u8Scs3TransferByteCnt;          ///< transfer byte count for CS 3
    boolean_t           bScs0En;                        ///< TRUE: SCS0 Enable  
    boolean_t           bScs1En;                        ///< TRUE: SCS1 Enable  
    boolean_t           bScs2En;                        ///< TRUE: SCS2 Enable  
    boolean_t           bScs3En;                        ///< TRUE: SCS3 Enable  
} stc_csio_cs_t;

/**
 ******************************************************************************
 ** \brief CSIO configuration structure
 ******************************************************************************/
typedef struct stc_mfs_csio_config
{
    uint16_t                    u16BaudRate;            ///< Baudrate configuration
    en_csio_ms_mode_t           enMsMode;               ///< 0 : Master and 1 : Slave
    en_csio_act_mode_t          enActMode;              ///< 0 : Normal Synchronous and 1 : SPI
    en_csio_access_width_t      enAWCbit;
    en_csio_sync_wait_time_t    enSyncWaitTime;         ///< Required number of waits for the transmission or reception of continuous data
    en_csio_data_len_t          enDataLength;           ///< data length of transmission and reception data
    en_csio_data_dir_t          enBitDirection;         ///< 0 : LsbFirst and  1 : MsbFirst
    boolean_t                   bInvertClk;             ///< Serial clock invert bit
    stc_csio_serial_timer_t*    pstcSerialTimer;        ///< Serial Timer congfiguration
    stc_csio_cs_t*              pstcCsConfig;           ///< Chip select configuration
    stc_mfs_fifo_config_t*      pstcFifoConfig;         ///< FIFO config
    stc_csio_irq_en_t*          pstcIrqEn;              ///< Interrupt Enable
    stc_csio_irq_cb_t*          pstcIrqCb;              ///< Interrupt call back function
} stc_mfs_csio_config_t;

/**
 ******************************************************************************
 ** \brief CSIO status structure
 ******************************************************************************/
typedef struct stc_mfs_csio_status
{
    boolean_t bOREStatus;          ///< CSIO parity error
    boolean_t bRDRFStatus;         ///< CSIO receive buffer full
    boolean_t bTDREStatus;         ///< CSIO tranfer buffer empty
    boolean_t bTBIStatus;          ///< CSIO tranfer idle status
    boolean_t bFDRQStatus;         ///< CSIO transfer FIFO request
    boolean_t bCSEStatus;          ///< CSIO chip selection error occurrance
    boolean_t bTINTStatus;         ///< CSIO serial timer interrupt flag
} stc_mfs_csio_status_t;

/**
******************************************************************************
** \brief  CSIO linked data list.
**
** Used to connect various different buffer sources that shall be transmitted
** during the same transaction, e.g. Ethernet packet with different buffers for
** for the ISO/OSI layer headers (external Ethernet MAC connected via CSIO).
******************************************************************************/
//typedef struct stc_csio_linked_data_list
//{
//    const uint16_t* pu16Data;                     ///< Pointer to transfer data
//    uint16_t u16ListItemDataSize;               ///< Size of transfer data
//    struct stc_csio_linked_data_list* pstcNext;  ///< Pointer to next #stc_csio_linked_data_list_t item or NULL if end of list
//}stc_csio_linked_data_list_t;
    
/*****************************************************************************/
/* Global function prototypes ('extern', definition in C source)             */
/*****************************************************************************/

extern en_result_t Mfs_Csio_DisableIrq(volatile stc_cpg_mfsn_csio_t* pstcCsio, en_csio_irq_sel_t enIrqSel);
extern en_result_t Mfs_Csio_EnableIrq(volatile stc_cpg_mfsn_csio_t* pstcCsio, en_csio_irq_sel_t enIrqSel);
extern en_result_t Mfs_Csio_SetBaudRate(volatile stc_cpg_mfsn_csio_t* pstcCsio, uint16_t u16BaudRate);
extern en_result_t Mfs_Csio_Init (volatile stc_cpg_mfsn_csio_t * pstcCsio, const stc_mfs_csio_config_t * pstcConfig);
extern en_result_t Mfs_Csio_DeInit(volatile stc_cpg_mfsn_csio_t* pstcCsio);
extern en_result_t Mfs_Csio_SetTimerCompareValue(volatile stc_cpg_mfsn_csio_t* pstcCsio, uint16_t u16CompareValue);
extern en_result_t Mfs_Csio_SetCsHoldStatus(volatile stc_cpg_mfsn_csio_t* pstcCsio, boolean_t bHold);
extern stc_mfs_csio_status_t* Mfs_Csio_GetStatus(volatile stc_cpg_mfsn_csio_t* pstcCsio);
extern en_result_t Mfs_Csio_DisableFunc(volatile stc_cpg_mfsn_csio_t* pstcCsio, en_csio_func_t enFunc);
extern en_result_t Mfs_Csio_EnableFunc(volatile stc_cpg_mfsn_csio_t* pstcCsio, en_csio_func_t enFunc);
extern en_result_t Mfs_Csio_ClrStatus(volatile stc_cpg_mfsn_csio_t* pstcCsio, en_csio_status_t enStatus);
extern en_result_t Mfs_Csio_ResetFifo (volatile stc_cpg_mfsn_csio_t* pstcCsio, en_mfs_fifo_t enFifo);
extern en_result_t Mfs_Csio_SetFifoCount(volatile stc_cpg_mfsn_csio_t* pstcCsio, en_mfs_fifo_t enFifo, uint8_t u8Count);
extern uint8_t     Mfs_Csio_GetFifoCount(volatile stc_cpg_mfsn_csio_t* pstcCsio, en_mfs_fifo_t enFifo);
extern en_result_t Mfs_Csio_FifoEnable (volatile stc_cpg_mfsn_csio_t * pstcCsio);
extern en_result_t Mfs_Csio_SetTimerTransferByteCount(volatile stc_cpg_mfsn_csio_t* pstcCsio, uint8_t u8ByteCnt);
extern en_result_t Mfs_Csio_HalfDuplexComm(volatile stc_cpg_mfsn_csio_t* pstcCsio, en_cs_pin_sel_t enDeviceNumber, uint16_t* pu16TxData, uint16_t u16TxSize, uint16_t* pu16RxData, uint16_t u16RxSize);
extern en_result_t Mfs_Csio_FullDuplexComm(volatile stc_cpg_mfsn_csio_t* pstcCsio, en_cs_pin_sel_t enDeviceNumber, uint16_t* pu16TxData, uint16_t u16TxSize, uint16_t* pu16RxData);

/** @} */

#endif /* _MFS_CSIO_H_ */
