/******************************************************************************
* $Revision: 423 $
* $Date:: 2017-04-07 16:03:30 +0900#$
*****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************/
/** \file port.h
**
** Sample program for port configurations.
**
** History:
**   - 2014-05-16  0.01  HS  Initial version for Traveo
**   - 2014-10-17  0.02  CEy Added functions to configure several port pins
**                           and resource inputs, added GPIO Toggle function
*****************************************************************************/

#ifndef __PORT_H__
    #define __PORT_H__

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "base_types.h"

/**
*****************************************************************************
** \defgroup PortGroup Port functions (PORT)
**
** \brief This section describes the interface for the Port functions.
**
** Provided functions of Port module:
**
** - Port_SetPinConfig()
** - Port_SelectInputPort()
** - Port_SelectInputSource()
** - Port_EnableInput()
** - Port_SetPortPinLevel()
** - Port_GetPortPinLevel()
**
** Port_SetPinConfig() is used to set the configuration of a single port pin
** with parameter pstcPinConfig of type #stc_port_pin_config_t.
** Port_SelectInputPort() is used to select the input port of a function.
** \n e.g. select serial data input pin(SIN) from P411 or P420.\n
** Port_SelectInputSource() is used to select the input resource of a function.
** \n e.g. select a FRT ch that is input to OCU0.\n
** Port_EnableInput() is used to enable port inputs.
** Port_SetPortPinLevel() is used to set a port pin level.
** Port_GetPortPinLevel() is used to get a port pin level.
**
** If application use this module, it has to use these APIs by combining.
*****************************************************************************/
/*! @{ */

/*****************************************************************************/
    /* Global pre-processor symbols/macros ('#define')                           */
/*****************************************************************************/

/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/
/**
******************************************************************************
** \brief Port output function selection
**
    ** This enumeration lists all the possible output functions of a port.
    **
    ** \note Refer to POF in the Hardware Manual.
    ******************************************************************************/
    typedef enum en_port_output_resource_function
    {
        PortOutputResourceGPIO = 0,     /*!< Resource A output */
        PortOutputResourceB = 1,        /*!< Resource B output */
        PortOutputResourceC = 2,        /*!< Resource C output */
        PortOutputResourceD = 3,        /*!< Resource D output */
        PortOutputResourceE = 4,        /*!< Resource E output */
        PortOutputResourceF = 5,        /*!< Resource F output */
        PortOutputResourceG = 6,        /*!< Resource G output */
        PortOutputResourceH = 7         /*!< Resource H output */
    } en_port_output_resource_function_t;
    
    /**
    ******************************************************************************
    ** \brief Port output drive selection
    **
    ** This enumeration lists all the possible output drive strengths.
    **
    ** \note The PortOutputDrive settings are related to the specific circuit
    ** type of the used MCU. PLs. refer to ODR in the Hardware Manual to get information
    ** about the availble options for the used port.
    ******************************************************************************/
    typedef enum en_port_output_drive
    {
        PortOutputDriveA = 0,    /*!< Type A */
        PortOutputDriveB = 1,    /*!< Type B */
        PortOutputDriveC = 2,    /*!< Type C */
        PortOutputDriveD = 3     /*!< Type D */
    } en_port_output_drive_t;
    
    
    /**
    ******************************************************************************
    ** \brief Port pull-up/down selection
    **
    ** This enumeration lists all the possible pull-up/down for input mode
    ******************************************************************************/
    typedef enum en_port_pull_resistor
    {
        PortPullResistorNone = 0,    /*!< No pull up/down */
        PortPullResistorDown = 1,    /*!< Pull down */
        PortPullResistorUp = 2       /*!< Pull up */
    } en_port_pull_resistor_t;
    
    /**
    ******************************************************************************
    ** \brief Port Input Level selection
    **
    ** This enumeration lists all the possible input levels for input mode
    ******************************************************************************/
    typedef enum en_port_input_level
    {
        PortInputLevelCmosA = 0,         /*!< Type A */
        PortInputLevelAutomotive = 1,    /*!< Type B */
        PortInputLevelTTL = 2,           /*!< Type C */
        PortInputLevelCmosB = 3          /*!< Type D */
    } en_port_input_level_t;
    
    /**
    *****************************************************************************
    ** \brief GPIO direction
    **
    ** Set pin to input or output
    *****************************************************************************/
    typedef enum en_port_gpio_direction
    {
        PortGpioInput  = 0,       /*!< Set data direction to input */
        PortGpioOutput = 1        /*!< Set data direction to output */
    } en_port_gpio_direction_t;
    
    /**
    *****************************************************************************
    ** \brief Port pin level
    **
    *****************************************************************************/
    typedef enum en_port_gpio_level
    {
        PortGpioLow = 0,     /*!< Output "L" */
        PortGpioHigh = 1     /*!< Output "H" */
    } en_port_gpio_level_t;
    
    /**
    *****************************************************************************
    ** \brief Port input source selection.(Corresponds to PORTSEL:Resource selection bit)
    **
    ** \note Refer to PORTSEL in the Hardware Manua.
    *****************************************************************************/
    typedef enum en_port_input_port
    {
        PortInputPortA = 0,     /*!< Source A */
        PortInputPortB = 1,     /*!< Source B */
        PortInputPortC = 2,     /*!< Source C */
        PortInputPortD = 3,     /*!< Source D */
        PortInputPortE = 4,     /*!< Source E */
        PortInputPortF = 5,     /*!< Source F */
        PortInputPortG = 6,     /*!< Source G */
        PortInputPortH = 7,     /*!< Source H */
        PortInputPortI = 8,     /*!< Source I */
        PortInputPortJ = 9,     /*!< Source J */
        PortInputPortK = 10,    /*!< Source K */
        PortInputPortL = 11,    /*!< Source L */
        PortInputPortM = 12,    /*!< Source M */
        PortInputPortN = 13,    /*!< Source N */
        PortInputPortO = 14,    /*!< Source O */
        PortInputPortP = 15     /*!< Source P */
    } en_port_input_port_t;
    
    
    /**
    *****************************************************************************
    ** \brief Port input Resource selection.(Corresponds to RESSEL: Resource selection bit)
    **
    ** \note Refer to RESSEL in the Hardware Manual.
    *****************************************************************************/
    typedef enum en_port_input_source
    {
        PortInputSourceA = 0,     /*!< Source A */
        PortInputSourceB = 1,     /*!< Source B */
        PortInputSourceC = 2,     /*!< Source C */
        PortInputSourceD = 3,     /*!< Source D */
        PortInputSourceE = 4,     /*!< Source E */
        PortInputSourceF = 5,     /*!< Source F */
        PortInputSourceG = 6,     /*!< Source G */
        PortInputSourceH = 7,     /*!< Source H */
        PortInputSourceI = 8,     /*!< Source I */
        PortInputSourceJ = 9,     /*!< Source J */
        PortInputSourceK = 10,    /*!< Source K */
        PortInputSourceL = 11,    /*!< Source L */
        PortInputSourceM = 12,    /*!< Source M */
        PortInputSourceN = 13,    /*!< Source N */
        PortInputSourceO = 14,    /*!< Source O */
        PortInputSourceP = 15     /*!< Source P */
    } en_port_input_source_t;
    
    /**
    ******************************************************************************
    ** \brief Port Pin Configuration.
    **
    ** This struct is used to configure a port pin .
    ******************************************************************************/
    typedef struct stc_port_pin_config
    {
        /** Output function. See #en_port_output_resource_function_t. */
        en_port_output_resource_function_t enOutputFunction;
        /** Output drive. See #en_port_output_drive_t. */
        en_port_output_drive_t             enOutputDrive;
        /** Pull-up/down selection in input status. See #en_port_pull_resistor_t. */
        en_port_pull_resistor_t            enPullResistor;
        /** Input level selection. See #en_port_pull_resistor_t. */
        en_port_input_level_t              enInputLevel;
        /** Input enable setting. TRUE = Pin input is enabled, FALSE = Pin input is disabled or analog input */
        boolean_t                          bInputEnable;
        /** Noise filter setting. TRUE = Enable the noise filter, FALSE = Disable the noise filter */
        boolean_t                          bNoiseFilterEnable;
        /** Direction selection. See #en_port_gpio_direction_t. */
        en_port_gpio_direction_t           enGpioDirection;
        /** Initial output level. See #en_port_gpio_level_t. */
        en_port_gpio_level_t               enGpioInitOutputLevel;
    } stc_port_pin_config_t;
    
    
    /**
    ******************************************************************************
    ** \brief Port Pin Configuration for a collection of ports which are identified
    **        by their port and pin number
    ******************************************************************************/
    typedef struct stc_port_pin_set
    {
        uint8_t                u8PortIndex;        /*!< Index of port to set.(e.g. "0" of the P011) */
        uint8_t                u8PinNumber;        /*!< Channel number to operate on. (e.g."11" of the P011) */
        stc_port_pin_config_t  stcPortPinConfig;   /*!< Port Pin Configuration. See #stc_port_pin_config_t. */
    } stc_port_pin_set_t;
    
    /**
    ******************************************************************************
    ** \brief Resource Input Configuration for a collection of input signals
    ******************************************************************************/
    typedef struct stc_input_resource_set
    {
        volatile un_ric_resinn_t*  punRicResinReg;    /*!< Pointer of the Resource Input Setting Register (RIC_RESIN) */
        en_port_input_port_t       enInputPort;       /*!< Port input source selection (RIC_RESIN.PORTSEL) */
        en_port_input_source_t     enInputSource;     /*!< Port input Resource selection (RIC_RESIN.RESTSEL) */
    } stc_input_resource_set_t;
    
    /*****************************************************************************/
    /* Global variable declarations ('extern', definition in C source)           */
    /*****************************************************************************/
    
    /*****************************************************************************/
    /* Global function prototypes ('extern', definition in C source)             */
    /*****************************************************************************/
    extern  en_result_t Port_SetPinConfig(uint8_t u8PortIndex,
    uint8_t u8PinNumber,
    const stc_port_pin_config_t* pstcPinConfig);
    extern  void Port_SelectInputPort(volatile un_ric_resinn_t*  punRicResinReg,
    en_port_input_port_t enInputPort);
    extern  void Port_SelectInputSource(volatile un_ric_resinn_t*  punRicResinReg,
    en_port_input_source_t enInputSource);
    extern  void Port_EnableInput(void);
    extern  en_result_t Port_SetPortPinLevel(uint8_t u8PortIndex,
    uint8_t u8PinNumber,
    en_port_gpio_level_t enLevel);
    extern  en_result_t Port_GetPortPinLevel(uint8_t u8PortIndex,
    uint8_t u8PinNumber,
    en_port_gpio_level_t * enLevel);
    extern  en_result_t Port_TogglePortPin(uint8_t u8PortIndex,
    uint8_t u8ChNumber);
    extern  en_result_t Port_ConfigurePortPins(const stc_port_pin_set_t astcPortPins[], uint32_t u32Count);
    extern  en_result_t Port_ConfigureInputResources(const stc_input_resource_set_t astcInputResources[], uint32_t u32Count);
    
    extern  en_result_t Port_GetOutPortPinLevel(uint8_t u8PortIndex,uint8_t u8ChNumber,en_port_gpio_level_t * enLevel);
      
    //void     WPREG_PPC(unsigned long n,unsigned long d);


#endif /* __PORT_H__ */
