/******************************************************************************
* $Revision: 423 $
* $Date:: 2017-04-07 16:03:30 +0900#$
*****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************
** \file ppg.h
**
** Headerfile for PPG
**
** History:
**   - 2015-01-29  0.01  MAs  Initial version for Traveo
*****************************************************************************/

#ifndef __ppg_H__
    #define __ppg_H__

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "pdl.h"

/* C binding of definitions if building with C++ compiler */
#ifdef __cplusplus
    extern "C"
{
#endif


/*****************************************************************************/
/* Global pre-processor symbols/macros ('define')                            */
/*****************************************************************************/


/**
*****************************************************************************
** \defgroup ppgGroup Programmable Pulse Generation (PPG)
**
** \brief This section describes the interface for the Programmable Pulse Generation .
**
** Provided functions of ppg module:
** - Bt_Ppg_Init() (Initialization of ppg module)
** - Bt_Ppg_Start() (Set the cycle, the duty and to start a ppg)
** - Bt_Ppg_SetDuty16() (Set duty cycle)
** - Bt_Ppg_Stop() (Stop a ppg)
** - Bt_Ppg_EnableInterupt (Enable interrupt)
** - Bt_Ppg_DisableInterupt (Disable Interrupt)
** - Bt_Ppg_DeInit() (DeInitialize ppg)
**
**
** Programmable Pulse Generators (PPGs) are used to obtain one-shot
** (rectangular wave) output.
** With their software programmable cycle and duty capability, possibility to
** postpone the start of PWM output signal generation by software and trigger
** an AD conversion, the PPGs comfortably fit into a broad range of
** applications. To increase flexibility, the PPGs can be configured as a
** 16-bit resolution PPG channel.
**
** The first thing to do is to initialize the PPG by calling Bt_Ppg_Init().
** The configuration is depending on on parameters(stc_ppg_config).
** Calling Ppg_Start() starts the PPG.
** Next the duty cycle (i.e. how much of the whole cycle the PPG signal is
** high) has to be set. Depending on the used resolution this has to be done
** with Ppg_SetDuty16().
**
** If the duty should be altered during PPG operation, this can be done with
** Bt_Ppg_SetDuty16(). The new value comes into effect when the current cycle is
** finished.
**
** \note Please read the hardware manual for a detailed description of the
**       Programmable Pulse Generator
**
*****************************************************************************/
//@{

/*****************************************************************************
* Global type definitions
*****************************************************************************/

/**
*****************************************************************************
** \brief ppg trigger edge selection
*****************************************************************************/
    typedef enum en_ppg_trigger_input_edge_selection
    {
        PpgSoftwareOnly = 0x0,   /*!< ppg is triggered by software */
        PpgRisingEdge   = 0x1,   /*!< ppg is triggered by a rising edge */
        PpgFallingEdge  = 0x2,   /*!< ppg is triggered by a falling edge */
        PpgBothEdges    = 0x3    /*!< ppg is triggered by a rising or a falling edge */
    } en_ppg_trigger_input_edge_selection_t ;
    
    
    
    /**
    *****************************************************************************
    ** \brief ppg clock prescaler
    *****************************************************************************/
    typedef enum en_ppg_clock_prescaler
    {
        PpgDiv1     = 0x0, /*!< Internal clock                      */
        PpgDiv4     = 0x1, /*!< Internal clock divided by 4         */
        PpgDiv16    = 0x2, /*!< Internal clock divided by 16        */
        PpgDiv128   = 0x3, /*!< Internal clock divided by 128       */
        PpgDiv256   = 0x4, /*!< Internal clock divided by 256       */
        PpgExClkRis = 0x5, /*!< External clock (rising-edge event)  */
        PpgExClkfal = 0x6, /*!< External clock (falling-edge event) */
        PpgExClkbot = 0x7, /*!< External clock (both-edges event)   */
        PpgDiv512   = 0x8, /*!< Internal clock divided by 512       */
        PpgDiv1024  = 0x9, /*!< Internal clock divided by 1024      */
        PpgDiv2048  = 0xa, /*!< Internal clock divided by 2048      */
        PpgDiv2     = 0xb, /*!< Internal clock divided by 2         */
        PpgDiv8     = 0xc, /*!< Internal clock divided by 8         */
        PpgDiv32    = 0xd, /*!< Internal clock divided by 32        */
        PpgDiv64    = 0xe  /*!< Internal clock divided by 64        */
    } en_ppg_clock_prescaler_t;
    
    
    
    /**
    *****************************************************************************
    ** \brief Cycle configuration
    *****************************************************************************/
    typedef struct stc_ppg_cycle_config
    {
        uint16_t           u16HighWidth ;               /*!< ppg High PRLH*/
        uint16_t           u16LowWidth  ;               /*!< ppg low PRLL*/
    } stc_ppg_cycle_config_t ;
    
    /**
    *****************************************************************************
    ** \brief ppg configuration
    *****************************************************************************/
    typedef struct stc_ppg_config
    {
        /** Sets clock prescaler. See #en_ppg_clock_prescaler_t. */
        en_ppg_clock_prescaler_t                enPrescaler;
        /** Trigger restarts ppg operation. TRUE: Enable restart, FALSE: Disable restart */
        boolean_t                               bRestartEnable;
        /** Pulse output mask. TRUE: Fixed at "L" output, FALSE: Normal output */
        boolean_t                               bPulseOutputMaskEnable;
        /** Select edge that triggers ppg. See #en_ppg_trigger_input_edge_selection_t. */
        en_ppg_trigger_input_edge_selection_t   enTriggerInputEdgeSelection;
        /** Normal or Inverse output polarity. TRUE: Inverse polarity, FALSE: Normal polarity */
        boolean_t                               bOutputPolarityInverse;
        /** Mode selection. TRUE: One shot, FALSE: Continuous */
        boolean_t                               bOneShotOperation;
        /** Callback for trigger interrupt */
        func_ptr_t                              pfnTriggerCallback;
        /** Callback for Underflow interrupt */
        func_ptr_t                              pfnUnderflowCallback;
        
    } stc_ppg_config_t ;
    
    
    /**
    *****************************************************************************
    ** \brief Datatype for holding internal data needed for ppg
    **
    ** This struct is used to store the interrupt callback function for the ppg
    *****************************************************************************/
    typedef struct stc_ppg_intern_data
    {
        func_ptr_t       pfnTriggerCallback ;     /*!< Interrupt callback function pointer */
        func_ptr_t       pfnUnderflowCallback ;   /*!< Interrupt callback function pointer */
    } stc_ppg_intern_data_t ;
    
    
    /*****************************************************************************/
    /* Global variable declarations ('extern', definition in C source)           */
    /*****************************************************************************/
    
    /*****************************************************************************
    * Global function prototypes
    *****************************************************************************/
    extern en_result_t Bt_Ppg_Init( volatile stc_btn_ppg_t* pstcppg,
    stc_ppg_config_t* pstcConfig);
    
    extern en_result_t Bt_Ppg_Start( volatile stc_btn_ppg_t* pstcppg,
    stc_ppg_cycle_config_t* pstcCycle,
    boolean_t bSwTrigger );
    
    extern en_result_t Bt_Ppg_SetDuty16( volatile stc_btn_ppg_t* pstcppg,
    uint16_t u16HighWidth,
    uint16_t u16LowWidth);
    
    extern en_result_t Bt_Ppg_Stop( volatile stc_btn_ppg_t* pstcppg );
    
    //For interupts
    extern en_result_t Bt_Ppg_EnableInterupt(volatile stc_btn_ppg_t* pstcppg,
    boolean_t bEnableTrigger,
    boolean_t bEnableUnderflow);
    
    extern en_result_t Bt_Ppg_DisableInterupt(volatile stc_btn_ppg_t* pstcppg,
    boolean_t bDisableTrigger,
    boolean_t bDisableUnderflow);
    
    extern en_result_t Bt_Ppg_DeInit(volatile stc_btn_ppg_t* pstcppg);
    
//@}

#ifdef __cplusplus
}
#endif

#endif /* __ppg_H__ */
/*****************************************************************************/
/* EOF (not truncated)                                                       */
/*****************************************************************************/
