/******************************************************************************
 * $Revision: 423 $
 * $Date:: 2017-04-07 16:03:30 +0900#$
 *****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************/
/** \file swdg.h
 **
 ** Sample program for software watchdog timer.
 **
 ** History:
 **   - 2014-05-16  0.01  HS  Initial version for Traveo
 *****************************************************************************/

#ifndef __SWDG_H__
#define __SWDG_H__

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "base_types.h"

/* C binding of definitions if building with C++ compiler */
#ifdef __cplusplus
extern "C" {
#endif

/**
 *****************************************************************************
 ** \defgroup SwdgGroup Software watchdog (SWDG)
 **
 ** \brief This section describes the interface for the Software Watchdog.
 **
 ** Provided functions of Swdg module:
 **
 ** - Swdg_Init()
 ** - Swdg_GetResetCause()
 ** - Swdg_Clear()
 ** - Swdg_GetCounterValue()
 **
 ** Swdg_Init() is used to initialize Swdg with parameter pstcConfig of type #stc_swdg_config_t.
 ** This function can be used only once after a reset.
 ** Swdg_GetResetCause() is used to get Swdg reset cause. Must be use before
 ** calling Swdg_Init().
 ** Swdg_Clear() is used to clear watchdog counter.
 ** Swdg_GetCounterValue() is used to current watchdog counter.
 **
 ** If application use this module, it has to use these APIs by combining.
 **
 *****************************************************************************/
/*! @{ */


/*****************************************************************************/
/* Global pre-processor symbols/macros ('#define')                           */
/*****************************************************************************/


/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/
/**
 *****************************************************************************
 ** \brief SWDG clock selection
 **
 ** The SWDG counter starts on Fast RC clock on startup
 *****************************************************************************/
typedef enum en_swdg_clk_selection
{
    SwdgRcClk       = 0,       /*!< Fast RC clock will be used for SWDG */
    SwdgSlowRcClk   = 1,       /*!< Slow RC clock will be used for SWDG */
                               /* "2" is prohibit for this device */
    SwdgMainClk     = 3        /*!< Main clock will be used for SWDG */
} en_swdg_clk_selection_t;


/**
 *****************************************************************************
 ** \brief RUN/PSS mode specific SWDG configuration
 *****************************************************************************/
typedef struct stc_swdg_mode_settings
{
    boolean_t bWdgEnable;         /*!< Enable or disable SWDG */
    uint32_t u32WindowLowerLimit; /*!< Value for SWDG counter lower window limit */
    uint32_t u32WindowUpperLimit; /*!< Value for SWDG counter upper window limit */
} stc_swdg_mode_settings_t;


/**
 *****************************************************************************
 ** \brief SWDG configuration
 *****************************************************************************/
typedef struct stc_swdg_config
{
    /*! SWDG settings specific to RUN mode. See #stc_swdg_mode_settings_t */
    stc_swdg_mode_settings_t stcRunModeSettings;
    /*! SWDG settings specific to PSS mode. See #stc_swdg_mode_settings_t */
    stc_swdg_mode_settings_t stcPssModeSettings;
    /*! Allow SWDG clock to stop in PSS mode */
    boolean_t               bAllowClockStopInPss;
    /*! See #en_swdg_clk_selection_t */
    en_swdg_clk_selection_t enClockSelection;
    /*! Watchdog counter monitor bit output selection bits */
    uint8_t                 u8OBSSelection;
    /*! Key value for trigger key 0 */
    uint8_t                 u8TriggerKey0;
    /*! Key value for trigger key 1 */
    uint8_t                 u8TriggerKey1;
    /*! Delay cycle for WDG counter to wait until reset/NMI in case of WDG error */
    uint16_t                u16ResetDelay;
    /*! Callback to get prior warning notification in case of WDG error */
    func_ptr_t              pfnPreWarnCallback;
    /*! Callback to get NMI notification in case of WDG error */
    func_ptr_t              pfnNmiCallback;
} stc_swdg_config_t;

/**
 *****************************************************************************
 ** \brief SWDG reset cause
 *****************************************************************************/
typedef struct stc_swdg_reset_cause
{
    /*! SWDG counter was triggered while SWDG was still unlocked */
    boolean_t bSwdgTriggerWhileUnlocked;
    /*! SWDG counter was triggered before lower limit was reached */
    boolean_t bSwdgTriggerTooEarly;
    /*! SWDG counter reached upper limit */
    boolean_t bSwdgNotTriggered;
    /*! The trigger functions were called in the wrong order */
    boolean_t bSwdgTriggerSequenceViolated;
    /*! The key value for trigger 0 or 1 was wrong */
    boolean_t bSwdgWrongTriggerValue;
} stc_swdg_reset_cause_t;

/*****************************************************************************/
/* Global variable declarations ('extern', definition in C source)           */
/*****************************************************************************/

/*****************************************************************************/
/* Global function prototypes ('extern', definition in C source)             */
/*****************************************************************************/
en_result_t Swdg_Init(const stc_swdg_config_t *pstcConfig);
en_result_t Swdg_GetResetCause(stc_swdg_reset_cause_t* pstcResetCause);
en_result_t Swdg_Clear(uint8_t u8TrgVal0, uint8_t u8TrgVal1);
uint32_t Swdg_GetCounterValue(void);

/*! @} */
#ifdef __cplusplus
}
#endif

#endif /* __SWDG_H__ */
