/*===========================================================================*/
/* Module      = cpu.h                                                       */
/* Version     = V1.10                                                       */
/*===========================================================================*/
/*                                  COPYRIGHT                                */
/*===========================================================================*/
/* Copyright (c) 2014 by Renesas Electronics Europe GmbH,                    */
/*               a company of the Renesas Electronics Corporation            */
/*===========================================================================*/
/* Purpose:      Definition of Macros to access CPU System Registers         */
/*                                                                           */
/*===========================================================================*/
/*                                                                           */
/* Warranty Disclaimer                                                       */
/*                                                                           */
/* Because the Product(s) is licensed free of charge, there is no warranty   */
/* of any kind whatsoever and expressly disclaimed and excluded by Renesas,  */
/* either expressed or implied, including but not limited to those for       */
/* non-infringement of intellectual property, merchantability and/or         */
/* fitness for the particular purpose.                                       */
/* Renesas shall not have any obligation to maintain, service or provide bug */
/* fixes for the supplied Product(s) and/or the Application.                 */
/*                                                                           */
/* Each User is solely responsible for determining the appropriateness of    */
/* using the Product(s) and assumes all risks associated with its exercise   */
/* of rights under this Agreement, including, but not limited to the risks   */
/* and costs of program errors, compliance with applicable laws, damage to   */
/* or loss of data, programs or equipment, and unavailability or             */
/* interruption of operations.                                               */
/*                                                                           */
/* Limitation of Liability                                                   */
/*                                                                           */
/* In no event shall Renesas be liable to the User for any incidental,       */
/* consequential, indirect, or punitive damage (including but not limited    */
/* to lost profits) regardless of whether such liability is based on breach  */
/* of contract, tort, strict liability, breach of warranties, failure of     */
/* essential purpose or otherwise and even if advised of the possibility of  */
/* such damages. Renesas shall not be liable for any services or products    */
/* provided by third party vendors, developers or consultants identified or  */
/* referred to the User by Renesas in connection with the Product(s) and/or  */
/* the Application.                                                          */
/*                                                                           */
/*===========================================================================*/
/* Environment:                                                              */
/*         Device:         RH850G3M core devices                             */
/*                         RH850G3K core devices                             */
/*===========================================================================*/

#ifndef __CPU_H__
#define __CPU_H__

#ifdef __ghs
#pragma ghs startnomisra

#define _INLINE_ __inline


/*
 * Utilize the GHS specific intrinsic functions
 *       __LDSR( n,m,v)
 *    v= __STSR((n,m)
 */
#include <..\include\v800\v800_ghs.h>
#else
/* Other Tool vendors might have different approaches ... */
#define _INLINE_ inline
#endif

#ifndef u32_T
#define u32_T unsigned long
#endif

/* Basic control registers type definitions */
typedef union {
    u32_T     psw;               /* PSW, 32-bit access */
    struct {
        u32_T   Z:1,             /* Bit 0, zero */
                S:1,             /* Bit 1, sign */
                OV:1,            /* Bit 2, overflow */
                CY:1,            /* Bit 3, carry */
                SAT:1,           /* Bit 4, saturated */
                ID:1,            /* Bit 5, interrupt enable (0:enabled) */
                EP:1,            /* Bit 6, exception in process (0:interupt is processed */
                NP:1,            /* Bit 7, FE exception enable (0:enabled) */
                rsvd1:1,         /* Bit 8, reserved */
                DBG:3,           /* Bit 9-11, debug */
                rsvd2:3,         /* Bit 12-14, reserved */
                EBV:1,           /* Bit 15, 0:RBASE or 1:EBASE is used as exception processing base */
                CU0:1,           /* Bit 16, FPU */
                CU1:1,           /* Bit 17, reserved */
                CU2:1,           /* Bit 18, reserved */
                rsvd3:11,        /* Bit 19-29, reserved */
                UM:1,            /* Bit 30, User mode (0:supervisor, 1:user mode) */
                rsvd4:1;         /* Bit 31, reserved */
    };
} PSW_T;

typedef struct {
    u32_T   ctpsw;               /* CTPSW, 32-bit access */
    struct {                    
        u32_T   Z:1,             /* Bit 0, zero */
                S:1,             /* Bit 1, sign */
                OV:1,            /* Bit 2, overflow */
                CY:1,            /* Bit 3, carry */
                SAT:1;           /* Bit 4, saturated */
    };
} CTPSW_T;

typedef struct {
    const u32_T htcfg0;          /* HTCFG0, 32-bit access */
    struct {
        const u32_T rsvd1:16,    /* Bit 0-15, zero */
                    PEID:3,      /* processor entity identifier */
                    rsvd2:13;    /* Bit19-31, reserved */
    };
} HTCFG0_T;

typedef struct {
    u32_T   asid;                /* ASID, 32-bit access */
    struct {
        u32_T  ASID:10,          /* Bit 0-15, Address space identifier */
               rsvd1:22;         /* Bit10-31, reserved */
    };
} ASID_T;

typedef struct {
    u32_T   mei;                 /* MEI, 32-bit access */
    struct {                     
        u32_T   RW:1,            /* Bit 0, Address space identifier */
                ITYPE:5,         /* Bit 1-5, Instruction type */
                rsvd1:2,         /* Bit 6-7, reserved */
                U:1,             /* Bit 8, sign extension method of ITYPE */
                DS:2,            /* Bit 9-10, Acces size 0:byte,1:short,2:long,3:long long */
                rsvd2:5,         /* Bit 11-15, reserved */
                REG:5,           /* Bit 16-20, source/destination register accessed */
                rsvd3:11;        /* Bit 21-31, reserved */
    };
} MEI_T;

typedef union {
    u32_T     pid;               /* PID, 32-bit access */
    struct {                     
        u32_T   VID:8,           /* Bit 0-7, Instruction type */
                MPU:1,           /* Bit 8, MPU available */
                SFPU:1,          /* Bit 9, single precision FPU available */
                DFPU:1;          /* Bit 10, double precision FPU available */
    };
} PID_T;

typedef union {
    u32_T     mctl;              /* MCTL, 32-bit access */
    struct {
       u32_T  UIC:1,             /* Bit 0, 1:allow EI/DI instructions in user mode */
              MA:1;              /* Bit 1, Operate misalign access (if H/W supports) */
    };
} MCTL_T;

typedef union {
    u32_T mcfg0;                 /* MCFG0, 32-bit access */
    struct {                     
        u32_T HVTE:1,            /* Bit 0, HVTRAP allowed (R) */
              HVCE:1,            /* Bit 1, HVCALL allowed  (must be 1!) (R) */
              HVP:1,             /* Bit 2, reserved, always 0 (R) */
              rsvd1:13,          /* Bit 3-15, reserved */
              SPID:8,            /* System protection number (h/w dependant!) */
              rsvd2:8;           /* Bit 24-31, reserved */
    };
} MCFG0_T;

typedef union {
    u32_T   sccfg;               /* SCCFG, 32-bit access */
    struct {                    
        u32_T   SIZE:8,          /* Bit 0-8, max. number of syscall entries */
                rsvd1:24;        /* Bit 8-31, resevred */
    };                          
} SCCFG_T;

typedef union {
    u32_T    ispr;               /* ISPR, 32-bit access */
    struct {
        u32_T     ISP:16,        /* Bit 0-15, interrupts priorities pending */
                  rsvd1:16;      /* Bit 16-31, resevred */
    };
} ISPR_T;


typedef union {
    u32_T   mpm;                 /* MPM, 32-bit access */
    struct {                     
        u32_T   MPE:1,           /* Bit 0, 1:Enable */
                SVP:1,           /* Bit 1, restrict access in SV mode */
                rsvd1:6,         /* Bit 2-7, reserved */
                DR:1,            /* Bit 8, 1:enable read from address outside p-area */
                DW:1,            /* BIt 9, 1:enable write to address outside p-area */
                DX:1,            /* BIt 10, 1:enable execute from address outside p-area */
                rsvd2:21;        /* Bit 11-31, reserved */
    };
} MPM_T;

typedef union {
    u32_T   mprc;                /* MPRC, 32-bit access */
    struct {                    
        u32_T   E0:1,            /* Bit  0, protection area enable bits 1:enables */
                E1:1,            /* Bit  1 */
                E2:1,            /* Bit  2 */
                E3:1,            /* Bit  3 */
                E4:1,            /* Bit  4 */
                E5:1,            /* Bit  5 */
                E6:1,            /* Bit  6 */
                E7:1,            /* Bit  7 */
                E8:1,            /* Bit  8 */
                E9:1,            /* Bit  9 */
                E10:1,           /* Bit 10 */
                E11:1,           /* Bit 11 */
                E12:1,           /* Bit 12 */
                E13:1,           /* Bit 13 */
                E14:1,           /* Bit 14 */
                E15:1;           /* Bit 15 */
    };
} MPRC_T;

typedef union {
    u32_T   fpsr;                /* FPSR, 32-bit access */
    struct {                    
        u32_T   XPI:1,           /* Bit  0, preservation bit */
                XPU:1,           /* Bit  1, preservation bit */
                XPO:1,           /* Bit  2, preservation bit */
                XPZ:1,           /* Bit  3, preservation bit */
                XPV:1,           /* Bit  4, preservation bit */
                                 
                XEI:1,           /* Bit  5, enable bit */
                XEU:1,           /* Bit  6, enable bit */
                XEO:1,           /* Bit  7, enable bit */
                XEZ:1,           /* Bit  8, enable bit */
                XEV:1,           /* Bit  9, enable bit */
                                 
                XCI:1,           /* Bit 10, preservation bit */
                XCU:1,           /* Bit 11, preservation bit */
                XCO:1,           /* Bit 12, preservation bit */
                XCZ:1,           /* Bit 13, preservation bit */
                XCV:1,           /* Bit 14, preservation bit */
                                 
                rsvd1:1,         /* Bit 16, reserved */
                FS:1,            /* Bit 17, subnormal number enable */
                RM:2,            /* Bit 18-19, rounding mode control register */
                rsvd2:1,         /* Bit 20, reserved */
                PEM:1,           /* Bit 21, precise exception mode */
                IF:1,            /* Bit 22, flushing of subnormal numbers */
                FN:1,            /* Bit 23, flushing of subnormal numbers to nearest */
                CC:8;            /* Bit 24-31, comparison result condition bits */
    };                          
} FPSR_T;                       
                                
typedef union {                 
    u32_T   ictagl;              /* ICCTAGL, 32-bit access */
    struct {                    
        u32_T    V:1,            /* Bit 0, valid bit */
                 rsvd1:1,        /* Bit 1, reserved */
                 Lock:1,         /* Bit 2, lock bit */
                 rsvd2:1,        /* Bit 3, reserved */
                 LRU:2,          /* Bit 4-5, valid bit */
                 rsvd3:4,        /* Bit 6-9, reserved */
                 LPN:22;         /* Bit 10-31, physical page number */
    };                          
} ICCTAGL_T;                    
                                
typedef union {                 
    u32_T   ictagh;              /* ICCTAGH, 32-bit access */
    struct {                    
        u32_T    rsvd1:8,        /* Bit 0-7, reserved */
                 TAGECC:8,       /* Bit 8-15, TAG ECC */
                 DATAECC:8,      /* Bit 16-23, data ECC */
                 rsvd2:4,        /* Bit 24-27, reserved */
                 PT:1,           /* Bit 28, automatic ECC generation TAG */
                 WT:1,           /* Bit 29, automatic ECC generation TAG */
                 PD:1,           /* Bit 30, automatic ECC generation data */
                 WD:1;           /* Bit 31, automatic ECC generation data */
    };                          
} ICCTAGH_T;                    
                                
typedef union {                 
    u32_T   icctrl;              /* ICCTRL, 32-bit access */
    struct {                    
        u32_T    ICHEN:1,        /* Bit 0, Instruction cahe 0:invalid 1:valid */
                 ICHEMK:1,       /* Bit 1, mask cache error expceptions */
                 ICHEIV:1,       /* Bit 2, 1:automatically invalidate instr. cache */
                 rsvd1:5,        /* Bit 3-7, reserved */
                 ICHCLR:1,       /* Bit 8, 1:clear entire instr. cache */
                 rsvd2:7,        /* Bit 9-15, reserved set to 0*/
                 rsvd3:1,        /* Bit 16, reserved set to 1*/
                 D1EV:1,         /* Bit 17, ECC correction */
                 rsvd4:14;       /* Bit 31, automatic ECC generation data */
    };                          
} ICCTRL_T;                     
                                
typedef union {                 
    u32_T   iccfg;               /* ICCFG, 32-bit access */
    struct {                    
        u32_T    ICHWAY:4,       /* no. of ways in ICACHE */
                                 /* 0: no icache, 1: 1 way */
                                 /* 2: 2 ways,   4: 4 ways */
                                 /* 8: 8 ways */
                 ICHLINE:4,      /* no of lines/way in ICACHE */
                                 /* 0: no icache,  1: 32 lines */
                                 /* 2: 64 lines,   4: 128 lines */
                                 /* 8: 256 liness */
                                
                 ICHSIZE:7,      /* Size in kBytes of ICACHE */
                                 /*  0:  no icache,   1:  1 kByte */
                                 /*  2:  2 kBytes,    4:  4 kBytes */
                                 /*  8:  8 kBytes,   16: 16 kBytes */
                                 /* 32: 32 kBytes,   64: 64 kBytes */
                                
                 rsvd1:1,        /* Bit 15, reserved, always 0 */
                 rsvd2:1,        /* Bit 16, reserved, always 1 */
                 rsvd3:15;       /* Bit 17-31, reserved, always 0 */
    };                          
} ICCFG_T;                      
                                
typedef union {                 
    u32_T   icerr;               /* ICERR, 32-bit access */
    struct {                   
        u32_T    ICHERR:1,       /* Bit 0, is 1, if error occured */
                 rsvd1:1,        /* Bit 1, reserved, always 0 */
                 ICHET:1,        /* Bit 2, error in tag RAM */
                 ICHED:1,        /* Bit 3, error in data RAM */
                 ICHERQ:1,       /* Bit 4, chache error exception in progress */
                 ICHEIX:8,       /* Bit 5-12, retain cache index, if chache error occured */
                 ICHEWY:2,       /* Bit 13-14, retain way number, if chache error occured */
                 rsvd2:1,        /* Bit 15, reserved, always 0 */
                                
                 ERMDE:1,        /* Bit 16, Error exception notification mask : Data RAM 2-bit error */
                 ERMDC:1,        /* Bit 17, Error exception notification mask : Data RAM 1-bit correction */
                 ERMTE2:1,       /* Bit 18, Error exception notification mask : Tag RAM 2-bit error */
                 ERMTE1:1,       /* Bit 19, Error exception notification mask : Tag RAM 1-bit error */
                 ERMPBSE:1,      /* Bit 20, Error exception notification mask : WAY error */
                 ERMMH:1,        /* Bit 21, Error exception notification mask : Multi bit */
                                
                 rsvd3:2,        /* Bit 22-23, reserved, always 0 */
                                
                 ESDE:1,         /* Bit 24, Error status: Data RAM 2-bit error */
                 ESDC:1,         /* Bit 25, Error status: Data RAM 1-bit correction R/W Undefined */
                 ESTE2:1,        /* Bit 26, Error status: Tag RAM 2-bit error R/W Undefined */
                 ESTE1:1,        /* Bit 27, Error status: Tag RAM 1-bit error R/W Undefined */
                 ESPBSE:1,       /* Bit 28, Error status: WAY error R/W Undefined */
                 ESMH:1,         /* Bit 29, Error status: Multi hit R/W Undefined */
                 rsvd4:2,        /* Bit 30, reserved, always 0 */
                 CISTW:1;        /* Bit 31, Indicates an error in the destibnation way of CIST instr. */
    };
} ICERR_T;

typedef union {                 
    u32_T   cdbcr;               /* CDBCR, 32-bit access */
    struct {                   
        u32_T   CDBEN:1,         /* Bit 0, This bit specifies enables or disables of the data buffer. */
                                 /*        0: Data buffer disabled.  1: Data buffer enabled. */
                CDBCLR:1,        /* Bit 1, 1: data buffer is cleared. */
                rsvd1:30;        /* Bit 2-31, Reserved for future expansion. Be sure to set to 0. */ 
    };
} CDBCR_T;

typedef union {
    u32_T   mpat;
    struct  {
        u32_T   UR:1,            /* Indicates the read permission for user mode. */
                UW:1,            /* Indicates the write permission for user mode */
                UX:1,            /* Indicates the execution privilege for user mode. */
                SR:1,            /* Indicates the write permission for supervisor mode */
                SW:1,            /* Indicates the write permission for supervisor mode. */
                SX:1,            /* Indicates the execution privilege for supervisor mode */
                G:1,             /* Global specification of an MPU area for all VMs. */
                                 /* If 1, the ASID bit has no effect on the area-matching */
                                 /* condition. */
                E:1,             /* This bit indicates whether area n is enabled or disabled. */
                rsvd1:8,         /* (Reserved for future expansion. Be sure to clear to 0.) */
                ASID:10,         /* These bits indicate the ASID value to be used as the */
                                 /* area match condition */
                rsvd2:6;         /* Reserved. Be sure to clear to 0.) */
    };
} MPAT_T;


/*
 * Groups
 */

/*      Name      Reg, Group        Comment */
#define EIPC         0, 0        /* Status save registers when acknowledging EI level exception SV */
#define EIPSW        1, 0        /* Status save registers when acknowledging EI level exception SV */
#define FEPC         2, 0        /* Status save registers when acknowledging FE level exception SV */
#define FEPSW        3, 0        /* Status save registers when acknowledging FE level exception SV */
#define PSW          5, 0        /* Program status word Note 1 */
#define EIIC        13, 0        /* EI level exception cause SV */
#define FEIC        14, 0        /* FE level exception cause SV */
#define CTPC        16, 0        /* CALLT execution status save register UM */
#define CTPSW       17, 0        /* CALLT execution status save register UM */
#define CTBP        20, 0        /* CALLT base pointer UM */
#define EIWR        28, 0        /* EI level exception working register SV */
#define FEWR        29, 0        /* FE level exception working register SV */
#define BSEL        31, 0        /* (Reserved for backward compatibility with V850E2 series)Note 2 SV */

#define MCFG0        0, 1        /* Machine configuration SV */
#define RBASE        2, 1        /* Reset vector base address SV */
#define EBASE        3, 1        /* Exception handler vector address SV */
#define INTBP        4, 1        /* Base address of the interrupt handler table SV */
#define MCTL         5, 1        /* CPU control SV */
#define PID          6, 1        /* Processor ID SV */
#define SCCFG       11, 1        /* SYSCALL operation setting SV */
#define SCBP        12, 1        /* SYSCALL base pointer SV */

#define HTCFG0       0, 2        /* Thread configuration SV */
#define MEA          6, 2        /* Memory error address SV */
#define ASID         7, 2        /* Address space ID SV */
#define MEI          8, 2        /* Memory error information SV */
#define ISPR        10, 2        /* Priority of interrupt being serviced SV */
#define PMR         11, 2        /* Interrupt priority masking SV */
#define ICSR        12, 2        /* Interrupt control status SV */
#define INTCFG      13, 2        /* Interrupt function setting SV */

#define MPM          0, 5        /* Memory protection operation mode setting SV */
#define MPRC         1, 5        /* MPU region control SV */
#define MPBRGN       4, 5        /* MPU base region number SV */
#define MPTRGN       5, 5        /* MPU end region number SV */
#define MCA          8, 5        /* Memory protection setting check address SV */
#define MCS          9, 5        /* Memory protection setting check size SV */
#define MCC         10, 5        /* Memory protection setting check command SV */
#define MCR         11, 5        /* Memory protection setting check result SV */

#define MPLA0        0, 6        /* Protection area minimum address SV */
#define MPUA0        1, 6        /* Protection area maximum address SV */
#define MPAT0        2, 6        /* Protection area attribute SV */
#define MPLA1        4, 6        /* Protection area minimum address SV */
#define MPUA1        5, 6        /* Protection area maximum address SV */
#define MPAT1        6, 6        /* Protection area attribute SV */
#define MPLA2        8, 6        /* Lower address of the protection area SV */
#define MPUA2        9, 6        /* Protection area maximum address SV */
#define MPAT2       10, 6        /* Protection area attribute SV */
#define MPLA3       12, 6        /* Protection area minimum address SV */
#define MPUA3       13, 6        /* Protection area maximum address SV */
#define MPAT3       14, 6        /* Protection area attribute SV */
#define MPLA4       16, 6        /* Protection area minimum address SV */
#define MPUA4       17, 6        /* Protection area maximum address SV */
#define MPAT4       18, 6        /* Protection area attribute SV */
#define MPLA5       20, 6        /* Protection area minimum address SV */
#define MPUA5       21, 6        /* Protection area maximum address SV */
#define MPAT5       22, 6        /* Protection area attribute SV */
#define MPLA6       24, 6        /* Protection area minimum address SV */
#define MPUA6       25, 6        /* Protection area maximum address SV */
#define MPAT6       26, 6        /* Protection area attribute SV */
#define MPLA7       28, 6        /* Protection area minimum address SV */
#define MPUA7       29, 6        /* Protection area maximum address SV */
#define MPAT7       30, 6        /* Protection area attribute SV */
#define MPLA8        0, 7        /* Protection area minimum address SV */
#define MPUA8        1, 7        /* Protection area maximum address SV */
#define MPAT8        2, 7        /* Protection area attribute SV */
#define MPLA9        4, 7        /* Protection area minimum address SV */
#define MPUA9        5, 7        /* Protection area maximum address SV */
#define MPAT9        6, 7        /* Protection area attribute SV */
#define MPLA10       8, 7        /* Protection area minimum address SV */
#define MPUA10       9, 7        /* Protection area maximum address SV */
#define MPAT10      10, 7        /* Protection area attribute SV */
#define MPLA11      12, 7        /* Protection area minimum address SV */
#define MPUA11      13, 7        /* Protection area maximum address SV */
#define MPAT11      14, 7        /* Protection area attribute SV */
#define MPLA12      16, 7        /* Protection area minimum address SV */
#define MPUA12      17, 7        /* Protection area maximum address SV */
#define MPAT12      18, 7        /* Protection area attribute SV */
#define MPLA13      20, 7        /* Protection area minimum address SV */
#define MPUA13      21, 7        /* Protection area maximum address SV */
#define MPAT13      22, 7        /* Protection area attribute SV */
#define MPLA14      24, 7        /* Protection area minimum address SV */
#define MPUA14      25, 7        /* Protection area maximum address SV */
#define MPAT14      26, 7        /* Protection area attribute SV */
#define MPLA15      28, 7        /* Protection area minimum address SV */
#define MPUA15      29, 7        /* Protection area maximum address SV */
#define MPAT15      30, 7        /* Protection area attribute SV */



/* HAZARD RESOLUTION PROCEDURE FOR SYSTEM REGISTERS */

_INLINE_ void SET_PSW_USERMODE( void)
{
    PSW_T p;
    p.psw= __STSR( PSW); 
    p.UM= 1; 
    __LDSR( PSW, p.psw); 
    __SYNCI(); 
}

_INLINE_ void SET_FEPSW_SVMODE( void)
{ 
    PSW_T p;
    p.psw= __STSR( FEPSW); 
    p.UM= 0; 
    __LDSR( FEPSW, p.psw); 
    __SYNCI(); 
}

_INLINE_ void SET_MCFG0_SPID( u32_T v)
{ 
    MCFG0_T p;
    p.mcfg0= __STSR( MCFG0); 
    p.SPID= v; 
    __LDSR( MCFG0, p.mcfg0); 
    __SYNCI(); 
}

_INLINE_ void SET_SCCFG( u32_T emax) 
{ 
    SCCFG_T p; 
    p.sccfg= __STSR( SCCFG); 
    p.SIZE= emax; 
    __LDSR( SCCFG, p.sccfg); 
    __SYNCP(); 
}


_INLINE_ void SET_ASID( u32_T v)   { __LDSR( ASID  , v); __SYNCP();}

_INLINE_ void SET_MPM   (u32_T v)  { __LDSR( MPM   , v); __SYNCI();}
_INLINE_ void SET_MPRC  (u32_T v)  { __LDSR( MPRC  , v); __SYNCI();}
_INLINE_ void SET_MPBRGN(u32_T v)  { __LDSR( MPBRGN, v); __SYNCI();}
_INLINE_ void SET_MPTRGN(u32_T v)  { __LDSR( MPTRGN, v); __SYNCI();}
_INLINE_ void SET_MCA   (u32_T v)  { __LDSR( MCA   , v); __SYNCI();}
_INLINE_ void SET_MCS   (u32_T v)  { __LDSR( MCS   , v); __SYNCI();}
_INLINE_ void SET_MCC   (u32_T v)  { __LDSR( MCC   , v); __SYNCI();}
_INLINE_ void SET_MCR   (u32_T v)  { __LDSR( MCR   , v); __SYNCI();}

_INLINE_ void SET_MPLA0(u32_T v)   { __LDSR( MPLA0 , v); __SYNCP();}
_INLINE_ void SET_MPUA0(u32_T v)   { __LDSR( MPUA0 , v); __SYNCP();}
_INLINE_ void SET_MPAT0(u32_T v)   { __LDSR( MPAT0 , v); __SYNCP();}
_INLINE_ void SET_MPLA1(u32_T v)   { __LDSR( MPLA1 , v); __SYNCP();}
_INLINE_ void SET_MPUA1(u32_T v)   { __LDSR( MPUA1 , v); __SYNCP();}
_INLINE_ void SET_MPAT1(u32_T v)   { __LDSR( MPAT1 , v); __SYNCP();}
_INLINE_ void SET_MPLA2(u32_T v)   { __LDSR( MPLA2 , v); __SYNCP();}
_INLINE_ void SET_MPUA2(u32_T v)   { __LDSR( MPUA2 , v); __SYNCP();}
_INLINE_ void SET_MPAT2(u32_T v)   { __LDSR( MPAT2 , v); __SYNCP();}
_INLINE_ void SET_MPLA3(u32_T v)   { __LDSR( MPLA3 , v); __SYNCP();}
_INLINE_ void SET_MPUA3(u32_T v)   { __LDSR( MPUA3 , v); __SYNCP();}
_INLINE_ void SET_MPAT3(u32_T v)   { __LDSR( MPAT3 , v); __SYNCP();}
_INLINE_ void SET_MPLA4(u32_T v)   { __LDSR( MPLA4 , v); __SYNCP();}
_INLINE_ void SET_MPUA4(u32_T v)   { __LDSR( MPUA4 , v); __SYNCP();}
_INLINE_ void SET_MPAT4(u32_T v)   { __LDSR( MPAT4 , v); __SYNCP();}
_INLINE_ void SET_MPLA5(u32_T v)   { __LDSR( MPLA5 , v); __SYNCP();}
_INLINE_ void SET_MPUA5(u32_T v)   { __LDSR( MPUA5 , v); __SYNCP();}
_INLINE_ void SET_MPAT5(u32_T v)   { __LDSR( MPAT5 , v); __SYNCP();}
_INLINE_ void SET_MPLA6(u32_T v)   { __LDSR( MPLA6 , v); __SYNCP();}
_INLINE_ void SET_MPUA6(u32_T v)   { __LDSR( MPUA6 , v); __SYNCP();}
_INLINE_ void SET_MPAT6(u32_T v)   { __LDSR( MPAT6 , v); __SYNCP();}
_INLINE_ void SET_MPLA7(u32_T v)   { __LDSR( MPLA7 , v); __SYNCP();}
_INLINE_ void SET_MPUA7(u32_T v)   { __LDSR( MPUA7 , v); __SYNCP();}
_INLINE_ void SET_MPAT7(u32_T v)   { __LDSR( MPAT7 , v); __SYNCP();}
_INLINE_ void SET_MPLA8(u32_T v)   { __LDSR( MPLA8 , v); __SYNCP();}
_INLINE_ void SET_MPUA8(u32_T v)   { __LDSR( MPUA8 , v); __SYNCP();}
_INLINE_ void SET_MPAT8(u32_T v)   { __LDSR( MPAT8 , v); __SYNCP();}
_INLINE_ void SET_MPLA9(u32_T v)   { __LDSR( MPLA9 , v); __SYNCP();}
_INLINE_ void SET_MPUA9(u32_T v)   { __LDSR( MPUA9 , v); __SYNCP();}
_INLINE_ void SET_MPAT9(u32_T v)   { __LDSR( MPAT9 , v); __SYNCP();}
_INLINE_ void SET_MPLA10(u32_T v)  { __LDSR( MPLA10, v); __SYNCP();}
_INLINE_ void SET_MPUA10(u32_T v)  { __LDSR( MPUA10, v); __SYNCP();}
_INLINE_ void SET_MPAT10(u32_T v)  { __LDSR( MPAT10, v); __SYNCP();}
_INLINE_ void SET_MPLA11(u32_T v)  { __LDSR( MPLA11, v); __SYNCP();}
_INLINE_ void SET_MPUA11(u32_T v)  { __LDSR( MPUA11, v); __SYNCP();}
_INLINE_ void SET_MPAT11(u32_T v)  { __LDSR( MPAT11, v); __SYNCP();}
_INLINE_ void SET_MPLA12(u32_T v)  { __LDSR( MPLA12, v); __SYNCP();}
_INLINE_ void SET_MPUA12(u32_T v)  { __LDSR( MPUA12, v); __SYNCP();}
_INLINE_ void SET_MPAT12(u32_T v)  { __LDSR( MPAT12, v); __SYNCP();}
_INLINE_ void SET_MPLA13(u32_T v)  { __LDSR( MPLA13, v); __SYNCP();}
_INLINE_ void SET_MPUA13(u32_T v)  { __LDSR( MPUA13, v); __SYNCP();}
_INLINE_ void SET_MPAT13(u32_T v)  { __LDSR( MPAT13, v); __SYNCP();}
_INLINE_ void SET_MPLA14(u32_T v)  { __LDSR( MPLA14, v); __SYNCP();}
_INLINE_ void SET_MPUA14(u32_T v)  { __LDSR( MPUA14, v); __SYNCP();}
_INLINE_ void SET_MPAT14(u32_T v)  { __LDSR( MPAT14, v); __SYNCP();}
_INLINE_ void SET_MPLA15(u32_T v)  { __LDSR( MPLA15, v); __SYNCP();}
_INLINE_ void SET_MPUA15(u32_T v)  { __LDSR( MPUA15, v); __SYNCP();}
_INLINE_ void SET_MPAT15(u32_T v)  { __LDSR( MPAT15, v); __SYNCP();}

_INLINE_ u32_T GET_EIIC(  void )   { return __STSR( EIIC  );  }
_INLINE_ u32_T GET_FEIC(  void )   { return __STSR( FEIC  );  }
_INLINE_ u32_T GET_EIPSW( void )   { return __STSR( EIPSW  ); }
_INLINE_ u32_T GET_FEPSW( void )   { return __STSR( FEPSW  ); }

_INLINE_ u32_T GET_ASID( void )    { return __STSR( ASID  ); }

_INLINE_ u32_T GET_MPM   (void )   { return __STSR( MPM   ); } 
_INLINE_ u32_T GET_MPRC  (void )   { return __STSR( MPRC  ); } 
_INLINE_ u32_T GET_MPBRGN(void )   { return __STSR( MPBRGN); } 
_INLINE_ u32_T GET_MPTRGN(void )   { return __STSR( MPTRGN); } 
_INLINE_ u32_T GET_MCA   (void )   { return __STSR( MCA   ); } 
_INLINE_ u32_T GET_MCS   (void )   { return __STSR( MCS   ); } 
_INLINE_ u32_T GET_MCC   (void )   { return __STSR( MCC   ); } 
_INLINE_ u32_T GET_MCR   (void )   { return __STSR( MCR   ); } 
_INLINE_ u32_T GET_PID   (void )   { return __STSR( PID   ); } 

_INLINE_ u32_T GET_MPLA0(void )    { return __STSR( MPLA0 ); }
_INLINE_ u32_T GET_MPUA0(void )    { return __STSR( MPUA0 ); }
_INLINE_ u32_T GET_MPAT0(void )    { return __STSR( MPAT0 ); }
_INLINE_ u32_T GET_MPLA1(void )    { return __STSR( MPLA1 ); }
_INLINE_ u32_T GET_MPUA1(void )    { return __STSR( MPUA1 ); }
_INLINE_ u32_T GET_MPAT1(void )    { return __STSR( MPAT1 ); }
_INLINE_ u32_T GET_MPLA2(void )    { return __STSR( MPLA2 ); }
_INLINE_ u32_T GET_MPUA2(void )    { return __STSR( MPUA2 ); }
_INLINE_ u32_T GET_MPAT2(void )    { return __STSR( MPAT2 ); }
_INLINE_ u32_T GET_MPLA3(void )    { return __STSR( MPLA3 ); }
_INLINE_ u32_T GET_MPUA3(void )    { return __STSR( MPUA3 ); }
_INLINE_ u32_T GET_MPAT3(void )    { return __STSR( MPAT3 ); }
_INLINE_ u32_T GET_MPLA4(void )    { return __STSR( MPLA4 ); }
_INLINE_ u32_T GET_MPUA4(void )    { return __STSR( MPUA4 ); }
_INLINE_ u32_T GET_MPAT4(void )    { return __STSR( MPAT4 ); }
_INLINE_ u32_T GET_MPLA5(void )    { return __STSR( MPLA5 ); }
_INLINE_ u32_T GET_MPUA5(void )    { return __STSR( MPUA5 ); }
_INLINE_ u32_T GET_MPAT5(void )    { return __STSR( MPAT5 ); }
_INLINE_ u32_T GET_MPLA6(void )    { return __STSR( MPLA6 ); }
_INLINE_ u32_T GET_MPUA6(void )    { return __STSR( MPUA6 ); }
_INLINE_ u32_T GET_MPAT6(void )    { return __STSR( MPAT6 ); }
_INLINE_ u32_T GET_MPLA7(void )    { return __STSR( MPLA7 ); }
_INLINE_ u32_T GET_MPUA7(void )    { return __STSR( MPUA7 ); }
_INLINE_ u32_T GET_MPAT7(void )    { return __STSR( MPAT7 ); }
_INLINE_ u32_T GET_MPLA8(void )    { return __STSR( MPLA8 ); }
_INLINE_ u32_T GET_MPUA8(void )    { return __STSR( MPUA8 ); }
_INLINE_ u32_T GET_MPAT8(void )    { return __STSR( MPAT8 ); }
_INLINE_ u32_T GET_MPLA9(void )    { return __STSR( MPLA9 ); }
_INLINE_ u32_T GET_MPUA9(void )    { return __STSR( MPUA9 ); }
_INLINE_ u32_T GET_MPAT9(void )    { return __STSR( MPAT9 ); }
_INLINE_ u32_T GET_MPLA10(void )   { return __STSR( MPLA10); }
_INLINE_ u32_T GET_MPUA10(void )   { return __STSR( MPUA10); }
_INLINE_ u32_T GET_MPAT10(void )   { return __STSR( MPAT10); }
_INLINE_ u32_T GET_MPLA11(void )   { return __STSR( MPLA11); }
_INLINE_ u32_T GET_MPUA11(void )   { return __STSR( MPUA11); }
_INLINE_ u32_T GET_MPAT11(void )   { return __STSR( MPAT11); }
_INLINE_ u32_T GET_MPLA12(void )   { return __STSR( MPLA12); }
_INLINE_ u32_T GET_MPUA12(void )   { return __STSR( MPUA12); }
_INLINE_ u32_T GET_MPAT12(void )   { return __STSR( MPAT12); }
_INLINE_ u32_T GET_MPLA13(void )   { return __STSR( MPLA13); }
_INLINE_ u32_T GET_MPUA13(void )   { return __STSR( MPUA13); }
_INLINE_ u32_T GET_MPAT13(void )   { return __STSR( MPAT13); }
_INLINE_ u32_T GET_MPLA14(void )   { return __STSR( MPLA14); }
_INLINE_ u32_T GET_MPUA14(void )   { return __STSR( MPUA14); }
_INLINE_ u32_T GET_MPAT14(void )   { return __STSR( MPAT14); }
_INLINE_ u32_T GET_MPLA15(void )   { return __STSR( MPLA15); }
_INLINE_ u32_T GET_MPUA15(void )   { return __STSR( MPUA15); }
_INLINE_ u32_T GET_MPAT15(void )   { return __STSR( MPAT15); }


/* Access restrictions, write only if PSW.ID=1 */
_INLINE_ void SET_PSW_EBV( void)
{ 
    PSW_T p;
    __DI() ;
    p.psw= __STSR( PSW); 
    p.EBV= 1;
    __LDSR( PSW, p.psw);
}

_INLINE_ void SET_MCTL_MA( u32_T v)
{ 
    MCTL_T p;
    p.mctl= __STSR( MCTL);
    p.MA=!!v;
    __LDSR( MCTL, p.mctl);
    __SYNCP();
}

_INLINE_ void SET_MCTL_UIC( u32_T v)
{ 
    MCTL_T p;
    p.mctl= __STSR( MCTL);
    p.UIC=!!v;
    __LDSR( MCTL, p.mctl);
    __SYNCP();
}

_INLINE_ void SET_EBASE( u32_T v)
{ 
    u32_T __t=__DIR(); 
    __LDSR( EBASE, v);
    __RIR( __t);
}

_INLINE_ void SET_INTBP( u32_T v)
{ 
    u32_T __t=__DIR(); 
    __LDSR( INTBP, v);
    __RIR( __t);
}

_INLINE_ void SET_ISPR(u32_T v)
{ 
    u32_T __t=__DIR(); 
    __LDSR( ISPR, v);
    __RIR( __t);
}

_INLINE_ void SET_PMR(u32_T v)
{ 
    u32_T __t=__DIR(); 
    __LDSR( PMR, v);
    __RIR( __t);
}

_INLINE_ void SET_ICSR( u32_T v)
{ 
    u32_T __t=__DIR(); 
    __LDSR( ICSR, v);
    __RIR( __t);
}

_INLINE_ void SET_INTCFG( u32_T v)
{ 
    u32_T __t=__DIR(); 
    __LDSR( INTCFG, v);
    __RIR( __t);
}

#ifdef __RH850G3M__
/* 
 * RH850 G3M related handling 
 */


/* FPU Operation Registers */
/*      Name       Reg, Group       Comment */
#define FPSR         6, 0        /* (See 3.4 FPU Function Registers) CU0 and SV */
#define FPEPC        7, 0        /* (See 3.4 FPU Function Registers) CU0 and SV */
#define FPST         8, 0        /* (See 3.4 FPU Function Registers) CU0 */
#define FPCC         9, 0        /* (See 3.4 FPU Function Registers) CU0 */
#define FPCFG       10, 0        /* (See 3.4 FPU Function Registers) CU0 */
#define FPEC        11, 0        /* (See 3.4 FPU Function Registers) CU0 and SV */

/* Cache Operation Function Registers */
/*      Name       Reg, Group       Comment */
#define ICTAGL      16, 4        /* Instruction cache tag Lo access SV */
#define ICTAGH      17, 4        /* Instruction cache tag Hi access SV */
#define ICDATL      18, 4        /* Instruction cache data Lo access SV */
#define ICDATH      19, 4        /* Instruction cache data Hi access SV */
#define ICCTRL      24, 4        /* Instruction cache control SV */
#define ICCFG       26, 4        /* Instruction cache configuration SV */
#define ICERR       28, 4        /* Instruction cache error SV */

#define CDBCR       24,13        /* Data buffer control register */

_INLINE_ void SET_FPSR(u32_T v) { __LDSR( FPSR ,v); __SYNCP();}
_INLINE_ void SET_FPEP(u32_T v) { __LDSR( FPEPC,v); __SYNCP();}
_INLINE_ void SET_FPST(u32_T v) { __LDSR( FPST ,v); __SYNCP();}
_INLINE_ void SET_FPCC(u32_T v) { __LDSR( FPCC ,v); __SYNCP();}
_INLINE_ void SET_FPCF(u32_T v) { __LDSR( FPCFG,v); __SYNCP();}
_INLINE_ void SET_FPEC(u32_T v) { __LDSR( FPEC ,v); __SYNCP();}

_INLINE_ void SET_PSW_ENABLEFPU( void)
{ 
    PSW_T p;
    p.psw= __STSR( PSW); 
    p.CU0= 1; 
    __LDSR( PSW, p.psw); 
    __SYNCI(); 
}

_INLINE_ void SET_FPSR_PEM( void)
{ 
    FPSR_T p;
    __SYNCP(); 
    p.fpsr= __STSR( FPSR); 
    p.PEM= 1; 
    __LDSR( FPSR, p.fpsr); 
    __SYNCP(); 
}

_INLINE_ void CLR_FPSR_PEM( void)
{ 
    FPSR_T p;
    __SYNCP(); 
    p.fpsr= __STSR( FPSR); 
    p.PEM= 0; 
    __LDSR( FPSR, p.fpsr); 
    __SYNCP(); 
}

_INLINE_ u32_T GetPEID( void)
{
	u32_T	__t;

	__t= (__STSR( 0, 2) >>16) & 0x7;

	return __t;
}


#endif  /* __RH850G3M__ */
#ifdef __ghs
#pragma ghs endnomisra
#endif  /* __ghs */
#endif
