#include "RTE_Memory.h"
#include "Analog_Signals.h"

uint8_t  SessionType = DefaultSession;

uint16_t S3_ServerCnt = S3_SERVER;
uint8_t  S3_ServerClear = 0u;
uint8_t  S3_ServerTimeOut = 1u;

uint8_t  UDSKL15Status = 0u;
uint8_t  UDSSleepStatus = 0u;

uint16_t UDSTimer10ms = 0u;

uint8_t  S31DID0203Status = 0u;

uint8_t  isDTCOpen = 1u;
uint8_t  isReceive = 1u;
uint8_t  isEnable = 0u;
uint16_t enableTimer = 0u;
uint16_t enableMaxTimer = KL15DTCEnableTimer;

DiagMSGUnion DiagMSG;
DiagFlagUnion DiagFlag;
Ser27_FlowCtrlUnion Ser27_FlowCtrl;
Ser27_FlowCtrlCntUnion Ser27_FlowCtrlCnt;
BootToDFlashInfoUnion BootToDFlashInfo;
Ser2EToDFlashInfoUnion Ser2EToDFlashInfo;
DTCToDFlashInfoUnion DTCToDFlashInfo;
CAFDToDFlashInfoUnion CAFDToDFlashInfo;

/**
  *设置会话类型
  *@param status 会话类型
  */
void setSessionType(uint8_t status)
{
    if ((status == DefaultSession) || (status == ProgrammingSession) || (status == ExtendedDiagnosticSession))
    {
        SessionType = status;
    }
    else
    {
        SessionType = DefaultSession;
    }
}

/**
  *获取会话类型
  *@return 当前会话类型
  */
uint8_t getSessionType(void)
{
    return SessionType;
}

/**
  *设置诊断信息
  *@param Identifier 收到的诊断报文ID
  *@param dlc 收到的诊断报文长度
  *@param Data 收到的诊断报文数据
  */
void setDiagMSG(uint32_t Identifier, uint16_t dlc, uint8_t Data[])
{
    uint16_t i;
    DiagMSG.msgID = Identifier;
    DiagMSG.msgDLC = dlc;
    for (i = 0u; i < dlc; ++i)
    {
        DiagMSG.msgData.Data[i] = Data[i];
    }
}

/**
  *清除诊断信息
  */
void clearDiagMSG(void)
{
    uint16_t i;
    DiagMSG.msgID = 0u;
    DiagMSG.msgDLC = 0u;
    for (i = 0u; i < DIM(DiagMSG.msgData.Data); ++i)
    {
        DiagMSG.msgData.Data[i] = 0u;
    }
}

/**
  *获取当前车速
  *@return 返回当前车速
  */
uint16_t getSpeed(void)
 {
    return 0;// (VSpeed / 10u);
}

/**
  *获取当前转速
  *@return 返回当前转速
  */
uint16_t getESpeed(void)
{

    return 0;
}

/**
  *获取当前30电压
  *@return 返回当前30电压
  */
uint16_t getKL30Voltage(void)
{
    // return ADC_Read_Signal(ADC_CH_KL30_VOLTAGE);
}

/**
  *获取15电状态
  *@return 15点状态 0为IGN OFF,1为IGN ON
  */
uint8_t getKL15Status(void)
{
    return 1;
}

/**
  *获取当前安全级别
  *@return 返回当前安全级别
  */
uint8_t getSecurityLevel(void)
{
    uint8_t Status = NormalKeyLock;
    if(Ser27_FlowCtrl.LV1SafetyStatus == NormalKeyUnlock)
    {
        Status = NormalKeyUnlock;
    }
    if(Ser27_FlowCtrl.LV11SafetyStatus == NormalKeyUnlock)
    {
        Status = BootKeyUnlock;
    }
    return Status;
}

/**
  *重置安全等级
  */
void Ser27_ChangeSessionHandle(void)
{
    Ser27_FlowCtrl.LV1Ctrl = SER27_CLEAR_REQ_SEEDED;
    Ser27_FlowCtrl.LV1Key[0]  = 0x00u;
    Ser27_FlowCtrl.LV1Key[1]  = 0x00u;
    Ser27_FlowCtrl.LV1Key[2]  = 0x00u;
    Ser27_FlowCtrl.LV1Key[3]  = 0x00u;
    Ser27_FlowCtrl.LV1SafetyStatus = NormalKeyLock;
    Ser27_FlowCtrl.LV1Seed[0] = 0x00u;
    Ser27_FlowCtrl.LV1Seed[1] = 0x00u;
    Ser27_FlowCtrl.LV1Seed[2] = 0x00u;
    Ser27_FlowCtrl.LV1Seed[3] = 0x00u;
    Ser27_FlowCtrl.LV11Ctrl = SER27_CLEAR_REQ_SEEDED;
    Ser27_FlowCtrl.LV11Key[0]  = 0x00u;
    Ser27_FlowCtrl.LV11Key[1]  = 0x00u;
    Ser27_FlowCtrl.LV11Key[2]  = 0x00u;
    Ser27_FlowCtrl.LV11Key[3]  = 0x00u;
    Ser27_FlowCtrl.LV11SafetyStatus = NormalKeyLock;
    Ser27_FlowCtrl.LV11Seed[0] = 0x00u;
    Ser27_FlowCtrl.LV11Seed[1] = 0x00u;
    Ser27_FlowCtrl.LV11Seed[2] = 0x00u;
    Ser27_FlowCtrl.LV11Seed[3] = 0x00u;
}

/**
  *复位函数
  */
void UDS_RESET(void)
{
    ControlMessageCommunication(networkManagementMsgAndnormalMsg, disableRxAndTx);
    UDS_Delay(30000u);
    hardware_reset();
}

/**
  *诊断延时
  *@param count 延时次数
  */
void UDS_Delay(uint16_t count)
{
    uint16_t tmpcnt;
    tmpcnt = count;
    while (tmpcnt--)
    {
        //WDT_Clear();
        nop( );
    }
}

/**
  *发送诊断11服务正反馈
  */
void UDS_S11_Sand(uint8_t subid)
{
    _CAN_Msg canMag;
    canMag.MsgID = DIAG_ID_Tx;
    canMag.Msg[0] = 0x02u;
    canMag.Msg[1] = 0x51u;
    canMag.Msg[2] = subid;
    canMag.Msg[3] = FILLER_BYTE;
    canMag.Msg[4] = FILLER_BYTE;
    canMag.Msg[5] = FILLER_BYTE;
    canMag.Msg[6] = FILLER_BYTE;
    canMag.Msg[7] = FILLER_BYTE;
    canMag.MsgDLC = 8u;
    canMag.MsgPro = 25u;
    Can_Write(&canMag);
}

/**
  *诊断11服务正反馈确认
  */
void UDS_S11_Confirm(void)
{
    UDS_RESET();
}

/**
  *发送诊断78负反馈
  */
void UDS_NRC78_Sand(uint8_t sid)
{
    _CAN_Msg canMag;
    canMag.MsgID = DIAG_ID_Tx;
    canMag.Msg[0] = 0x03u;
    canMag.Msg[1] = 0x7Fu;
    canMag.Msg[2] = sid;
    canMag.Msg[3] = 0x78u;
    canMag.Msg[4] = FILLER_BYTE;
    canMag.Msg[5] = FILLER_BYTE;
    canMag.Msg[6] = FILLER_BYTE;
    canMag.Msg[7] = FILLER_BYTE;
    canMag.MsgDLC = 8u;
    canMag.MsgPro = 26u;
    Can_Write(&canMag);
}

/**
  *诊断50us任务
  */
void UDS_50us_Service(void)
{
    if(UDSSleepStatus == 1u)
    {
        ++UDSTimer10ms;
    }
}

/**
  *诊断10ms任务
  */
void UDS_10ms_Service(void)
{
    uint16_t count;
    UDS_KL15_init();
    if(UDSTimer10ms >= 200u)
    {
        count = UDSTimer10ms / 200u;
        UDSTimer10ms %= 200u;
        enableDTCTimer(count * 10u);
        MsgLostTimer(count * 10u);
        TestDTC(count * 10u);
        S3_Service(count * 10u);
        Ser27_ErrDelayTimerService(count * 10u);
    }
}

/**
  *诊断KL15init
  */
void UDS_KL15_init(void)
{
    uint8_t KL15Status = getKL15Status();
    if(UDSKL15Status == 0u)
    {
        if((KL15Status == 1u))
        {
            UDSKL15Status = 1u;
            UDSSleepStatus = 1u;
            setDTCEnableTimer(KL15DTCEnableTimer);
            S3_ServiceTimeOut();
            DTCAgeing();
        }
    }
    else
    {
        if(KL15Status == 0u)
        {
            UDSKL15Status = 0u;
            setDTCEnableTimer(KL15DTCEnableTimer);
        }
    }
}

void UDS_Slepp(void)
{
    UDSSleepStatus = 0u;
    UDSTimer10ms = 0u;
}

/**
  *报文掉线计时器
  *@param timer 计时器增加的时间，单位：ms
  */
void MsgLostTimer(uint16_t timer)
{
    uint8_t i;
    uint8_t BUS_OFF_Status = getBusoffStatus();
    if((isEnable == 1u) && (isReceive == 1u) && (UDSKL15Status == 1u) && (BUS_OFF_Status == 0u) && (HVoltageStatus == 0u) && (LVoltageStatus == 0u))
    {
        for(i = 0u; i < (sizeof(msgList) / sizeof(msgList[0u])); ++i)
        {
            if(msgList[i][2u] < 200u)
            {
                if(msgList[i][3u] < 200u * OfflineLoopMultiple)
                {
                    msgList[i][3u] += timer;
                }
            }
            else
            {
                if(msgList[i][3u] < msgList[i][2u] * OfflineLoopMultiple)
                {
                    msgList[i][3u] += timer;
                }
            }
        }
    }
}

/**
  *收到报文清除计时器
  *@param MsgID 报文ID
  *@param DLC   报文长度
  */
void ReceivedMsg(uint32_t MsgID, uint8_t DLC)
{
    uint8_t i;
    if(isReceive == 1u)
    {
        for(i = 0u; i < (sizeof(msgList) / sizeof(msgList[0u])); ++i)
        {
            if((MsgID == msgList[i][0u]) && (DLC == msgList[i][1u]))
            {
                msgList[i][3] = 0u;
                break;
            }
        }
    }
}

/**
  *S3服务
  */
void S3_Service(uint16_t timer)
{
    if (S3_ServerClear == 1u)
    {
        S3_ServerClear = 0u;
        S3_ServerCnt = 0u;
        S3_ServerTimeOut = 0u;
    }
    else
    {
        if (S3_ServerCnt < S3_SERVER)
        {
            S3_ServerCnt += timer;
        }
        else
        {
            if (S3_ServerTimeOut == 0u)
            {
                S3_ServerTimeOut = 1u;
                S3_ServiceTimeOut();
            }
        }
    }
}

/**
  *S3重置服务
  */
void S3_ServiceClear(void)
{
    S3_ServerClear = 1u;
}

/**
  *S3等待服务
  */
void S3_ServiceWait(void)
{
    S3_ServerClear = 0u;
    S3_ServerCnt = S3_SERVER;
    S3_ServerTimeOut = 1u;
}

/**
  *S3超时服务
  */
void S3_ServiceTimeOut(void)
{
    /*10服务重置*/
    setSessionType(DefaultSession);
    /*27服务重置*/
    Ser27_ChangeSessionHandle();
    /*28服务重置*/
    ControlMessageCommunication(networkManagementMsgAndnormalMsg, enableRxAndTx);
    /*2F服务重置*/
    UDS_10Service_S2FInit();
    /*85服务重置*/
    OpenDTCTest();
    /*诊断发送重置*/
    clearDiagMSG();
}

/**
  *27服务10s延时
  */
void Ser27_ErrDelayTimerService(uint16_t timer)
{
    if ( ( Ser27_FlowCtrl.LV1DelayCtrl == SER27_ERR_DELAY_TIMER_EN ) )
    {
        if (Ser27_FlowCtrl.LV1DelayTimer < 0xfff0u )
        {
            Ser27_FlowCtrl.LV1DelayTimer += timer;
            if (Ser27_FlowCtrl.LV1DelayTimer >= SER27_ERR_DELAY_TIMEOUT)
            {
                if (Ser27_FlowCtrlCnt.LV1ReqSeedCnt >= SER27_MAX_REQ_SEED_NUM )
                {
                    Ser27_FlowCtrlCnt.LV1ReqSeedCnt  = SER27_MAX_REQ_SEED_NUM - 1u;
                }
                if (Ser27_FlowCtrlCnt.LV11ReqSeedCnt >= SER27_MAX_REQ_SEED_NUM)
                {
                    Ser27_FlowCtrlCnt.LV11ReqSeedCnt  = SER27_MAX_REQ_SEED_NUM - 1u;
                }
                Ser27_FlowCtrl.LV1Ctrl = SER27_CLEAR_REQ_SEEDED;
                Ser27_FlowCtrl.LV1DelayTimer = 0u;
                Ser27_FlowCtrl.LV1DelayCtrl = 0u;
                /*调用向DFlash写入函数。*/
                (void)WriteDFlashData(0x02u, (uint32_t *)&Ser27_FlowCtrlCnt.Flag, sizeof(Ser27_FlowCtrlCnt) , NoNeedWait);
            }
        }
    }
}

/**
  *控制报文通信
  *@param Type 控制报文的类型
  *@param Status 控制的标志位
  */
void ControlMessageCommunication(uint8_t Type, uint8_t Status)
{
    if ((uint8_t)(Type & normalMsg) == normalMsg)
    {
    }
    if ((uint8_t)(Type & networkManagementMsg) == networkManagementMsg)
    {

    }
}

/**
  *DTC使能定时
  */
void enableDTCTimer(uint16_t Timer)
{
    uint8_t BUS_OFF_Status = getBusoffStatus();
    if((UDSKL15Status == 1u) && (BUS_OFF_Status == 0u) && (HVoltageStatus == 0u) && (LVoltageStatus == 0u))
    {
        if(isEnable == 0u)
        {
            if(enableTimer >= enableMaxTimer)
            {
                isEnable = 1u;
                enableTimer = 0u;
            }
            else
            {
                enableTimer += Timer;
            }
        }
    }
}

/**
  *DTC是否使能
  *@return DTC使能状态 0为未使能,1为使能
  */
uint8_t isDTCEnable(void)
{
    return isEnable;
}

/**
  *设置DTC使能时间
  */
void setDTCEnableTimer(uint16_t Timer)
{
    uint8_t i;
    if(isEnable == 1u)
    {
        isEnable = 0u;
        enableTimer = 0u;
        enableMaxTimer = Timer;
        for(i = 0u; i < (sizeof(msgList) / sizeof(msgList[0u])); ++i)
        {
            msgList[i][3] = 0u;
        }
    }
    else
    {
        if(enableMaxTimer == Timer)
        {
            enableTimer = 0u;
        }
        else
        {
            if(enableMaxTimer < enableTimer + Timer)
            {
                enableMaxTimer = enableTimer + Timer;
            }
        }
    }
}

/**
  *开启DTC检测
  */
void OpenDTCTest(void)
{
    isDTCOpen = 1u;
}

/**
  *关闭DTC检测
  */
void CloseDTCTest(void)
{
    isDTCOpen = 0u;
}

/**
  *DTC是否开启
  *@return DTC状态 0关闭,1开启
  */
uint8_t DTCTestIsOpen(void)
{
    return isDTCOpen;
}

/**
  *DTC老化
  */
void DTCAgeing(void)
{
    uint8_t i;
    for (i = 0u; i < sizeof(DTCToDFlashInfo.DTC) / sizeof(DTCUnion); ++i)
    {
        DTCToDFlashInfo.DTC[i].DTCStatus |= (DTCNoTestThisCycle);
        if ((uint8_t)(DTCToDFlashInfo.DTC[i].DTCStatus & DTCErrorThisCycle) != DTCErrorThisCycle)
        {
            DTCToDFlashInfo.DTC[i].DTCStatus = (uint8_t) (DTCToDFlashInfo.DTC[i].DTCStatus & (~DTCPending));
            DTCToDFlashInfo.DTC[i].DTCPendingCount = 0u;
            if ((uint8_t)(DTCToDFlashInfo.DTC[i].DTCStatus & DTCConfirmed) == DTCConfirmed)
            {
                DTCToDFlashInfo.DTC[i].DTCAgeing++;
                if (DTCToDFlashInfo.DTC[i].DTCAgeing >= AgeingCount)
                {
                    DTCToDFlashInfo.DTC[i].DTCStatus =  (uint8_t)(DTCToDFlashInfo.DTC[i].DTCStatus & (~DTCConfirmed));
                }
            }
        }
        else
        {
            DTCToDFlashInfo.DTC[i].DTCStatus = (uint8_t)(DTCToDFlashInfo.DTC[i].DTCStatus & (~DTCErrorThisCycle));
        }
        DTCToDFlashInfo.DTC[i].DTCStatus = (uint8_t)(DTCToDFlashInfo.DTC[i].DTCStatus & DTCNoError);
    }
}
const uint8_t DID_F189[] = {'S','W',':','A','.','0','.','4'};
/**
  *读取所有的诊断信息
  */
void ReadAllDFlashData(void)
{
    uint32_t i = 0;
    (void)ReadDFlashData(INTERNAL_EE_BLOCK_00, (uint32_t *)&DiagFlag.Flag         , sizeof(DiagFlag)          , AllFlagInit);
    (void)ReadDFlashData(INTERNAL_EE_BLOCK_01, (uint32_t *)&Ser27_FlowCtrlCnt.Flag, sizeof(Ser27_FlowCtrlCnt) , Ser27ToDFlashInfoInit);
    (void)ReadDFlashData(INTERNAL_EE_BLOCK_02, (uint32_t *)&BootToDFlashInfo.Flag , sizeof(BootToDFlashInfo)  , BootToDFlashInfoInit);
    (void)ReadDFlashData(INTERNAL_EE_BLOCK_03, (uint32_t *)&Ser2EToDFlashInfo.Flag, sizeof(Ser2EToDFlashInfo) , Ser2EToDFlashInfoInit);
    (void)ReadDFlashData(INTERNAL_EE_BLOCK_04, (uint32_t *)&DTCToDFlashInfo.Flag  , sizeof(DTCToDFlashInfo)   , DTCToDFlashInfoInit);
    (void)ReadDFlashData(INTERNAL_EE_BLOCK_08, (uint32_t *)&CAFDToDFlashInfo.Flag , sizeof(CAFDToDFlashInfo)  , CAFDToDFlashInfoInit);

    if (Ser27_FlowCtrlCnt.LV1ReqSeedCnt >= SER27_MAX_REQ_SEED_NUM )
    {
        Ser27_FlowCtrl.LV1DelayCtrl = SER27_ERR_DELAY_TIMER_EN;
    }
    if (Ser27_FlowCtrlCnt.LV11ReqSeedCnt >= SER27_MAX_REQ_SEED_NUM)
    {
        Ser27_FlowCtrl.LV11DelayCtrl = SER27_ERR_DELAY_TIMER_EN;
    }


    DTCConfigurationInit();
    for(i = 0u; i < DIM(DID_F189); ++i)
    {
        if(DID_F189[i] != BootToDFlashInfo.DID_F189[i])
        {
            (void)memset(BootToDFlashInfo.DID_F189, 0x00, sizeof(BootToDFlashInfo.DID_F189));
            BootToDFlashInfo.DID_F189[0] = DID_F189[0];
            BootToDFlashInfo.DID_F189[1] = DID_F189[1];
            BootToDFlashInfo.DID_F189[2] = DID_F189[2];
            BootToDFlashInfo.DID_F189[3] = DID_F189[3];
            BootToDFlashInfo.DID_F189[4] = DID_F189[4];
            BootToDFlashInfo.DID_F189[5] = DID_F189[5];
            BootToDFlashInfo.DID_F189[6] = DID_F189[6];
            BootToDFlashInfo.DID_F189[7] = DID_F189[7];
            WriteDFlashData(0x03u, (uint32_t *)&BootToDFlashInfo.Flag, sizeof(BootToDFlashInfo) , NeedWait);
            break;
        }
    }
}

/**
  *读取信息
  */
uint8_t ReadDFlashData(uint8_t u8BlockNum, uint32_t u32Data[], uint16_t u16Len, InitFunction initFunction)
{
    uint8_t  enResult = OK;
    enResult = DFlash_ReadData(u8BlockNum, (uint8_t *)u32Data, u16Len);
    if(enResult == OK)
    {
        //u32Data = (uint32_t *)u8Data;
        if (*u32Data != 0x7AA7A55Au)
        {
            initFunction();
            enResult = WriteDFlashData(u8BlockNum, u32Data, u16Len, NeedWait);
        }
    }else
    {
        initFunction();
        enResult = WriteDFlashData(u8BlockNum, u32Data, u16Len, NeedWait);
    }
    return enResult;
}

/**
  *写入信息
  */
uint8_t WriteDFlashData(uint8_t u8BlockNum, uint32_t u32Data[], uint16_t u16Len, uint8_t isWait)
{
    uint8_t  enResult = OK;
    (void)isWait;
    enResult = DFlash_WriteData(u8BlockNum, (uint8_t *)u32Data, u16Len);
    return enResult;
}

/**
  *所有标志位初始化
  */
void AllFlagInit(void)
{
    DiagFlag.Flag = 0x7AA7A55Au;
    DiagFlag.JumpToBootFlag = 0x00000000u;
    DiagFlag.AppValidFlag = 0x00000000u;
    DiagFlag.S11Flag = 0x00000000u;
}

/**
  *27信息初始化
  */
void Ser27ToDFlashInfoInit(void)
{
    Ser27_FlowCtrlCnt.Flag = 0x7AA7A55Au;
    Ser27_FlowCtrlCnt.LV11InvaidKyeCnt = 0u;
    Ser27_FlowCtrlCnt.LV11ReqSeedCnt = 0u;
    Ser27_FlowCtrlCnt.LV1InvaidKyeCnt = 0u;
    Ser27_FlowCtrlCnt.LV1ReqSeedCnt = 0u;
    Ser27_FlowCtrlCnt.RandNum = 0u;
}

/**
  *boot信息初始化
  */
void BootToDFlashInfoInit(void)
{
    BootToDFlashInfo.Flag = 0x7AA7A55Au;
    BootToDFlashInfo.DID_0200 = 0u;
    BootToDFlashInfo.DID_0201 = 0u;
    /*长安定义的ECU软件版本编号*/
    (void)memset(BootToDFlashInfo.DID_F189, 0x00, sizeof(BootToDFlashInfo.DID_F189));
    BootToDFlashInfo.DID_F189[0] = DID_F189[0];
    BootToDFlashInfo.DID_F189[1] = DID_F189[1];
    BootToDFlashInfo.DID_F189[2] = DID_F189[2];
    BootToDFlashInfo.DID_F189[3] = DID_F189[3];
    BootToDFlashInfo.DID_F189[4] = DID_F189[4];
    BootToDFlashInfo.DID_F189[5] = DID_F189[5];
    BootToDFlashInfo.DID_F189[6] = DID_F189[6];
    BootToDFlashInfo.DID_F189[7] = DID_F189[7];
    /*长安定义的ECU硬件版本编号*/
    (void)memset(BootToDFlashInfo.DID_F089, 0x00, sizeof(BootToDFlashInfo.DID_F089));
    BootToDFlashInfo.DID_F089[0] = 'H';
    BootToDFlashInfo.DID_F089[1] = 'W';
    BootToDFlashInfo.DID_F089[2] = ':';
    BootToDFlashInfo.DID_F089[3] = 'A';
    BootToDFlashInfo.DID_F089[4] = '.';
    BootToDFlashInfo.DID_F089[5] = '0';
    BootToDFlashInfo.DID_F089[6] = '.';
    BootToDFlashInfo.DID_F089[7] = '2';
    /*零件号*/
    (void)memset(BootToDFlashInfo.DID_F187, 0x00, sizeof(BootToDFlashInfo.DID_F187));
    BootToDFlashInfo.DID_F187[0]  = '7';
    BootToDFlashInfo.DID_F187[1]  = '9';
    BootToDFlashInfo.DID_F187[2]  = '1';
    BootToDFlashInfo.DID_F187[3]  = '2';
    BootToDFlashInfo.DID_F187[4]  = '0';
    BootToDFlashInfo.DID_F187[5]  = '8';
    BootToDFlashInfo.DID_F187[6]  = '0';
    BootToDFlashInfo.DID_F187[7]  = '-';
    BootToDFlashInfo.DID_F187[8]  = 'B';
    BootToDFlashInfo.DID_F187[9]  = 'N';
    BootToDFlashInfo.DID_F187[10] = '0';
    BootToDFlashInfo.DID_F187[11] = '8';
    /*系统供应商标识符*/
    (void)memset(BootToDFlashInfo.DID_F18A, 0x00, sizeof(BootToDFlashInfo.DID_F18A));
    BootToDFlashInfo.DID_F18A[0] = 'S';
    BootToDFlashInfo.DID_F18A[1] = '0';
    BootToDFlashInfo.DID_F18A[2] = '7';
    BootToDFlashInfo.DID_F18A[3] = '9';
    BootToDFlashInfo.DID_F18A[4] = '6';
    BootToDFlashInfo.DID_F18A[5] = '6';
    /*FBL软件版本号*/
    (void)memset(BootToDFlashInfo.DID_F170, 0x00, sizeof(BootToDFlashInfo.DID_F170));
    BootToDFlashInfo.DID_F170[0] = 'V';
    BootToDFlashInfo.DID_F170[1] = '1';
    BootToDFlashInfo.DID_F170[2] = '.';
    BootToDFlashInfo.DID_F170[3] = '4';
    /*FBL规范版本*/
    (void)memset(BootToDFlashInfo.DID_F171, 0x00, sizeof(BootToDFlashInfo.DID_F171));
    BootToDFlashInfo.DID_F171[0] = 'V';
    BootToDFlashInfo.DID_F171[1] = '2';
    BootToDFlashInfo.DID_F171[2] = '.';
    BootToDFlashInfo.DID_F171[3] = '5';
    /*应用程序刷写指纹信息*/
    (void)memset(BootToDFlashInfo.DID_F184, 0x00, sizeof(BootToDFlashInfo.DID_F184));
    /*软件件号*/
    (void)memset(BootToDFlashInfo.DID_F188, 0x00, sizeof(BootToDFlashInfo.DID_F188));
    BootToDFlashInfo.DID_F188[0]  = '9';
    BootToDFlashInfo.DID_F188[1]  = '2';
    BootToDFlashInfo.DID_F188[2]  = '1';
    BootToDFlashInfo.DID_F188[3]  = '0';
    BootToDFlashInfo.DID_F188[4]  = '1';
    BootToDFlashInfo.DID_F188[5]  = '0';
    BootToDFlashInfo.DID_F188[6]  = '1';
    BootToDFlashInfo.DID_F188[7]  = '-';
    BootToDFlashInfo.DID_F188[8]  = 'B';
    BootToDFlashInfo.DID_F188[9]  = 'N';
    BootToDFlashInfo.DID_F188[10] = '0';
    BootToDFlashInfo.DID_F188[11] = '8';
    /*处理器上的操作系统的软件版本号*/
    (void)memset(BootToDFlashInfo.DID_F087, 0x00, sizeof(BootToDFlashInfo.DID_F087));

}

/**
  *2E信息初始化
  */
void Ser2EToDFlashInfoInit(void)
{

    /*下线配置*/
    /*VIN码*/
    (void)memset(Ser2EToDFlashInfo.DID_F190, 0x00, sizeof(Ser2EToDFlashInfo.DID_F190));
    (void)memset(Ser2EToDFlashInfo.DID_F1F0, 0x00, sizeof(Ser2EToDFlashInfo.DID_F1F0));
}

/**
  *DTC信息初始化
  */
void DTCToDFlashInfoInit(void)
{
    uint8_t i;
    DTCToDFlashInfo.Flag = 0x7AA7A55Au;
    for (i = 0u; i < sizeof(DTCToDFlashInfo.DTC) / sizeof(DTCUnion); ++i)
    {
        DTCToDFlashInfo.DTC[i].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[i].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[i].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[i].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[i].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[i].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[i].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[i].Snapshot.DTCLocalTime_second = 0x00u;
        DTCToDFlashInfo.DTC[i].Snapshot.DTCLocalTime_minute = 0x00u;
        DTCToDFlashInfo.DTC[i].Snapshot.DTCLocalTime_hour = 0x00u;
        DTCToDFlashInfo.DTC[i].Snapshot.DTCLocalTime_day = 0x00u;
        DTCToDFlashInfo.DTC[i].Snapshot.DTCLocalTime_month = 0x00u;
        DTCToDFlashInfo.DTC[i].Snapshot.DTCLocalTime_year = 0x00u;
    }
    for(i = 0u; i < (sizeof(msgList) / sizeof(msgList[0u])); ++i)
    {
        msgList[i][3] = 0u;
    }
    for(i = 0u; i < (sizeof(DTCTimerList) / sizeof(DTCTimerList[0u])); ++i)
    {
        DTCTimerList[i] = 0u;
    }
}

/**
  *燃油初始化
  */
void CAFDToDFlashInfoInit(void)
{

}

/**
  *DTC信息配置初始化
  */
void DTCConfigurationInit(void)
{
    /*if(!Val_EOL_F110_EMS())
    {
        DTCToDFlashInfo.DTC[cnU010087].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU010087].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU010087].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU010087].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU010087].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU010087].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU010087].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU010087].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_TCU())
    {
        DTCToDFlashInfo.DTC[cnU010187].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU010187].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU010187].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU010187].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU010187].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU010187].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU010187].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU010187].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_ESC())
    {
        DTCToDFlashInfo.DTC[cnU012287].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU012287].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU012287].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU012287].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU012287].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU012287].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU012287].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU012287].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_EPS())
    {
        DTCToDFlashInfo.DTC[cnU013187].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU013187].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU013187].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU013187].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU013187].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU013187].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU013187].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU013187].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_GW())
    {
        DTCToDFlashInfo.DTC[cnU014687].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU014687].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU014687].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU014687].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU014687].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU014687].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU014687].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU014687].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_BCM())
    {
        DTCToDFlashInfo.DTC[cnU014087].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU014087].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU014087].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU014087].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU014087].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU014087].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU014087].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU014087].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_ACU())
    {
        DTCToDFlashInfo.DTC[cnU015187].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU015187].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU015187].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU015187].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU015187].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU015187].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU015187].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU015187].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_AC())
    {
        DTCToDFlashInfo.DTC[cnU016487].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU016487].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU016487].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU016487].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU016487].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU016487].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU016487].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU016487].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_PEPS())
    {
        DTCToDFlashInfo.DTC[cnU021487].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU021487].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU021487].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU021487].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU021487].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU021487].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU021487].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU021487].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_BMSH())
    {
        DTCToDFlashInfo.DTC[cnU011287].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU011287].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU011287].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU011287].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU011287].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU011287].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU011287].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU011287].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_AVAS())
    {
        DTCToDFlashInfo.DTC[cnU016087].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU016087].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU016087].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU016087].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU016087].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU016087].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU016087].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU016087].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_VCU())
    {
        DTCToDFlashInfo.DTC[cnU111487].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU111487].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU111487].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU111487].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU111487].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU111487].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU111487].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU111487].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_OBC())
    {
        DTCToDFlashInfo.DTC[cnU111587].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU111587].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU111587].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU111587].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU111587].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU111587].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU111587].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU111587].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_IPU())
    {
        DTCToDFlashInfo.DTC[cnU011087].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU011087].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU011087].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU011087].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU011087].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU011087].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU011087].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU011087].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_MMI())
    {
        DTCToDFlashInfo.DTC[cnU015687].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU015687].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU015687].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU015687].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU015687].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU015687].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU015687].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU015687].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_DSCU4())
    {
        DTCToDFlashInfo.DTC[cnU020887].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU020887].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU020887].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU020887].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU020887].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU020887].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU020887].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU020887].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_IB())
    {
        DTCToDFlashInfo.DTC[cnU111A87].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU111A87].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU111A87].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU111A87].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU111A87].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU111A87].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU111A87].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU111A87].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_ESCL())
    {
        DTCToDFlashInfo.DTC[cnU023687].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU023687].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU023687].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU023687].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU023687].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU023687].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU023687].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU023687].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_EGSM())
    {
        DTCToDFlashInfo.DTC[cnU010387].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU010387].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU010387].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU010387].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU010387].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU010387].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU010387].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU010387].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_FRS())
    {
        DTCToDFlashInfo.DTC[cnU023587].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU023587].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU023587].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU023587].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU023587].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU023587].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU023587].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU023587].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_FCS())
    {
        DTCToDFlashInfo.DTC[cnU120387].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU120387].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU120387].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU120387].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU120387].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU120387].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU120387].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU120387].Snapshot.DTCLocalTime = 0x00000000u;
    }
    if(!Val_EOL_F110_PAS())
    {
        DTCToDFlashInfo.DTC[cnU017087].DTCStatus = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC[cnU017087].DTCAgeing = 0x00u;
        DTCToDFlashInfo.DTC[cnU017087].DTCCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU017087].DTCPendingCount = 0x00u;
        DTCToDFlashInfo.DTC[cnU017087].Snapshot.OdometerValue = 0x00000000u;
        DTCToDFlashInfo.DTC[cnU017087].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC[cnU017087].Snapshot.VehicleSpeed = 0x0000u;
        DTCToDFlashInfo.DTC[cnU017087].Snapshot.DTCLocalTime = 0x00000000u;
    }*/
}
