/******************************************************************************
 * $Revision: 423 $
 * $Date:: 2017-04-07 16:03:30 +0900#$
 *****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************
 ** \file workflash.c
 **
 ** Headerfile for WorkFlash functions
 **
 ** History:
 **   - 2014-06-02  0.01  HS  Initial version for Traveo
 *****************************************************************************/

#ifndef __WORKFLASH_H__
#define __WORKFLASH_H__


/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "base_types.h"

/* C binding of definitions if building with C++ compiler */
#ifdef __cplusplus
extern "C" {
#endif


/**
 *****************************************************************************
 ** \defgroup WorkFlashGroup Work FLash (WFLASH)
 **
 ** \brief This section describes the interface for the Work Flash.
 **
 ** Provided functions of WFlash module:
 **
 ** - WFlash_EnableWriteAccess()
 ** - WFlash_DisableWriteAccess()
 ** - WFlash_Reset()
 ** - WFlash_ReadResetCmd()
 ** - WFlash_Erase()
 ** - WFlash_Write8()
 ** - WFlash_Write16()
 ** - WFlash_Write32()
 **
 ** WFlash_EnableWriteAccess() is used to get write permission to the Work Flash.
 ** When SHE has write permission, this function return an Error(ErrorInvalidMode).
 ** In that condition, retry this function until returning Ok if needed.
 ** Other functions have to be used after calling this function.
 ** WFlash_DisableWriteAccess() is used to release write permission to the Work Flash.
 ** WFlash_Reset() is used to reset the flash macro.
 ** WFlash_ReadResetCmd() is used to clear hanging status.
 ** If WFlash_Erase() or write functions(WFlash_WriteXX()) are used by non-blocking
 ** (parameter bBlocking==FALSE), functions do not wait until flash status is ready.
 ** Please wait by application code. At this time, to check the Work Flash status,
 ** WFlash_GetStatus() can be used. If flash is hanging, WFlash_ReadResetCmd()
 ** can be used to clear hanging status.
 ** If WFlash_Erase() or write functions(WFlash_WriteXX()) are used by blocking
 ** (parameter bBlocking==TRUE), those functions wait until status is ready.
 **
 ** If application use this module, it has to use these APIs by combining.
 **
 *****************************************************************************/
/*! @{ */


/*****************************************************************************/
/* Global pre-processor symbols/macros ('define')                            */
/*****************************************************************************/


/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/
/**
 ******************************************************************************
 ** \brief Work Flash mirror area selection
 ******************************************************************************/
typedef enum en_wflash_mirror_area{
    WFlashMirror1,          /*!< Work Flash memory mirror area 1 (Read/Write with ECC if enabled.) */
    WFlashMirror3           /*!< Work Flash memory mirror area 3 (Read/Write with no ECC) */
}en_wflash_mirror_area_t;

/**
 ******************************************************************************
 ** \brief Work Flash status
 ******************************************************************************/
typedef enum en_wflash_status
{
    WFlashReady = 0,        /*!< Flash memory can start the execution of the next command */
    WFlashNotReady  = 1,    /*!< Writing or erasing is on going. */
    WFlashHang  = 2         /*!< Writing or erasing has failed. */
}en_wflash_status_t;


/*****************************************************************************/
/* Global variable declarations ('extern', definition in C source)           */
/*****************************************************************************/


/*****************************************************************************/
/* Global function prototypes ('extern', definition in C source)             */
/*****************************************************************************/
extern en_result_t WFlash_EnableWriteAccess(void);
extern en_result_t WFlash_DisableWriteAccess(void);
extern en_result_t WFlash_Reset(void);
extern en_result_t WFlash_Erase(uint8_t u8SectorNum, boolean_t bBlocking);
extern en_result_t WFlash_Write8(uint32_t u32OffsetAddress,
                                 uint8_t u8Data,
                                 en_wflash_mirror_area_t enMirrorArea,
                                 boolean_t bBlocking);
extern en_result_t WFlash_Write16(uint32_t u32OffsetAddress,
                                  uint16_t u16Data,
                                  en_wflash_mirror_area_t enMirrorArea,
                                  boolean_t bBlocking);
extern en_result_t WFlash_Write32(uint32_t u32OffsetAddress,
                                  uint32_t u32Data,
                                  en_wflash_mirror_area_t enMirrorArea,
                                  boolean_t bBlocking);
#if 0
extern en_result_t WFlash_Write64(uint32_t u32OffsetAddress,
                                  uint64_t u64Data,
                                  en_wflash_mirror_area_t enMirrorArea,
                                  boolean_t bBlocking);
#endif
extern en_wflash_status_t WFlash_GetStatus(void);
extern en_result_t WFlash_ReadResetCmd(void);

/*! @} */
#ifdef __cplusplus
}
#endif

#endif /* __WORKFLASH_H__ */
/*****************************************************************************/
/* EOF (not truncated)                                                       */
/*****************************************************************************/
