/******************************************************************************
 * $Revision: 423 $
 * $Date:: 2017-04-07 16:03:30 +0900#$
 *****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************/
/** \file mfs_i2c.h
 **
 ** Headerfile for I2C functions
 **
 ** History:
 **   - 2014-10-24  1.0  HS  Initial version for Traveo I2C Interface
 **   - 2014-11-25  1.1  PDL Version including interrupts handling and Rx/Tx fifo 
 *****************************************************************************/

#ifndef __I2C_H__
#define __I2C_H__

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
// #include "base_types.h"
#include "pdl.h"

/* C binding of definitions if building with C++ compiler */
#ifdef __cplusplus
extern "C"
{
#endif

/**
 *****************************************************************************
 ** \defgroup I2CGroup Inter-Integrated Circuit (I2C)
 **
 ** \brief This section describes the interface for the Inter-Integrated Circuit.
 **
 ** Provided functions of I2C module:
 **- Mfs_I2c_Init()
 **- Mfs_I2c_Start()
 **- Mfs_I2c_Write()
 **- Mfs_I2c_Acknowlegde()
 **- Mfs_I2c_Continue()
 **- Mfs_I2c_Read()
 **- Mfs_I2c_ReadLast()
 **- Mfs_I2c_Stop()
 **- Mfs_I2c_EnableInterupt()
 **- Mfs_I2c_DisableInterupt()
 **- Mfs_I2c_WriteSequence()
 **- Mfs_I2c_ReadSequence()
 **- Mfs_I2c_EnableTxFifo()
 **- Mfs_I2c_WriteString()
 **- Mfs_I2c_EnableRxFifo()
 **- Mfs_I2c_ReadString()
 **- Mfs_I2c_DisableRxTxFifo()
 **- Mfs_I2c_GetReadValues()
 **- Mfs_I2c_DeInit()
 ** 
 ** Mfs_I2c_Init() must be used for activation and setup of an I2C channel. All
 ** I2C connection related parameters can be set with parameter
 ** pstcConfig of type #stc_i2c_config_t.
 ** Mfs_I2c_Start() is used to Setup Communication channel between master and slave.
 ** Mfs_I2c_Write() to write a byte at a particular location in slave 
 ** Mfs_I2c_Acknowlegde()for getting acknowledge from slave
 ** Mfs_I2c_Continue() to generate a repeated start condition
 ** Mfs_I2c_Read() is used to read a byte from slave
 ** Mfs_I2c_ReadLast() master need to acknowledge last byte only
 ** Mfs_I2c_Stop() to release I2C Bus
 ** Mfs_I2c_EnableInterupt() to enable an interupt(Status, error, transmission or reception)
 ** Mfs_I2c_DisableInterupt()to disable an interupt(Status, error, transmission or reception)
 ** Mfs_I2c_WriteSequence() to Write given number of  bytes from a buffer in a sequence 
 ** to a slave at a given location
 ** Mfs_I2c_ReadSequence()to read given number of bytes from slave in a sequence 
 ** from a given location and return in provided buffer
 ** Mfs_I2c_EnableTxFifo() to enable transmission FIFO
 ** Mfs_I2c_WriteString() to write a string using transmission fifo
 ** Mfs_I2c_EnableRxFifo()to enable reception FIFO
 ** Mfs_I2c_ReadString()to read a string using reception fifo
 ** Mfs_I2c_DisableRxTxFifo()to disable transmission and reception FIFO
 ** Mfs_I2c_GetReadValues() to get read values back stored in static buffer
 ** Mfs_I2c_DeInit() (Will reset all i2c related register to their
 **                 default values.)
 *****************************************************************************/
/** @{ */

/*****************************************************************************/
/* Global pre-processor symbols/macros ('#define')                           */
/*****************************************************************************/
/** I2C ch0 */
#define I2C0_Type   ((pstc_i2c_type_t)&CPG_MFS00_I2C)
#define I2C0_TIE    CPG_MFS00_I2C_SCR_TIE
#define I2C0_RIE    CPG_MFS00_I2C_SCR_RIE
/** I2C ch1 */
#define I2C1_Type   ((pstc_i2c_type_t)&CPG_MFS01_I2C)
#define I2C1_TIE    CPG_MFS01_I2C_SCR_TIE
#define I2C1_RIE    CPG_MFS01_I2C_SCR_RIE
/** I2C ch2 */
#define I2C2_Type   ((pstc_i2c_type_t)&CPG_MFS02_I2C)
#define I2C2_TIE    CPG_MFS02_I2C_SCR_TIE
#define I2C2_RIE    CPG_MFS02_I2C_SCR_RIE
/** I2C ch3 */
#define I2C3_Type   ((pstc_i2c_type_t)&CPG_MFS03_I2C)
#define I2C3_TIE    CPG_MFS03_I2C_SCR_TIE
#define I2C3_RIE    CPG_MFS03_I2C_SCR_RIE
/** I2C ch4 */
#define I2C4_Type   ((pstc_i2c_type_t)&CPG_MFS04_I2C)
#define I2C4_TIE    CPG_MFS04_I2C_SCR_TIE
#define I2C4_RIE    CPG_MFS04_I2C_SCR_RIE
/** I2C ch5 */
#define I2C5_Type   ((pstc_i2c_type_t)&CPG_MFS05_I2C)
#define I2C5_TIE    CPG_MFS05_I2C_SCR_TIE
#define I2C5_RIE    CPG_MFS05_I2C_SCR_RIE
/** I2C ch6 */
#define I2C6_Type   ((pstc_i2c_type_t)&CPG_MFS06_I2C)
#define I2C6_TIE    CPG_MFS06_I2C_SCR_TIE
#define I2C6_RIE    CPG_MFS06_I2C_SCR_RIE
/** I2C ch7 */
#define I2C7_Type   ((pstc_i2c_type_t)&CPG_MFS07_I2C)
#define I2C7_TIE    CPG_MFS07_I2C_SCR_TIE
#define I2C7_RIE    CPG_MFS07_I2C_SCR_RIE
/** I2C ch8 */
#define I2C8_Type   ((pstc_i2c_type_t)&CPG_MFS08_I2C)
#define I2C8_TIE    CPG_MFS08_I2C_SCR_TIE
#define I2C8_RIE    CPG_MFS08_I2C_SCR_RIE
/** I2C ch9 */
#define I2C9_Type   ((pstc_i2c_type_t)&CPG_MFS09_I2C)
#define I2C9_TIE    CPG_MFS09_I2C_SCR_TIE
#define I2C9_RIE    CPG_MFS09_I2C_SCR_RIE
/** I2C ch10 */
#define I2C10_Type   ((pstc_i2c_type_t)&CPG_MFS10_I2C)
#define I2C10_TIE    CPG_MFS10_I2C_SCR_TIE
#define I2C10_RIE    CPG_MFS10_I2C_SCR_RIE
/** I2C ch11 */
#define I2C11_Type   ((pstc_i2c_type_t)&CPG_MFS11_I2C)
#define I2C11_TIE    CPG_MFS11_I2C_SCR_TIE
#define I2C11_RIE    CPG_MFS11_I2C_SCR_RIE

/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/
/**
 *****************************************************************************
 ** \brief I2C Unit Type
 *****************************************************************************/
typedef  volatile stc_cpg_mfsn_i2c_t*  pstc_i2c_type_t;

/**
 *****************************************************************************
 ** \brief I2C configuration
 ** This settings are per I2C instance.
 *****************************************************************************/
typedef struct stc_i2c_config
{
    uint32_t                  u32DataRate;                        ///< Bits per second 
    uint8_t                   u8SlaveAddress;                     ///< Slave Address
    boolean_t                 bSlaveAddressEnable;                ///< Enable slave address detection. 
    boolean_t                 bStartConditionContinue;            ///< Start condition Continue
    boolean_t                 bAcknowledgeEnable;                 ///< Enable Acknowledge
    boolean_t                 bWaitSelect;                        ///< TRUE = Wait after data transmission is completed (8th bit);FALSE= Wait after acknowledgment (9th bit).
    boolean_t                 bInterruptEnable;                   ///< Enable Interupt
    boolean_t                 bDmaModeEnable;                     ///< Enable DMA Mode
    func_ptr_t                pfnStopConditionCallback ;          ///< IRQ Callback
    func_ptr_t                pfnRepeatedStartCallback ;          ///< IRQ Callback
    func_ptr_t                pfnBusErrorCallback;                ///< IRQ Callback
    func_ptr_t                pfnArbitrationLostCallback;         ///< IRQ Callback
    func_ptr_t                pfnReservedAddressDetectedCallback; ///< IRQ Callback
    func_ptr_t                pfnNACKCallback;                    ///< IRQ Callback
    func_ptr_t                pfnTransmissionFifoOverRunCallback; ///< IRQ Callback
    func_ptr_t                pfnReceptionFifoUnderRunCallback;   ///< IRQ Callback
    func_ptr_t                pfnReceptionFifoFullCallback ;      ///< IRQ Callback
    unsigned char             m_readBuffer[6];                    ///< Stores read data
    unsigned char             m_writeBuffer[6];                   ///< Stores data to be written
      
}stc_i2c_config_t;


/*****************************************************************************/
/* Global variable declarations ('extern', definition in C source)           */
/*****************************************************************************/


/*****************************************************************************/
/* Global function prototypes ('extern', definition in C source)             */
/*****************************************************************************/

extern en_result_t Mfs_I2c_Init(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                stc_i2c_config_t *pstcConfig);

extern  en_result_t Mfs_I2c_Start(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                  uint8_t u8SlaveAddress);

extern  en_result_t Mfs_I2c_Write(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                  unsigned char cValue);

extern  en_result_t Mfs_I2c_Acknowlegde(volatile stc_cpg_mfsn_i2c_t *pstcI2c);

extern  en_result_t Mfs_I2c_Continue(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                     uint8_t u8SlaveAddress);

//To send Acknowledge for every byte read
extern  en_result_t Mfs_I2c_Read(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                 unsigned char* cReadValue);

//To not send Acknowledge for last byte read to indicate end of data reception
extern  en_result_t Mfs_I2c_ReadLast(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                     unsigned char* cReadValue);

extern  en_result_t Mfs_I2c_Stop(volatile stc_cpg_mfsn_i2c_t *pstcI2c);

//For interupts
extern en_result_t Mfs_I2c_EnableInterupt(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                          boolean_t bEnableStatus,
                                          boolean_t bEnableRx,
                                          boolean_t bEnableTx);


extern en_result_t Mfs_I2c_DisableInterupt(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                           boolean_t bDisableStatus,
                                           boolean_t bDisableRx, 
                                           boolean_t bDisableTx);

extern en_result_t Mfs_I2c_WriteSequence(volatile stc_cpg_mfsn_i2c_t* pstcI2c,
                                         uint8_t u8SlaveAddress,
                                         uint8_t  u8WriteLocation,
                                         uint8_t  u8Data[],
                                         uint8_t  u8DataLength);

extern en_result_t Mfs_I2c_ReadSequence(volatile stc_cpg_mfsn_i2c_t* pstcI2c,
                                        uint8_t u8SlaveAddress,
                                        uint8_t  u8ReadLocation,
                                        uint8_t  u8Data[],
                                        uint8_t  u8DataLength);
//
//for FIFO
extern en_result_t Mfs_I2c_EnableTxFifo(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                        boolean_t bFifoTxInterruptEnable);

extern en_result_t Mfs_I2c_WriteString(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                       const char *cString);


extern en_result_t Mfs_I2c_EnableRxFifo(volatile stc_cpg_mfsn_i2c_t *pstcI2c, 
                                     boolean_t bFifoRxInterruptEnable,
                                     uint32_t  u32Readbytecount);

extern en_result_t Mfs_I2c_ReadString(volatile stc_cpg_mfsn_i2c_t *pstcI2c);

extern en_result_t Mfs_I2c_DisableRxTxFifo(volatile stc_cpg_mfsn_i2c_t *pstcI2c);

extern en_result_t Mfs_I2c_GetReadValues(volatile stc_cpg_mfsn_i2c_t *pstcI2c,
                                         unsigned char cReadBuffer[]);

extern  en_result_t Mfs_I2c_DeInit(volatile stc_cpg_mfsn_i2c_t *pstcI2c);

/** @} */

#ifdef __cplusplus
}
#endif

#endif /* __I2C_H__ */
/*****************************************************************************/
/* EOF (not truncated)                                                       */
/*****************************************************************************/
