
#ifndef __CANFD_H__
    #define __CANFD_H__

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/


/**
*****************************************************************************
** \defgroup CanFD320Group CAN with Flexible Data rate (CAN FD) MCAN3.2.0
**
** \brief This section describes the interface for the CAN with Flexible Data rate.
**
** Provided functions of CAN-FD module:
** - CanFD_Init() (Initialization of CAN-FD module)
** - CanFD_UpdateAndTransmitMsgBuffer() (Used to update the data in a message buffer and to transmit a message)
*
** \brief How to use the CAN-FD module:
** First, to initialise a CAN-FD module, CanFD_Init() must be called.
** Initialize the CAN-FD module with the configuration parameters, and start
** the CAN-FD communication.
** The callback functions are optional, but recommended, otherwise there is no
** report to the API in case of any error and transmission/reception.
**
** For transmitting CAN-FD messages the function CanFD_UpdateAndTransmitMsgBuffer()
** has to be used.
** It is possible to set a callback function which will be notified whenever a
** message has been received.
**
** \description
** Specification of sample software is as follows:
**   - Selectable CAN-FD mode or CAN mode.
**   - CAN clock : 40MHz
**   - CAN Bit rate : 500kHz
**   - CAN-FD Bit rate : 5MHz (CAN-FD mode)
**   - Standard ID filter : 0x10
**                          store Rx buffer 0
**   - Extended Message ID Filter : 0x10001
**                          store Rx buffer 1
**   - Tx/Rx Element Size : 8 byte (CAN mode)
**     Tx/Rx Element Size : 64 byte (CAN-FD mode)
**   - Tx/Rx FIFO : Not use
**   - Interrupt : DRXE=1 (Message stored to Dedicated Rx Buffer)
**                 (Inactive other)
**
** Send data of ID:0x200 periodically in the main().
** Echoback by incrementing the ID of the Rx data received by
** the receiving interrupt.(CanFDIrqHandler() --- CanFDCallbackRx())
**
** \note Following settings are not implemented.
**         - Using Tx/warning/error interrupt
**         - Using Tx/Rx FIFO
**         - Variable ID Filter configuration
**         - Variable Tx/Rx Element size configuration
**         - Variable Message buffer size configuration
**
*****************************************************************************/
//@{

/*****************************************************************************/
/* Global pre-processor symbols/macros ('define')                            */
/*****************************************************************************/

/** Size of Word data bytes in a receive or transmit operation (16word = 64byte) */
    #define CANFD_MESSAGE_DATA_BUFFER_SIZE     16

/** Offset of Message RAM */
    #define	MSGRAMOFFSET    (0x00008000ul)

/** Size of Word data bytes in Message RAM */
    #define	CANFD0MRWORDS   (4096)

#define CANFD0_Type   ((pstc_canfd_type_t)&CPG_CANFD0)
/** CAN-FD ch1 */
#define CANFD1_Type   ((pstc_canfd_type_t)&CPG_CANFD1)
/** CAN-FD ch5 */
#define CANFD2_Type   ((pstc_canfd_type_t)&CPG_CANFD2)
/** CAN-FD ch6 */
#define CANFD3_Type   ((pstc_canfd_type_t)&CPG_CANFD3)


/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/
/**
*****************************************************************************
** \brief CAN-FD Unit Type
*****************************************************************************/
    typedef  volatile stc_cpg_canfdn_t*  pstc_canfd_type_t;
    
    /**
    ******************************************************************************
    ** \brief CANFD Unit configuration structures
    ******************************************************************************/
    
    /**
    *****************************************************************************
    ** \brief DLC-Word conversion
    *****************************************************************************/
    static const int iDlcInWord[] = {
        2,			/*!<  8 bytes */
        3,			/*!< 12 bytes */
        4,			/*!< 16 bytes */
        5,			/*!< 20 bytes */
        6,			/*!< 24 bytes */
        8,			/*!< 32 bytes */
        12,			/*!< 48 bytes */
        16			/*!< 64 bytes */
    };
    
    /**
    ******************************************************************************
    ** \brief Redefinition of R0 register bitfields
    ******************************************************************************/
    typedef struct stc_canfd_r0
    {
        volatile uint32_t  ID         :29;  /*!< Identifier */
        volatile uint32_t  RTR        : 1;  /*!< Remote Transmission Request. 0:data frame, 1:remote frame */
        volatile uint32_t  XTD        : 1;  /*!< Extended Identifier. 0:11-bit standard identifier, 1:29-bit extended identifier */
        volatile uint32_t  ESI        : 1;  /*!< Error State Indicator. 0:error active, 1:error passive */
    } stc_canfd_r0_t;
    
    /**
    ******************************************************************************
    ** \brief Redefinition of R1 register bitfields
    ******************************************************************************/
    typedef struct stc_canfd_r1
    {
        volatile uint32_t  RXTS       :16;  /*!< Rx Timestamp */
        volatile uint32_t  DLC        : 4;  /*!< Data Length Code */
        volatile uint32_t  BRS        : 1;  /*!< Bit Rate Switch. 0:without bit rate switching, 1:with bit rate switching */
        volatile uint32_t  FDF        : 1;  /*!< Extended Data Length. 0:Classic CAN frame format, 1:CAN FD frame format */
        uint32_t  RESERVED1  : 2;  /*!< Reserved bits */
        volatile uint32_t  FIDX       : 7;  /*!< Filter Index */
        volatile uint32_t  ANMF       : 1;  /*!< Accepted Non-matching Frame. 0:matching filter index FIDX, 1:did not match any Rx filter element */
    } stc_canfd_r1_t;
    
    /**
    ******************************************************************************
    ** \brief Redefinition of Rx Buffer Register
    ******************************************************************************/
    typedef struct stc_canfd_rx_buffer
    {
        stc_canfd_r0_t  R0_f;               /*!< Rx buffer element for R0. See #stc_canfd_r0_t. */
        stc_canfd_r1_t  R1_f;               /*!< Rx buffer element for R1. See #stc_canfd_r1_t. */
        uint32_t        DATA_AREA_f[16];    /*!< Rx buffer element for Rn */
    } stc_canfd_rx_buffer_t;
    
    
    /**
    ******************************************************************************
    ** \brief Redefinition of T0 register bitfields
    ******************************************************************************/
    typedef struct stc_canfd_t0
    {
        volatile uint32_t  ID         :29;  /*!< Identifier */
        volatile uint32_t  RTR        : 1;  /*!< Remote Transmission Request. 0:data frame, 1:remote frame */
        volatile uint32_t  XTD        : 1;  /*!< Extended Identifier. 0:11-bit standard identifier, 1:29-bit extended identifier */
        volatile uint32_t  ESI        : 1;  /*!< Error State Indicator. 0:ESI bit in CAN FD format depends only on error passive flag, 1:ESI bit in CAN FD format transmitted recessive. */
    } stc_canfd_t0_t;
    
    /**
    ******************************************************************************
    ** \brief Redefinition of T1 register bitfields
    ******************************************************************************/
    typedef struct stc_canfd_t1
    {
        uint32_t  RESERVED1  :16;  /*!< Reserved bits */
        volatile uint32_t  DLC        : 4;  /*!< Data Length Code */
        volatile uint32_t  BRS        : 1;  /*!< Bit Rate Switching. 0:CAN FD frames transmitted without bit rate switching, CAN FD frames transmitted with bit rate switching */
        volatile uint32_t  FDF        : 1;  /*!< FD Format. 0:Frame transmitted in Classic CAN format, 1:Frame transmitted in CAN FD format */
        volatile uint32_t  EFC        : 1;  /*!< Event FIFO Control. 0:Donft store Tx events, 1:Store Tx events */
        volatile uint32_t  MM         : 8;  /*!< Message Marker */
    } stc_canfd_t1_t;
    
    /**
    ******************************************************************************
    ** \brief Redefinition of Tx Buffer Register
    ******************************************************************************/
    typedef struct stc_canfd_tx_buffer
    {
        stc_canfd_t0_t  T0_f;               /*!< Tx buffer element for T0. See #stc_canfd_t0_t. */
        stc_canfd_t1_t  T1_f;               /*!< Tx buffer element for T1. See #stc_canfd_t1_t. */
        uint32_t        DATA_AREA_f[16];    /*!< Tx buffer element for Tn */
    } stc_canfd_tx_buffer_t;
    
     /**
    ******************************************************************************
    ** \brief Redefinition of E1 register bitfields
    ******************************************************************************/
    typedef struct stc_canfd_e1
    {
        volatile uint32_t  TXTS       :16;  /*!< Tx Timestamp */
        volatile uint32_t  DLC        : 4;  /*!< Data Length Code */
        volatile uint32_t  BRS        : 1;  /*!< Bit Rate Switching. 0:CAN FD frames transmitted without bit rate switching, CAN FD frames transmitted with bit rate switching */
        volatile uint32_t  FDF        : 1;  /*!< FD Format. 0:Frame transmitted in Classic CAN format, 1:Frame transmitted in CAN FD format */
        volatile uint32_t  ET         : 2;  /*!< Event Type. 1:Tx event. 2:Transmission in spite of cancellation.*/
        volatile uint32_t  MM         : 8;  /*!< Message Marker */
    } stc_canfd_e1_t;
    
    /**
    ******************************************************************************
    ** \brief Redefinition of Tx Event Register
    ******************************************************************************/
    typedef struct stc_canfd_tx_event
    {
        stc_canfd_t0_t  E0_f;               /*!< Tx buffer element for T0. See #stc_canfd_t0_t. */
        stc_canfd_e1_t  E1_f;               /*!< Tx buffer element for T1. See #stc_canfd_t1_t. */
    } stc_canfd_tx_event_t;
    
    /**
    ******************************************************************************
    ** \brief Redefinition of CANP_CTR register bitfields
    ******************************************************************************/
    typedef struct stc_canfd_canp_ctr
    {
        volatile uint32_t CANPRE     : 6;   /*!< CAN prescaler division setting bits */
        uint32_t RESERVED1  : 2;   /*!< Reserved bits */
        volatile uint32_t CPCKS      : 1;   /*!< CAN prescaler source clock selection bit */
        uint32_t RESERVED2  :23;   /*!< Reserved bits */
    } stc_canfd_canp_ctr_t;
    
    /**
    ******************************************************************************
    ** \brief Redefinition of CANP_STR register bitfields
    ******************************************************************************/
    typedef struct stc_canfd_canp_str
    {
        volatile uint32_t BUSY       : 1;   /*!< status of switching by the CAN prescaler */
        volatile uint32_t SCKM       : 2;   /*!< Source clock display */
        uint32_t RESERVED1  :29;   /*!< Reserved bits */
    } stc_canfd_canp_str_t;
    
    /**
    ******************************************************************************
    ** \brief Redefinition of CAN PRESCALER registers
    ******************************************************************************/
    typedef struct stc_canprescaler
    {
        stc_canfd_canp_ctr_t   CTR_f;   /*!< CAN Prescaler Control */
        stc_canfd_canp_str_t   STR_f;   /*!< CAN Prescaler Status */
    } stc_canprescaler_t;
    
    
    /**
    ******************************************************************************
    ** \brief Redefinition of Message ID Filter register
    ******************************************************************************/
    typedef struct stc_id_filter
    {
        volatile uint32_t SFID2      :11;   /*!< Standard Filter ID 2 */
        uint32_t RESERVED4  : 5;   /*!< Reserved bits */
        volatile uint32_t SFID1      :11;   /*!< Standard Filter ID 1 */
        volatile uint32_t SFEC       : 3;   /*!< Standard Filter Element Configuration */
        volatile uint32_t SFT        : 2;   /*!< Standard Filter Type */
    } stc_id_filter_t;
    
    
    /**
    ******************************************************************************
    ** \brief Redefinition of F0 register bitfields
    ******************************************************************************/
    typedef struct stc_canfd_f0
    {
        volatile uint32_t EFID1      :29;   /*!< Extended Filter ID 1 */
        volatile uint32_t EFEC       : 3;   /*!< Extended Filter Element Configuration */
    } stc_canfd_f0_t;
    
    /**
    ******************************************************************************
    ** \brief Redefinition of F1 register bitfields
    ******************************************************************************/
    typedef struct stc_canfd_f1
    {
        volatile uint32_t EFID2      :29;   /*!< Extended Filter ID 2 */
        uint32_t RESERVED1  : 1;   /*!< Reserved bits */
        volatile uint32_t EFT        : 2;   /*!< Extended Filter Type */
    } stc_canfd_f1_t;
    
    /**
    ******************************************************************************
    ** \brief Redefinition of Extended Message ID Filter register
    ******************************************************************************/
    typedef struct stc_extid_filter
    {
        stc_canfd_f0_t  F0_f;   /*!< Extended Message ID Filter Element for F0. See #stc_canfd_f0_t. */
        stc_canfd_f1_t  F1_f;   /*!< Extended Message ID Filter Element for F1. See #stc_canfd_f1_t. */
    } stc_extid_filter_t;
    
    
    /**
    *****************************************************************************
    ** \brief CAN-FD bitrate.
    **
    ** This structure is used to set the bitrate register.
    *****************************************************************************/
    typedef struct stc_canfd_bitrate
    {
        uint16_t  u16Prescaler;    /*!< Baud Rate Prescaler */
        uint8_t   u8TimeSegment1;  /*!< Time segment before sample point */
        uint8_t   u8TimeSegment2;  /*!< Time segment after sample point */
        uint8_t   u8SyncJumpWidth; /*!< Synchronization Jump Width */
    } stc_canfd_bitrate_t;
    
    
    /**
    *****************************************************************************
    ** \brief CAN-FD message identifier
    **
    ** This structure is used to set the CAN-FD message identifier for transmit and
    ** receive operations.
    *****************************************************************************/
    typedef struct stc_canfd_msg_id
    {
        uint32_t u32Identifier;     /*!< 11- or 29-bit identifier. The valid bit length depends on bExtended. */
        boolean_t bExtended;        /*!< TRUE: 29-bit identifier,  FALSE: 11-bit identifier. */
    } stc_canfd_msg_id_t;
    
    
    /**
    *****************************************************************************
    ** \brief CAN-FD message data.
    **
    ** Data structure for transmit and receive operations
    *****************************************************************************/
    typedef struct stc_canfd_msg_data
    {
        uint32_t au32Data[CANFD_MESSAGE_DATA_BUFFER_SIZE];  /*!< Data of CAN-FD message. */
        uint8_t u8DataLengthCode;  /*!< Number of valid bytes in au8Data and DLC of CAN-FD message. */
    } stc_canfd_msg_data_t;
    
    /**
    *****************************************************************************
    ** \brief CAN-FD status.
    **
    ** These state values represent the current CAN-FD state, and are used in the
    ** CAN-FD status changed callback function (canfd_status_chg_func_ptr_t).
    *****************************************************************************/
    typedef enum en_canfd_status
    {
        CanFDNoError    = 0,   /*!< No error pending. */
        CanFDStuffError = 1,   /*!< More than 5 equal bits in a sequence have occurred in a part of a received message. */
        CanFDFormError  = 2,   /*!< A fixed format part of a received frame has the wrong format. */
        CanFDAckError   = 3,   /*!< The message this CAN-FD Core transmitted was not acknowledged by another node. */
        CanFDBit1Error  = 4,   /*!< While trying to send a recessive bit (1) a dominant level (0) was sampled. */
        CanFDBit0Error  = 5,   /*!< While trying to send a dominant bit (0) a recessive level (1) was sampled. */
        CanFDCRCError   = 6    /*!< The CRC checksum was incorrect. */
    } en_canfd_status_t;
    
    
    /**
    *****************************************************************************
    ** \brief CAN-FD error.
    **
    ** These error values are used to report any CAN-FD related error to the API via
    ** the CAN-FD error callback function (canfd_error_func_ptr_t).
    *****************************************************************************/
    typedef enum en_canfd_error
    {
        CanFDBusOff        = 0,    /*!< The CAN-FD module is in busoff state. */
        CanFDWarning       = 1     /*!< At least one error counter has reached error warning. */
    } en_canfd_error_t;
    
    
    /**
    *****************************************************************************
    ** \brief CAN-FD message.
    **
    ** This structure stores a CAN-FD message, including the identifier, data
    ** and data length code (DLC).
    *****************************************************************************/
    typedef struct stc_canfd_msg
    {
        boolean_t             bCanFDFormat;   /*!< TRUE:CAN FD format, FALSE:Classic CAN format. */
        stc_canfd_msg_id_t    stcIdentifier;  /*!< 11- or 29-bit identifier (ID) */
        stc_canfd_msg_data_t  stcData;        /*!< Data and DLC */
    } stc_canfd_msg_t;
    
    
    /**
    *****************************************************************************
    ** \brief Message transmission complete callback function (canfd_tx_msg_func_ptr_t).
    **
    ** Signals a successful completed transmission.
    *****************************************************************************/
    typedef void (*canfd_tx_msg_func_ptr_t)( void );
    
    
    /**
    *****************************************************************************
    ** \brief Message reception callback function (canfd_rx_msg_func_ptr_t).
    **
    ** Signals that CAN-FD has received a new message.
    *****************************************************************************/
    typedef void (*canfd_rx_msg_func_ptr_t)(stc_canfd_msg_t* stcCanFDmsg);
    
    
    /**
    *****************************************************************************
    ** \brief Status changed callback function (canfd_status_chg_func_ptr_t).
    **
    ** Any status change will be reported to the API (see #en_canfd_status_t for
    ** possible status change codes).
    *****************************************************************************/
    typedef void (*canfd_status_chg_func_ptr_t)(en_canfd_status_t enCanFDStatus);
    
    
    /**
    *****************************************************************************
    ** \brief Error callback function (canfd_error_func_ptr_t).
    **
    ** Any error will be reported to the API (see #en_canfd_error_t for
    ** possible error codes).
    *****************************************************************************/
    typedef void (*canfd_error_func_ptr_t)(en_canfd_error_t enCanFDError);
    
    
    /**
    *****************************************************************************
    ** \brief CAN-FD configuration.
    **
    ** Is used to set the bitrate and select CAN mode or CAN-FD mode.
    ** Optionally Callbacks notification can be set.
    *****************************************************************************/
    typedef struct stc_canfd_config
    {
        
        canfd_tx_msg_func_ptr_t       pfnTxCallback;      /*!< Callback function for transmit completed, can be NULL. */
        canfd_rx_msg_func_ptr_t       pfnRxCallback;      /*!< Callback function for receive completed, can be NULL. */
        canfd_rx_msg_func_ptr_t       pfnRxFIFO_0_Callback;
        canfd_status_chg_func_ptr_t   pfnStatusCallback;  /*!< Callback function for CAN status changes, can be NULL. */
        canfd_error_func_ptr_t        pfnErrorCallback;   /*!< Callback function for CAN related errors, can be NULL. */
        boolean_t                     bCanFDMode;         /*!< TRUE:CAN FD mode, FALSE:Classic CAN mode. */
        stc_canfd_bitrate_t           stcBitrate;         /*!< CAN bitrate setting.  */
        stc_canfd_bitrate_t           stcFastBitrate;     /*!< CAN Fast bitrate setting.  */
        
    } stc_canfd_config_t;
    
    
    /**
    *****************************************************************************
    ** \brief CAN-FD interrupt pointer structure
    **
    ** Holds some pointers to callback functions and buffer
    *****************************************************************************/
    typedef struct stc_canfd_interrupt_handling
    {
        canfd_tx_msg_func_ptr_t pfnCanFDTxInterruptFunction; /*!< pointer to transmit interrupt callback */
        canfd_rx_msg_func_ptr_t pfnCanFDRxInterruptFunction; /*!< pointer to receive interrupt callback */
        canfd_rx_msg_func_ptr_t pfnCanFDRxFIFO_0_InterruptFunction; /*!< pointer to receive interrupt callback */
    } stc_canfd_interrupt_handling_t;
    
    /**
    *****************************************************************************
    ** \brief CAN-FD notification pointer structure
    **
    ** Holds some pointers to callback functions for status and error notification
    *****************************************************************************/
    typedef struct stc_canfd_notification
    {
        canfd_status_chg_func_ptr_t pfnCanFDStatusInterruptFunction ; /*!< pointer to status interrupt callback */
        canfd_error_func_ptr_t      pfnCanFDErrorInterruptFunction ;  /*!< pointer to error interrupt callback */
    } stc_canfd_notification_t;
    
    /**
    *****************************************************************************
    ** \brief CAN-FD internal data structure
    **
    ** Pointer to callback function interrupt handler
    *****************************************************************************/
    typedef struct stc_canfd_intern_data
    {
        stc_canfd_interrupt_handling_t stcCanFDInterruptHandling;   /*!< Interrupt callback. See #stc_canfd_interrupt_handling_t. */
        stc_canfd_notification_t       stcCanFDNotificationCb;      /*!< Interrupt callback. See #stc_canfd_notification_t. */
    } stc_canfd_intern_data_t;
    
#endif /* __CANFD_H__ */
/*****************************************************************************/
/* EOF (not truncated)                                                       */
/*****************************************************************************/
