/*****************************************************************************
* $Revision: 423 $
* $Date: 2015-03-27 15:10:42 +0900$
*****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************/
/** \file dma.c
**
** DMA Functions
**
** History:
** - 2015-02-10  1.0  MAs  First version for DMA
*****************************************************************************/

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "bsp_dma.h"
#include "common_include.h"

uint32_t *paddr;
volatile boolean_t DmaCompleted;
/*****************************************************************************/
/* Local pre-processor symbols/macros ('#define')                            */
/*****************************************************************************/

#define CpuUserMode   0x10     // User mode

#define DMA_CHANNEL_REG(u8Channel)          (((volatile stc_dma_channel_registers_t*)&DMA0)[u8Channel])
#define DMA_INT_CLIENT_REG(u8IntClient)     (((volatile un_dma0_cmicicn_t*)&DMA0.unCMICIC8)[u8IntClient])
#define DMA_CLIENT_CHANNEL_REG(u8Channel)   (((volatile un_dma0_cmchicn_t*)&DMA0.unCMCHIC0)[u8Channel])

/*****************************************************************************/
/* Global variable definitions (declared in header file with 'extern')       */
/*****************************************************************************/

/*****************************************************************************/
/* Local type definitions ('typedef')                                        */
/*****************************************************************************/



/**
*****************************************************************************
** \brief Data type for holding channel registers
**
** This struct is used to store the channel registers in the same alignment as
** the originals.
*****************************************************************************/
typedef struct stc_dma_channel_registers
{
    un_dma0_an_t	unAn;	        /* 0x00000000 */
    un_dma0_bn_t	unBn;	        /* 0x00000004 */
    un_dma0_san_t	unSAn;	        /* 0x00000008 */
    un_dma0_dan_t	unDAn;	        /* 0x0000000C */
    un_dma0_cn_t	unCn;	        /* 0x00000010 */
    uint_io8_t	au8Reserved0[1];	/* 0x00000014 */
    un_dma0_dn_1_t	unDn_1;	        /* 0x00000015 */
    uint_io8_t	au8Reserved16[1];	/* 0x00000016 */
    un_dma0_dn_3_t	unDn_3;	        /* 0x00000017 */
    un_dma0_sashdwn_t	unSASHDWn;	/* 0x00000018 */
    un_dma0_dashdwn_t	unDASHDWn;	/* 0x0000001C */
    un_dma0_en_t	unEn;	        /* 0x00000020 */
    uint_io8_t	au8Reserved32[28];	/* 0x00000024 */
} stc_dma_channel_registers_t;



/*****************************************************************************/
/* Wrapper to map each channel associated interrupt to the default handler   */
/*****************************************************************************/

#define DmaIrqHandlerChannel0(instance, internData)  DmaIrqHandler(instance, internData, 0)
#define DmaIrqHandlerChannel1(instance, internData)  DmaIrqHandler(instance, internData, 1)
#define DmaIrqHandlerChannel2(instance, internData)  DmaIrqHandler(instance, internData, 2)
#define DmaIrqHandlerChannel3(instance, internData)  DmaIrqHandler(instance, internData, 3)
#define DmaIrqHandlerChannel4(instance, internData)  DmaIrqHandler(instance, internData, 4)
#define DmaIrqHandlerChannel5(instance, internData)  DmaIrqHandler(instance, internData, 5)
#define DmaIrqHandlerChannel6(instance, internData)  DmaIrqHandler(instance, internData, 6)
#define DmaIrqHandlerChannel7(instance, internData)  DmaIrqHandler(instance, internData, 7)
#define DmaIrqHandlerChannel8(instance, internData)  DmaIrqHandler(instance, internData, 8)
#define DmaIrqHandlerChannel9(instance, internData)  DmaIrqHandler(instance, internData, 9)
#define DmaIrqHandlerChannel10(instance, internData) DmaIrqHandler(instance, internData, 10)
#define DmaIrqHandlerChannel11(instance, internData) DmaIrqHandler(instance, internData, 11)
#define DmaIrqHandlerChannel12(instance, internData) DmaIrqHandler(instance, internData, 12)
#define DmaIrqHandlerChannel13(instance, internData) DmaIrqHandler(instance, internData, 13)
#define DmaIrqHandlerChannel14(instance, internData) DmaIrqHandler(instance, internData, 14)
#define DmaIrqHandlerChannel15(instance, internData) DmaIrqHandler(instance, internData, 15)

/*****************************************************************************/
/* Local variable definitions ('static')                                     */
/*****************************************************************************/
/******************************************************************************/
/* Local variable definitions ('static')                                      */
/******************************************************************************/

// Look-up table for all enabled DMA instances and their internal data
static stc_dma_instance_data_t m_astcDmaInstanceDataLut[] =
{
    { .pstcInstance = &DMA0 },
};
/*****************************************************************************/
/* Function implementation - global ('extern') and local ('static')          */
/*****************************************************************************/

//#include "dma_generated.c"

/**
*****************************************************************************
** \addtogroup DmaGroup Direct Memory Access (DMA)
*****************************************************************************/
//@{
    
    /**
    *****************************************************************************
    ** \brief DMA interrupt service routine.
    *****************************************************************************/
    void DmaIrqHandler(volatile stc_dma0_t* pstcDma,
                       stc_dma_intern_data_t* pstcDmaInternData,
                       uint8_t u8Channel)
    {
        // Check if interrupt flag is set, otherwise do nothing
        // Note: This is recommended to prevent "ghost" interrupts due to ARM VIC port protocol
        if (DMA_CHANNEL_REG(u8Channel).unBn.stcField.u1DQ == 0)
        {
            return;
        }
        
        // Retrieve stop status (will be cleared automatically when clearing the interrupt flag)
        en_dma_stop_status_t enStopStatus = (en_dma_stop_status_t)DMA_CHANNEL_REG(u8Channel).unBn.stcField.u3SS;
        
        // Disable channel
        DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1EB = 0;
        
        // Clear interrupt flag by clear bit
        DMA_CHANNEL_REG(u8Channel).unCn.stcField.u1CD = 1;
        
        // Dummy read to force waiting for DMA register write completion
        // for protection against double ISR execution
        DMA_CHANNEL_REG(u8Channel).unCn;
        
        // Check for valid callback function.
        if (pstcDmaInternData->apfnCallbackFunction[u8Channel] != NULL)
        {
            uint32_t u32Address = 0;
            
            // Call callback function with channel and stop status
            pstcDmaInternData->apfnCallbackFunction[u8Channel](u8Channel, enStopStatus, u32Address);
        }
    } /* DmaIrqHandler */
    
    /**
    *****************************************************************************
    ** \brief DMA error interrupt service routine.
    *****************************************************************************/
    void DmaIrqHandlerError(volatile stc_dma0_t* pstcDma,
                            stc_dma_intern_data_t* pstcDmaInternData)
    {
        uint8_t u8Channel;
        
        // Determine DMA channel that generated the error interrupt
        u8Channel = 31 - CLZ(pstcDma->unEDIRQ1.u32Register);
        
        // Just to be on the safe side :-)
        if (u8Channel < DMA_CHANNEL_COUNT)
        {
            // Check if interrupt flag is set, otherwise do nothing
            // Note: This is recommended to prevent "ghost" interrupts due to ARM VIC port protocol
            if (DMA_CHANNEL_REG(u8Channel).unBn.stcField.u1EQ == 0)
            {
                return;
            }
            
            // Retrieve stop status (will be cleared automatically when clearing the interrupt flag)
            en_dma_stop_status_t enStopStatus = (en_dma_stop_status_t)DMA_CHANNEL_REG(u8Channel).unBn.stcField.u3SS;
            
            // Disable channel
            DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1EB = 0;
            
            // Clear interrupt flag by clear bit
            DMA_CHANNEL_REG(u8Channel).unCn.stcField.u1CE = 1;
            
            // Dummy read to force waiting for DMA register write completion
            // for protection against double ISR execution
            DMA_CHANNEL_REG(u8Channel).unCn;
            
            // Check for valid callback function.
            if (pstcDmaInternData->apfnCallbackFunction[u8Channel] != NULL)
            {
                uint32_t u32Address;
                switch (enStopStatus)
                {
                    case DmaStopStatusSourceAccessError:
                        u32Address = DMA_CHANNEL_REG(u8Channel).unSASHDWn.u32Register;
                        break;
                    case DmaStopStatusDestAccessError:
                        u32Address = DMA_CHANNEL_REG(u8Channel).unDASHDWn.u32Register;
                        break;
                    default:
                        u32Address = 0;
                        break;
                }
                
                // Call callback with channel and stop status
                pstcDmaInternData->apfnCallbackFunction[u8Channel](u8Channel, enStopStatus, u32Address);
            }
        }
    } /* DmaIrqHandlerError */
    
    /**
    *****************************************************************************
    ** \brief Initialize global DMA settings and disable all channels
    **
    ** \param [in]  pstcConfig         Global DMA configuration.
    **
    ** \retval Ok                      DMA module successfully initialized.
    ** \retval ErrorAccessRights       System is not in privileged (system) mode.
    ** \retval ErrorInvalidParameter   If following condition is met:
    **                                   - pstcConfig == NULL
    *****************************************************************************/
    en_result_t Dma_Init(const stc_dma_config_t* pstcConfig)
    {
        // Check if in privilege mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Check for NULL-Pointer
        if (pstcConfig == NULL)
        {
            return ErrorInvalidParameter;
        }
        
        // Preset local register variable to zero
        un_dma0_r_t unDmaConfigRegister = { 0 };
        
        // Set priority type
        unDmaConfigRegister.stcField.u2PR = pstcConfig->enPriorityType;
        
        // Set debug behavior
        unDmaConfigRegister.stcField.u1DBE = 1; // react on debug events (behavior set by DMAi_R:DB)
        unDmaConfigRegister.stcField.u2DB  = pstcConfig->enDebugBehavior;
        
        // Enable DMA controller
        unDmaConfigRegister.stcField.u1DE = 1;
        
        // Copy settings to hardware register
        DMA0.unR = unDmaConfigRegister;
        
        return Ok;
        
    } /* Dma_Init */
    
    /**
    *****************************************************************************
    ** \brief De-initialization of DMA module.
    **
    ** All DMA related register are reset to their default value.
    **
    ** \pre Must be in privileged mode
    **
    ** \retval Ok                      DMA module successfully de-initialized.
    ** \retval ErrorAccessRights       System is not in privileged (system) mode.
    ** \retval ErrorUninitialized      if DMA is not enabled
    *****************************************************************************/
    en_result_t Dma_DeInit(void)
    {
        
        //Loop counter
        uint32_t u32Index;
        
        //Get the internal pointer and check for null pointer
        stc_dma_intern_data_t* pstcInternData = DmaGetInternDataPtr(&DMA0);
        
        if (pstcInternData == NULL)
        {
            return ErrorUninitialized;
        }
        
        // Check if in prividge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Disable DMA globally
        DMA0.unR.u32Register = 0;
        
        // Clear all channel related registers
        for (u32Index = 0; u32Index < DMA_CHANNEL_COUNT; u32Index++)
        {
            DMA_CHANNEL_REG(u32Index).unAn.u32Register       = 0x000F0000;
            DMA_CHANNEL_REG(u32Index).unBn.u32Register       = 0x0200337F;
            DMA_CHANNEL_REG(u32Index).unCn.u32Register       = 0;
            DMA_CHANNEL_REG(u32Index).unDn_1.u8Register      = 0;
            DMA_CHANNEL_REG(u32Index).unDn_3.u8Register      = 0;
            DMA_CHANNEL_REG(u32Index).unSAn.u32Register      = 0;
            DMA_CHANNEL_REG(u32Index).unDAn.u32Register      = 0;
            
            // Clear internal clients
            DMA_CLIENT_CHANNEL_REG(u32Index).u32Register     = 0;
            DMA_INT_CLIENT_REG(u32Index).u32Register         = 0;
            
            //Clear internal data
            pstcInternData->abChannelConfigured[u32Index]  = FALSE;
            pstcInternData->apfnCallbackFunction[u32Index] = NULL;
        }
        
        return Ok;
        
    } /* Dma_DeInit */
    
    /**
    *****************************************************************************
    ** \brief Set DMA channel configuration parameters.
    **
    ** Client index (#pstcConfig.u16ClientIndex) is used for client interface
    **
    ** \pre Must be in privileged mode.
    ** \pre Channel must be disabled before calling this function.
    **
    ** \param [in]  u8Channel              DMA channel, valid range from 0 to #DMA_CHANNEL_COUNT-1.
    ** \param [in]  pstcConfig             DMA configuration parameters.
    ** \param [in]  pfnCallbackFunction    DMA callback function, can be NULL
    **
    ** \retval Ok                    DMA channel successfully configured.
    ** \retval ErrorAccessRights     System is not in privileged (system) mode.
    ** \retval ErrorUninitialized    DMA is not initialized, call Dma_Init() to initialize
    ** \retval ErrorInvalidMode      DMA channel not disabled, call Dma_DisableChannel() to disable
    ** \retval ErrorInvalidParameter If one of following conditions are met:
    **                                   - pstcConfig == NULL
    **                                   - u8Channel out of valid range (0 to #DMA_CHANNEL_COUNT-1).
    **                                   - pstcConfig->u16ClientIndex out of valid range (0 to #DMA_CLIENT_COUNT-1).
    **                                   - pstcConfig->u8BlockCount invalid (0 or > #DMA_MAX_BLOCK_COUNT).
    **                                   - pstcConfig->u8PriorityNumber invalid (> #DMA_LOWEST_PRIORITY).
    *****************************************************************************/
    en_result_t Dma_SetChannelConfig(uint8_t u8Channel,
                                     const stc_dma_channel_config_t* pstcConfig,
                                     dma_func_ptr_t pfnCallbackFunction)
    {
        // Declare pointers to local variables.
        un_dma0_an_t    unDmaChannelRegisterA    = { 0 };
        un_dma0_bn_t    unDmaChannelRegisterB    = { 0 };
        un_dma0_dn_3_t  unDmaChannelRegisterD_3 = { 0 };   //Source
        un_dma0_dn_1_t  unDmaChannelRegisterD_1 = { 0 };  //Destination
        
        //Get the internal pointer and check for null pointer
        stc_dma_intern_data_t* pstcInternData = DmaGetInternDataPtr(&DMA0);
        
        if (pstcInternData == NULL)
        {
            return ErrorUninitialized;
        }
        
        // Check if in prividge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Check if DMA is configured (enabled) globally
        if (DMA0.unR.stcField.u1DE == 0)
        {
            return ErrorUninitialized;
        }
        
        // Check if Channel is in valid range
        if (u8Channel >= DMA_CHANNEL_COUNT)
        {
            return ErrorInvalidParameter;
        }
        
        // Check for NULL-Pointer
        if (pstcConfig == NULL)
        {
            return ErrorInvalidParameter;
        }
        
        // Check if Channel is in valid range
        if (pstcConfig->u16ClientIndex >= DMA_CLIENT_COUNT)
        {
            return ErrorInvalidParameter;
        }
        
        // Check if Block Count is in valid range
        if (   (pstcConfig->u8BlockCount == 0)
            || (pstcConfig->u8BlockCount > DMA_MAX_BLOCK_COUNT) )
        {
            return ErrorInvalidParameter;
        }
        
        // Check if priority is in valid range
        if (pstcConfig->u8PriorityNumber > DMA_LOWEST_PRIORITY)
        {
            return ErrorInvalidParameter;
        }
        
        // Check if channel is disabled (to prevent changing a running DMA transfer)
        if (DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1EB != 0)
        {
            return ErrorInvalidMode;
        }
        
        // Set channel related settings
        
        // Prepare register A in local variable (faster than single RMW on DMA registers)
        unDmaChannelRegisterA.stcField.u2IS  = pstcConfig->enTrigger;
        unDmaChannelRegisterA.stcField.u1AL  = 0;    // set to contiguous R/W
        unDmaChannelRegisterA.stcField.u2BL  = pstcConfig->enBeatLimit;
        unDmaChannelRegisterA.stcField.u4BC  = pstcConfig->u8BlockCount - 1;
        unDmaChannelRegisterA.stcField.u4TO  = 0xF;  // set 15 DMAC clock cycles for timeout
        unDmaChannelRegisterA.stcField.u16TC = DMA_CHANNEL_REG(u8Channel).unAn.stcField.u16TC;  // keep current TC setting
        
        // Copy settings to hardware register
        DMA_CHANNEL_REG(u8Channel).unAn = unDmaChannelRegisterA;
        
        // Prepare register B in local variable (faster than single RMW on DMA registers)
        unDmaChannelRegisterB.stcField.u2MS = pstcConfig->enTransferMode;
        unDmaChannelRegisterB.stcField.u2TW = pstcConfig->enTransferWidth;
        unDmaChannelRegisterB.stcField.u1EI = (pfnCallbackFunction == NULL) ? 0 : 1;    // enable error interrupt only if callback is given
        unDmaChannelRegisterB.stcField.u1CI = (pfnCallbackFunction == NULL) ? 0 : 1;    // enable completion interrupt only if callback is given
        unDmaChannelRegisterB.stcField.u4SP = (pstcConfig->enSourceAccessRights << 1) | 1;  // not cacheable, not bufferable, data access, access rights according to configuration
        unDmaChannelRegisterB.stcField.u4DP = (pstcConfig->enDestAccessRights   << 1) | 1;  // not cacheable, not bufferable, data access, access rights according to configuration
            unDmaChannelRegisterB.stcField.u7PN = pstcConfig->u8PriorityNumber;
        
        // Copy settings to hardware register
        DMA_CHANNEL_REG(u8Channel).unBn = unDmaChannelRegisterB;
        
        // Prepare register D in local variable (faster than single RMW on DMA registers)
        unDmaChannelRegisterD_3.stcField.u1FS  = (pstcConfig->enSourceItemAddressMode == DmaItemAddressModeFixed)     ? 1 : 0;
        unDmaChannelRegisterD_3.stcField.u1DES = (pstcConfig->enSourceItemAddressMode == DmaItemAddressModeDecrement) ? 1 : 0;
        unDmaChannelRegisterD_3.stcField.u1US  = (pstcConfig->bUpdateSourceAddress == TRUE) ? 1 : 0;
        unDmaChannelRegisterD_3.stcField.u1FBS = (pstcConfig->enSourceBlockAddressMode == DmaBlockAddressModeFixed)   ? 1 : 0;
        
        unDmaChannelRegisterD_1.stcField.u1FD  = (pstcConfig->enDestItemAddressMode == DmaItemAddressModeFixed)     ? 1 : 0;
        unDmaChannelRegisterD_1.stcField.u1DED = (pstcConfig->enDestItemAddressMode == DmaItemAddressModeDecrement) ? 1 : 0;
        unDmaChannelRegisterD_1.stcField.u1UD  = (pstcConfig->bUpdateDestAddress == TRUE) ? 1 : 0;
        unDmaChannelRegisterD_1.stcField.u1FBD = (pstcConfig->enDestBlockAddressMode == DmaBlockAddressModeFixed)   ? 1 : 0;
        
        // Copy settings to hardware register
        DMA_CHANNEL_REG(u8Channel).unDn_3 = unDmaChannelRegisterD_3;
        DMA_CHANNEL_REG(u8Channel).unDn_1 = unDmaChannelRegisterD_1;
        
        // Disable channel
        DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1EB = 0;
        
        // Connect client to the channel
        DMA_CLIENT_CHANNEL_REG(u8Channel).u32Register = pstcConfig->u16ClientIndex;
        
        // DMA request enable signal to the corresponding peripheral which send DMA request by IRQ by Cutting Offset
        DMA_INT_CLIENT_REG(pstcConfig->u16ClientIndex - 8).stcField.u1BEHREQACK = 1;
        
        // Store callback function
        pstcInternData->apfnCallbackFunction[u8Channel] = pfnCallbackFunction;
        
        // ... and set configuration done flag
        pstcInternData->abChannelConfigured[u8Channel] = TRUE;
        
        return Ok;
        
    } /* Dma_SetChannelConfig */
    
    /**
    *****************************************************************************
    ** \brief Get DMA channel configuration parameters.
    **
    ** \pre Must be in privileged mode.
    **
    ** \param [in]  u8Channel          DMA channel, valid range from 0 to #DMA_CHANNEL_COUNT-1.
    ** \param [out] pstcConfig         DMA configuration parameters.
    **
    ** \retval Ok                      DMA channel configuration valid.
    ** \retval ErrorAccessRights       System is not in privileged (system) mode.
    ** \retval ErrorInvalidParameter   If one of following conditions are met:
    **                                   - pstcConfig == NULL
    **                                   - u8Channel out of valid range (0 to #DMA_CHANNEL_COUNT-1).
    *****************************************************************************/
    en_result_t Dma_GetChannelConfig(uint8_t u8Channel,
                                     stc_dma_channel_config_t* pstcConfig)
    {
        // Declare pointers to local variables
        un_dma0_an_t    unDmaChannelRegisterA    = { 0 };
        un_dma0_bn_t    unDmaChannelRegisterB    = { 0 };
        un_dma0_dn_3_t  unDmaChannelRegisterDn_3 = { 0 };   //Source
        un_dma0_dn_1_t  unDmaChannelRegisterDn_1 = { 0 };   //Destination
        
        //Get the internal pointer and check for null pointer
        stc_dma_intern_data_t* pstcInternData = DmaGetInternDataPtr(&DMA0);
        
        if (pstcInternData == NULL)
        {
            return ErrorUninitialized;
        }
        
        // Check if in prividge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Check if Channel is in valid range
        if (u8Channel >= DMA_CHANNEL_COUNT)
        {
            return ErrorInvalidParameter;
        }
        
        // Check for NULL-Pointer
        if (pstcConfig == NULL)
        {
            return ErrorInvalidParameter;
        }
        
        // Get channel related settings
        
        // Read register A into local variable (faster than multiple reads on DMA registers)
        unDmaChannelRegisterA = DMA_CHANNEL_REG(u8Channel).unAn;
        pstcConfig->enTrigger    = (en_dma_trigger_t)unDmaChannelRegisterA.stcField.u2IS;
        pstcConfig->enBeatLimit  = (en_dma_beat_limit_t)unDmaChannelRegisterA.stcField.u2BL;
        pstcConfig->u8BlockCount = unDmaChannelRegisterA.stcField.u4BC + 1;
        
        // Read register B into local variable (faster than multiple reads on DMA registers)
        unDmaChannelRegisterB = DMA_CHANNEL_REG(u8Channel).unBn;
        pstcConfig->enTransferMode  = (en_dma_transfer_mode_t)unDmaChannelRegisterB.stcField.u2MS;
        pstcConfig->enTransferWidth = (en_dma_transfer_width_t)unDmaChannelRegisterB.stcField.u2TW;
        pstcConfig->enSourceAccessRights = (en_dma_access_rights_t)((unDmaChannelRegisterB.stcField.u4SP & 0x2) >> 1);
        pstcConfig->enDestAccessRights   = (en_dma_access_rights_t)((unDmaChannelRegisterB.stcField.u4DP & 0x2) >> 1);
        pstcConfig->u8PriorityNumber = unDmaChannelRegisterB.stcField.u7PN;
        
        // Read register D into local variable (faster than multiple reads on DMA registers)
        unDmaChannelRegisterDn_3=DMA_CHANNEL_REG(u8Channel).unDn_3;
        unDmaChannelRegisterDn_1=DMA_CHANNEL_REG(u8Channel).unDn_1;
        
        //Read Source settings
        if (unDmaChannelRegisterDn_3.stcField.u1FS == 0)
        {
            pstcConfig->enSourceItemAddressMode = (unDmaChannelRegisterDn_3.stcField.u1DES == 0) ? DmaItemAddressModeIncrement : DmaItemAddressModeDecrement;
        }
        else
        {
            pstcConfig->enSourceItemAddressMode = DmaItemAddressModeFixed;
        }
        pstcConfig->bUpdateSourceAddress = (unDmaChannelRegisterDn_3.stcField.u1US == 0) ? FALSE : TRUE;
        pstcConfig->enSourceBlockAddressMode = (unDmaChannelRegisterDn_3.stcField.u1FBS == 0) ? DmaBlockAddressModeUpdate : DmaBlockAddressModeFixed;
        
        //Read Destination settings
        if (unDmaChannelRegisterDn_1.stcField.u1FD == 0)
        {
            pstcConfig->enDestItemAddressMode = (unDmaChannelRegisterDn_1.stcField.u1DED == 0) ? DmaItemAddressModeIncrement : DmaItemAddressModeDecrement;
        }
        else
        {
            pstcConfig->enDestItemAddressMode = DmaItemAddressModeFixed;
        }
        pstcConfig->bUpdateDestAddress = (unDmaChannelRegisterDn_1.stcField.u1UD == 0) ? FALSE : TRUE;
        pstcConfig->enDestBlockAddressMode = (unDmaChannelRegisterDn_1.stcField.u1FBD == 0) ? DmaBlockAddressModeUpdate : DmaBlockAddressModeFixed;
        
        // Get connected client index
        pstcConfig->u16ClientIndex = DMA_CLIENT_CHANNEL_REG(u8Channel).u32Register;
        
        return Ok;
        
    } /* Dma_GetChannelConfig */
    
    /**
    *****************************************************************************
    ** \brief Set transfer parameters for DMA channel.
    **
    ** \pre Must be in privileged mode.
    ** \pre Channel must be disabled before calling this function.
    ** \pre Channel must be configured before by calling Dma_SetChannelConfig().
    **
    ** \param [in]  u8Channel            DMA channel, valid range from 0 to #DMA_CHANNEL_COUNT-1.
    ** \param [in]  pvSourceAddress      Source address of the DMA transfer. Must be aligned
    **                                   according to configured transfer width.
    ** \param [in]  pvDestinationAddress Destination address of the DMA transfer. Must be aligned
    **                                   according to configured transfer width.
    ** \param [in]  u32TransferSizeByte  Transfer size in byte. This value must be
    **                                   a multiple of transfer width (byte count) *
    **                                   block count (item count). For example,
    **                                   transfer width = 32-bit (DmaTransferWidthWord)
    **                                   and block count = 8 (8 x 32-bit) requires
    **                                   #u32TransferSizeByte to be a multiple of 32 byte.
    **                                   Invalid values will result in ErrorInvalidParameter error.
    **
    ** \retval Ok                      DMA channel transfer parameter successfully set.
    ** \retval ErrorAccessRights       System is not in privileged (system) mode.
    ** \retval ErrorUninitialized      DMA is not initialized or channel is not configured.
    ** \retval ErrorInvalidMode        DMA channel is not disabled.
    ** \retval ErrorInvalidParameter   If one of following conditions are met:
    **                                   - u8Channel out of valid range (0 to #DMA_CHANNEL_COUNT-1).
    **                                   - pvSourceAddress or pvDestinationAddress not aligned according to configured transfer width
    **                                   - u32TransferSizeByte if not multiple of transfer width (byte count) * block count
        *****************************************************************************/
    en_result_t Dma_SetTransferParameter(uint8_t u8Channel,
                                         uint32_t pvSourceAddress,
                                         uint32_t pvDestinationAddress,
                                         uint32_t u32TransferSizeByte)
    {
        uint32_t u32AddressMask;
        uint32_t u32BlockSizeByte;
        
        //Get the internal pointer and check for null pointer
        stc_dma_intern_data_t* pstcInternData = DmaGetInternDataPtr(&DMA0);
        
        if (pstcInternData == NULL)
        {
            return ErrorUninitialized;
        }
        
        // Check if in prividge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Check if DMA is configured (enabled) globally
        if (DMA0.unR.stcField.u1DE == 0)
        {
            return ErrorUninitialized;
        }
        
        // Check if Channel is in valid range
        if (u8Channel >= DMA_CHANNEL_COUNT)
        {
            return ErrorInvalidParameter;
        }
        
        // Check if channel is configured (done by Dma_SetChannelConfig())
        if (pstcInternData->abChannelConfigured[u8Channel] == FALSE)
        {
            return ErrorUninitialized;
        }
        
        // Check if channel is disabled.
        if (DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1EB != 0)
        {
            return ErrorInvalidMode;
        }
        
        // Check address alignment
        u32AddressMask = ~(0xFFFFFFF0 | (0xF0 >> (4 - DMA_CHANNEL_REG(u8Channel).unBn.stcField.u2TW)));
        if (((uint32_t)pvSourceAddress & u32AddressMask) != 0)
        {
            return ErrorInvalidParameter;
        }
        if (((uint32_t)pvDestinationAddress & u32AddressMask) != 0)
        {
            return ErrorInvalidParameter;
        }
        
        // Check transfer size (to be multiple of transfer width (byte count) * block count)
        u32BlockSizeByte = (1 << DMA_CHANNEL_REG(u8Channel).unBn.stcField.u2TW);
        u32BlockSizeByte = u32BlockSizeByte * (DMA_CHANNEL_REG(u8Channel).unAn.stcField.u4BC + 1);
        if ((u32TransferSizeByte % u32BlockSizeByte) != 0)
        {
            return ErrorInvalidParameter;
        }
        
        // ... and minimum one block size
        if (u32TransferSizeByte < u32BlockSizeByte)
        {
            return ErrorInvalidParameter;
        }
        
        // Set transfer count (number of blocks, derived from size in byte)
        // Value of 0 means one block of data to be transferred.
        DMA_CHANNEL_REG(u8Channel).unAn.stcField.u16TC = (u32TransferSizeByte / u32BlockSizeByte) - 1;
        DMA_CHANNEL_REG(u8Channel).unSAn.u32Register   = (uint32_t)pvSourceAddress;
        DMA_CHANNEL_REG(u8Channel).unDAn.u32Register   = (uint32_t)pvDestinationAddress;
        
        return Ok;
        
    } /* Dma_SetTransferParameter */
    
    /**
    *****************************************************************************
    ** \brief Enable a DMA channel.
    **
    ** Enabling a DMA channel will not touch its pause state i.e. if a DMA is in
    ** pause state it must be resumed separately by calling Dma_ResumeChannel().
    **
    ** If the channel is configured for software trigger (#DmaTriggerSoftware) a
    ** SW trigger is required to finally start the transfer, either by #bTriggerChannel
    ** or by calling Dma_TriggerChannel().
    **
    ** \pre Must be in privileged mode
    ** \pre DMA and channel must be configured before (Dma_Init(), Dma_SetChannelConfig())
    **
    ** \param [in]  u8Channel        DMA channel, valid range from 0 to #DMA_CHANNEL_COUNT-1.
    ** \param [in]  bTriggerChannel  Issue a SW trigger to start the DMA transfer (in case of #DmaTriggerSoftware)
    **
    ** \retval Ok                    DMA channel successfully enabled.
    ** \retval ErrorAccessRights     System is not in privileged (system) mode.
    ** \retval ErrorUninitialized    DMA is not initialized or channel is not configured.
    ** \retval ErrorInvalidParameter If one of following conditions are met:
    **                                   - u8Channel out of valid range (0 to #DMA_CHANNEL_COUNT-1).
    ** \retval ErrorInvalidMode      SW trigger not possible either due to internal state (global halt, channel pause) or #DmaTriggerHardware setting.
    *****************************************************************************/
    en_result_t Dma_EnableChannel(uint8_t u8Channel,
                                  boolean_t bTriggerChannel)
    {
        
        //Get the internal pointer and check for null pointer
        stc_dma_intern_data_t* pstcInternData = DmaGetInternDataPtr(&DMA0);
        
        if (pstcInternData == NULL)
        {
            return ErrorUninitialized;
        }
        
        // Check if in prividge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Check if DMA is configured (enabled) globally
        if (DMA0.unR.stcField.u1DE == 0)
        {
            return ErrorUninitialized;
        }
        
        // Check if Channel is in valid range
        if (u8Channel >= DMA_CHANNEL_COUNT)
        {
            return ErrorInvalidParameter;
        }
        
        // Check if channel is configured (done by Dma_SetChannelConfig())
        if (pstcInternData->abChannelConfigured[u8Channel] == FALSE)
        {
            return ErrorUninitialized;
        }
        
        // Clear interrupt flags by clear bits
        DMA_CHANNEL_REG(u8Channel).unCn.stcField = (stc_dma0_cn_field_t) { .u1CE = 1, .u1CD = 1 };
        
        // Enable specified channel
        DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1EB = 1;
        
        // Trigger channel, if requested
        if (bTriggerChannel == TRUE)
        {
            return Dma_TriggerChannel(u8Channel);
        }
        
        return Ok;
        
    } /* Dma_EnableChannel */
    
    /**
    *****************************************************************************
    ** \brief Disable a DMA channel.
    **
    ** \pre Must be in privileged mode
    **
    ** \param [in]  u8Channel          DMA channel, valid range from 0 to #DMA_CHANNEL_COUNT-1.
    **
    ** \retval Ok                      DMA channel successfully disabled.
    ** \retval ErrorAccessRights       System is not in privileged (system) mode.
    ** \retval ErrorInvalidParameter   If one of following conditions are met:
    **                                   - u8Channel out of valid range (0 to #DMA_CHANNEL_COUNT-1).
    *****************************************************************************/
    en_result_t Dma_DisableChannel(uint8_t u8Channel)
    {
        // Check if in prividge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Check if Channel is in valid range
        if (u8Channel >= DMA_CHANNEL_COUNT)
        {
            return ErrorInvalidParameter;
        }
        
        // Disable specified channel
        DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1EB = 0;
        
        return Ok;
        
    } /* Dma_DisableChannel */
    
    /**
    *****************************************************************************
    ** \brief Issue a software trigger to (re)-start a DMA transfer on specified channel.
        **
    ** \pre Must be in privileged mode
    ** \pre Channel must be configured for SW trigger (#DmaTriggerSoftware)
    **
    ** \param [in]  u8Channel  DMA channel, valid range from 0 to #DMA_CHANNEL_COUNT-1.
    **
    ** \retval Ok                    DMA transfer successfully started.
    ** \retval ErrorAccessRights     System is not in privileged (system) mode.
    ** \retval ErrorUninitialized    DMA is not initialized or channel is not configured.
    ** \retval ErrorInvalidParameter If u8Channel out of valid range (0 to #DMA_CHANNEL_COUNT-1).
    ** \retval ErrorInvalidMode      SW trigger not possible either due to internal state (global halt, channel pause,
                                                                                           **                               transfer ongoing) or #DmaTriggerHardware setting.
    *****************************************************************************/
    en_result_t Dma_TriggerChannel(uint8_t u8Channel)
    {
        //Get the internal pointer and check for null pointer
        stc_dma_intern_data_t* pstcInternData = DmaGetInternDataPtr(&DMA0);
        
        if (pstcInternData == NULL)
        {
            return ErrorUninitialized;
        }
        
        // Check if in prividge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Check if DMA is configured (enabled) globally
        if (DMA0.unR.stcField.u1DE == 0)
        {
            return ErrorUninitialized;
        }
        
        // Check if Channel is in valid range
        if (u8Channel >= DMA_CHANNEL_COUNT)
        {
            return ErrorInvalidParameter;
        }
        
        // Check if channel is configured (done by Dma_SetChannelConfig())
        if (pstcInternData->abChannelConfigured[u8Channel] == FALSE)
        {
            return ErrorUninitialized;
        }
        
        // Check if channel is enabled.
        if (DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1EB == 0)
        {
            return ErrorInvalidMode;
        }
        
        // Check if DMA is software trigger ready.
        if (DMA_CHANNEL_REG(u8Channel).unBn.stcField.u1SR == 0)
        {
            return ErrorInvalidMode;
        }
        
        // Issue SW trigger
        DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1ST = 1;
        
        return Ok;
        
    } /* Dma_TriggerChannel */
    
    /**
    *****************************************************************************
    ** \brief Pause DMA channel.
    **
    ** This functions sets a DMA channel into pause state i.e. halts an ongoing
    ** transfer (at next transfer gap). The operation can be resumed by calling
    ** Dma_ResumeChannel().
    **
    ** \pre Must be in privileged mode
    **
    ** \param [in]  u8Channel  DMA channel, valid range from 0 to #DMA_CHANNEL_COUNT-1.
    **
    ** \retval Ok                    DMA channel successfully paused.
    ** \retval ErrorAccessRights     System is not in privileged (system) mode.
    ** \retval ErrorInvalidParameter If u8Channel out of valid range (0 to #DMA_CHANNEL_COUNT-1).
    *****************************************************************************/
    en_result_t Dma_PauseChannel(uint8_t u8Channel)
    {
        // Check if in prividge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Check if Channel is in valid range
        if (u8Channel >= DMA_CHANNEL_COUNT)
        {
            return ErrorInvalidParameter;
        }
        
        // Set Pause bit
        DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1PB = 1;
        
        return Ok;
        
    } /* Dma_PauseChannel */
    
    /**
    *****************************************************************************
    ** \brief Resume a paused DMA channel.
    **
    ** Resumes DMA channel which were paused by Dma_PauseChannel().
    **
    ** \pre Must be in privileged mode
    **
    ** \retval Ok                    DMA channel successfully paused.
    ** \retval ErrorAccessRights     System is not in privileged (system) mode.
    ** \retval ErrorInvalidParameter If u8Channel out of valid range (0 to #DMA_CHANNEL_COUNT-1).
    *****************************************************************************/
    en_result_t Dma_ResumeChannel(uint8_t u8Channel)
    {
        // Check if in prividge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Check if Channel is in valid range
        if (u8Channel >= DMA_CHANNEL_COUNT)
        {
            return ErrorInvalidParameter;
        }
        
        // Clear Pause bit
        DMA_CHANNEL_REG(u8Channel).unAn.stcField.u1PB = 0;
        
        return Ok;
        
    } /* Dma_ResumeChannel */
    
    /**
    *****************************************************************************
    ** \brief Pause overall DMA i.e. all channels.
    **
    ** This functions sets the global DMA halt request i.e. halts all an ongoing
    ** transfers (at next transfer gap). The halt request can be cleared by calling
    ** Dma_ResumeDma().
    **
    ** This function will not touch individual channel pause states.
    **
    ** \pre Must be in privileged mode
    **
    ** \retval Ok                    DMA channel successfully paused.
    ** \retval ErrorAccessRights     System is not in privileged (system) mode.
    *****************************************************************************/
    en_result_t Dma_PauseDma(void)
    {
        // Check for priviledge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Set global halt bit
        DMA0.unR.stcField.u1DH = 1;
        
        return Ok;
    } /* Dma_PauseDma */
    
    /**
    *****************************************************************************
    ** \brief Resumes overall DMA i.e. all channels.
    **
    ** This functions clears the global DMA halt request which was set by
    ** Dma_PauseChannel() i.e. allows to resume all ongoing transfers (if any).
        **
    ** The function will not touch individual channel pause states i.e. if
    ** one or more channels are in pause state (set by Dma_PauseChannel())
    ** this state stays active.
    **
    ** \pre Must be in privileged mode
    **
    ** \retval Ok                    DMA channel successfully paused.
    ** \retval ErrorAccessRights     System is not in privileged (system) mode.
    *****************************************************************************/
    en_result_t Dma_ResumeDma(void)
    {
        // Check for priviledge mode
        if (IsCpuInPrivilegedMode() != TRUE)
        {
            return ErrorAccessRights;
        }
        
        // Set global halt bit
        DMA0.unR.stcField.u1DH = 1;
        
        return Ok;
        
    } /* Dma_ResumeDma */
    
    /**
    ****************************************************************************
    ** \brief  Check if CPU is in privileged mode
    **
    ** \retval TRUE if CPU is any mode other than the user mode
    **         FALSE if CPU is in user mode
    ****************************************************************************/
    static boolean_t IsCpuInPrivilegedMode(void)
    {
        
        uint32_t u32Cpsr ;
        u32Cpsr = GET_CPSR();
        
        return ((u32Cpsr & 0x0000001F) != (uint32_t)CpuUserMode) ? TRUE : FALSE ;
        
    } /* IsCpuInPrivilegedMode */
    
    /**
    ******************************************************************************
    ** \brief Return the internal data for a certain DMA instance.
    **
    ** \param pstcDma Pointer to DMA instance
    **
    ** \return Pointer to internal data or NULL if instance is not enabled (or not known)
        **
    ******************************************************************************/
    stc_dma_intern_data_t* DmaGetInternDataPtr(volatile stc_dma0_t* pstcDma)
    {
        uint32_t u32Instance;
        
        for (u32Instance = 0; u32Instance < DMA_INSTANCE_COUNT; u32Instance++)
        {
            if (pstcDma == m_astcDmaInstanceDataLut[u32Instance].pstcInstance)
            {
                return &m_astcDmaInstanceDataLut[u32Instance].stcInternData;
            }
        }
        
        return NULL;
        
    }
    
    // -------------------------------------------------------------------
    // Definition of all individual ISRs for each enabled DMA instance.
    //
    // The individual ISR calls the generic IRQ handler with
    // the instance pointer and its internal data to handle the interrupt.
    // -------------------------------------------------------------------
    
FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Error(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_198;
    DmaIrqHandlerError((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel0(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_199;
    DmaIrqHandlerChannel0((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}
    
FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel1(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_200;
    DmaIrqHandlerChannel1((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}
    
FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel2(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_201;  
    DmaIrqHandlerChannel2((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}
    
FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel3(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_202;   
    DmaIrqHandlerChannel3((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}
    
FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel4(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_203;  
    DmaIrqHandlerChannel4((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel5(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_204;    
    DmaIrqHandlerChannel5((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel6(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_205; 
    DmaIrqHandlerChannel6((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel7(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_206;  
    DmaIrqHandlerChannel7((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel8(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_207; 
    DmaIrqHandlerChannel8((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel9(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_208;     
    DmaIrqHandlerChannel9((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel10(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_209;  
    DmaIrqHandlerChannel10((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel11(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_210;   
    DmaIrqHandlerChannel11((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel12(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_211; 
    DmaIrqHandlerChannel12((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel13(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_212;  
    DmaIrqHandlerChannel13((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel14(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_213;  
    DmaIrqHandlerChannel14((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

FN_IRQ_NOT_NESTED void Dma_Isr_Dma0Channel15(void)
{
    const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_214; 
    DmaIrqHandlerChannel15((stc_dma0_t*)&DMA0, &(m_astcDmaInstanceDataLut[DmaInstanceIndexDma0].stcInternData));
    IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
}

    
    void DmaCallback(uint8_t u8Channel, en_dma_stop_status_t enStopStatus, uint32_t u32ErrorAddress)
    {
        if (enStopStatus == DmaStopStatusNormalEnd)
        {
            DmaCompleted = TRUE;
        }
    }
    
    en_result_t BSP_DMA_Init(void)
    {
        en_result_t result;
        // Initialize DMA
        stc_dma_config_t stcDmaConfig = {  .enPriorityType  = DmaPriorityTypeFixed,
        .enDebugBehavior = DmaDebugBehaviorContinueOnBreak };
        result = Dma_Init(&stcDmaConfig);
        
        // Configure DMA channel 0 for SW triggering (no connection to a client)
        stc_dma_channel_config_t stcChannelConfig = {   .u16ClientIndex = 10,                                   // Client Matrix channel
            .enTransferMode = DmaTransferModeBurst,                 // do burst transfer (i.e. transfer all data at once by one trigger)
            .enTransferWidth = DmaTransferWidthWord,                // make 32-bit accesses
            .enBeatLimit = DmaBeatLimitInc16,                       // (try to) make 16-items burst accesses on bus
            .u8BlockCount = 16,                                     // required for 16-items burst
            .enTrigger = DmaTriggerSoftware,                        // Software triggered
            .enSourceItemAddressMode = DmaItemAddressModeIncrement, // increment source address
            .enDestItemAddressMode = DmaItemAddressModeIncrement,   // increment destination address
            .enSourceBlockAddressMode = DmaBlockAddressModeUpdate,  // update source block address after each block
            .enDestBlockAddressMode = DmaBlockAddressModeUpdate,    // update destination block address after each block
            .enSourceAccessRights = DmaAccessRightUser,             // DMA uses "user" privileges for source access
            .enDestAccessRights = DmaAccessRightUser,               // DMA uses "user" privileges for dest. access
            .bUpdateSourceAddress = FALSE,                          // reload source address after DMA completion
            .bUpdateDestAddress = TRUE,                             // update dest. address after DMA completion (to allow consecutive transfers)
        .u8PriorityNumber = 0 };                                // highest priority for that channel
        Dma_SetChannelConfig(0, &stcChannelConfig, DmaCallback);
        
        
        // Setup example transfer: SYSRAM >> SYSRAM, 64 byte (1 burst E16 x 32-bit)
        uint32_t u32SrcAddress = 0x02000000;
        paddr = (uint32_t*)u32SrcAddress;
        paddr[0] = 0x11223344;
        paddr[1] = 0x55667788;
        paddr[2] = 0x99AABBCC;
        uint32_t u32DstAddress = 0x02000200;
        uint32_t u32Size       = 64;
        Dma_SetTransferParameter(0, u32SrcAddress, u32DstAddress, u32Size);
        
        // Enable DMA channel 0
        Dma_EnableChannel(0, TRUE);
        return result;
    }
//@} // DmaGroup
//#endif // (PDL_PERIPHERAL_DMA_ACTIVE)

