#include "RIIC.h"
#include "dr7f701684.dvf.h"
#include "rh850_macros.h"
#define RIIC_WAIT_MAX 0XFFFFF0UL
#define RIIC_GPIO_PROTECTED_WRITE(preg, pstatus, reg, value) \
	do                                                       \
	{                                                        \
		(preg) = 0xA5U;                                      \
		(reg) = (value);                                     \
		(reg) = ~(value);                                    \
		(reg) = (value);                                     \
	} while ((pstatus) == 1U)
#define RIIC_PCLK 40000000UL
typedef enum
{
	RIIC_STATUS_IDLE = 0,
	RIIC_STATUS_START_WRITE,
	RIIC_STATUS_START_READ,
	RIIC_STATUS_SEND_DATA,
	RIIC_STATUS_RECEIVE_DATA,
	RIIC_STATUS_STOP,
} RIIC_Status_en_t;
typedef enum
{
	RIIC_MODE_SLAVE = 0,
	RIIC_MODE_MASTER,
} RIIC_Mode_en_t;
typedef struct
{
	RIIC_Mode_en_t enRIICMode;
	RIIC_Status_en_t enRIICStatus;
	uint32_t u32RIICDeviceAddr;
	uint32_t u32RIICWriteAndRead;
	uint32_t u32RIICStartFlag;
	uint32_t u32RIICTxCount;
	uint32_t u32RIICMasterTxLen;
	uint32_t u32RIICSlaveTxLen;
	uint32_t u32RIICSlaveTxCount;
	uint32_t u32RIICSlaveTxFlag;
	uint8_t *pu8RIICMasterTxData;
	uint8_t *pu8RIICSlaveTxData;
	uint8_t *pu8RIICSlaveTxDataRem;
	uint32_t u32RIICRxCount;
	uint32_t u32RIICMasterRxLen;
	uint32_t u32RIICSlaveRxLen;
	uint32_t u32RIICSlaveRxCount;
	uint32_t u32RIICSlaveRxFlag;
	uint8_t *pu8RIICMasterRxData;
	uint8_t *pu8RIICSlaveRxData;
	uint8_t *pu8RIICSlaveRxDataRem;
} RIIC_Ctr_st_t;
static RIIC_Ctr_st_t RIIC0Ctr;
static void RIIC0_Default_Mode_Status(void)
{
	RIIC0Ctr.enRIICMode = RIIC_MODE_SLAVE;
	RIIC0Ctr.enRIICStatus = RIIC_STATUS_IDLE;
	RIIC0Ctr.u32RIICStartFlag = 0UL;
	RIIC0Ctr.u32RIICRxCount = 0UL;
	RIIC0Ctr.u32RIICTxCount = 0UL;
	RIIC0Ctr.pu8RIICSlaveRxData = RIIC0Ctr.pu8RIICSlaveRxDataRem;
	RIIC0Ctr.pu8RIICSlaveTxData = RIIC0Ctr.pu8RIICSlaveTxDataRem;
}
uint32_t RIIC0_IsBusy(void)
{
	uint32_t u32RIICBusy = 0UL;
	if (RIIC0CR2LL & 0x80U)
	{
		u32RIICBusy = 1UL;
	}
	if (RIIC0CR2LL & 0x02U)
	{
		u32RIICBusy = 1UL;
	}
	if (RIIC0Ctr.enRIICStatus != RIIC_STATUS_IDLE)
	{
		u32RIICBusy = 1UL;
	}
	return u32RIICBusy;
}
void RIIC0_WaitForBusy_Block(void)
{
	uint32_t u32RIICCount = 0UL;
	while ((RIIC0_IsBusy() != 0UL) && (u32RIICCount < RIIC_WAIT_MAX))
	{
		u32RIICCount++;
	}
}
static void RIIC0_GPIO_Init(void)
{
#if (RIIC0_PORT_P0_11_ENABLE)
	PORTPBDC0 = 0x1800U;
	RIIC_GPIO_PROTECTED_WRITE(PORTPPCMD0, PORTPPROTS0, PORTPODC0, 0x1800UL);
#endif
#if (RIIC0_PORT_P10_2_ENABLE)
	PORTPBDC10 = 0x0CU;
	RIIC_GPIO_PROTECTED_WRITE(PORTPPCMD10, PORTPPROTS10, PORTPODC10, 0x0CUL);
#endif
}
uint32_t RIIC0_Init(RIIC_Init_st_t *pstRIICInit)
{
	uint32_t u32RIICbps = 0;
	peripheral_IRQ_disable(INTRIIC0EE);
	peripheral_IRQ_disable(INTRIIC0RI);
	peripheral_IRQ_disable(INTRIIC0TEI);
	peripheral_IRQ_disable(INTRIIC0TI);
	RIIC0_GPIO_Init();
	RIIC0CR1 &= 0xFFFFFF7FUL;
	RIIC0CR1 |= 0x00000040UL;
	RIIC0CR1 |= 0x00000080UL;
	RIIC0SAR0 = (uint32_t)(pstRIICInit->u32RIICSlaveAddr & 0x000000FEUL);
	RIIC0SER = 0x00000001UL;
	if (pstRIICInit->u32RIICBaud == 100000UL)
	{
		RIIC0MR1 = 0x38UL;
		RIIC0BRL = 0XF7UL;
		RIIC0BRH = 0xF3UL;
		u32RIICbps = 100000UL;
	}
	else if (pstRIICInit->u32RIICBaud == 200000UL)
	{
		RIIC0MR1 = 0x28UL;
		RIIC0BRL = 0XF5UL;
		RIIC0BRH = 0xF5UL;
		u32RIICbps = 200000UL;
	}
	else if (pstRIICInit->u32RIICBaud == 400000UL)
	{
		RIIC0MR1 = 0x18UL;
		RIIC0BRL = 0XF9UL;
		RIIC0BRH = 0xEBUL;
		u32RIICbps = 400000UL;
	}
	else
	{
		RIIC0MR1 = 0x08UL;
		RIIC0BRL = 0XFFUL;
		RIIC0BRH = 0xFFUL;
		u32RIICbps = 0;
	}
	RIIC0MR2 = 0x00000006UL;
	RIIC0MR3 = 0x00000070UL;
	RIIC0FER = 0x00000022UL;
	RIIC0IER = 0x000000FFUL;
	RIIC0CR1 &= 0xFFFFFFBFUL;
	RIIC0Ctr.enRIICMode = RIIC_MODE_SLAVE;
	RIIC0Ctr.enRIICStatus = RIIC_STATUS_IDLE;
	RIIC0Ctr.u32RIICDeviceAddr = 0UL;
	RIIC0Ctr.u32RIICWriteAndRead = 0UL;
	RIIC0Ctr.u32RIICTxCount = 0U;
	RIIC0Ctr.u32RIICMasterTxLen = 0UL;
	RIIC0Ctr.u32RIICSlaveTxLen = pstRIICInit->u32RIICSlaveTxLen;
	RIIC0Ctr.pu8RIICMasterTxData = 0UL;
	RIIC0Ctr.pu8RIICSlaveTxData = pstRIICInit->pu8RIICSlaveTxData;
	RIIC0Ctr.pu8RIICSlaveTxDataRem = pstRIICInit->pu8RIICSlaveTxData;
	RIIC0Ctr.u32RIICRxCount = 0U;
	RIIC0Ctr.u32RIICMasterRxLen = 0UL;
	RIIC0Ctr.u32RIICSlaveRxLen = pstRIICInit->u32RIICSlaveRxLen;
	RIIC0Ctr.pu8RIICMasterRxData = 0UL;
	RIIC0Ctr.pu8RIICSlaveRxData = pstRIICInit->pu8RIICSlaveRxData;
	RIIC0Ctr.pu8RIICSlaveRxDataRem = pstRIICInit->pu8RIICSlaveRxData;
	peripheral_IRQ_enable(INTRIIC0EE);
	peripheral_IRQ_enable(INTRIIC0RI);
	peripheral_IRQ_enable(INTRIIC0TEI);
	peripheral_IRQ_enable(INTRIIC0TI);
	return u32RIICbps;
}
void RIIC0_Event_ISR(void)
{
	uint32_t u32RIIC_SR2 = 0UL;
	u32RIIC_SR2 = RIIC0SR2;
	RIIC0SR2 &= 0xE0UL;
	if (u32RIIC_SR2 & 0x04U)
	{
		if (RIIC0Ctr.enRIICMode == RIIC_MODE_MASTER)
		{
			if (RIIC0Ctr.enRIICStatus == RIIC_STATUS_START_WRITE)
			{
				RIIC0Ctr.enRIICStatus = RIIC_STATUS_SEND_DATA;
				RIIC0DRT = (RIIC0Ctr.u32RIICDeviceAddr);
			}
			else if (RIIC0Ctr.enRIICStatus == RIIC_STATUS_START_READ)
			{
				RIIC0Ctr.enRIICStatus = RIIC_STATUS_RECEIVE_DATA;
				RIIC0DRT = (RIIC0Ctr.u32RIICDeviceAddr + 1U);
			}
			else
			{
				RIIC0CR2LL |= 0x08UL;
				RIIC0_Default_Mode_Status();
			}
		}
		else
		{
			RIIC0Ctr.u32RIICStartFlag = 1UL;
		}
	}
	if (u32RIIC_SR2 & 0x08U)
	{
		if (RIIC0Ctr.enRIICMode == RIIC_MODE_SLAVE)
		{
			if (RIIC0Ctr.enRIICStatus == RIIC_STATUS_SEND_DATA)
			{
				if ((RIIC0Ctr.u32RIICSlaveTxFlag == 1UL) && (RIIC0Ctr.u32RIICStartFlag))
				{
					RIIC0Ctr.u32RIICSlaveTxFlag = 2UL;
				}
			}
			else if (RIIC0Ctr.enRIICStatus == RIIC_STATUS_RECEIVE_DATA)
			{
				if ((RIIC0Ctr.u32RIICSlaveRxFlag == 1UL) && (RIIC0Ctr.u32RIICStartFlag))
				{
					RIIC0Ctr.u32RIICSlaveRxFlag = 2UL;
				}
			}
		}
		RIIC0_Default_Mode_Status();
	}
	if (u32RIIC_SR2 & 0x3U)
	{
		if (RIIC0Ctr.enRIICMode == RIIC_MODE_MASTER)
		{
			RIIC0CR2LL |= 0x08UL;
		}
		RIIC0_Default_Mode_Status();
	}
}
void RIIC0_TX_ISR(void)
{
	uint32_t u32RIIC_SR2 = 0UL;
	u32RIIC_SR2 = RIIC0SR2;
	RIIC0SR2 &= 0x7FUL;
	if (RIIC0Ctr.enRIICMode == RIIC_MODE_SLAVE)
	{
		if (RIIC0SR1 & 0x01UL)
		{
			if (RIIC0CR2 & 0x20)
			{
				RIIC0Ctr.enRIICStatus = RIIC_STATUS_SEND_DATA;
				if (u32RIIC_SR2 & 0x80UL)
				{
					if ((RIIC0Ctr.pu8RIICSlaveTxDataRem == RIIC0Ctr.pu8RIICSlaveTxData))
					{
						if (RIIC0Ctr.pu8RIICSlaveTxData)
						{
							RIIC0DRT = (*RIIC0Ctr.pu8RIICSlaveTxData);
							RIIC0Ctr.pu8RIICSlaveTxData++;
							RIIC0Ctr.u32RIICSlaveTxFlag = 1UL;
							RIIC0Ctr.u32RIICSlaveTxCount = 0UL;
						}
					}
				}
			}
			else
			{
				RIIC0_Default_Mode_Status();
			}
		}
		else
		{
			RIIC0_Default_Mode_Status();
		}
	}
}
void RIIC0_TX_End_ISR(void)
{
	uint32_t u32RIIC_SR2 = 0UL;
	uint32_t u32RIIC_MR3 = 0UL;
	uint32_t u32RIICDummy = 0UL;
	u32RIIC_SR2 = RIIC0SR2;
	u32RIIC_MR3 = RIIC0MR3;
	RIIC0SR2 &= 0xBFUL;
	if (RIIC0Ctr.enRIICMode == RIIC_MODE_SLAVE)
	{
		if (u32RIIC_MR3 & 0x04UL)
		{
			u32RIICDummy = RIIC0DRR;
		}
		else
		{
			if (RIIC0SR1 & 0x01UL)
			{
				if (RIIC0CR2 & 0x20)
				{
					RIIC0Ctr.enRIICStatus = RIIC_STATUS_SEND_DATA;
					if (u32RIIC_SR2 & 0x40UL)
					{
						RIIC0Ctr.u32RIICTxCount++;
						RIIC0Ctr.u32RIICSlaveTxCount++;
						if (RIIC0Ctr.u32RIICTxCount < RIIC0Ctr.u32RIICSlaveTxLen)
						{
							if (RIIC0Ctr.pu8RIICSlaveTxData)
							{
								RIIC0DRT = (*RIIC0Ctr.pu8RIICSlaveTxData);
								RIIC0Ctr.pu8RIICSlaveTxData++;
							}
						}
						else
						{
							RIIC0DRT = 0XFFU;
						}
					}
				}
				else
				{
					RIIC0_Default_Mode_Status();
				}
			}
			else
			{
				RIIC0_Default_Mode_Status();
			}
		}
	}
	else
	{
		if (u32RIIC_MR3 & 0x04UL)
		{
			RIIC0CR2LL |= 0x08UL;
			RIIC0_Default_Mode_Status();
		}
		else
		{
			if (u32RIIC_SR2 & 0x40UL)
			{
				if (RIIC0Ctr.enRIICStatus == RIIC_STATUS_SEND_DATA)
				{
					if (RIIC0Ctr.pu8RIICMasterTxData)
					{
						RIIC0DRT = (*RIIC0Ctr.pu8RIICMasterTxData);
						RIIC0Ctr.pu8RIICMasterTxData++;
					}
					if (RIIC0Ctr.u32RIICTxCount < RIIC0Ctr.u32RIICMasterTxLen)
					{
						RIIC0Ctr.u32RIICTxCount++;
					}
					else
					{
						if (RIIC0Ctr.u32RIICWriteAndRead)
						{
							RIIC0Ctr.enRIICStatus = RIIC_STATUS_START_READ;
							RIIC0CR2LL |= 0x04UL;
						}
						else
						{
							RIIC0Ctr.enRIICStatus = RIIC_STATUS_STOP;
							RIIC0CR2LL |= 0x08UL;
						}
					}
				}
				else
				{
					RIIC0CR2LL |= 0x08UL;
					RIIC0_Default_Mode_Status();
				}
			}
		}
	}
}
void RIIC0_RX_End_ISR(void)
{
	uint32_t u32RIIC_SR2 = 0UL;
	uint32_t u32RIIC_MR3 = 0UL;
	uint32_t u32RIICDummy = 0UL;
	u32RIIC_SR2 = RIIC0SR2;
	u32RIIC_MR3 = RIIC0MR3;
	RIIC0SR2 &= 0xDFUL;
	if (RIIC0Ctr.enRIICMode == RIIC_MODE_SLAVE)
	{
		if (RIIC0SR1 & 0x01UL)
		{
			if (RIIC0CR2 & 0x20)
			{
				RIIC0_Default_Mode_Status();
			}
			else
			{
				RIIC0Ctr.enRIICStatus = RIIC_STATUS_RECEIVE_DATA;
				if (u32RIIC_SR2 & 0x20U)
				{
					RIIC0MR3LL &= 0xF7U;
					if ((RIIC0Ctr.u32RIICStartFlag) && ((RIIC0Ctr.u32RIICSlaveRxFlag == 0UL) || (RIIC0Ctr.u32RIICSlaveRxFlag == 1UL)))
					{
						if (RIIC0Ctr.u32RIICRxCount == 0UL)
						{
							u32RIICDummy = RIIC0DRR;
							RIIC0Ctr.u32RIICSlaveRxFlag = 1UL;
							RIIC0Ctr.u32RIICSlaveRxCount = 0UL;
						}
						else
						{
							if (RIIC0Ctr.u32RIICRxCount <= RIIC0Ctr.u32RIICSlaveRxLen)
							{
								if (RIIC0Ctr.pu8RIICSlaveRxData)
								{
									*RIIC0Ctr.pu8RIICSlaveRxData = RIIC0DRR;
									RIIC0Ctr.pu8RIICSlaveRxData++;
								}
								else
								{
									u32RIICDummy = RIIC0DRR;
								}
							}
							else
							{
								u32RIICDummy = RIIC0DRR;
							}
							RIIC0Ctr.u32RIICSlaveRxCount++;
						}
						RIIC0Ctr.u32RIICRxCount++;
					}
					else
					{
						u32RIICDummy = RIIC0DRR;
					}
				}
			}
		}
		else
		{
			RIIC0_Default_Mode_Status();
		}
	}
	else
	{
		if (u32RIIC_MR3 & 0x04UL)
		{
			RIIC0CR2LL |= 0x08UL;
			u32RIICDummy = RIIC0DRR;
			RIIC0_Default_Mode_Status();
		}
		else
		{
			if (u32RIIC_SR2 & 0x20U)
			{
				if (RIIC0Ctr.enRIICStatus == RIIC_STATUS_RECEIVE_DATA)
				{
					if (RIIC0Ctr.u32RIICRxCount)
					{
						RIIC0MR3LL &= 0xF7;
						if (RIIC0Ctr.u32RIICRxCount == RIIC0Ctr.u32RIICMasterRxLen)
						{
							u32RIICDummy = RIIC0DRR;
						}
						else
						{
							if (RIIC0Ctr.pu8RIICMasterRxData)
							{
								*RIIC0Ctr.pu8RIICMasterRxData = RIIC0DRR;
								RIIC0Ctr.pu8RIICMasterRxData++;
							}
						}
						RIIC0Ctr.u32RIICRxCount--;
					}
					else
					{
						RIIC0MR3LL |= 0x08;
						RIIC0Ctr.enRIICStatus = RIIC_STATUS_STOP;
						RIIC0CR2LL |= 0x08UL;
						if (RIIC0Ctr.pu8RIICMasterRxData)
						{
							*RIIC0Ctr.pu8RIICMasterRxData = RIIC0DRR;
							RIIC0Ctr.pu8RIICMasterRxData++;
						}
					}
				}
				else
				{
					RIIC0CR2LL |= 0x08UL;
					u32RIICDummy = RIIC0DRR;
					RIIC0_Default_Mode_Status();
				}
			}
		}
	}
}
uint32_t RIIC0_Master_SendMultiByte_Interrupt(uint32_t u32RIICDevice, uint8_t *pu8RIICData, uint32_t RIICLEN)
{
	uint32_t u32RIICSet = 1;
	if ((RIIC0_IsBusy() == 0) && (pu8RIICData != 0U))
	{
		u32RIICSet = 0;
		RIIC0Ctr.enRIICMode = RIIC_MODE_MASTER;
		RIIC0Ctr.enRIICStatus = RIIC_STATUS_START_WRITE;
		RIIC0Ctr.u32RIICDeviceAddr = (u32RIICDevice & 0x000000FEUL);
		RIIC0Ctr.u32RIICWriteAndRead = 0U;
		RIIC0Ctr.u32RIICTxCount = 0U;
		RIIC0Ctr.u32RIICMasterTxLen = RIICLEN;
		RIIC0Ctr.pu8RIICMasterTxData = pu8RIICData;
		RIIC0Ctr.u32RIICRxCount = 0U;
		RIIC0Ctr.u32RIICMasterRxLen = 0U;
		RIIC0Ctr.pu8RIICMasterRxData = 0U;
		RIIC0CR2LL |= 0x02UL;
	}
	return u32RIICSet;
}
uint32_t RIIC0_Master_RecvMultiByte_Interrupt(uint32_t u32RIICDevice, uint8_t *pu8RIICData, uint32_t RIICLEN)
{
	uint32_t u32RIICSet = 1;
	if ((RIIC0_IsBusy() == 0) && (pu8RIICData != 0U))
	{
		u32RIICSet = 0;
		RIIC0Ctr.enRIICMode = RIIC_MODE_MASTER;
		RIIC0Ctr.enRIICStatus = RIIC_STATUS_START_READ;
		RIIC0Ctr.u32RIICDeviceAddr = (u32RIICDevice & 0x000000FEUL);
		RIIC0Ctr.u32RIICWriteAndRead = 0U;
		RIIC0Ctr.u32RIICTxCount = 0U;
		RIIC0Ctr.u32RIICMasterTxLen = 0U;
		RIIC0Ctr.pu8RIICMasterTxData = 0U;
		RIIC0Ctr.u32RIICRxCount = RIICLEN;
		RIIC0Ctr.u32RIICMasterRxLen = RIICLEN;
		RIIC0Ctr.pu8RIICMasterRxData = pu8RIICData;
		RIIC0CR2LL |= 0x02UL;
	}
	return u32RIICSet;
}
uint32_t RIIC0_Master_SendRecvMultiByte_Interrupt(uint32_t u32RIICDevice, uint8_t *pu8RIICTXData, uint32_t RIICTXLEN, uint8_t *pu8RIICRXData, uint32_t RIICRXLEN)
{
	uint32_t u32RIICSet = 1;
	if ((RIIC0_IsBusy() == 0) && (pu8RIICRXData != 0U) && (pu8RIICTXData != 0U))
	{
		u32RIICSet = 0;
		RIIC0Ctr.enRIICMode = RIIC_MODE_MASTER;
		RIIC0Ctr.enRIICStatus = RIIC_STATUS_START_WRITE;
		RIIC0Ctr.u32RIICDeviceAddr = (u32RIICDevice & 0x000000FEUL);
		RIIC0Ctr.u32RIICWriteAndRead = 1U;
		RIIC0Ctr.u32RIICTxCount = 0U;
		RIIC0Ctr.u32RIICMasterTxLen = RIICTXLEN;
		RIIC0Ctr.pu8RIICMasterTxData = pu8RIICTXData;
		RIIC0Ctr.u32RIICRxCount = RIICRXLEN;
		RIIC0Ctr.u32RIICMasterRxLen = RIICRXLEN;
		RIIC0Ctr.pu8RIICMasterRxData = pu8RIICRXData;
		RIIC0CR2LL |= 0x02UL;
	}
	return u32RIICSet;
}
uint32_t RIIC0_Slave_SendMultiByte(uint8_t *pu8RIICData, uint32_t RIICLEN)
{
	uint32_t u32RIICSet = 1;
	if ((RIIC0_IsBusy() == 0) && (pu8RIICData != 0U))
	{
		u32RIICSet = 0;
		RIIC0Ctr.enRIICMode = RIIC_MODE_SLAVE;
		RIIC0Ctr.enRIICStatus = RIIC_STATUS_IDLE;
		RIIC0Ctr.u32RIICTxCount = 0U;
		RIIC0Ctr.u32RIICSlaveTxLen = RIICLEN;
		RIIC0Ctr.pu8RIICSlaveTxData = pu8RIICData;
		RIIC0Ctr.pu8RIICSlaveTxDataRem = pu8RIICData;
	}
	return u32RIICSet;
}
uint32_t RIIC0_Slave_RecvMultiByte(uint8_t *pu8RIICData, uint32_t RIICLEN)
{
	uint32_t u32RIICSet = 1;
	if ((RIIC0_IsBusy() == 0) && (pu8RIICData != 0U))
	{
		u32RIICSet = 0;
		RIIC0Ctr.enRIICMode = RIIC_MODE_SLAVE;
		RIIC0Ctr.enRIICStatus = RIIC_STATUS_IDLE;
		RIIC0Ctr.u32RIICRxCount = 0U;
		RIIC0Ctr.u32RIICSlaveRxLen = RIICLEN;
		RIIC0Ctr.pu8RIICSlaveRxData = pu8RIICData;
		RIIC0Ctr.pu8RIICSlaveRxDataRem = pu8RIICData;
	}
	return u32RIICSet;
}
uint32_t RIIC0_Slave_Recv_Flag(void)
{
	return RIIC0Ctr.u32RIICSlaveRxFlag;
}
void RIIC0_Slave_Recv_Flag_Reset(void)
{
	RIIC0Ctr.u32RIICStartFlag = 0UL;
	RIIC0Ctr.u32RIICSlaveRxFlag = 0UL;
	RIIC0Ctr.u32RIICSlaveRxCount = 0UL;
}
uint32_t RIIC0_Slave_Recv_Count(void)
{
	return RIIC0Ctr.u32RIICSlaveRxCount;
}
uint32_t RIIC0_Slave_Send_Flag(void)
{
	return RIIC0Ctr.u32RIICSlaveTxFlag;
}
void RIIC0_Slave_Send_Flag_Reset(void)
{
	RIIC0Ctr.u32RIICSlaveTxFlag = 0UL;
	RIIC0Ctr.u32RIICSlaveTxCount = 0UL;
}
uint32_t RIIC0_Slave_Send_Count(void)
{
	return RIIC0Ctr.u32RIICSlaveTxCount;
}
static RIIC_Ctr_st_t RIIC1Ctr;
static void RIIC1_Default_Mode_Status(void)
{
	RIIC1Ctr.enRIICMode = RIIC_MODE_SLAVE;
	RIIC1Ctr.enRIICStatus = RIIC_STATUS_IDLE;
	RIIC1Ctr.u32RIICStartFlag = 0UL;
	RIIC1Ctr.u32RIICRxCount = 0UL;
	RIIC1Ctr.u32RIICTxCount = 0UL;
	RIIC1Ctr.pu8RIICSlaveRxData = RIIC1Ctr.pu8RIICSlaveRxDataRem;
	RIIC1Ctr.pu8RIICSlaveTxData = RIIC1Ctr.pu8RIICSlaveTxDataRem;
}
uint32_t RIIC1_IsBusy(void)
{
	uint32_t u32RIICBusy = 0UL;
	if (RIIC1CR2LL & 0x80U)
	{
		u32RIICBusy = 1UL;
	}
	if (RIIC1CR2LL & 0x02U)
	{
		u32RIICBusy = 1UL;
	}
	if (RIIC1Ctr.enRIICStatus != RIIC_STATUS_IDLE)
	{
		u32RIICBusy = 1UL;
	}
	return u32RIICBusy;
}
void RIIC1_WaitForBusy_Block(void)
{
	uint32_t u32RIICCount = 0UL;
	while ((RIIC1_IsBusy() != 0UL) && (u32RIICCount < RIIC_WAIT_MAX))
	{
		u32RIICCount++;
	}
}
static void RIIC1_GPIO_Init(void)
{
#if (RIIC1_PORT_P8_0_ENABLE)
	PORTPBDC8 = 0x03U;
	RIIC_GPIO_PROTECTED_WRITE(PORTPPCMD8, PORTPPROTS8, PORTPODC8, 0x03UL);
#endif
#if (RIIC1_PORT_P9_0_ENABLE)
	PORTPBDC9 = 0x03U;
	RIIC_GPIO_PROTECTED_WRITE(PORTPPCMD9, PORTPPROTS9, PORTPODC9, 0x03UL);
#endif
}
uint32_t RIIC1_Init(RIIC_Init_st_t *pstRIICInit)
{
	uint32_t u32RIICbps = 0;
	peripheral_IRQ_disable(INTRIIC1EE);
	peripheral_IRQ_disable(INTRIIC1RI);
	peripheral_IRQ_disable(INTRIIC1TEI);
	peripheral_IRQ_disable(INTRIIC1TI);
	RIIC1_GPIO_Init();
	RIIC1CR1 &= 0xFFFFFF7FUL;
	RIIC1CR1 |= 0x00000040UL;
	RIIC1CR1 |= 0x00000080UL;
	RIIC1SAR0 = (uint32_t)(pstRIICInit->u32RIICSlaveAddr & 0x000000FEUL);
	RIIC1SER = 0x00000001UL;
	if (pstRIICInit->u32RIICBaud == 100000UL)
	{
		RIIC1MR1 = 0x38UL;
		RIIC1BRL = 0XF7UL;
		RIIC1BRH = 0xF3UL;
		u32RIICbps = 100000UL;
	}
	else if (pstRIICInit->u32RIICBaud == 200000UL)
	{
		RIIC1MR1 = 0x28UL;
		RIIC1BRL = 0XF5UL;
		RIIC1BRH = 0xF5UL;
		u32RIICbps = 200000UL;
	}
	else if (pstRIICInit->u32RIICBaud == 400000UL)
	{
		RIIC1MR1 = 0x18UL;
		RIIC1BRL = 0XF9UL;
		RIIC1BRH = 0xEBUL;
		u32RIICbps = 400000UL;
	}
	else
	{
		RIIC1MR1 = 0x08UL;
		RIIC1BRL = 0XFFUL;
		RIIC1BRH = 0xFFUL;
		u32RIICbps = 0;
	}
	RIIC1MR2 = 0x00000006UL;
	RIIC1MR3 = 0x00000070UL;
	RIIC1FER = 0x00000022UL;
	RIIC1IER = 0x000000FFUL;
	RIIC1CR1 &= 0xFFFFFFBFUL;
	RIIC1Ctr.enRIICMode = RIIC_MODE_SLAVE;
	RIIC1Ctr.enRIICStatus = RIIC_STATUS_IDLE;
	RIIC1Ctr.u32RIICDeviceAddr = 0UL;
	RIIC1Ctr.u32RIICWriteAndRead = 0UL;
	RIIC1Ctr.u32RIICTxCount = 0U;
	RIIC1Ctr.u32RIICMasterTxLen = 0UL;
	RIIC1Ctr.u32RIICSlaveTxLen = pstRIICInit->u32RIICSlaveTxLen;
	RIIC1Ctr.pu8RIICMasterTxData = 0UL;
	RIIC1Ctr.pu8RIICSlaveTxData = pstRIICInit->pu8RIICSlaveTxData;
	RIIC1Ctr.pu8RIICSlaveTxDataRem = pstRIICInit->pu8RIICSlaveTxData;
	RIIC1Ctr.u32RIICRxCount = 0U;
	RIIC1Ctr.u32RIICMasterRxLen = 0UL;
	RIIC1Ctr.u32RIICSlaveRxLen = pstRIICInit->u32RIICSlaveRxLen;
	RIIC1Ctr.pu8RIICMasterRxData = 0UL;
	RIIC1Ctr.pu8RIICSlaveRxData = pstRIICInit->pu8RIICSlaveRxData;
	RIIC1Ctr.pu8RIICSlaveRxDataRem = pstRIICInit->pu8RIICSlaveRxData;
	peripheral_IRQ_enable(INTRIIC1EE);
	peripheral_IRQ_enable(INTRIIC1RI);
	peripheral_IRQ_enable(INTRIIC1TEI);
	peripheral_IRQ_enable(INTRIIC1TI);
	return u32RIICbps;
}
void RIIC1_Event_ISR(void)
{
	uint32_t u32RIIC_SR2 = 0UL;
	u32RIIC_SR2 = RIIC1SR2;
	RIIC1SR2 &= 0xE0UL;
	if (u32RIIC_SR2 & 0x04U)
	{
		if (RIIC1Ctr.enRIICMode == RIIC_MODE_MASTER)
		{
			if (RIIC1Ctr.enRIICStatus == RIIC_STATUS_START_WRITE)
			{
				RIIC1Ctr.enRIICStatus = RIIC_STATUS_SEND_DATA;
				RIIC1DRT = (RIIC1Ctr.u32RIICDeviceAddr);
			}
			else if (RIIC1Ctr.enRIICStatus == RIIC_STATUS_START_READ)
			{
				RIIC1Ctr.enRIICStatus = RIIC_STATUS_RECEIVE_DATA;
				RIIC1DRT = (RIIC1Ctr.u32RIICDeviceAddr + 1U);
			}
			else
			{
				RIIC1CR2LL |= 0x08UL;
				RIIC1_Default_Mode_Status();
			}
		}
		else
		{
			RIIC1Ctr.u32RIICStartFlag = 1UL;
		}
	}
	if (u32RIIC_SR2 & 0x08U)
	{
		if (RIIC1Ctr.enRIICMode == RIIC_MODE_SLAVE)
		{
			if (RIIC1Ctr.enRIICStatus == RIIC_STATUS_SEND_DATA)
			{
				if ((RIIC1Ctr.u32RIICStartFlag) && (RIIC1Ctr.u32RIICSlaveTxFlag == 1UL))
				{
					RIIC1Ctr.u32RIICSlaveTxFlag = 2UL;
				}
			}
			else if (RIIC1Ctr.enRIICStatus == RIIC_STATUS_RECEIVE_DATA)
			{
				if ((RIIC1Ctr.u32RIICStartFlag) && (RIIC1Ctr.u32RIICSlaveRxFlag == 1UL))
				{
					RIIC1Ctr.u32RIICSlaveRxFlag = 2UL;
				}
			}
		}
		RIIC1_Default_Mode_Status();
	}
	if (u32RIIC_SR2 & 0x3U)
	{
		if (RIIC1Ctr.enRIICMode == RIIC_MODE_MASTER)
		{
			RIIC1CR2LL |= 0x08UL;
		}
		RIIC1_Default_Mode_Status();
	}
}
void RIIC1_TX_ISR(void)
{
	uint32_t u32RIIC_SR2 = 0UL;
	u32RIIC_SR2 = RIIC1SR2;
	RIIC1SR2 &= 0x7FUL;
	if (RIIC1Ctr.enRIICMode == RIIC_MODE_SLAVE)
	{
		if (RIIC1SR1 & 0x01UL)
		{
			if (RIIC1CR2 & 0x20)
			{
				RIIC1Ctr.enRIICStatus = RIIC_STATUS_SEND_DATA;
				if (u32RIIC_SR2 & 0x80UL)
				{
					if ((RIIC1Ctr.pu8RIICSlaveTxDataRem == RIIC1Ctr.pu8RIICSlaveTxData))
					{
						if (RIIC1Ctr.pu8RIICSlaveTxData)
						{
							RIIC1DRT = (*RIIC1Ctr.pu8RIICSlaveTxData);
							RIIC1Ctr.pu8RIICSlaveTxData++;
							RIIC1Ctr.u32RIICSlaveTxFlag = 1UL;
							RIIC1Ctr.u32RIICSlaveTxCount = 0UL;
						}
					}
				}
			}
			else
			{
				RIIC1_Default_Mode_Status();
			}
		}
		else
		{
			RIIC1_Default_Mode_Status();
		}
	}
}
void RIIC1_TX_End_ISR(void)
{
	uint32_t u32RIIC_SR2 = 0UL;
	uint32_t u32RIIC_MR3 = 0UL;
	uint32_t u32RIICDummy = 0UL;
	u32RIIC_SR2 = RIIC1SR2;
	u32RIIC_MR3 = RIIC1MR3;
	RIIC1SR2 &= 0xBFUL;
	if (RIIC1Ctr.enRIICMode == RIIC_MODE_SLAVE)
	{
		if (u32RIIC_MR3 & 0x04UL)
		{
			u32RIICDummy = RIIC1DRR;
		}
		else
		{
			if (RIIC1SR1 & 0x01UL)
			{
				if (RIIC1CR2 & 0x20)
				{
					RIIC1Ctr.enRIICStatus = RIIC_STATUS_SEND_DATA;
					if (u32RIIC_SR2 & 0x40UL)
					{
						RIIC1Ctr.u32RIICTxCount++;
						RIIC1Ctr.u32RIICSlaveTxCount++;
						if (RIIC1Ctr.u32RIICTxCount < RIIC1Ctr.u32RIICSlaveTxLen)
						{
							if (RIIC1Ctr.pu8RIICSlaveTxData)
							{
								RIIC1DRT = (*RIIC1Ctr.pu8RIICSlaveTxData);
								RIIC1Ctr.pu8RIICSlaveTxData++;
							}
						}
						else
						{
							RIIC1DRT = 0XFFU;
						}
					}
				}
				else
				{
					RIIC1_Default_Mode_Status();
				}
			}
			else
			{
				RIIC1_Default_Mode_Status();
			}
		}
	}
	else
	{
		if (u32RIIC_MR3 & 0x04UL)
		{
			RIIC1CR2LL |= 0x08UL;
			RIIC1_Default_Mode_Status();
		}
		else
		{
			if (u32RIIC_SR2 & 0x40UL)
			{
				if (RIIC1Ctr.enRIICStatus == RIIC_STATUS_SEND_DATA)
				{
					if (RIIC1Ctr.pu8RIICMasterTxData)
					{
						RIIC1DRT = (*RIIC1Ctr.pu8RIICMasterTxData);
						RIIC1Ctr.pu8RIICMasterTxData++;
					}
					if (RIIC1Ctr.u32RIICTxCount < RIIC1Ctr.u32RIICMasterTxLen)
					{
						RIIC1Ctr.u32RIICTxCount++;
					}
					else
					{
						if (RIIC1Ctr.u32RIICWriteAndRead)
						{
							RIIC1Ctr.enRIICStatus = RIIC_STATUS_START_READ;
							RIIC1CR2LL |= 0x04UL;
						}
						else
						{
							RIIC1Ctr.enRIICStatus = RIIC_STATUS_STOP;
							RIIC1CR2LL |= 0x08UL;
						}
					}
				}
				else
				{
					RIIC1CR2LL |= 0x08UL;
					RIIC1_Default_Mode_Status();
				}
			}
		}
	}
}
void RIIC1_RX_End_ISR(void)
{
	uint32_t u32RIIC_SR2 = 0UL;
	uint32_t u32RIIC_MR3 = 0UL;
	uint32_t u32RIICDummy = 0UL;
	u32RIIC_SR2 = RIIC1SR2;
	u32RIIC_MR3 = RIIC1MR3;
	RIIC1SR2 &= 0xDFUL;
	if (RIIC1Ctr.enRIICMode == RIIC_MODE_SLAVE)
	{
		if (RIIC1SR1 & 0x01UL)
		{
			if (RIIC1CR2 & 0x20)
			{
				RIIC1_Default_Mode_Status();
			}
			else
			{
				RIIC1Ctr.enRIICStatus = RIIC_STATUS_RECEIVE_DATA;
				if (u32RIIC_SR2 & 0x20U)
				{
					RIIC1MR3LL &= 0xF7U;
					if ((RIIC1Ctr.u32RIICStartFlag) && ((RIIC1Ctr.u32RIICSlaveRxFlag == 0UL) || (RIIC1Ctr.u32RIICSlaveRxFlag == 1UL)))
					{
						if (RIIC1Ctr.u32RIICRxCount == 0UL)
						{
							u32RIICDummy = RIIC1DRR;
							RIIC1Ctr.u32RIICSlaveRxFlag = 1UL;
							RIIC1Ctr.u32RIICSlaveRxCount = 0UL;
						}
						else
						{
							if (RIIC1Ctr.u32RIICRxCount <= RIIC1Ctr.u32RIICSlaveRxLen)
							{
								if (RIIC1Ctr.pu8RIICSlaveRxData)
								{
									*RIIC1Ctr.pu8RIICSlaveRxData = RIIC1DRR;
									RIIC1Ctr.pu8RIICSlaveRxData++;
								}
								else
								{
									u32RIICDummy = RIIC1DRR;
								}
							}
							else
							{
								u32RIICDummy = RIIC1DRR;
							}
							RIIC1Ctr.u32RIICSlaveRxCount++;
						}
						RIIC1Ctr.u32RIICRxCount++;
					}
					else
					{
						u32RIICDummy = RIIC1DRR;
					}
				}
			}
		}
		else
		{
			RIIC1_Default_Mode_Status();
		}
	}
	else
	{
		if (u32RIIC_MR3 & 0x04UL)
		{
			RIIC1CR2LL |= 0x08UL;
			u32RIICDummy = RIIC1DRR;
			RIIC1_Default_Mode_Status();
		}
		else
		{
			if (u32RIIC_SR2 & 0x20U)
			{
				if (RIIC1Ctr.enRIICStatus == RIIC_STATUS_RECEIVE_DATA)
				{
					if (RIIC1Ctr.u32RIICRxCount)
					{
						RIIC1MR3LL &= 0xF7;
						if (RIIC1Ctr.u32RIICRxCount == RIIC1Ctr.u32RIICMasterRxLen)
						{
							u32RIICDummy = RIIC1DRR;
						}
						else
						{
							if (RIIC1Ctr.pu8RIICMasterRxData)
							{
								*RIIC1Ctr.pu8RIICMasterRxData = RIIC1DRR;
								RIIC1Ctr.pu8RIICMasterRxData++;
							}
						}
						RIIC1Ctr.u32RIICRxCount--;
					}
					else
					{
						RIIC1MR3LL |= 0x08;
						RIIC1Ctr.enRIICStatus = RIIC_STATUS_STOP;
						RIIC1CR2LL |= 0x08UL;
						if (RIIC1Ctr.pu8RIICMasterRxData)
						{
							*RIIC1Ctr.pu8RIICMasterRxData = RIIC1DRR;
							RIIC1Ctr.pu8RIICMasterRxData++;
						}
					}
				}
				else
				{
					RIIC1CR2LL |= 0x08UL;
					u32RIICDummy = RIIC1DRR;
					RIIC1_Default_Mode_Status();
				}
			}
		}
	}
}
uint32_t RIIC1_Master_SendMultiByte_Interrupt(uint32_t u32RIICDevice, uint8_t *pu8RIICData, uint32_t RIICLEN)
{
	uint32_t u32RIICSet = 1;
	if ((RIIC1_IsBusy() == 0) && (pu8RIICData != 0U))
	{
		u32RIICSet = 0;
		RIIC1Ctr.enRIICMode = RIIC_MODE_MASTER;
		RIIC1Ctr.enRIICStatus = RIIC_STATUS_START_WRITE;
		RIIC1Ctr.u32RIICDeviceAddr = (u32RIICDevice & 0x000000FEUL);
		RIIC1Ctr.u32RIICWriteAndRead = 0U;
		RIIC1Ctr.u32RIICTxCount = 0U;
		RIIC1Ctr.u32RIICMasterTxLen = RIICLEN;
		RIIC1Ctr.pu8RIICMasterTxData = pu8RIICData;
		RIIC1Ctr.u32RIICRxCount = 0U;
		RIIC1Ctr.u32RIICMasterRxLen = 0U;
		RIIC1Ctr.pu8RIICMasterRxData = 0U;
		RIIC1CR2LL |= 0x02UL;
	}
	return u32RIICSet;
}
uint32_t RIIC1_Master_RecvMultiByte_Interrupt(uint32_t u32RIICDevice, uint8_t *pu8RIICData, uint32_t RIICLEN)
{
	uint32_t u32RIICSet = 1;
	if ((RIIC1_IsBusy() == 0) && (pu8RIICData != 0U))
	{
		u32RIICSet = 0;
		RIIC1Ctr.enRIICMode = RIIC_MODE_MASTER;
		RIIC1Ctr.enRIICStatus = RIIC_STATUS_START_READ;
		RIIC1Ctr.u32RIICDeviceAddr = (u32RIICDevice & 0x000000FEUL);
		RIIC1Ctr.u32RIICWriteAndRead = 0U;
		RIIC1Ctr.u32RIICTxCount = 0U;
		RIIC1Ctr.u32RIICMasterTxLen = 0U;
		RIIC1Ctr.pu8RIICMasterTxData = 0U;
		RIIC1Ctr.u32RIICRxCount = RIICLEN;
		RIIC1Ctr.u32RIICMasterRxLen = RIICLEN;
		RIIC1Ctr.pu8RIICMasterRxData = pu8RIICData;
		RIIC1CR2LL |= 0x02UL;
	}
	return u32RIICSet;
}
uint32_t RIIC1_Master_SendRecvMultiByte_Interrupt(uint32_t u32RIICDevice, uint8_t *pu8RIICTXData, uint32_t RIICTXLEN, uint8_t *pu8RIICRXData, uint32_t RIICRXLEN)
{
	uint32_t u32RIICSet = 1;
	if ((RIIC1_IsBusy() == 0) && (pu8RIICRXData != 0U) && (pu8RIICTXData != 0U))
	{
		u32RIICSet = 0;
		RIIC1Ctr.enRIICMode = RIIC_MODE_MASTER;
		RIIC1Ctr.enRIICStatus = RIIC_STATUS_START_WRITE;
		RIIC1Ctr.u32RIICDeviceAddr = (u32RIICDevice & 0x000000FEUL);
		RIIC1Ctr.u32RIICWriteAndRead = 1U;
		RIIC1Ctr.u32RIICTxCount = 0U;
		RIIC1Ctr.u32RIICMasterTxLen = RIICTXLEN;
		RIIC1Ctr.pu8RIICMasterTxData = pu8RIICTXData;
		RIIC1Ctr.u32RIICRxCount = RIICRXLEN;
		RIIC1Ctr.u32RIICMasterRxLen = RIICRXLEN;
		RIIC1Ctr.pu8RIICMasterRxData = pu8RIICRXData;
		RIIC1CR2LL |= 0x02UL;
	}
	return u32RIICSet;
}
uint32_t RIIC1_Slave_SendMultiByte(uint8_t *pu8RIICData, uint32_t RIICLEN)
{
	uint32_t u32RIICSet = 1;
	if ((RIIC1_IsBusy() == 0) && (pu8RIICData != 0U))
	{
		u32RIICSet = 0;
		RIIC1Ctr.enRIICMode = RIIC_MODE_SLAVE;
		RIIC1Ctr.enRIICStatus = RIIC_STATUS_IDLE;
		RIIC1Ctr.u32RIICTxCount = 0U;
		RIIC1Ctr.u32RIICSlaveTxLen = RIICLEN;
		RIIC1Ctr.pu8RIICSlaveTxData = pu8RIICData;
		RIIC1Ctr.pu8RIICSlaveTxDataRem = pu8RIICData;
	}
	return u32RIICSet;
}
uint32_t RIIC1_Slave_RecvMultiByte(uint8_t *pu8RIICData, uint32_t RIICLEN)
{
	uint32_t u32RIICSet = 1;
	if ((RIIC1_IsBusy() == 0) && (pu8RIICData != 0U))
	{
		u32RIICSet = 0;
		RIIC1Ctr.enRIICMode = RIIC_MODE_SLAVE;
		RIIC1Ctr.enRIICStatus = RIIC_STATUS_IDLE;
		RIIC1Ctr.u32RIICRxCount = 0U;
		RIIC1Ctr.u32RIICSlaveRxLen = RIICLEN;
		RIIC1Ctr.pu8RIICSlaveRxData = pu8RIICData;
		RIIC1Ctr.pu8RIICSlaveRxDataRem = pu8RIICData;
	}
	return u32RIICSet;
}
uint32_t RIIC1_Slave_Recv_Flag(void)
{
	return RIIC1Ctr.u32RIICSlaveRxFlag;
}
void RIIC1_Slave_Recv_Flag_Reset(void)
{
	RIIC1Ctr.u32RIICStartFlag = 0UL;
	RIIC1Ctr.u32RIICSlaveRxFlag = 0UL;
	RIIC1Ctr.u32RIICSlaveRxCount = 0UL;
}
uint32_t RIIC1_Slave_Recv_Count(void)
{
	return RIIC1Ctr.u32RIICSlaveRxCount;
}
uint32_t RIIC1_Slave_Send_Flag(void)
{
	return RIIC1Ctr.u32RIICSlaveTxFlag;
}
void RIIC1_Slave_Send_Flag_Reset(void)
{
	RIIC1Ctr.u32RIICSlaveTxFlag = 0UL;
	RIIC1Ctr.u32RIICSlaveTxCount = 0UL;
}
uint32_t RIIC1_Slave_Send_Count(void)
{
	return RIIC1Ctr.u32RIICSlaveTxCount;
}