/******************************************************************************
 * $Revision: 423 $
 * $Date:: 2017-04-07 16:03:30 +0900#$
 *****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************
 ** \file cpu.c
 **
 ** A detailed description is available at
 ** @link CpuGroup CPU Module description @endlink
 **
 ** History:
 **   - 2014-06-02  0.01  HS  Initial version for Traveo
 **   - 2014-11-21  0.02  CEy Added Cpu_InvalidateDCacheArea()
 *****************************************************************************/


/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "common_include.h"

//#if defined(PDL_PERIPHERAL_CPU_ACTIVE)

/*****************************************************************************/
/* Local pre-processor symbols/macros ('define')                             */
/*****************************************************************************/


/*****************************************************************************/
/* Global variable definitions (declared in header file with 'extern')       */
/*****************************************************************************/

/*****************************************************************************/
/* Local type definitions ('typedef')                                        */
/*****************************************************************************/


/*****************************************************************************/
/* Local variable definitions ('static')                                     */
/*****************************************************************************/


/*****************************************************************************/
/* Local function prototypes ('static')                                      */
/*****************************************************************************/

/*****************************************************************************/
/* Function implementation - global ('extern') and local ('static')          */
/*****************************************************************************/

/**
 *****************************************************************************
 ** \addtogroup CpuGroup CPU Mode Functions
 *****************************************************************************/
/* @{ */

/**
 *****************************************************************************
 ** \brief  Check if CPU is in privileged mode
 **
 ** \retval TRUE if CPU is any mode other than the user mode
 **         FALSE if CPU is in user mode
 *****************************************************************************/
boolean_t Cpu_CpuIsInPrivilegedMode(void)
{
    return (Cpu_GetSystemMode() != (uint32_t)CpuUserMode) ? TRUE : FALSE ;
} /* Cpu_CpuIsInPrivilegedMode */


/**
 *****************************************************************************
 ** \brief    Function to read the current system mode.
 **
 ** This needs to be done in assembler since the compiler intrinsic functions
 ** to read CPSR are only allowed in privileged mode. This routine also runs in
 ** user mode.
 *****************************************************************************/
uint32_t Cpu_GetSystemMode(void)
{
    uint32_t u32Cpsr ;
    u32Cpsr = GET_CPSR();
    return (u32Cpsr & 0x0000001F);
} /* Cpu_GetSystemMode */


/**
 *****************************************************************************
 ** \brief  Invalidates the cache content of certain memory area
 **
 ** If the content of a cached RAM area has been changed by a different master
 ** than the CPU, this function can be used to invalidate the cache content of
 ** this area, so that the CPU can work with the updated data.
 ** Alternatively, this memory area could be assigned a non-cacheable attribute
 ** by using the Cortex-R5 MPU
 **
 ** \note This function expects arguments based on the cache line granularity
 **       (1 cache line is 32 bytes), so that user does not invalidate other
 **       areas by accident
 **
 ** \retval Ok                    Invalidation was successful
 ** \retval ErrorInvalidParameter If one of the following conditions are met:
 **                               - u32StartAddress is not 32-byte aligned
 **                               - u32ByteSize is not an integer multiple
 **                                 of 32 (bytes)
 *****************************************************************************/
en_result_t Cpu_InvalidateDCacheArea(uint32_t u32StartAddress, uint32_t u32ByteSize)
{
    const uint32_t u32EndAddress = u32StartAddress + u32ByteSize - 1;
    
    // Check start address alignment to avoid invalidating of unwanted areas (cache lines are 32-byte aligned)
    if((u32StartAddress & 0x1f) != 0)
    {
        return ErrorInvalidParameter;
    }
    // Check that size is a multiple of the cache line size to avoid invalidating of unwanted areas
    if((u32ByteSize % 32) != 0)
    {
        return ErrorInvalidParameter;
    }
    
    for( ; u32StartAddress < u32EndAddress; u32StartAddress += 32)
    {
        // Invalidate data cache line by MVA to PoC (refer ARM Cortex-R5 TRM)
        MCR(15, 0, u32StartAddress, 7, 6, 1);    
    }

    return Ok;
} /* Cpu_InvalidateDCacheArea */


/*! @} */

//#endif /* (PDL_PERIPHERAL_CPU_ACTIVE) */

/*****************************************************************************/
/* EOF (not truncated)                                                       */
/*****************************************************************************/
