/******************************************************************************
* $Revision: 423 $
* $Date:: 2017-04-07 16:03:30 +0900#$
*****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************
** \file eic.c
**
** A detailed description is available at
** @link EicGroup EIC Module description @endlink
**
** History:
** - 2014-06-27  0.01  ST  First version
*****************************************************************************/

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "common_include.h"



/**
*****************************************************************************
** \addtogroup EicGroup
**
*****************************************************************************/
//@{
    
    /*****************************************************************************/
    /* Local pre-processor symbols/macros ('define')                             */
    /*****************************************************************************/
    
    
    /*****************************************************************************/
    /* Global variable definitions (declared in header file with 'extern')       */
    /*****************************************************************************/
    
    
    /*****************************************************************************/
    /* Local type definitions ('typedef')                                        */
    /*****************************************************************************/
    
    
    /*****************************************************************************/
    /* Local variable definitions ('static')                                     */
    /*****************************************************************************/
    
    
    /*****************************************************************************/
    /* Local function prototypes ('static')                                      */
    /*****************************************************************************/
    static void EicIrqHandler( volatile stc_eic00_t* pstcEic,
                              stc_eic_intern_data_t* pstcEicInternData,
                              uint8_t u8Channel
                              ) ;
    
    stc_eic_intern_data_t* EicGetInternDataPtr(volatile stc_eic00_t* pstcEic) ;
    
    /*****************************************************************************/
    /* Wrapper to map each channel associated interrupt to the default handler   */
    /*****************************************************************************/
#define EicIrqHandlerChannel0( instance, internData) EicIrqHandler(instance, internData,  0)
#define EicIrqHandlerChannel1( instance, internData) EicIrqHandler(instance, internData,  1)
#define EicIrqHandlerChannel2( instance, internData) EicIrqHandler(instance, internData,  2)
#define EicIrqHandlerChannel3( instance, internData) EicIrqHandler(instance, internData,  3)
#define EicIrqHandlerChannel4( instance, internData) EicIrqHandler(instance, internData,  4)
#define EicIrqHandlerChannel5( instance, internData) EicIrqHandler(instance, internData,  5)
#define EicIrqHandlerChannel6( instance, internData) EicIrqHandler(instance, internData,  6)
#define EicIrqHandlerChannel7( instance, internData) EicIrqHandler(instance, internData,  7)
#define EicIrqHandlerChannel8( instance, internData) EicIrqHandler(instance, internData,  8)
#define EicIrqHandlerChannel9( instance, internData) EicIrqHandler(instance, internData,  9)
#define EicIrqHandlerChannel10(instance, internData) EicIrqHandler(instance, internData, 10)
#define EicIrqHandlerChannel11(instance, internData) EicIrqHandler(instance, internData, 11)
#define EicIrqHandlerChannel12(instance, internData) EicIrqHandler(instance, internData, 12)
#define EicIrqHandlerChannel13(instance, internData) EicIrqHandler(instance, internData, 13)
#define EicIrqHandlerChannel14(instance, internData) EicIrqHandler(instance, internData, 14)
#define EicIrqHandlerChannel15(instance, internData) EicIrqHandler(instance, internData, 15)
#define EicIrqHandlerChannel16(instance, internData) EicIrqHandler(instance, internData, 16)
#define EicIrqHandlerChannel17(instance, internData) EicIrqHandler(instance, internData, 17)
#define EicIrqHandlerChannel18(instance, internData) EicIrqHandler(instance, internData, 18)
#define EicIrqHandlerChannel19(instance, internData) EicIrqHandler(instance, internData, 19)
#define EicIrqHandlerChannel20(instance, internData) EicIrqHandler(instance, internData, 20)
#define EicIrqHandlerChannel21(instance, internData) EicIrqHandler(instance, internData, 21)
#define EicIrqHandlerChannel22(instance, internData) EicIrqHandler(instance, internData, 22)
#define EicIrqHandlerChannel23(instance, internData) EicIrqHandler(instance, internData, 23)
    
    
    /*****************************************************************************/
    /* Function implementation - global ('extern') and local ('static')          */
    /*****************************************************************************/
    
    
    /**
    *****************************************************************************
    ** \brief ISR for EIC
    **
    ** This ISR calls the callback function that has been during EIC configuration.
    ** (see Eic_Init and #stc_eic_config_t) and resets the interrupt flag.
    ** The interrupt request flag will be reset in the ISR.
    **
    ** \param [in] pstcEic           EIC instance that caused the interrupt (always
                                                                             **                               the same, as there is only one)
    ** \param [in] pstcEicInternData Internal data associated with the EIC instance
    *****************************************************************************/
    static void EicIrqHandler( volatile stc_eic00_t* pstcEic,
                              stc_eic_intern_data_t* pstcEicInternData,
                              uint8_t u8Channel
                              )
    {
        uint32_t u32EircrValue;
        
        /* Reset interrupt flag */
        u32EircrValue = (1 << u8Channel) ;
        PDL_WRITE_REG_SYNC(pstcEic->unEIRCR.u32Register, u32EircrValue);
        
        /* Check if callback function is not NULL */
        if ( pstcEicInternData->fptrEicCbFunction[u8Channel] != NULL )
        {
            /* Call callback function */
            pstcEicInternData->fptrEicCbFunction[u8Channel]( u8Channel ) ;
        }
    } /* EicIrqHandler */
    
    
    /**
    *****************************************************************************
    ** \brief Initialisation of external interrupt
    **
    ** All channel related parameters are set. The external interrupt number
    ** has to be less than #EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS.
    ** The callback function in parameter \a pstcConfig must not be NULL.
    **
    ** \param [in]  pstcEic          External interrupt instance
    ** \param [in]  u8Eic            External interrupt channel number
    ** \param [in]  pstcConfig       External interrupt configuration
    **
    ** \retval Ok                    External interrupt pin successfully
    **                                initialised
    ** \retval ErrorInvalidParameter If one of following conditions are met:
    **         - pstcEic == NULL
    **         - pstcConfig == NULL
    **         - callback given in \a pstcConfig->pfnCallback is NULL
    **         - u8Eic is greater or equal #EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS
    **         - pstcEicInternData == NULL (invalid or disabled ICU unit
                                            **                                                 (PDL_PERIPHERAL_ENABLE_EICn))
    *****************************************************************************/
    en_result_t Eic_Init( volatile stc_eic00_t* pstcEic,
                         uint8_t u8Eic,
                         stc_eic_config_t* pstcConfig
                         )
    {
        stc_eic_intern_data_t* pstcEicInternData ;
        un_eic00_elvrn_t       unELVR0 ;
        un_eic00_elvrn_t       unELVR1 ;
        un_eic00_elvrn_t       unELVR2 ;
        un_eic00_elvrn_t       unELVR3 ;
        uint8_t                u8LevelBitShift ;
        uint32_t               u32EnicrValue;
        
        /* Parameter check for NULL pointer */
        if ( pstcEic    == NULL ||
            pstcConfig == NULL
            )
        {
            return ErrorInvalidParameter ;
        }
        
        /* Parameter check for NULL pointer */
        if ( pstcConfig->pfnCallback == NULL )
        {
            return ErrorInvalidParameter ;
        }
        
        /* Check for channel is in range */
        if ( u8Eic >= EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS )
        {
            return ErrorInvalidParameter ;
        }
        
        /* Get pointer to internal data */
        pstcEicInternData = EicGetInternDataPtr( pstcEic ) ;
        /* ... and check for NULL */
        if ( NULL == pstcEicInternData )
        {
            return ErrorInvalidParameter ;
        }
        
        /* Set external interrupt enable clear flag */
        u32EnicrValue = (1 << u8Eic);
        PDL_SAFELY_DISABLE_INTERRUPT(pstcEic->unENICR.u32Register, u32EnicrValue)
        
        
        /* Calculate bitshift within Interrupt Level register */
        /* ((interrupt number % 8) * 4) */
        u8LevelBitShift = (u8Eic % 8) << 2 ;
            
        /* Set specific level to channel in ELVR register */
        /* (8 configuration nibble per 32 bit register) */
        switch ( u8Eic / EIC_PINS_PER_LEVEL_REGISTER )
        {
            case 0:
                /* Set ELVR0 register */
                
                /* Only change settings for one channel, preserve the rest */
                unELVR0.u32Register = pstcEic->unELVR0.u32Register ;
                
                /* Pull level setting bits to zero */
                unELVR0.u32Register &= ~(0x07 << u8LevelBitShift) ;
                    
                /* Set specific request event type */
                unELVR0.u32Register |= \
                (uint32_t)pstcConfig->enRequestEvent << u8LevelBitShift ;
                    
                /* Write back register value */
                pstcEic->unELVR0.u32Register = unELVR0.u32Register ;
                break ;
                
            case 1:
                /* Set ELVR1 register */
                
                /* Only change settings for one channel, preserve the rest */
                unELVR1.u32Register = pstcEic->unELVR1.u32Register ;
                
                /* Pull level setting bits to zero */
                unELVR1.u32Register &= ~(0x07 << u8LevelBitShift) ;
                    
                /* Set specific request event type */
                unELVR1.u32Register |= \
                (uint32_t)pstcConfig->enRequestEvent << u8LevelBitShift ;
                    
                /* Write back register value */
                pstcEic->unELVR1.u32Register = unELVR1.u32Register ;
                break ;
                
            case 2:
                /* Set ELVR2 register */
                
                /* Only change settings for one channel, preserve the rest */
                unELVR2.u32Register = pstcEic->unELVR2.u32Register ;
                
                /* Pull level setting bits to zero */
                unELVR2.u32Register &= ~(0x07 << u8LevelBitShift) ;
                    
                /* Set specific request event type */
                unELVR2.u32Register |= \
                (uint32_t)pstcConfig->enRequestEvent << u8LevelBitShift ;
                    
                /* Write back register value */
                pstcEic->unELVR2.u32Register = unELVR2.u32Register ;
                break ;
                
            case 3:
                /* Set ELVR3 register */
                
                /* Only change settings for one channel, preserve the rest */
                unELVR3.u32Register = pstcEic->unELVR3.u32Register ;
                
                /* Pull level setting bits to zero */
                unELVR3.u32Register &= ~(0x07 << u8LevelBitShift) ;
                    
                /* Set specific request event type */
                unELVR3.u32Register |= \
                (uint32_t)pstcConfig->enRequestEvent << u8LevelBitShift ;
                    
                /* Write back register value */
                pstcEic->unELVR3.u32Register = unELVR3.u32Register ;
                break ;
                
            default:
                return ErrorInvalidParameter ;
                
        } /* switch ( u8Eic / EIC_PINS_PER_LEVEL_REGISTER ) */
        
        
        /* Input noise filter settting */
        if ( pstcConfig->bInputNoiseFilterEnable == FALSE )
        {
            pstcEic->unNFECR.u32Register = (1 << u8Eic) ;
        }
        else
        {
            /* Enable input noise filter using set bit */
            pstcEic->unNFESR.u32Register = (1 << u8Eic) ;
        }
        
        /* DMA request settings */
        if ( pstcConfig->bDmaRequestEnable == FALSE )
        {
            /* Disable DMA requests using clear bit */
            pstcEic->unDRECR.u32Register = (1 << u8Eic) ;
        }
        else
        {
            /* Enable DMA requests using set bit */
            pstcEic->unDRESR.u32Register = (1 << u8Eic) ;
        }
        
        /* Store callback function to intern data */
        pstcEicInternData->fptrEicCbFunction[u8Eic] = pstcConfig->pfnCallback ;
        
        return Ok ;
    } /* Eic_Init */
    
    
    /**
    *****************************************************************************
    ** \brief Enable external interrupt channel
    **
    ** Selected external interrupt pin will be set active. The external interrupt
    ** number must be less than #EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS.
    **
    ** \param [in]  pstcEic          External interrupt instance
    ** \param [in]  u8Eic            External interrupt channel number
    **
    ** \retval Ok                    External interrupt channel successfully
    **                                enabled
    ** \retval ErrorInvalidParameter If one of following conditions are met:
    **         - pstcEic == NULL
    **         - u8Eic is greater or equal #EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS
    *****************************************************************************/
    en_result_t Eic_Enable( volatile stc_eic00_t* pstcEic, uint8_t u8Eic )
    {
        uint32_t u32EircrValue;
        
        /* Parameter check for NULL pointer */
        if ( pstcEic == NULL )
        {
            return ErrorInvalidParameter ;
        }
        
        /* Check for channel is in range */
        if ( u8Eic >= EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS )
        {
            return ErrorInvalidParameter ;
        }
        
        /* Reset/clear external interrupt request */
        u32EircrValue = (1 << u8Eic) ;
        PDL_WRITE_REG_SYNC(pstcEic->unEIRCR.u32Register, u32EircrValue);
        
        /* Activate external interrupt enable bit in register */
        pstcEic->unENISR.u32Register = (1 << u8Eic) ;
        
        return Ok ;
    } /* Eic_Enable */
    
    
    /**
    *****************************************************************************
    ** \brief Disable external interrupt pin.
    **
    ** Selected external interrupt pin will be deactivated. The external interrupt
    ** number must be less than #EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS.
    **
    ** \param [in]  pstcEic          External interrupt instance
    ** \param [in]  u8Eic            External interrupt channel number
    **
    ** \retval Ok                    External interrupt channel successfully
    **                                disabled
    ** \retval ErrorInvalidParameter If one of following conditions are met:
    **         - pstcEic == NULL
    **         - u8Eic is greater or equal #EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS
    *****************************************************************************/
    en_result_t Eic_Disable( volatile stc_eic00_t* pstcEic, uint8_t u8Eic )
    {
        uint32_t u32EnicrValue;
        
        /* Parameter check for NULL pointer */
        if ( pstcEic    == NULL )
        {
            return ErrorInvalidParameter ;
        }
        
        /* Check for channel is in range */
        if ( u8Eic >= EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS )
        {
            return ErrorInvalidParameter ;
        }
        
        /* Deactivate external interrupt enable bit in register */
        u32EnicrValue = (1 << u8Eic);
        PDL_SAFELY_DISABLE_INTERRUPT(pstcEic->unENICR.u32Register, u32EnicrValue)
        
        
        return Ok ;
    } /* Eic_Disable */
    
    /* Macro to return the number of enabled EIC instances */
#define EIC_INSTANCE_COUNT (uint32_t)(sizeof(m_astcEicInstanceDataLut) / sizeof(m_astcEicInstanceDataLut[0]))
    
    
    typedef enum en_eic_instance_index
    {
        EicInstanceIndexEic0,
        EicInstanceIndexMax
    } en_eic_instance_index_t;
    
    /* EIC instance data type */
    typedef struct stc_eic_instance_data
    {
        volatile stc_eic00_t* pstcInstance;  /*!< pointer to registers of an instance */
        stc_eic_intern_data_t stcInternData; /*!< module internal data of instance */
    } stc_eic_instance_data_t;
    
    
    static stc_eic_instance_data_t m_astcEicInstanceDataLut[] =
    {
        &EIC00,         /* pstcInstance */
        { NULL }        /* stcInternData */
    };
    
    /**
    *****************************************************************************
    ** \brief Return the internal data for a certain EIC instance.
    **
    ** \param pstcEic Pointer to EIC instance
    **
    ** \return Pointer to internal data or NULL if instance is not enabled (or not known)
        **
    *****************************************************************************/
    stc_eic_intern_data_t* EicGetInternDataPtr(volatile stc_eic00_t* pstcEic)
    {
        uint32_t u32Instance;
        
        for (u32Instance = 0; u32Instance < EIC_INSTANCE_COUNT; u32Instance++)
        {
            if (pstcEic == m_astcEicInstanceDataLut[u32Instance].pstcInstance)
            {
                return &m_astcEicInstanceDataLut[u32Instance].stcInternData;
            }
        }
        
        return NULL;
    }
    
    /**
    *****************************************************************************
    ** Definition of all individual ISRs for each enabled FRT instance.
    **
    ** The individual ISR calls the generic IRQ handler with
    ** the instance pointer and its internal data to handle the interrupt.
    *****************************************************************************/
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel0(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_24;
        EicIrqHandlerChannel0((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        EicIrqHandlerChannel16((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel1(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_25;
        EicIrqHandlerChannel1((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        EicIrqHandlerChannel17((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel2(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_26;
        EicIrqHandlerChannel2((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        EicIrqHandlerChannel18((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel3(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_27;
        EicIrqHandlerChannel3((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        EicIrqHandlerChannel19((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel4(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_28;
        EicIrqHandlerChannel4((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        EicIrqHandlerChannel20((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel5(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_29;
        EicIrqHandlerChannel5((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        EicIrqHandlerChannel21((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel6(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_30;
        EicIrqHandlerChannel6((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        EicIrqHandlerChannel22((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel7(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_31;
        EicIrqHandlerChannel7((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        EicIrqHandlerChannel23((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel8(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_32;
        EicIrqHandlerChannel8((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel9(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_33;
        EicIrqHandlerChannel9((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel10(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_34;
        EicIrqHandlerChannel10((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel11(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_35;
        EicIrqHandlerChannel11((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel12(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_36;
        EicIrqHandlerChannel12((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel13(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_37;
        EicIrqHandlerChannel13((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel14(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_38;
        EicIrqHandlerChannel14((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    FN_IRQ_NOT_NESTED void Eic_Isr_Eic0Channel15(void)
    {
        const uint32_t u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN = INTERRUPTS_IRQ_NUMBER_39;
        EicIrqHandlerChannel15((stc_eic00_t*)&EIC00, &(m_astcEicInstanceDataLut[EicInstanceIndexEic0].stcInternData));
        IRC0_IRQHC = u32IrqNumSetBy_FN_IRQ_DEFINE_BEGIN;
    }
    
    
    
//@} /* EicGroup */
/*****************************************************************************/
/* EOF (not truncated)                                                       */
/*****************************************************************************/
