/******************************************************************************
 * $Revision: 423 $
 * $Date:: 2017-04-07 16:03:30 +0900#$
 *****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************
 ** \file bt_rlt.h
 **
 ** Headerfile for Reload Timer functions
 **
 ** History:
 ** - 2014-05-30  0.01  ST  First version
 *****************************************************************************/

#ifndef __BT_RLT_H__
#define __BT_RLT_H__


/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "common_include.h"

/*****************************************************************************/
/* Global pre-processor symbols/macros ('define')                            */
/*****************************************************************************/


/**
 *****************************************************************************
 ** \defgroup RltGroup Reload Timer (RLT)
 **
 ** \brief This section describes the interface for the Reload Timer.
 **
 ** Provided functions of RLT module:
 ** - Bt_Rlt_Init() (Initialization of RLT module)
 ** - Bt_Rlt_Start() (Set the Reload value and to start a RLT)
 ** - Bt_Rlt_Stop() (Stop a RLT)
 **
 ** The Reload Timer is a counter that starts at a user defined value
 ** (aka. reload value) and goes down to zero. At zero an underflow
 ** interrupt is triggered and the counter either stops or restarts at
 ** the reload value. Counting can be triggered either by an external
 ** trigger (see #en_rlt_operation_mode_t) or by software.
 ** During active counting, an applied trigger restarts the counter with the
 ** reload value.
 **
 ** The usual procedure would be to first configure the RLT as desired using
 ** Bt_Rlt_Init().
 ** Next the RLT is armed by calling Bt_Rlt_Start(). Now the RLT
 ** is waiting for a trigger to start counting. The trigger
 ** can be applied either externally or by software trigger. 
 ** If the RLT is triggered in 32-bit mode, set the instance of odd-numbered 
 ** channel to pstcRltOdd.
 ** Whenever there was enough counting the timer can be stopped using
 ** Bt_Rlt_Stop().
 **
 ** External triggers are applied to the RLT input pin. The RLT
 ** is triggered by a rising, a falling or both types of edges on this pin.
 ** In gated mode, the RLT is counting as long as the pin is high or low
 ** (depending on the configuration).
 **
 *****************************************************************************/
//@{

/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/

/**
 *****************************************************************************
 ** \brief Reload Timer operation mode
 **
 ** This enumeration is used to specify how and when the RLT is counting.
 *****************************************************************************/
typedef enum en_rlt_operation_mode
{
    RltSoftwareTriggerOnly = 0x00,  /*!< Trigger timer only on software trigger */
    RltTriggerRisingEdge   = 0x01,  /*!< Reload timer on a rising edge */
    RltTriggerFallingEdge  = 0x02,  /*!< Reload timer on a falling edge */
    RltTriggerBothEdges    = 0x03   /*!< Reload timer on both edges */
} en_rlt_operation_mode_t;

/**
 *****************************************************************************
 ** \brief Reload Timer clock prescale divisors
 **
 ** This enumeration is used to specify the source clock divisor for the RLT
 ** to slow down counting of the RLT.
 *****************************************************************************/
typedef enum en_rlt_clock_prescaler
{
    RltDiv1     = 0x0,  /*!< Internal clock                      */
    RltDiv4     = 0x1,  /*!< Internal clock divided by 4         */
    RltDiv16    = 0x2,  /*!< Internal clock divided by 16        */
    RltDiv128   = 0x3,  /*!< Internal clock divided by 128       */
    RltDiv256   = 0x4,  /*!< Internal clock divided by 256       */
    RltExClkRis = 0x5,  /*!< External clock (rising-edge event)  */
    RltExClkfal = 0x6,  /*!< External clock (falling-edge event) */
    RltExClkbot = 0x7,  /*!< External clock (both-edges event)   */
    RltDiv512   = 0x8,  /*!< Internal clock divided by 512       */
    RltDiv1024  = 0x9,  /*!< Internal clock divided by 1024      */
    RltDiv2048  = 0xa,  /*!< Internal clock divided by 2048      */
    RltDiv2     = 0xb,  /*!< Internal clock divided by 2         */
    RltDiv8     = 0xc,  /*!< Internal clock divided by 8         */
    RltDiv32    = 0xd,  /*!< Internal clock divided by 32        */
    RltDiv64    = 0xe   /*!< Internal clock divided by 64        */
} en_rlt_clock_prescaler_t;

/**
 *****************************************************************************
 ** \brief Reload Timer configuration
 **
 ** This struct holds the configuration parameters of a Reload Timer.
 *****************************************************************************/
typedef struct stc_rlt_config
{
    /** Sets clock prescaler. See #en_pwm_clock_prescaler_t. */
    en_rlt_clock_prescaler_t  enPrescaler;
    /** Select operation mode of timer. See #en_rlt_operation_mode_t. */
    en_rlt_operation_mode_t   enOperationMode;
    /** 32bit timer Selection. TRUE: 32-bit timer mode, FALSE: 16-bit timer mode */
    boolean_t                 b32bitTimerModeEnable;
    /** Normal or Inverse output polarity. TRUE: Inverse polarity, FALSE: Normal polarity */
    boolean_t                 bOutputPolarityInverse;
    /** Mode selection. TRUE: One-shot mode, FALSE: Reload mode */
    boolean_t                 bOneshotEnable;
    /** Gate input enable. TRUE:Trigger input function, FALSE: Gate function */
    boolean_t                 bGateInputEnable;
    /** Callback for Trigger interrupt */
    func_ptr_t                pfnTriggerCallback;
    /** Callback for Underflow interrupt */
    func_ptr_t                pfnUnderflowCallback;
} stc_rlt_config_t;


/**
 *****************************************************************************
 ** \brief Data type for holding internal data needed for RLT
 **
 ** This struct is used to store the interrupt callback function for the RLT.
 *****************************************************************************/
typedef struct stc_rlt_intern_data
{
    func_ptr_t       pfnTriggerCallback ;    /*!< Interrupt callback function pointer */
    func_ptr_t       pfnUnderflowCallback ;  /*!< Interrupt callback function pointer */
} stc_rlt_intern_data_t;


/*****************************************************************************/
/* Global variable declarations ('extern', definition in C source)           */
/*****************************************************************************/

/*****************************************************************************/
/* Global function prototypes ('extern', definition in C source)             */
/*****************************************************************************/

/*****************************************************************************
 * Global function prototypes (doxygen can be found in .c file)
 *****************************************************************************/
extern en_result_t Bt_Rlt_Init(volatile stc_btn_rt_t* pstcRlt, const stc_rlt_config_t* pstcConfig);  
extern en_result_t Bt_Rlt_Start(volatile stc_btn_rt_t* pstcRlt,  volatile stc_btn_rt_t* pstcRltodd, uint32_t u32ReloadValue, boolean_t bTrigger);
extern en_result_t Bt_Rlt_Stop(volatile stc_btn_rt_t* pstcRlt );
extern void BtRltIrqHandler( volatile stc_btn_rt_t* pstcRlt, stc_rlt_intern_data_t* pstcRltInternData);
extern stc_rlt_intern_data_t* BtRltGetInternDataPtr(volatile stc_btn_rt_t* pstcRlt); 
extern void BSP_ReloadTimer_Init(uint8_t ch, en_rlt_clock_prescaler_t RltDiv, uint32_t ReloadValue, func_ptr_t Callback);

#endif /* __BT_RLT_H__ */
