/******************************************************************************
* $Revision: 423 $
* $Date:: 2017-04-07 16:03:30 +0900#$
*****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************
** \file eic.h
**
** Headerfile for EIC functions
**
** History:
** - 2014-06-27  0.01  ST  First version
*****************************************************************************/


#ifndef __EIC_H__
#define __EIC_H__

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/

/**
*****************************************************************************
** \defgroup EicGroup External Interrupt Controller (EIC)
**
** \brief This section describes the interface for the
**           External Interrupt Controller.
**
** Provided functions of EIC module:
** - Eic_Init() (Initialization of EIC module)
** - Eic_Enable() (Enable the EIC channel)
** - Eic_Disable() (Disable the EIC channel)
**
** The EIC detects a signal input to an External Interrupt pin and
** generates an interrupt request. The interrupt can be generated on five
** different pin levels
** (#en_eic_request_event_t):
** - high
** - low
** - rising edge
** - falling edge
** - any edge
**
** Eic_Init() is used to set the configuration of one interrupt channel.
** Here a notification callback (Eic_Init#pfnCallback) must be set,
** which is called when an interrupt on the configure channel occurs.
** Optionally the input noise filter and DAM requests can be enabled on the
** specified channel.
**
    ** After the external interrupt channel has been initialized, the function
    ** Eic_Enable() has to be called to activate the external interrupt channel.
** To deactivate the channel, Eic_Disable() can to called.
**
*****************************************************************************/
//@{

/*****************************************************************************/
/* Global pre-processor symbols/macros ('define')                            */
/*****************************************************************************/
/**
*****************************************************************************
** \brief Represents the maximum count of pins which can be used for interrupt
**        generation.
*****************************************************************************/
    #define EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS    24

/**
*****************************************************************************
** \brief Number of pins in one Interrupt Level register.
*****************************************************************************/
    #define EIC_PINS_PER_LEVEL_REGISTER              8

/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/

/**
*****************************************************************************
    ** \brief Callback function which is triggered when external event occures
**
    ** Will return triggered external interrupt in
** range 0 to (#EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS-1).
*****************************************************************************/
    typedef void (*eic_func_ptr_t)( uint8_t u8Eic ) ;
    
    
    /**
    *****************************************************************************
    ** \brief External Interrupt Request Event
    **
    ** Interrupt level is used to set request event type
    *****************************************************************************/
    typedef enum en_eic_request_event
    {
        EicLowLevel     = 0, /*!< raise interrupt on low level detection  */
        EicHighLevel    = 1, /*!< raise interrupt on high level detection */
        EicRisingEdge   = 2, /*!< raise interrupt on rising edge          */
        EicFallingEdge  = 3, /*!< raise interrupt on falling edge         */
        EicAnyEdge      = 4  /*!< raise interrupt on both edges           */
    } en_eic_request_event_t;
    
    
    /**
    *******************************************************************************
    ** \brief External Interrupt Controller Configuration
    **
    ** This struct includes parameter for activate/deactivate noise filtering
    ** and DMA access. The callback function, which notifies the API in case of
    ** external event must not be null.
    ******************************************************************************/
    typedef struct stc_eic_config
    {
        /** Request event type. See #en_eic_request_event_t. */
        en_eic_request_event_t  enRequestEvent ;
        /** Noise filter control. TRUE:Enable the noise filter, FALSE:Disable the noise filter */
        boolean_t               bInputNoiseFilterEnable ;
        /** DMA request. TRUE:Enable DMA, FALSE:Disable DMA */
        boolean_t               bDmaRequestEnable ;
        /** Callback when external interrupt event occured */
        eic_func_ptr_t          pfnCallback ;
    } stc_eic_config_t ;
    
    
    /**
    *****************************************************************************
    ** \brief Datatype for holding internal data needed for EIC
    **
    ** This struct is used to store the interrupt callback function for the EIC
    *****************************************************************************/
    typedef struct stc_eic_intern_data
    {
        /** Callback for interrupts */
        eic_func_ptr_t fptrEicCbFunction[EIC_MAX_NUMBER_OF_SUPPORTED_CHANNELS] ;
    } stc_eic_intern_data_t ;
    
    
    /*****************************************************************************/
    /* Global variable declarations ('extern', definition in C source)           */
    /*****************************************************************************/
    
    
    /*****************************************************************************/
    /* Global function prototypes ('extern', definition in C source)             */
    /*****************************************************************************/
    extern en_result_t Eic_Init( volatile stc_eic00_t* pstcEic, uint8_t u8Eic, stc_eic_config_t* pstcConfig);
    extern en_result_t Eic_Enable( volatile stc_eic00_t* pstcEic,  uint8_t u8Eic );
    extern en_result_t Eic_Disable( volatile stc_eic00_t* pstcEic, uint8_t u8Eic );
    

#endif /* __EIC_H__ */
/*****************************************************************************/
/* EOF (not truncated)                                                       */
/*****************************************************************************/
