/******************************************************************************
 * $Revision: 423 $
 * $Date:: 2017-04-07 16:03:30 +0900#$
 *****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************
 ** \file frt.h
 **
 ** Headerfile for Free Run Timer functions
 **
 ** History:
 ** - 2014-05-30  0.01  ST  First version
 *****************************************************************************/

#ifndef __FRT_H__
#define __FRT_H__


/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "pdl.h"

/* C binding of definitions if building with C++ compiler */
#ifdef __cplusplus
extern "C"
{
#endif

/*****************************************************************************/
/* Global pre-processor symbols/macros ('#define')                           */
/*****************************************************************************/


/**
 *****************************************************************************
 ** \defgroup FrtGroup Free Running Timer (FRT)
 **
 ** \brief This section describes the interface for the Free Running Timer.
 **
 ** Provided functions of FRT module:
 ** - Frt_Init() (Initialization of FRT module)
 ** - Frt_Start() (Set the CompareClearValue and to start a FRT)
 ** - Frt_Stop() (Stop a FRT)
 ** - Frt_GetCount() (Get a Timer data)
 **
 ** The FRT is a counter that starts at zero and goes up to a
 ** user defined value (aka. compare clear value). When the value is reached
 ** a compare clear interrupt is triggered. The counter can be operated in
 ** up-counting oder up/down-conting mode. Depending on that, the counter
 ** restarts counting from zero or goes down from compare clear value
 ** on a compare clear value. In up/down-counting mode the zereo detect
 ** interrupt is triggered when reaching zero.
 **
 ** A FRT can be operated bufferd or unbuffered. In buffered mode, compare clear
 ** values, that are set while the timer is running are stored in a compare
 ** clear buffer register (CPCLRB) and transferred to the main compare clear
 ** register (CPCLR) when zero is detected.
 **
 ** The FRT is closely coupled with the Input Capture Unit (ICU) and the
 ** Output Compare Unit (OCU) as this units use the FRT counter value for their
 ** operation. 
 **
 ** The typical sequence of using the FRT is to first initialize it by calling
 ** Frt_Init(). This sets up the timers operational parameters
 ** (see also #stc_frt_config_t).
 ** To start the timer Frt_Start() is used. The timer can be stopped at 
 ** anytime by calling Frt_Stop().
 **
 ** To get the count of FRT to call Frt_GetCount().
 **
 *****************************************************************************/
//@{

/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/


/**
 *****************************************************************************
 ** \brief Free Running Timer clock prescaler
 **
 ** Enumeration for prescaling primary clock
 *****************************************************************************/
typedef enum en_frt_clock_prescaler
{
    FrtDiv1     = 0x0,  /*!< Count clock is not divided */
    FrtDiv2     = 0x1,  /*!< Count clock is divide by 2 */
    FrtDiv4     = 0x2,  /*!< Count clock is divide by 4 */
    FrtDiv8     = 0x3,  /*!< Count clock is divide by 8 */
    FrtDiv16    = 0x4,  /*!< Count clock is divide by 16 */
    FrtDiv32    = 0x5,  /*!< Count clock is divide by 32 */
    FrtDiv64    = 0x6,  /*!< Count clock is divide by 64 */
    FrtDiv128   = 0x7,  /*!< Count clock is divide by 128 */
    FrtDiv256   = 0x8,  /*!< Count clock is divide by 256 */
} en_frt_clock_prescaler_t;

/**
 *****************************************************************************
 ** \brief Free Running Timer Interrupt mask selection
 *****************************************************************************/
typedef enum en_frt_msi
{
    FrtCnt1     = 0x0,  /*!< Generate an interrupt flag at the first match occurrence   */
    FrtCnt2     = 0x1,  /*!< Generate an interrupt flag at the second match occurrence  */
    FrtCnt3     = 0x2,  /*!< Generate an interrupt flag at the third match occurrence   */
    FrtCnt4     = 0x3,  /*!< Generate an interrupt flag at the fourth match occurrence  */
    FrtCnt5     = 0x4,  /*!< Generate an interrupt flag at the fifth match occurrence   */
    FrtCnt6     = 0x5,  /*!< Generate an interrupt flag at the sixth match occurrence   */
    FrtCnt7     = 0x6,  /*!< Generate an interrupt flag at the seventh match occurrence */
    FrtCnt8     = 0x7,  /*!< Generate an interrupt flag at the eighth match occurrence  */
} en_frt_msi_t;


/**
 *****************************************************************************
 ** \brief Free Running Timer configuration
 **
 ** This struct holds the configuration parameters of a single FRT.
 *****************************************************************************/
typedef struct stc_frt_config
{
    /** Set interrupt mask selection. See #en_frt_msi_t. */
    en_frt_msi_t              enInterruptMaskSelection;
    /** Buffer functionality. TRUE:Enable the compare clear buffer, FALSE=Disable the compare clear buffer */
    boolean_t                 bCompareClearBufferEnable;
    /** Timer count mode. TRUE:Up/down count mode, FALSE:Up count mode */
    boolean_t                 bUpDownCountEnable;
    /** Set clock prescaler. See #en_frt_clock_prescaler_t. */
    en_frt_clock_prescaler_t  enPrescaler;
    /** Interrupt mask mode2. See MODE2 of hardware manual. */
    boolean_t                 bInterruptMaskMode2;
    /** Set Compare clear interrupt mask selection. See #en_frt_msi_t. */
    en_frt_msi_t              enCompareClearInterruptMaskSelection;

    /** Callback for zero detect interrupts, can be NULL for synchronous operation */
    func_ptr_t                pfnZeroDetectCallback;
    /** Callback for compare clear interrupts, can be NULL for synchronous operation */
    func_ptr_t                pfnCompareClearCallback;

} stc_frt_config_t;


/**
 *****************************************************************************
 ** \brief Data type for holding internal data needed for FRT
 **
 ** This struct is used to store the interrupt callback functions for the FRT.
 *****************************************************************************/
typedef struct stc_frt_intern_data
{

    func_ptr_t  pfnZeroDetectCallback;   /*!< Called when timer reaches zero (in up-down count mode),  */
                                         /*!< can be NULL for synchronous operation without interrupt. */
    func_ptr_t  pfnCompareClearCallback; /*!< Called when timer reaches the compare value,             */
                                         /*!< can be NULL for synchronous operation without interrupt. */
} stc_frt_intern_data_t;


/*****************************************************************************/
/* Global variable declarations ('extern', definition in C source)           */
/*****************************************************************************/

/*****************************************************************************/
/* Global function prototypes ('extern', definition in C source)             */
/*****************************************************************************/

/*****************************************************************************
 * Global function prototypes
 *****************************************************************************/
extern en_result_t Frt_Init( volatile stc_frtn_t* pstcFrt,
                             const stc_frt_config_t* pstcConfig
                           ) ;
extern en_result_t Frt_Start( volatile stc_frtn_t* pstcFrt,
                              uint32_t u32CompareClearValue
                            ) ;
extern en_result_t Frt_Stop( volatile stc_frtn_t* pstcFrt ) ;
extern en_result_t Frt_GetCount( volatile stc_frtn_t* pstcFrt, uint32_t* data ) ;


//@}

#ifdef __cplusplus
}
#endif

#endif /* __RLT_H__ */
/*****************************************************************************/
/* EOF (not truncated)                                                       */
/*****************************************************************************/
