#include <string.h>
#include <esp_log.h>
#include <nvs_flash.h>
#include <sys/param.h>
#include <sys/stat.h>
#include "esp_netif.h"
#include <esp_wifi.h>
#include <esp_system.h>
#include "nvs_flash.h"
#include "esp_ota_ops.h"
#include "http_server.h"

static const char *TAG = "http_server";

char buf[MAX_OTA_BUFF]; // 接收服务端传来的文件缓存, 必须使用全局变量, 否则会触发看门狗复位, 原理未知

httpd_handle_t http_server = NULL;

static esp_err_t http_resp_index_html(httpd_req_t *req)
{
    extern const unsigned char index_html_start[] asm("_binary_index_html_start");
    extern const unsigned char index_html_end[] asm("_binary_index_html_end");
    const size_t index_html_size = (index_html_end - index_html_start);
    ESP_LOGI(TAG, "---- DevKit首页");
    httpd_resp_set_type(req, "text/HTML");
    httpd_resp_send(req, (const char *)index_html_start, index_html_size);
    return ESP_OK;
}

static esp_err_t http_resp_favicon_ico(httpd_req_t *req)
{
    extern const unsigned char favicon_ico_start[] asm("_binary_favicon_ico_start");
    extern const unsigned char favicon_ico_end[] asm("_binary_favicon_ico_end");
    const size_t favicon_ico_size = (favicon_ico_end - favicon_ico_start);
    
    ESP_LOGI(TAG, "---- 下载网站图标");
    httpd_resp_set_type(req, "image/x-icon");
    httpd_resp_send(req, (const char *)favicon_ico_start, favicon_ico_size);
    return ESP_OK;
}

static esp_err_t http_resp_ota_html(httpd_req_t *req)
{
    extern const unsigned char ota_html_start[] asm("_binary_ota_html_start");
    extern const unsigned char ota_html_end[] asm("_binary_ota_html_end");
    const size_t ota_html_size = (ota_html_end - ota_html_start);

    ESP_LOGI(TAG, "---- 进入固件升级页面");
    httpd_resp_set_type(req, "text/HTML");
    httpd_resp_send(req, (const char *)ota_html_start, ota_html_size);
    return ESP_OK;
}

static esp_err_t http_resp_handler(httpd_req_t *req)
{
    const char *uri_get = req->uri;
    ESP_LOGI(TAG, "---- request uri: 192.168.4.1%s", req->uri);
    if (strcmp(uri_get, "/") == 0)
    {
        return http_resp_index_html(req);
    }
    else if (strcmp(uri_get, "/favicon.ico") == 0)
    {
        return http_resp_favicon_ico(req);
    }
    else if (strcmp(uri_get, "/ota") == 0)
    {
        return http_resp_ota_html(req);
    }
    else
    {
        ESP_LOGI(TAG, "---- 未知链接, 不处理, 直接回到起始页");
        return http_resp_index_html(req);
    }

    return ESP_OK;
}

static esp_err_t http_ota_handler(httpd_req_t *req)
{
    int ret = 0;
    int recv_block = 0;
    int remaining = req->content_len;
    int total = remaining;
    double percent = 0.0;

    ESP_LOGI(TAG, "---- 应用程序更新, 需接收数据长度(bytes): %d", remaining);
    esp_ota_handle_t app_update_handle = 0;
    const esp_partition_t *app_update_partition = esp_ota_get_next_update_partition(NULL);
    esp_err_t err = esp_ota_begin(app_update_partition, OTA_WITH_SEQUENTIAL_WRITES, &app_update_handle);
    while (remaining > 0)
    {
        /* Read the data for the request */
        if ((ret = httpd_req_recv(req, buf, MIN(remaining, sizeof(buf)))) <= 0)
        {
            if (ret == HTTPD_SOCK_ERR_TIMEOUT)
            {
                /* Retry receiving if timeout occurred */
                continue;
            }
            return ESP_FAIL;
        }

        err = esp_ota_write(app_update_handle, buf, ret);
        if (err != ESP_OK)
        {
            ESP_LOGI(TAG, "---- 写入数据失败, 错误信息:%s", esp_err_to_name(err));
        }
        else
        {
            remaining -= ret;
            recv_block++;
            if ((recv_block % 20) == 0)
            {
                percent = 100.0 - (double)(remaining * 100) / (double)total;
                ESP_LOGI(TAG, "---- 写入OTA升级数据: %.2f%%", percent);
            }
        }

        // 增加20ms延时, 解决CPU0看门狗超时的问题
        vTaskDelay(pdMS_TO_TICKS(20));
    }
    err = esp_ota_end(app_update_handle);
    if (err == ESP_OK)
    {
        esp_err_t err = esp_ota_set_boot_partition(app_update_partition);
        if (err != ESP_OK)
        {
            ESP_LOGI(TAG, "---- 设备启动分区失败, 错误信息:%s", esp_err_to_name(err));
        }
        else
        {
            ESP_LOGI(TAG, "---- 应用程序更新完成, 3秒后自动重启.");
            httpd_resp_send_chunk(req, NULL, 0);
            // httpd_resp_sendstr(req, "<p>应用程序更新完成，3秒后自动重启.</p>");
            vTaskDelay(pdMS_TO_TICKS(3000));
            esp_restart();
        }
    }

    // End response
    ESP_LOGI(TAG, "---- 数据接收完成, OTA失败, 1秒后自动重启.");
    // httpd_resp_sendstr(req, "<p>数据上传完成，但应用更新不成功，请重试.</p>");
    httpd_resp_send_chunk(req, NULL, 0);
    vTaskDelay(pdMS_TO_TICKS(1000));
    esp_restart();
    return ESP_OK;
}

httpd_handle_t start_webserver(void)
{
    httpd_handle_t server = NULL;
    httpd_config_t config = HTTPD_DEFAULT_CONFIG();
    config.lru_purge_enable = true;
    config.uri_match_fn = httpd_uri_match_wildcard;

    // Start the httpd server
    ESP_LOGI(TAG, "---- 服务器端口号: '%d'", config.server_port);
    if (httpd_start(&server, &config) == ESP_OK)
    {
        // Set URI handlers
        ESP_LOGI(TAG, "---- 注册URI管理程序");

        // 注册客户端请求行为
        httpd_uri_t http_resp = {
            .uri = "/*",
            .method = HTTP_GET,
            .handler = http_resp_handler,
            .user_ctx = NULL};
        httpd_register_uri_handler(server, &http_resp);

        // 注册客户端OTA行为
        httpd_uri_t http_ota = {
            .uri = "/ota/*",
            .method = HTTP_POST,
            .handler = http_ota_handler,
            .user_ctx = NULL};
        httpd_register_uri_handler(server, &http_ota);

        return server;
    }

    ESP_LOGI(TAG, "---- 服务器启动时发生错误.");
    return NULL;
}

esp_err_t stop_webserver(httpd_handle_t server)
{
    return httpd_stop(server);
}

void connect_handler(void *arg)
{
    httpd_handle_t *server = (httpd_handle_t *)arg;
    if (*server == NULL)
    {
        ESP_LOGI(TAG, "---- 开启HTTP服务器");
        *server = start_webserver();
    }
}

void disconnect_handler(void *arg)
{
    httpd_handle_t *server = (httpd_handle_t *)arg;
    if (*server)
    {
        ESP_LOGI(TAG, "---- 关闭HTTP服务器");
        if (stop_webserver(*server) == ESP_OK)
        {
            *server = NULL;
        }
        else
        {
            ESP_LOGE(TAG, "Failed to stop http server");
        }
    }
}

httpd_handle_t http_service(void)
{
    static httpd_handle_t server = NULL;

    // 首次启动http_server
    server = start_webserver();
    if (server)
    {
        ESP_LOGI(TAG, "---- HTTP服务器开启成功.");
    }
    else
    {
        ESP_LOGI(TAG, "---- HTTP服务器开启失败.");
    }

    return server;
}
