/******************************************************************************
 * $Revision: 423 $
 * $Date:: 2017-04-07 16:03:30 +0900#$
 *****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************/
/** \file mfs_uart.h
 **
 ** Headerfile for UART functions
 **  
 ** History:
 **   - 2014-06-04  0.01  HS  Initial version for Traveo
 **   - 2014-07-04  0.02  HS  Add interrupt and FIFO handling.
 **   - 2015-09-09  0.03  ST  Fixed interrupt enable/disable to use the bit band unit
 *****************************************************************************/

#ifndef __MFS_UART_H__
#define __MFS_UART_H__

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
#include "base_types.h"


/**
 *****************************************************************************
 ** \defgroup UartGroup Asynchronous Serial Interface (UART)
 **
 ** \brief This section describes the interface for
 **        the asynchronous serial interface.
 **
 ** Provided functions of UART module:
 ** 
 ** - Mfs_Uart_Init()
 ** - Mfs_Uart_Read()
 ** - Mfs_Uart_Write()
 **
 ** Mfs_Uart_Init() must be used for activation and setup of an UART channel. All
 ** UART connection related parameters can be set with parameter 
 ** pstcConfig of type #stc_uart_config_t. 
 ** Mfs_Uart_Read() and Mfs_Uart_Write() can be used for communication.
 ** See the description of these functions for detail.
 **
 ** \note In this sample, following settings are not supported.
 **         - Using Multi Processor mode and data length over 8bits
 **
 *****************************************************************************/
/*!@{ */

/*****************************************************************************/
/* Global pre-processor symbols/macros ('#define')                           */
/*****************************************************************************/
/** UART ch0 */
#define UART0_Type   ((pstc_uart_type_t)&CPG_MFS00_UART)
#define UART0_TIE    CPG_MFS00_UART_SCR_TIE
#define UART0_RIE    CPG_MFS00_UART_SCR_RIE
/** UART ch1 */
#define UART1_Type   ((pstc_uart_type_t)&CPG_MFS01_UART)
#define UART1_TIE    CPG_MFS01_UART_SCR_TIE
#define UART1_RIE    CPG_MFS01_UART_SCR_RIE
/** UART ch2 */
#define UART2_Type   ((pstc_uart_type_t)&CPG_MFS02_UART)
#define UART2_TIE    CPG_MFS02_UART_SCR_TIE
#define UART2_RIE    CPG_MFS02_UART_SCR_RIE
/** UART ch3 */
#define UART3_Type   ((pstc_uart_type_t)&CPG_MFS03_UART)
#define UART3_TIE    CPG_MFS03_UART_SCR_TIE
#define UART3_RIE    CPG_MFS03_UART_SCR_RIE
/** UART ch4 */
#define UART4_Type   ((pstc_uart_type_t)&CPG_MFS04_UART)
#define UART4_TIE    CPG_MFS04_UART_SCR_TIE
#define UART4_RIE    CPG_MFS04_UART_SCR_RIE
/** UART ch5 */
#define UART5_Type   ((pstc_uart_type_t)&CPG_MFS05_UART)
#define UART5_TIE    CPG_MFS05_UART_SCR_TIE
#define UART5_RIE    CPG_MFS05_UART_SCR_RIE
/** UART ch6 */
#define UART6_Type   ((pstc_uart_type_t)&CPG_MFS06_UART)
#define UART6_TIE    CPG_MFS06_UART_SCR_TIE
#define UART6_RIE    CPG_MFS06_UART_SCR_RIE
/** UART ch7 */
#define UART7_Type   ((pstc_uart_type_t)&CPG_MFS07_UART)
#define UART7_TIE    CPG_MFS07_UART_SCR_TIE
#define UART7_RIE    CPG_MFS07_UART_SCR_RIE
/** UART ch8 */
#define UART8_Type   ((pstc_uart_type_t)&CPG_MFS08_UART)
#define UART8_TIE    CPG_MFS08_UART_SCR_TIE
#define UART8_RIE    CPG_MFS08_UART_SCR_RIE
/** UART ch9 */
#define UART9_Type   ((pstc_uart_type_t)&CPG_MFS09_UART)
#define UART9_TIE    CPG_MFS09_UART_SCR_TIE
#define UART9_RIE    CPG_MFS09_UART_SCR_RIE
/** UART ch10 */
#define UART10_Type   ((pstc_uart_type_t)&CPG_MFS10_UART)
#define UART10_TIE    CPG_MFS10_UART_SCR_TIE
#define UART10_RIE    CPG_MFS10_UART_SCR_RIE
/** UART ch11 */
#define UART11_Type   ((pstc_uart_type_t)&CPG_MFS11_UART)
#define UART11_TIE    CPG_MFS11_UART_SCR_TIE
#define UART11_RIE    CPG_MFS11_UART_SCR_RIE

/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/
/**
 *****************************************************************************
 ** \brief UART Unit Type
 *****************************************************************************/
typedef  volatile stc_cpg_mfsn_uart_t*  pstc_uart_type_t;

/**
 *****************************************************************************
 ** \brief UART Enable/Disable Interrupt Type
 *****************************************************************************/
typedef  volatile uint_io8_t  u8_uart_int_t;

/**
 *****************************************************************************
 ** \brief UART tx callback function prototypes.
 *****************************************************************************/
typedef void (*uart_tx_cb_func_ptr_t)(void);

/**
 *****************************************************************************
 ** \brief UART rx callback function prototypes.
 **        This callback is called when rx buffer is greater than or equal to
 **        u16RxCbFillLevel.
 **        If use FIFO, reception interrupt is generated per 64bytes.
 **        So, callback timing is per 64bytes.
 **
 ** \param  u16FillCnt  Number of received data.
 *****************************************************************************/
typedef void (*uart_rx_cb_func_ptr_t)(uint16_t u16FillCnt);

/**
 *****************************************************************************
 ** \brief Uart data length
 **
 ** Over 9bits data length are not supported in this sample.
 *****************************************************************************/
typedef enum en_uart_data_length
{
  UartLength5Bits = 0,      /*!< 5-bit length */
  UartLength6Bits = 1,      /*!< 6-bit length */
  UartLength7Bits = 2,      /*!< 7-bit length */
  UartLength8Bits = 3       /*!< 8-bit length */
} en_uart_data_length_t;


/**
 *****************************************************************************
 ** \brief Uart parity.
 *****************************************************************************/
typedef enum en_uart_parity
{
  UartParityNone  = 0,    /*!< No parity bit is used. */
  UartParityEven  = 2,    /*!< Even parity bit is used. */
  UartParityOdd   = 3     /*!< Odd parity bit is used. */
} en_uart_parity_t;


/**
 *****************************************************************************
 ** \brief Uart start/stop bit.
 **
 ** Enum value #UartNoStartNoStop is not allowed in asynchronous mode! 
 *****************************************************************************/
typedef enum en_uart_stopbit
{
    UartOneStopBit   = 1,    /*!< 1 stop bit */
    UartTwoStopBit   = 2,    /*!< 2 stop bits */
    UartThreeStopBit = 3,    /*!< 3 stop bits */
    UartFourStopBit  = 4     /*!< 4 stop bits */
} en_uart_stopbit_t;


/**
 *****************************************************************************
 ** \brief Uart data format.
 **
 *****************************************************************************/
typedef enum en_uart_data_format
{
    UartFormatNRZ = 0,              /*!< NRZ format */
    UartFormatInvertedNRZ = 1       /*!< Inverted NRZ format */
} en_uart_data_format_t;


/**
 *****************************************************************************
 ** \brief Uart ring buffer.
 ** 
 ** Contains all parameter for ring buffer handling. 
 *****************************************************************************/
typedef struct stc_uart_buffer
{
    uint8_t*  pu8Buffer;      /*!< Pointer to character buffer. */
    uint16_t  u16BufferSize;  /*!< Size of buffer. */
    uint16_t  u16InIndex;     /*!< Index of next element to store on buffer. */
    uint16_t  u16OutIndex;    /*!< Index of next element to read from buffer. */
    uint16_t  u16FillCount;   /*!< Indicates if elements are available in buffer. */
} stc_uart_buffer_t;


/**
 ******************************************************************************
 ** \brief UART configuration.
 **
 ** Contains all parameter for configuratin a Uart channel, if set to UART mode.
 ******************************************************************************/
typedef struct stc_uart_config
{
    uint32_t                u32DataRate;        /*!< Bits per second */
    en_uart_parity_t        enParity;           /*!< See description of #en_uart_parity_t. */
    en_uart_stopbit_t       enStopBit;          /*!< See description of #en_uart_start_stopbit_t. */
    en_uart_data_length_t   enDataLength;       /*!< 0-7bit, 1-8bit Character Length */
    boolean_t               bHardwareFlow;      /*!< TRUE=Enable hardware flow, FALSE=Disable */
    en_uart_data_format_t   enDataFormat;       /*!< See description of #en_uart_data_format_t. */
    boolean_t               bTransferDirecton;  /*!< TRUE=MSB first, FALSE=LSB first */
    uint8_t*                pu8TxBuf;           /*!< Pointer to tranasmit FIFO buffer. */
    uint8_t*                pu8RxBuf;           /*!< Pointer to receive FIFO buffer. */
    uint16_t                u16TxBufSize;       /*!< Size of transmit FIFO buffer. */
    uint16_t                u16RxBufSize;       /*!< Size of receive FIFO buffer. */
    boolean_t               bFifoUsage;         /*!< Usage of MFS FIFO, TRUE=use FIFO */
    uint16_t                u16RxCbFillLevel;   /*!< Unread counts of data buffer to call RX Callback function */
    uart_rx_cb_func_ptr_t   pfnRxCb;            /*!< Callback function, if RX Buffer is filled more than u16RxCbBufFillLvl */
    uart_tx_cb_func_ptr_t   pfnTxCb;            /*!< Callback function, if TX Buffer is empty */
} stc_uart_config_t;
/**
 *****************************************************************************
 ** \brief UART module internal data,
 **        storing internal information for each enabled UART instance.
 *****************************************************************************/
typedef struct stc_mfs_hl_intern_data
{
    uart_tx_cb_func_ptr_t       pfnTransmitCbFunction;      /*!< Transmission complete callback function */
    uart_rx_cb_func_ptr_t       pfnReceiveCbFunction;       /*!< Receive callback function */
    stc_uart_buffer_t           stcRxBuffer;                /*!< Store location for receive Fifo buffer */
    stc_uart_buffer_t           stcTxBuffer;                /*!< Store location for transmission Fifo buffer */
    uint16_t                    u16RxCbFillLevel;           /*!< Unread counts of data buffer to call RX Callback function */
    boolean_t                   bFifoUsage;                 /*!< TRUE=use FIFO, FALSE=not use */
} stc_uart_intern_data_t;

/*****************************************************************************/
/* Global variable declarations ('extern', definition in C source)           */
/*****************************************************************************/


/*****************************************************************************/
/* Global function prototypes ('extern', definition in C source)             */
/*****************************************************************************/
extern   __irq __arm void  Uart_Isr_Uart4_Rx(void);
extern   __irq __arm void  Uart_Isr_Uart4_Tx(void);
extern en_result_t Mfs_Uart_Init(pstc_uart_type_t pstcUart, const stc_uart_config_t *pstcConfig);
extern en_result_t Mfs_Uart_Write(pstc_uart_type_t pstcUart,uint8_t *pu8Data, uint16_t u16WriteCnt, boolean_t bBlocking);
extern en_result_t Mfs_Uart_Read(pstc_uart_type_t pstcUart, uint8_t *pu8Buffer, uint16_t *pu16DataCount, uint16_t u16ReadCount, boolean_t bWait);
extern void MfsUartIrqHandlerRx(pstc_uart_type_t pstcUart, stc_uart_intern_data_t *pstcUartInternData);
extern void MfsUartIrqHandlerTx(pstc_uart_type_t pstcUart, stc_uart_intern_data_t* pstcUartInternData);
extern stc_uart_intern_data_t* MfsUartGetInternDataPtr(pstc_uart_type_t pstcUart);
extern u8_uart_int_t* MfsUartGetTIE(pstc_uart_type_t pstcUart);
extern u8_uart_int_t* MfsUartGetRIE(pstc_uart_type_t pstcUart);
//extern en_result_t UART_init(uint8_t ch, uint32_t BootRate, boolean_t MSB, uart_rx_cb_func_ptr_t Callback);

#endif /* __MFS_UART_H__ */
/*****************************************************************************/
/* EOF (not truncated)                                                       */
/*****************************************************************************/
