


#ifndef _POPUP_SCHEDULER_H_
#define _POPUP_SCHEDULER_H_

#include "TYW_stdint.h"

/******************************************************************************
报警状态定义
******************************************************************************/
#define   POPUP_STATUS_OFF                      0x00u
#define   POPUP_STATUS_NEW                      0x01u
#define   POPUP_STATUS_FIRST                    0x02u
#define   POPUP_STATUS_LOOP                     0x03u
#define   POPUP_STATUS_MASKED                   0x04u
#define   POPUP_STATUS_DELETE                   0x05u
#define   POPUP_STATUS_CLEAR                    0x06u

/******************************************************************************
报警切换等级
******************************************************************************/
#define   POPUP_CUT_NOT                         0x00u
#define   POPUP_CUT_IMMEDIATELY                 0x01u
#define   POPUP_CUT_WAIT_TIME_MIN               0x02u
#define   POPUP_CUT_WAIT_TIME_NORMAL            0x03u

/******************************************************************************
报警策略配置
******************************************************************************/
#define   POPUP_DEFINE_FIRST_DISPLAY            0u      /*首次显示，POPUP_CUT_WAIT_TIME_MIN和POPUP_CUT_WAIT_TIME_NORMAL切换需要等待，否则，立即切换*/
#define   POPUP_SAME_CLASS_CUT_IMMEDIATELY      1u      /*同类型的信息可以立即打断，主要是声音*/
#define   POPUP_HIGH_CUT_LOW_TYPE_IMMEDIATELY   1u      /*类型高立即打断类型低：=1使能，=0不使能*/
#define   POPUP_HINT_CUT_HINT_IMMEDIATELY       1u      /*提示类信息立即打断提示类信息：=1使能，=0不使能*/
#define   POPUP_WARNING_CUT_HINT_MIN            0u      /*报警类可以打断提示类，但需要等待最小时间：=1使能，=0不使能*/
#define   POPUP_CUT_RECOVER                     1u      /*被打断的信息，恢复显示时，继续从此开始显示*/
#define   POPUP_DOOR_OPEN_REAL                  1u      /*门开实时响应，打断其他的信息*/
#define   POPUP_DISP_TIME_NORMAL                3u*10u  /*unit = 100ms*/
#define   POPUP_DISP_TIME_MIN                   3u*10u  /*unit = 100ms*/
#define   POPUP_DISP_TIME_NORMAL_FLAG           9       /*unit = 100ms*/
/******************************************************************************
查询模式定义
******************************************************************************/
#define   POPUP_POLLING_FORWARD                 0x00u
#define   POPUP_POLLING_REVERSE                 0x01u

/******************************************************************************
最大优先级数
******************************************************************************/
#define   POPUP_PRIORITY_NUMBER_MAX     POPUP_PRIORITY_NUMBER_FATAL

/******************************************************************************
弹出信息调度结构
******************************************************************************/
typedef struct
{
    uint8_t   Timer;
    uint16_t   NewPopupID;
    uint16_t   PollPopupID;
    uint8_t   CutGrade;
    uint16_t   CurPopupID;
    uint8_t   CurStatus;
    uint8_t   SndReqCode;
    uint16_t   CurMaskedID;
    uint8_t   WarningExist;
} PopupSchedulingStruct;

/******************************************************************************
函数声明
******************************************************************************/

/**********************************************************************************************************
 *@Function:    Popup_Scheduler_Init
 *@Description: 初始化弹出信息调度器
 *@Input:             MY_MemInit / CommMemInit
 *@Output:          none.
 *@Calls:           none.
 *@Return:          none.
 *@Note:            存在内存管理函数，要求必须在硬件初始化之前，进行初始化操作
 **********************************************************************************************************/
void Popup_Scheduler_Init ( void );

/**********************************************************************************************************
 *@Function:        Popup_Scheduler_OFF
 *@Description:     销毁存在的链表，恢复为初始化状态
 *@Input:           MY_MemInit / CommMemInit
 *@Output:          none.
 *@Calls:           none.
 *@Return:          none.
 *@Note:            none.
 **********************************************************************************************************/
void Popup_Scheduler_OFF ( void );

/**********************************************************************************************************
 *@Function:        Popup_Scheduling_Service
 *@Description:   调度弹出信息，更新当前显示的信息名称,接口函数
 *                        可能发生切换的条件：
 *                        1、当前显示信息被删除/屏蔽
 *                        2、产生优先级高的新消息
 *                        3、显示时间到
 *@Input:               CommMemInit
 *@Output:            none.
 *@Calls:             none
 *@Return:            none
 *@Note:              该服务函数必须每100ms被调用一次
 **********************************************************************************************************/
void Popup_Scheduling_Service ( void );

/**********************************************************************************************************
 *@Function:        Popup_Request
 *@Description:   信息插入请求，包括全部类型，接口函数
 *@Input:               uint8_t PopupID -- 弹出信息的ID.
 *@Output:            none.
 *@Calls:             none.
 *@Return:            none.
 *@Note:              none.
 **********************************************************************************************************/
void Popup_Request ( uint16_t PopupID );

/**********************************************************************************************************
 *@Function:        Popup_Mask_Request
 *@Description:   屏蔽第三类（报警类）信息，只有显示过的才可以被屏蔽
 *                ***********屏蔽全部显示过的***************
 *@Input:               none.
 *@Output:            none.
 *@Calls:             none.
 *@Return:            none.
 *@Note:              none.
 **********************************************************************************************************/
void Popup_Mask_Request ( void );

/**********************************************************************************************************
 *@Function:    Popup_Del_Masked_Request
 *@Description: 在IGN_OFF时，需要将屏蔽过的报警信息显示出来，但不允许再次显示报警声音
 *@Input:       none.
 *@Output:      none.
 *@Calls:       none.
 *@Return:      none.
 *@Note:        目前将报警信息的状态切换为循环显示状态.
 **********************************************************************************************************/
void Popup_Del_Masked_Request ( void );

/**********************************************************************************************************
 *@Function:        Popup_Get_Status
 *@Description:   获取查询信息的状态
 *@Input:               uint8_t PopupID -- 弹出信息的ID
 *@Output:            none.
 *@Calls:             PopupList_Select
 *@Return:            POPUP_STATUS_OFF -- 此信息不在需要显示
 *                POPUP_STATUS_FIRST -- 此信息第一次显示
 *                POPUP_STATUS_LOOP -- 此信息处于轮询显示中
 *                POPUP_STATUS_MASKED -- 此信息被屏蔽
 *                POPUP_STATUS_DELETE -- 此信息等待被删除
 *                POPUP_STATUS_CLEAR -- 此信息马上被删除
 *@Note:              none.
 **********************************************************************************************************/
uint8_t Popup_Get_Status ( uint16_t PopupID );

/**********************************************************************************************************
 *@Function:        Popup_Get_Current_Message
 *@Description:   获取当前信息的文字编号
 *@Input:               none.
 *@Output:            none.
 *@Calls:             none.
 *@Return:            none.
 *@Note:              none.
 **********************************************************************************************************/
uint16_t Popup_Get_Current_Message ( void );

/**********************************************************************************************************
 *@Function:        Popup_Get_Current_Sound
 *@Description:   获取当前信息的声音编号
 *@Input:               none.
 *@Output:            none.
 *@Calls:             none.
 *@Return:            none.
 *@Note:              声音请求码影响声音的持续性及是否打断
 **********************************************************************************************************/
uint8_t Popup_Get_Current_Sound ( void );

/**********************************************************************************************************
 *@Function:        Popup_Get_Current_Sound_Req_Code
 *@Description:   获取当前信息的声音请求码
 *@Input:               none.
 *@Output:            none.
 *@Calls:             none.
 *@Return:            none.
 *@Note:              声音请求码影响声音的持续性及是否打断
 **********************************************************************************************************/
uint8_t Popup_Get_Current_Sound_Req_Code ( void );

/**********************************************************************************************************
 *@Function:        Popup_Delete
 *@Description:   清除当前显示信息，如果是第一次显示，要等待Tmin（第二类信息（提示类）除外）
 *@Input:               uint8_t PopupID -- 弹出信息的ID
 *@Output:            FatalListHead/HintListHead/WarningListHead/ExceptionListHead -- 新链表
 *@Calls:             PopupList_Select / PopupList_Delete
 *@Return:            none.
 *@Note:              none.
 **********************************************************************************************************/
void Popup_Delete ( uint16_t PopupID );

/**********************************************************************************************************
 *@Function:        Popup_Clear
 *@Description:   清除当前显示信息，不需要等待Tmin
 *@Input:               uint8_t PopupID -- 弹出信息的ID
 *@Output:            FatalListHead/HintListHead/WarningListHead/ExceptionListHead -- 新链表
 *@Calls:             PopupList_Select / PopupList_Delete
 *@Return:            none.
 *@Note:              none.
 **********************************************************************************************************/
void Popup_Clear ( uint16_t PopupID );

/**********************************************************************************************************
 *@Function:        Popup_Get_Warning_Status
 *@Description:   查询是否存在报警类信息/或者是获取显示致命信息下，是否存在报警类信息
 *@Input:               none
 *@Output:            none
 *@Calls:             none
 *@Return:            TRUE --存在报警类信息
 *                FALSE -- 不存在报警类信息
 *@Note:              none.
 **********************************************************************************************************/
uint8_t Popup_Get_Warning_Status ( void );

/**********************************************************************************************************
 *@Function:        Popup_Get_Masked_Warning_Msg
 *@Description:   获取当前显示的屏蔽信息的文字/图片编号
 *@Input:               none
 *@Output:            none
 *@Calls:             none
 *@Return:            文字/图片编号
 *@Note:              none.
 **********************************************************************************************************/
uint16_t Popup_Get_Masked_Warning_Msg ( void );

/**********************************************************************************************************
 *@Function:        Popup_Masked_Warning_Polling_Reset
 *@Description:   复位屏蔽信息当前显示的指针，向前，则指向最后一个；向后，则指向第一个
 *@Input:               none.
 *@Output:            none.
 *@Calls:             none.
 *@Return:            none.
 *@Note:              none.
 **********************************************************************************************************/
void Popup_Masked_Warning_Polling_Reset ( uint8_t Mode );

/**********************************************************************************************************
函数名：Popup_Masked_Warning_Polling
功  能：查询已屏蔽的报警

        每调用一次此函数,已屏蔽报警指针会按指定方向移动一位,当指针已移指定方向
        上的最后一个报警信息时,再次向同一方向查询,当前的已屏蔽报警信息会变为
        POPUP_NONE 表示报警查询已结束

        例如:当前有已屏蔽报警 A、B、C、D,调用Popup_Masked_Warning_Polling_Reset
        函数,设定方向为 POPUP_POLLING_FORWARD 时,已屏蔽报警指针会指向报警A,而后,
        调用该函数设定方向为 POPUP_POLLING_FORWARD 时已屏蔽报警指针即指向报警B.
        重复调用此函数且方向不变至已屏蔽报警指针指向报警D后,再以相同方向调用此函
        数则返回 POPUP_NONE,表示查询已结束

        通过调用Popup_Get_Masked_Warning可读出当前已屏蔽报警指针指向的报警

参  数：Mode：查询模式 POPUP_POLLING_FORWARD - 正向查找模式
                       POPUP_POLLING_REVERSE - 逆向查找模式
返回值：无
 **********************************************************************************************************/
void Popup_Masked_Warning_Polling ( uint8_t Mode );
void Popup_Hint_Mask_Request ( uint16_t PopupID );
uint16_t Popup_Get_First_Masked_Warning ( void );
uint8_t Popup_WarningRequested ( void );
#endif
