/**************************************************************************//**
  * \file     Init.c
  * \brief    Initializations during work state switching
  * \details
  * \author   Zhang Xuan
  * \version  V1.0.0
  * \date     18-Jan-2019
  * \par      History:
  *           V1.0.0 Initial release
  * \par      Copyright:
  *           (c) Heilongjiang TYW Electronics co., LTD
******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "debugger.h"
#include "r_typedefs.h"
#include "Init.h"
#include "r_dev_clk_types.h"
#include "r_ddb_api.h"

#include "Analog_Signals.h"
#include "Clock.h"
#include "Emulated_EEPROM.h"
#include "GPIO.h"
#include "Internal_Flash.h"
#include "RTC.h"
#include "BU98R10.h"
#include "Watchdog.h"


#include "r_dev_api.h"
#include "r_wm_api.h"
#include "r_tick_api.h"


#include "rh850_macros.h"
#include "cpu.h"
#include "Graphic.h"
#include "Sys_Tick.h"
#include "RSCAN.h"
#include "SDRAM.h"
#include "sflash.h"

#include "sys_scheduler.h"

#include "RTE_ADC.h"
#include "COM_CAN.h"
#include "UDS_Common.h"
#include "r_fdl_types.h"
#include "fdl_descriptor.h"

#define WITHOUT_BOOT 1
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

void DelayUs(uint32_t time)
{
    uint32_t i;
    for (i = 0; i < (time * 2); i++)
    {
        R_DEV_Nop();
        WDT_Clear();
    }
}
/**************************************************************************//**
  * \brief      Initializations during work state switching
  * \attention
  * \retval     None
******************************************************************************/
void Sys_Startup_Pre_Init(void)
{
    //R_DEV_WRITE_REG(32, LOC_XCRAMCFG0, 0x100100U);
    // *((volatile uint32_t*)(0xFFC06050u)) = (0x100100U);
    SET_EBASE(0x00200400); //set Ebase to 0x10000, set RINT to 1
    SET_PSW_EBV();
    /* Release all Modules (except SDRAM controller) */
    Sys_Peripheral_Reset(SYS_PERI_RESETTABLE & (~SYS_PERI_SDRB0));

    /* Prepare device (clear all internal config, prepare interrupts) */
    R_DEV_Prepare();
#if (WITHOUT_BOOT)
    {
        Clock_Pre_Init();
    }
#endif

    GPIO_Init(0);



    Mem_SDRAM_Init(MEM_SDRAM_2x_W9812G6KH_120MHz, Mem_SDRAM_DRV_25Ohm);
    //Mem_SDRAM_Init(MEM_SDRAM_2x_IS45S16800F_120MHz, Mem_SDRAM_DRV_25Ohm);

    Sys_Enable_Preload_Buffer();
}

/**************************************************************************//**
  * \brief      Initializations during system power up
  * \attention
  * \retval     None
******************************************************************************/
uint8_t u8ResetFlag = 0;
uint32_t u32FDLDataFlag[4] = {0, 0, 0, 0};
uint32_t u32FDLBlank = 0;

extern uint32_t u32ResetFlag;
extern uint32_t TestBackupRam;

void Sys_Startup_Init(void)
{
    Clock_Init();
    /* Enable interrupts globally */
    enable_interrupt();

    WDT_Init();

    /* init graphic tick driver */
    R_TICK_Init(0);

    Gfx_Init(GFX_VO_SINGLE_RGB888 | GFX_VI_ITU656);

    CPU_Init();

    //RTC_Init();
    Sys_Tick_Timer_Start();
    /*COM_CAN_Init(); Ô­CAN³õÊ¼»¯Î»ÖÃ*/


    RTE_ADC_Init();

    DelayUs(50000);
    DelayUs(50001);

    Int_Flash_Init();

    BU98R10_Init();

    d_printf("Init complete!\n");
    InitSeriFlashExternReadMode();
    u8ResetFlag = Clock_Get_Startup_Mode();
    if (u8ResetFlag == CLOCK_STARTUP_PWR_ON)
    {
        TYW_FDL_Init();

        TYW_FDL_Flash_Read(0xFF20FF00 , u32FDLDataFlag, 4 );

        if ((u32FDLDataFlag[0] == 0x66666666) && (u32FDLDataFlag[1] == 0x55555555) && (u32FDLDataFlag[2] == 0xA555A55A) && (u32FDLDataFlag[3] == 0X12345678))
        {

        }
        else
        {
            u32FDLDataFlag[0] = 0x66666666;
            u32FDLDataFlag[1] = 0x55555555;
            u32FDLDataFlag[2] = 0xA555A55A;
            u32FDLDataFlag[3] = 0X12345678;

            TYW_FDL_Erase(0XFF200000, 0XFF20FFFF);

            TYW_FDL_Write(0xFF20FF00 ,  u32FDLDataFlag,  4);
        }
    }

    EEPROM_Init();

    if (u32ResetFlag != 0xA55AA55AUL)
    {
        u8ResetFlag = 0;
    }
    else
    {
        if ((u8ResetFlag == 0) && (TestBackupRam == 0xaabbccddu))
        {
            u8ResetFlag = 1;
            Rtc_Restore_Time();
        }
    }

    /*EOL诊断 配置数据读取 DFLASH*/
    ReadAllDFlashData();

    /*读取K-LINE*/
    ReadDTCEOLValue_Meter_K_LINE();

    /*30电初始化*/
    APP_Startup_Init(u8ResetFlag);
    TestBackupRam = 0x11223344UL;

    /*外发要用到EEP数据，位置移动*/
    COM_CAN_Init();
    Gfx_Sys_Start();
}

/**************************************************************************//**
  * \brief      Initializations during work state switch to system sleep
  * \attention
  * \retval     None
******************************************************************************/
void Sys_Sleep_Init(void)
{
    Sys_Tick_Timer_Stop();
}

/**************************************************************************//**
  * \brief      Initializations during work state switch to system run
  * \attention
  * \retval     None
******************************************************************************/
void Sys_Wakeup_Init(void)
{

}

