/****************************************************************************
 *  License : All rights reserved for TES Electronic Solutions GmbH
 *            See included /docs/license.txt for details
 *  Project : D/AVE HD
 *  Purpose : Colorunit (COL) related inline functions
 ****************************************************************************
 * Version Control Information :
 *  $Revision: 3935 $
 *  $Date: 2014-09-23 10:52:32 +0200 (Di, 23. Sep 2014) $
 *  $LastChangedBy: golczewskim $
 ****************************************************************************
 * Change History (autogenerated):
 ****************************************************************************/    

#ifndef DAVEHD_DRIVER_COL_INL_H_INCLUDED
#define DAVEHD_DRIVER_COL_INL_H_INCLUDED

/* PRQA S 3406 ++ *//* $Misra: #INLINE_FUNCTION_DEFINED_IN_HEADER $*/
/* PRQA S 3480 ++ *//* $Misra: #INLINE_FUNCTION_DEFINED_IN_HEADER $*/

/*----------------------------------------------------------------------------------------------------------*/
/* Set a busy bit for active color unit. The specified register index (assumed to be for unit 0) is mapped */
/* to active unit and the corresponding pending bit is set.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_register : Color unit register for unit 0*/
DHD_INLINE void dhd_internal_set_col_busy(dhd_device_data_t *a_device, dhd_uint32_t a_register) { /* PRQA S 3450 *//* $Misra: #INLINE_FUNCTION_NO_DECLARATION $*/
  dhd_uint32_t bit;

  /* all col unit registers are group 11*/
  bit = (a_register - 11*64) + a_device->m_col_unit_shift;                   
  a_device->m_pending_mask.m_groups.m_group11[ bit / 32 ] |= (1u << (bit % 32));
}

/*----------------------------------------------------------------------------------------------------------*/
/* Set a constant color. Writes a single constant color register. The number of available registers depends on*/
/* the hardware configuration and can be queried using dhd_get_const_col_count. */
/* Writes to nonexistant register indices will be silently ignored.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_index : Constant index (0..15)*/
/*  a_color : 32bit color in ARGB8888 format*/
/**/
/* See Also:*/
/*   dhd_get_const_col_count, dhd_set_color_array*/
DHD_API_INLINE void dhd_set_color(dhd_handle_t a_device, dhd_uint32_t a_index, dhd_uint32_t a_color) {
  dhd_device_data_t *device = dhd_get_device( a_device );

  if (a_index < device->m_description.m_num_const_cols) {
    /* store into correct register slot (relative to group 8)*/
    device->m_shadow_state.m_groups.m_group8[(DHD_REG_COL_CONST_COLOR_0 - (8*64)) + a_index] = a_color;
    /* address correct dirty bit*/
    device->m_pending_mask.m_groups.m_group8[0] |= (1u << (DHD_REG_COL_CONST_COLOR_0 - (8*64))) << a_index;
  }
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get a constant color. Retrieve current content of a single constant color register (see dhd_set_color*/
/* for details). Reads from non existant indices will return 0.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/**/
/* Returns:*/
/*  32bit color in ARGB8888 format*/
/**/
/* See Also:*/
/*  dhd_set_color*/
DHD_API_INLINE dhd_uint32_t dhd_get_color(dhd_handle_t a_device, dhd_uint32_t a_index) {
  dhd_device_data_t *device = dhd_get_device( a_device );

  if (a_index < device->m_description.m_num_const_cols) {
    /* store into correct register slot (relative to group 8)*/
    return device->m_shadow_state.m_groups.m_group8[(DHD_REG_COL_CONST_COLOR_0 - (8*64)) + a_index];    
  } else {  
    /* index out of bounds*/
    return 0;
  }
}

/*----------------------------------------------------------------------------------------------------------*/
/* Set multiple constant colors. To avoid repeated consecutive calls to dhd_set_color a single call to */
/* dhd_set_color_array can be used.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_start : Index of the first constant register to set (0..15)*/
/*  a_count : Number of registers*/
/*  a_colors : Pointer to an array of 32bit colors in ARGB8888 format */
/**/
/* See Also:*/
/*   dhd_get_const_col_count, dhd_set_color*/
DHD_API_INLINE void dhd_set_color_array(dhd_handle_t a_device, dhd_uint32_t a_start, dhd_uint32_t a_count, const dhd_uint32_t *a_colors) {
  dhd_uint32_t i;

  for (i=0; i<a_count; i++) {
    dhd_set_color( a_device, a_start + i, a_colors[i] );
  }
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get multiple constant colors. Retrieve current content of a several constant color registers (see dhd_set_color_array*/
/* for details). Reads from non existant indices will return 0.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_start : Index of the first constant register to set (0..15)*/
/*  a_count : Number of registers*/
/*  a_colors : Pointer to an array of dhd_uint32_t that will receive 32bit colors in ARGB8888 format */
/**/
/* See Also:*/
/*  dhd_set_color_array*/
DHD_API_INLINE void dhd_get_color_array(dhd_handle_t a_device, dhd_uint32_t a_start, dhd_uint32_t a_count, dhd_uint32_t *a_colors) {
  dhd_uint32_t i;

  if (a_colors != 0) {  
    for (i=0; i<a_count; i++) {
      a_colors[i] = dhd_get_color( a_device, a_start + i );
    }
  }
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select the active color unit. All following color unit state changes refer to that color unit. The */
/* available number of color units can be queried using dhd_get_color_unit_count. Selecting and configuring*/
/* a color unit does not enable it. In order to enable units dhd_set_color_unit_enable has to be used.*/
/* Selections of a nonexistant unit will be silently ignored.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_index : Index of color unit to be selected (0..7)*/
/**/
/* See Also:*/
/*   dhd_get_color_unit_count, dhd_set_color_unit_enable*/
DHD_API_INLINE void dhd_active_color_unit(dhd_handle_t a_device, dhd_uint32_t a_index) {
  dhd_device_data_t *device = dhd_get_device( a_device );

  if (a_index < device->m_description.m_num_col_units) {
    /* store index of currently active color unit*/
    device->m_col_unit_index = (dhd_uint8_t)a_index;
    device->m_col_unit_shift = (dhd_uint8_t)(a_index * 6);    
    device->m_col_unit_base = ((dhd_uint32_t *) & device->m_shadow_state.m_named.m_col_argb_op1a_0) + (a_index * 6);            
  }
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get a active color unit index. Retrieve currently active color unit (see dhd_active_color_unit for details). */
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/**/
/* Returns:*/
/*  Index of selected color unit (0..7)*/
/**/
/* See Also:*/
/*  dhd_active_color_unit*/
DHD_API_INLINE dhd_uint32_t dhd_get_active_color_unit(dhd_handle_t a_device) {
  dhd_device_data_t *device = dhd_get_device( a_device );

  return device->m_col_unit_index;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Set color scaling. Selects a scalefactor to be applied to the active color units output rgb value. Possible*/
/* values range from 0.5 to 4. */
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_scale : One of the four possible scale factors (see dhd_outscale_t)*/
DHD_API_INLINE void dhd_set_color_scale(dhd_handle_t a_device, dhd_outscale_t a_scale) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_out_0_t *col_argb_out = (dhd_reg_col_argb_out_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OUT_0 - (11*64) ];
  
  col_argb_out->m_fields.m_col_rgb_scale_out_0 = a_scale;     
  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OUT_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get color scaling. Retrieve currently active rgb scale factor (see dhd_set_color_scale for details). */
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/**/
/* Returns:*/
/*   One of the four possible scale factors (see dhd_outscale_t)*/
/**/
/* See Also:*/
/*  dhd_set_color_scale*/
DHD_API_INLINE dhd_outscale_t dhd_get_color_scale(dhd_handle_t a_device) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_out_0_t *col_argb_out = (dhd_reg_col_argb_out_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OUT_0 - (11*64) ];
  
  return (dhd_outscale_t) col_argb_out->m_fields.m_col_rgb_scale_out_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Set alpha scaling. Selects a scalefactor to be applied to the active color units output alpha value. Possible*/
/* values range from 0.5 to 4. */
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_scale : One of the four possible scale factors (see dhd_outscale_t)*/
DHD_API_INLINE void dhd_set_alpha_scale(dhd_handle_t a_device, dhd_outscale_t a_scale) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_out_0_t *col_argb_out = (dhd_reg_col_argb_out_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OUT_0 - (11*64) ];  
  
  col_argb_out->m_fields.m_col_alpha_scale_out_0 = a_scale;
  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OUT_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get alpha scaling. Retrieve currently active alpha scale factor (see dhd_set_alpha_scale for details). */
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/**/
/* Returns:*/
/*   One of the four possible scale factors (see dhd_outscale_t)*/
/**/
/* See Also:*/
/*  dhd_set_alpha_scale*/
DHD_API_INLINE dhd_outscale_t dhd_get_alpha_scale(dhd_handle_t a_device) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_out_0_t *col_argb_out = (dhd_reg_col_argb_out_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OUT_0 - (11*64) ];  
  
  return (dhd_outscale_t) col_argb_out->m_fields.m_col_alpha_scale_out_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Enable color clamping. Chooses if the rgb output should be clamped to 0..255 at the active color units */
/* output.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_enable : Boolean flag. 0 to disable color clamping.*/
DHD_API_INLINE void dhd_set_color_clamp(dhd_handle_t a_device, dhd_bool_t a_enable) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_out_0_t *col_argb_out = (dhd_reg_col_argb_out_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OUT_0 - (11*64) ];  
  
  col_argb_out->m_fields.m_col_rgb_clamp_out_0 = a_enable; 
  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OUT_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get color clamping state. Retrieve if color clamping is currently active (see dhd_set_color_clamp for details). */
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/**/
/* Returns:*/
/*  DHD_TRUE if rgb output clamping is active, DHD_FALSE otherwise.*/
/**/
/* See Also:*/
/*  dhd_set_color_clamp*/
DHD_API_INLINE dhd_bool_t dhd_get_color_clamp(dhd_handle_t a_device) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_out_0_t *col_argb_out = (dhd_reg_col_argb_out_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OUT_0 - (11*64) ];  
  
  return (dhd_bool_t) col_argb_out->m_fields.m_col_rgb_clamp_out_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Enable alpha clamping. Chooses if the alpha output should be clamped to 0..255 at the active color units */
/* output.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_enable : Boolean flag. 0 to disable alpha clamping.*/
DHD_API_INLINE void dhd_set_alpha_clamp(dhd_handle_t a_device, dhd_bool_t a_enable) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_out_0_t *col_argb_out = (dhd_reg_col_argb_out_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OUT_0 - (11*64) ];
  
  col_argb_out->m_fields.m_col_alpha_clamp_out_0 = a_enable;
  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OUT_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get alpha clamping state. Retrieve if color clamping is currently active (see dhd_set_alpha_clamp for details). */
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/**/
/* Returns:*/
/*  DHD_TRUE if alpha output clamping is active, DHD_FALSE otherwise.*/
/**/
/* See Also:*/
/*  dhd_set_alpha_clamp*/
DHD_API_INLINE dhd_bool_t dhd_get_alpha_clamp(dhd_handle_t a_device) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_out_0_t *col_argb_out = (dhd_reg_col_argb_out_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OUT_0 - (11*64) ];  
  
  return (dhd_bool_t) col_argb_out->m_fields.m_col_alpha_clamp_out_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select color operand 1a. Each color unit has 5 operands (1a,1b,2a,2b and 3). For each operand a source, a*/
/* channel swizzeling and a modifier can be selected.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_source : Operand source register, texture or constant (see dhd_col_input_t)*/
/*  a_channels : Any combination of A,R,G and B for each of the three output channels (see dhd_channel_select_t)*/
/*  a_modifier : Negation, one minus x and scaling by two of the input (see dhd_col_modify_t)*/
/**/
/* See Also:*/
/*  dhd_set_color_op1b, dhd_set_color_op2a, dhd_set_color_op2b, dhd_set_color_op3*/
DHD_API_INLINE void dhd_set_color_op1a(dhd_handle_t a_device, dhd_col_input_t a_source, dhd_channel_select_t a_channels, dhd_col_modify_t a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1a_0_t *col_argb_op1a = (dhd_reg_col_argb_op1a_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1A_0 - (11*64) ];

  col_argb_op1a->m_fields.m_col_rgb_op1a_input_select_0  = a_source;
  col_argb_op1a->m_fields.m_col_red_op1a_argb_select_0   = (a_channels >> 4) & 3;    
  col_argb_op1a->m_fields.m_col_green_op1a_argb_select_0 = (a_channels >> 2) & 3; 
  col_argb_op1a->m_fields.m_col_blue_op1a_argb_select_0  = (a_channels     ) & 3; 
  col_argb_op1a->m_fields.m_col_rgb_op1a_invert_mode_0   = a_modifier & 3;
  col_argb_op1a->m_fields.m_col_rgb_op1a_mul2_0          = (a_modifier >> 2) & 1;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP1A_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get color operand 1a. Retrieve the current parameters for color operand 1a (see dhd_set_color_op1a*/
/* for details). Passing a null pointer indicates that a specific information is not requested. */
/* The most important information (channel source) is returned directly.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_channels : Pointer to dhd_channel_select_t that will receive the output channel selector (can be null)*/
/*  a_modifier : Pointer to dhd_col_modify_t that will receive the output channel modifier (can be null)*/
/**/
/* Returns:*/
/*  Operand source register, texture or constant (see dhd_col_input_t)*/
/* */
/* See Also:*/
/*  dhd_set_color_op1a*/
DHD_API_INLINE dhd_col_input_t dhd_get_color_op1a(dhd_handle_t a_device, dhd_channel_select_t *a_channels, dhd_col_modify_t *a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1a_0_t *col_argb_op1a = (dhd_reg_col_argb_op1a_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1A_0 - (11*64) ];
  
  if (a_channels != 0) {
    dhd_uint32_t chan32 =  col_argb_op1a->m_fields.m_col_blue_op1a_argb_select_0 | 
                          (col_argb_op1a->m_fields.m_col_green_op1a_argb_select_0 << 2) |
                          (col_argb_op1a->m_fields.m_col_red_op1a_argb_select_0 << 4);
    *a_channels = (dhd_channel_select_t) chan32;
  }
  if (a_modifier != 0) {
    dhd_uint32_t mod32 =  col_argb_op1a->m_fields.m_col_rgb_op1a_invert_mode_0 |                     
                          (col_argb_op1a->m_fields.m_col_rgb_op1a_mul2_0 << 2);
    *a_modifier = (dhd_col_modify_t) mod32;
  }

  return (dhd_col_input_t) col_argb_op1a->m_fields.m_col_rgb_op1a_input_select_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select alpha operand 1a. Each color unit has 5 operands (1a,1b,2a,2b and 3). For each operand a source, a*/
/* channel and a modifier can be selected.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_source : Operand source register, texture or constant (see dhd_col_input_t)*/
/*  a_channel : Any of the four input channels A,R,G and B (see dhd_channel_t)*/
/*  a_modifier : Negation, one minus x and scaling by two of the input (see dhd_col_modify_t)*/
/**/
/* See Also:*/
/*  dhd_set_alpha_op1b, dhd_set_alpha_op2a, dhd_set_alpha_op2b, dhd_set_alpha_op3*/
DHD_API_INLINE void dhd_set_alpha_op1a(dhd_handle_t a_device, dhd_col_input_t a_source, dhd_channel_t a_channel, dhd_col_modify_t a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1a_0_t *col_argb_op1a = (dhd_reg_col_argb_op1a_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1A_0 - (11*64) ];

  col_argb_op1a->m_fields.m_col_alpha_op1a_input_select_0 = a_source;
  col_argb_op1a->m_fields.m_col_alpha_op1a_argb_select_0  = a_channel;
  col_argb_op1a->m_fields.m_col_alpha_op1a_invert_mode_0  = a_modifier & 3;
  col_argb_op1a->m_fields.m_col_alpha_op1a_mul2_0         = (a_modifier >> 2) & 1;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP1A_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get alpha operand 1a. Retrieve the current parameters for color operand 1a (see dhd_set_alpha_op1a*/
/* for details). Passing a null pointer indicates that a specific information is not requested. */
/* The most important information (channel source) is returned directly.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_channel : Pointer to dhd_channel_t that will receive the output channel (can be null)*/
/*  a_modifier : Pointer to dhd_col_modify_t that will receive the output channel modifier (can be null)*/
/**/
/* Returns:*/
/*  Operand source register, texture or constant (see dhd_col_input_t)*/
/* */
/* See Also:*/
/*  dhd_set_alpha_op1a*/
DHD_API_INLINE dhd_col_input_t dhd_get_alpha_op1a(dhd_handle_t a_device, dhd_channel_t *a_channel, dhd_col_modify_t *a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1a_0_t *col_argb_op1a = (dhd_reg_col_argb_op1a_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1A_0 - (11*64) ];
  
  if (a_channel != 0) {    
    *a_channel = (dhd_channel_t) col_argb_op1a->m_fields.m_col_alpha_op1a_argb_select_0;
  }
  if (a_modifier != 0) {
    dhd_uint32_t mod32 =  col_argb_op1a->m_fields.m_col_alpha_op1a_invert_mode_0 |                     
                          (col_argb_op1a->m_fields.m_col_alpha_op1a_mul2_0 << 2);
    *a_modifier = (dhd_col_modify_t) mod32;
  }

  return (dhd_col_input_t) col_argb_op1a->m_fields.m_col_alpha_op1a_input_select_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select color operand 1b. Each color unit has 5 operands (1a,1b,2a,2b and 3). For each operand a source, a*/
/* channel swizzeling and a modifier can be selected.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_source : Operand source register, texture or constant (see dhd_col_input_t)*/
/*  a_channels : Any combination of A,R,G and B for each of the three output channels (see dhd_channel_select_t)*/
/*  a_modifier : Negation, one minus x and scaling by two of the input (see dhd_col_modify_t)*/
/**/
/* See Also:*/
/*  dhd_set_color_op1a, dhd_set_color_op2a, dhd_set_color_op2b, dhd_set_color_op3*/
DHD_API_INLINE void dhd_set_color_op1b(dhd_handle_t a_device, dhd_col_input_t a_source, dhd_channel_select_t a_channels, dhd_col_modify_t a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1b_0_t *col_argb_op1b = (dhd_reg_col_argb_op1b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1B_0 - (11*64) ];

  col_argb_op1b->m_fields.m_col_rgb_op1b_input_select_0  = a_source;
  col_argb_op1b->m_fields.m_col_red_op1b_argb_select_0   = (a_channels >> 4) & 3; 
  col_argb_op1b->m_fields.m_col_green_op1b_argb_select_0 = (a_channels >> 2) & 3; 
  col_argb_op1b->m_fields.m_col_blue_op1b_argb_select_0  = (a_channels     ) & 3; 
  col_argb_op1b->m_fields.m_col_rgb_op1b_invert_mode_0   = a_modifier & 3;
  col_argb_op1b->m_fields.m_col_rgb_op1b_mul2_0          = (a_modifier >> 2) & 1;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP1B_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get color operand 1b. Retrieve the current parameters for color operand 1b (see dhd_set_color_op1b*/
/* for details). Passing a null pointer indicates that a specific information is not requested. */
/* The most important information (channel source) is returned directly.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_channels : Pointer to dhd_channel_select_t that will receive the output channel selector (can be null)*/
/*  a_modifier : Pointer to dhd_col_modify_t that will receive the output channel modifier (can be null)*/
/**/
/* Returns:*/
/*  Operand source register, texture or constant (see dhd_col_input_t)*/
/* */
/* See Also:*/
/*  dhd_set_color_op1b*/
DHD_API_INLINE dhd_col_input_t dhd_get_color_op1b(dhd_handle_t a_device, dhd_channel_select_t *a_channels, dhd_col_modify_t *a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1b_0_t *col_argb_op1b = (dhd_reg_col_argb_op1b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1B_0 - (11*64) ];
  
  if (a_channels != 0) {
    dhd_uint32_t chan32 =  col_argb_op1b->m_fields.m_col_blue_op1b_argb_select_0 | 
                          (col_argb_op1b->m_fields.m_col_green_op1b_argb_select_0 << 2) |
                          (col_argb_op1b->m_fields.m_col_red_op1b_argb_select_0 << 4);
    *a_channels = (dhd_channel_select_t) chan32;
  }
  if (a_modifier != 0) {
    dhd_uint32_t mod32 =  col_argb_op1b->m_fields.m_col_rgb_op1b_invert_mode_0 |                     
                          (col_argb_op1b->m_fields.m_col_rgb_op1b_mul2_0 << 2);
    *a_modifier = (dhd_col_modify_t) mod32;
  }

  return (dhd_col_input_t) col_argb_op1b->m_fields.m_col_rgb_op1b_input_select_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select alpha operand 1b. Each color unit has 5 operands (1a,1b,2a,2b and 3). For each operand a source, a*/
/* channel and a modifier can be selected.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_source : Operand source register, texture or constant (see dhd_col_input_t)*/
/*  a_channel : Any of the four input channels A,R,G and B (see dhd_channel_t)*/
/*  a_modifier : Negation, one minus x and scaling by two of the input (see dhd_col_modify_t)*/
/**/
/* See Also:*/
/*  dhd_set_alpha_op1a, dhd_set_alpha_op2a, dhd_set_alpha_op2b, dhd_set_alpha_op3*/
DHD_API_INLINE void dhd_set_alpha_op1b(dhd_handle_t a_device, dhd_col_input_t a_source, dhd_channel_t a_channel, dhd_col_modify_t a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1b_0_t *col_argb_op1b = (dhd_reg_col_argb_op1b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1B_0 - (11*64) ];

  col_argb_op1b->m_fields.m_col_alpha_op1b_input_select_0 = a_source;
  col_argb_op1b->m_fields.m_col_alpha_op1b_argb_select_0  = a_channel;
  col_argb_op1b->m_fields.m_col_alpha_op1b_invert_mode_0  = a_modifier & 3;
  col_argb_op1b->m_fields.m_col_alpha_op1b_mul2_0         = (a_modifier >> 2) & 1;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP1B_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get alpha operand 1b. Retrieve the current parameters for color operand 1b (see dhd_set_alpha_op1b*/
/* for details). Passing a null pointer indicates that a specific information is not requested. */
/* The most important information (channel source) is returned directly.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_channel : Pointer to dhd_channel_t that will receive the output channel (can be null)*/
/*  a_modifier : Pointer to dhd_col_modify_t that will receive the output channel modifier (can be null)*/
/**/
/* Returns:*/
/*  Operand source register, texture or constant (see dhd_col_input_t)*/
/* */
/* See Also:*/
/*  dhd_set_alpha_op1b*/
DHD_API_INLINE dhd_col_input_t dhd_get_alpha_op1b(dhd_handle_t a_device, dhd_channel_t *a_channel, dhd_col_modify_t *a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1b_0_t *col_argb_op1b = (dhd_reg_col_argb_op1b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1B_0 - (11*64) ];
  
  if (a_channel != 0) {    
    *a_channel = (dhd_channel_t) col_argb_op1b->m_fields.m_col_alpha_op1b_argb_select_0;
  }
  if (a_modifier != 0) {
    dhd_uint32_t mod32 =  col_argb_op1b->m_fields.m_col_alpha_op1b_invert_mode_0 |                     
                          (col_argb_op1b->m_fields.m_col_alpha_op1b_mul2_0 << 2);
    *a_modifier = (dhd_col_modify_t) mod32;
  }

  return (dhd_col_input_t) col_argb_op1b->m_fields.m_col_alpha_op1b_input_select_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select color operand 2a. Each color unit has 5 operands (1a,1b,2a,2b and 3). For each operand a source, a*/
/* channel swizzeling and a modifier can be selected.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_source : Operand source register, texture or constant (see dhd_col_input_t)*/
/*  a_channels : Any combination of A,R,G and B for each of the three output channels (see dhd_channel_select_t)*/
/*  a_modifier : Negation, one minus x and scaling by two of the input (see dhd_col_modify_t)*/
/**/
/* See Also:*/
/*  dhd_set_color_op1a, dhd_set_color_op1b, dhd_set_color_op2b, dhd_set_color_op3*/
DHD_API_INLINE void dhd_set_color_op2a(dhd_handle_t a_device, dhd_col_input_t a_source, dhd_channel_select_t a_channels, dhd_col_modify_t a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op2a_0_t *col_argb_op2a = (dhd_reg_col_argb_op2a_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2A_0 - (11*64) ];

  col_argb_op2a->m_fields.m_col_rgb_op2a_input_select_0  = a_source;
  col_argb_op2a->m_fields.m_col_red_op2a_argb_select_0   = (a_channels >> 4) & 3; 
  col_argb_op2a->m_fields.m_col_green_op2a_argb_select_0 = (a_channels >> 2) & 3; 
  col_argb_op2a->m_fields.m_col_blue_op2a_argb_select_0  = (a_channels     ) & 3; 
  col_argb_op2a->m_fields.m_col_rgb_op2a_invert_mode_0   = a_modifier & 3;
  col_argb_op2a->m_fields.m_col_rgb_op2a_mul2_0          = (a_modifier >> 2) & 1;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP2A_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get color operand 2a. Retrieve the current parameters for color operand 2a (see dhd_set_color_op2a*/
/* for details). Passing a null pointer indicates that a specific information is not requested. */
/* The most important information (channel source) is returned directly.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_channels : Pointer to dhd_channel_select_t that will receive the output channel selector (can be null)*/
/*  a_modifier : Pointer to dhd_col_modify_t that will receive the output channel modifier (can be null)*/
/**/
/* Returns:*/
/*  Operand source register, texture or constant (see dhd_col_input_t)*/
/* */
/* See Also:*/
/*  dhd_set_color_op2a*/
DHD_API_INLINE dhd_col_input_t dhd_get_color_op2a(dhd_handle_t a_device, dhd_channel_select_t *a_channels, dhd_col_modify_t *a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op2a_0_t *col_argb_op2a = (dhd_reg_col_argb_op2a_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2A_0 - (11*64) ];
  
  if (a_channels != 0) {
    dhd_uint32_t chan32 =  col_argb_op2a->m_fields.m_col_blue_op2a_argb_select_0 | 
                          (col_argb_op2a->m_fields.m_col_green_op2a_argb_select_0 << 2) |
                          (col_argb_op2a->m_fields.m_col_red_op2a_argb_select_0 << 4);
    *a_channels = (dhd_channel_select_t) chan32;
  }
  if (a_modifier != 0) {
    dhd_uint32_t mod32 =  col_argb_op2a->m_fields.m_col_rgb_op2a_invert_mode_0 |                     
                          (col_argb_op2a->m_fields.m_col_rgb_op2a_mul2_0 << 2);
    *a_modifier = (dhd_col_modify_t) mod32;
  }

  return (dhd_col_input_t) col_argb_op2a->m_fields.m_col_rgb_op2a_input_select_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select alpha operand 2a. Each color unit has 5 operands (1a,1b,2a,2b and 3). For each operand a source, a*/
/* channel and a modifier can be selected.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_source : Operand source register, texture or constant (see dhd_col_input_t)*/
/*  a_channel : Any of the four input channels A,R,G and B (see dhd_channel_t)*/
/*  a_modifier : Negation, one minus x and scaling by two of the input (see dhd_col_modify_t)*/
/**/
/* See Also:*/
/*  dhd_set_alpha_op1a, dhd_set_alpha_op1b, dhd_set_alpha_op2b, dhd_set_alpha_op3*/
DHD_API_INLINE void dhd_set_alpha_op2a(dhd_handle_t a_device, dhd_col_input_t a_source, dhd_channel_t a_channel, dhd_col_modify_t a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op2a_0_t *col_argb_op2a = (dhd_reg_col_argb_op2a_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2A_0 - (11*64) ];

  col_argb_op2a->m_fields.m_col_alpha_op2a_input_select_0 = a_source;
  col_argb_op2a->m_fields.m_col_alpha_op2a_argb_select_0  = a_channel;
  col_argb_op2a->m_fields.m_col_alpha_op2a_invert_mode_0  = a_modifier & 3;
  col_argb_op2a->m_fields.m_col_alpha_op2a_mul2_0         = (a_modifier >> 2) & 1;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP2A_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get alpha operand 2a. Retrieve the current parameters for color operand 2a (see dhd_set_alpha_op2a*/
/* for details). Passing a null pointer indicates that a specific information is not requested. */
/* The most important information (channel source) is returned directly.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_channel : Pointer to dhd_channel_t that will receive the output channel (can be null)*/
/*  a_modifier : Pointer to dhd_col_modify_t that will receive the output channel modifier (can be null)*/
/**/
/* Returns:*/
/*  Operand source register, texture or constant (see dhd_col_input_t)*/
/* */
/* See Also:*/
/*  dhd_set_alpha_op2a*/
DHD_API_INLINE dhd_col_input_t dhd_get_alpha_op2a(dhd_handle_t a_device, dhd_channel_t *a_channel, dhd_col_modify_t *a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op2a_0_t *col_argb_op2a = (dhd_reg_col_argb_op2a_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2A_0 - (11*64) ];
  
  if (a_channel != 0) {    
    *a_channel = (dhd_channel_t) col_argb_op2a->m_fields.m_col_alpha_op2a_argb_select_0;
  }
  if (a_modifier != 0) {
    dhd_uint32_t mod32 =  col_argb_op2a->m_fields.m_col_alpha_op2a_invert_mode_0 |                     
                          (col_argb_op2a->m_fields.m_col_alpha_op2a_mul2_0 << 2);
    *a_modifier = (dhd_col_modify_t) mod32;
  }

  return (dhd_col_input_t) col_argb_op2a->m_fields.m_col_alpha_op2a_input_select_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select color operand 2b. Each color unit has 5 operands (1a,1b,2a,2b and 3). For each operand a source, a*/
/* channel swizzeling and a modifier can be selected.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_source : Operand source register, texture or constant (see dhd_col_input_t)*/
/*  a_channels : Any combination of A,R,G and B for each of the three output channels (see dhd_channel_select_t)*/
/*  a_modifier : Negation, one minus x and scaling by two of the input (see dhd_col_modify_t)*/
/**/
/* See Also:*/
/*  dhd_set_color_op1a, dhd_set_color_op1b, dhd_set_color_op2a, dhd_set_color_op3*/
DHD_API_INLINE void dhd_set_color_op2b(dhd_handle_t a_device, dhd_col_input_t a_source, dhd_channel_select_t a_channels, dhd_col_modify_t a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op2b_0_t *col_argb_op2b = (dhd_reg_col_argb_op2b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2B_0 - (11*64) ];

  col_argb_op2b->m_fields.m_col_rgb_op2b_input_select_0  = a_source;
  col_argb_op2b->m_fields.m_col_red_op2b_argb_select_0   = (a_channels >> 4) & 3; 
  col_argb_op2b->m_fields.m_col_green_op2b_argb_select_0 = (a_channels >> 2) & 3; 
  col_argb_op2b->m_fields.m_col_blue_op2b_argb_select_0  = (a_channels     ) & 3; 
  col_argb_op2b->m_fields.m_col_rgb_op2b_invert_mode_0   = a_modifier & 3;
  col_argb_op2b->m_fields.m_col_rgb_op2b_mul2_0          = (a_modifier >> 2) & 1;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP2B_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get color operand 2b. Retrieve the current parameters for color operand 2b (see dhd_set_color_op2b*/
/* for details). Passing a null pointer indicates that a specific information is not requested. */
/* The most important information (channel source) is returned directly.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_channels : Pointer to dhd_channel_select_t that will receive the output channel selector (can be null)*/
/*  a_modifier : Pointer to dhd_col_modify_t that will receive the output channel modifier (can be null)*/
/**/
/* Returns:*/
/*  Operand source register, texture or constant (see dhd_col_input_t)*/
/* */
/* See Also:*/
/*  dhd_set_color_op2b*/
DHD_API_INLINE dhd_col_input_t dhd_get_color_op2b(dhd_handle_t a_device, dhd_channel_select_t *a_channels, dhd_col_modify_t *a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op2b_0_t *col_argb_op2b = (dhd_reg_col_argb_op2b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2B_0 - (11*64) ];
  
  if (a_channels != 0) {
    dhd_uint32_t chan32 =  col_argb_op2b->m_fields.m_col_blue_op2b_argb_select_0 | 
                          (col_argb_op2b->m_fields.m_col_green_op2b_argb_select_0 << 2) |
                          (col_argb_op2b->m_fields.m_col_red_op2b_argb_select_0 << 4);
    *a_channels = (dhd_channel_select_t) chan32;
  }
  if (a_modifier != 0) {
    dhd_uint32_t mod32 =  col_argb_op2b->m_fields.m_col_rgb_op2b_invert_mode_0 |                     
                          (col_argb_op2b->m_fields.m_col_rgb_op2b_mul2_0 << 2);
    *a_modifier = (dhd_col_modify_t) mod32;
  }

  return (dhd_col_input_t) col_argb_op2b->m_fields.m_col_rgb_op2b_input_select_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select alpha operand 2b. Each color unit has 5 operands (1a,1b,2a,2b and 3). For each operand a source, a*/
/* channel and a modifier can be selected.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_source : Operand source register, texture or constant (see dhd_col_input_t)*/
/*  a_channel : Any of the four input channels A,R,G and B (see dhd_channel_t)*/
/*  a_modifier : Negation, one minus x and scaling by two of the input (see dhd_col_modify_t)*/
/**/
/* See Also:*/
/*  dhd_set_alpha_op1a, dhd_set_alpha_op1b, dhd_set_alpha_op2a, dhd_set_alpha_op3*/
DHD_API_INLINE void dhd_set_alpha_op2b(dhd_handle_t a_device, dhd_col_input_t a_source, dhd_channel_t a_channel, dhd_col_modify_t a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op2b_0_t *col_argb_op2b = (dhd_reg_col_argb_op2b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2B_0 - (11*64) ];

  col_argb_op2b->m_fields.m_col_alpha_op2b_input_select_0 = a_source;
  col_argb_op2b->m_fields.m_col_alpha_op2b_argb_select_0  = a_channel;
  col_argb_op2b->m_fields.m_col_alpha_op2b_invert_mode_0  = a_modifier & 3;
  col_argb_op2b->m_fields.m_col_alpha_op2b_mul2_0         = (a_modifier >> 2) & 1;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP2B_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get alpha operand 2b. Retrieve the current parameters for color operand 2b (see dhd_set_alpha_op2b*/
/* for details). Passing a null pointer indicates that a specific information is not requested. */
/* The most important information (channel source) is returned directly.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_channel : Pointer to dhd_channel_t that will receive the output channel (can be null)*/
/*  a_modifier : Pointer to dhd_col_modify_t that will receive the output channel modifier (can be null)*/
/**/
/* Returns:*/
/*  Operand source register, texture or constant (see dhd_col_input_t)*/
/* */
/* See Also:*/
/*  dhd_set_alpha_op2b*/
DHD_API_INLINE dhd_col_input_t dhd_get_alpha_op2b(dhd_handle_t a_device, dhd_channel_t *a_channel, dhd_col_modify_t *a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op2b_0_t *col_argb_op2b = (dhd_reg_col_argb_op2b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2B_0 - (11*64) ];
  
  if (a_channel != 0) {    
    *a_channel = (dhd_channel_t) col_argb_op2b->m_fields.m_col_alpha_op2b_argb_select_0;
  }
  if (a_modifier != 0) {
    dhd_uint32_t mod32 =  col_argb_op2b->m_fields.m_col_alpha_op2b_invert_mode_0 |                     
                          (col_argb_op2b->m_fields.m_col_alpha_op2b_mul2_0 << 2);
    *a_modifier = (dhd_col_modify_t) mod32;
  }

  return (dhd_col_input_t) col_argb_op2b->m_fields.m_col_alpha_op2b_input_select_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select color operand 3. Each color unit has 5 operands (1a,1b,2a,2b and 3). For each operand a source, a*/
/* channel swizzeling and a modifier can be selected.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_source : Operand source register, texture or constant (see dhd_col_input_t)*/
/*  a_channels : Any combination of A,R,G and B for each of the three output channels (see dhd_channel_select_t)*/
/*  a_modifier : Negation, one minus x and scaling by two of the input (see dhd_col_modify_t)*/
/**/
/* See Also:*/
/*  dhd_set_color_op1a, dhd_set_color_op1b, dhd_set_color_op2a, dhd_set_color_op2b*/
DHD_API_INLINE void dhd_set_color_op3(dhd_handle_t a_device, dhd_col_input_t a_source, dhd_channel_select_t a_channels, dhd_col_modify_t a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op3_0_t *col_argb_op3 = (dhd_reg_col_argb_op3_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP3_0 - (11*64) ];

  col_argb_op3->m_fields.m_col_rgb_op3_input_select_0  = a_source;
  col_argb_op3->m_fields.m_col_red_op3_argb_select_0   = (a_channels >> 4) & 3; 
  col_argb_op3->m_fields.m_col_green_op3_argb_select_0 = (a_channels >> 2) & 3; 
  col_argb_op3->m_fields.m_col_blue_op3_argb_select_0  = (a_channels     ) & 3; 
  col_argb_op3->m_fields.m_col_rgb_op3_invert_mode_0   = a_modifier & 3;
  col_argb_op3->m_fields.m_col_rgb_op3_mul2_0          = (a_modifier >> 2) & 1;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP3_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get color operand 3. Retrieve the current parameters for color operand 3 (see dhd_set_color_op3*/
/* for details). Passing a null pointer indicates that a specific information is not requested. */
/* The most important information (channel source) is returned directly.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_channels : Pointer to dhd_channel_select_t that will receive the output channel selector (can be null)*/
/*  a_modifier : Pointer to dhd_col_modify_t that will receive the output channel modifier (can be null)*/
/**/
/* Returns:*/
/*  Operand source register, texture or constant (see dhd_col_input_t)*/
/* */
/* See Also:*/
/*  dhd_set_color_op3*/
DHD_API_INLINE dhd_col_input_t dhd_get_color_op3(dhd_handle_t a_device, dhd_channel_select_t *a_channels, dhd_col_modify_t *a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op3_0_t *col_argb_op3 = (dhd_reg_col_argb_op3_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP3_0 - (11*64) ];
  
  if (a_channels != 0) {
    dhd_uint32_t chan32 =  col_argb_op3->m_fields.m_col_blue_op3_argb_select_0 | 
                          (col_argb_op3->m_fields.m_col_green_op3_argb_select_0 << 2) |
                          (col_argb_op3->m_fields.m_col_red_op3_argb_select_0 << 4);
    *a_channels = (dhd_channel_select_t) chan32;
  }
  if (a_modifier != 0) {
    dhd_uint32_t mod32 =  col_argb_op3->m_fields.m_col_rgb_op3_invert_mode_0 |                     
                          (col_argb_op3->m_fields.m_col_rgb_op3_mul2_0 << 2);
    *a_modifier = (dhd_col_modify_t) mod32;
  }

  return (dhd_col_input_t) col_argb_op3->m_fields.m_col_rgb_op3_input_select_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select alpha operand 3. Each color unit has 5 operands (1a,1b,2a,2b and 3). For each operand a source, a*/
/* channel and a modifier can be selected.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_source : Operand source register, texture or constant (see dhd_col_input_t)*/
/*  a_channel : Any of the four input channels A,R,G and B (see dhd_channel_t)*/
/*  a_modifier : Negation, one minus x and scaling by two of the input (see dhd_col_modify_t)*/
/**/
/* See Also:*/
/*  dhd_set_alpha_op1b, dhd_set_alpha_op2a, dhd_set_alpha_op2b, dhd_set_alpha_op3*/
DHD_API_INLINE void dhd_set_alpha_op3(dhd_handle_t a_device, dhd_col_input_t a_source, dhd_channel_t a_channel, dhd_col_modify_t a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op3_0_t *col_argb_op3 = (dhd_reg_col_argb_op3_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP3_0 - (11*64) ];

  col_argb_op3->m_fields.m_col_alpha_op3_input_select_0 = a_source;
  col_argb_op3->m_fields.m_col_alpha_op3_argb_select_0  = a_channel;
  col_argb_op3->m_fields.m_col_alpha_op3_invert_mode_0  = a_modifier & 3;
  col_argb_op3->m_fields.m_col_alpha_op3_mul2_0         = (a_modifier >> 2) & 1;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP3_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get alpha operand 3. Retrieve the current parameters for color operand 3 (see dhd_set_alpha_op3*/
/* for details). Passing a null pointer indicates that a specific information is not requested. */
/* The most important information (channel source) is returned directly.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_channel : Pointer to dhd_channel_t that will receive the output channel (can be null)*/
/*  a_modifier : Pointer to dhd_col_modify_t that will receive the output channel modifier (can be null)*/
/**/
/* Returns:*/
/*  Operand source register, texture or constant (see dhd_col_input_t)*/
/* */
/* See Also:*/
/*  dhd_set_alpha_op3*/
DHD_API_INLINE dhd_col_input_t dhd_get_alpha_op3(dhd_handle_t a_device, dhd_channel_t *a_channel, dhd_col_modify_t *a_modifier) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op3_0_t *col_argb_op3 = (dhd_reg_col_argb_op3_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP3_0 - (11*64) ];
  
  if (a_channel != 0) {    
    *a_channel = (dhd_channel_t) col_argb_op3->m_fields.m_col_alpha_op3_argb_select_0;
  }
  if (a_modifier != 0) {
    dhd_uint32_t mod32 =  col_argb_op3->m_fields.m_col_alpha_op3_invert_mode_0 |                     
                          (col_argb_op3->m_fields.m_col_alpha_op3_mul2_0 << 2);
    *a_modifier = (dhd_col_modify_t) mod32;
  }

  return (dhd_col_input_t) col_argb_op3->m_fields.m_col_alpha_op3_input_select_0;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Set color addition input carry. Each add operation has an additional carry in bit that enables doing a '+1'.*/
/* This can be used to calculate a two's complement by using the operand source modifier (see dhd_col_modify_t)*/
/* to negate all bits and the carry in to add an additional one.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_carry1 : Carry in for the addition of operands 1a and 1b (can be 0 or 1)*/
/*  a_carry2 : Carry in for the addition of operands 2a and 2b (can be 0 or 1)*/
DHD_API_INLINE void dhd_set_color_carry(dhd_handle_t a_device, dhd_bool_t a_carry1, dhd_bool_t a_carry2) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1b_0_t *col_argb_op1b = (dhd_reg_col_argb_op1b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1B_0 - (11*64) ];
  dhd_reg_col_argb_op2b_0_t *col_argb_op2b = (dhd_reg_col_argb_op2b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2B_0 - (11*64) ];

  col_argb_op1b->m_fields.m_col_rgb_carry1_0 = a_carry1;
  col_argb_op2b->m_fields.m_col_rgb_carry2_0 = a_carry2;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP1B_0);
  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP2B_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get color addition input carry. Retrieve the currently active rgb carry bit configuration (see */
/* dhd_set_color_carry for details). Passing a null pointer indicates that a specific information is not */
/* requested. */
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_carry1 : Pointer to dhd_bool_t that will receive carry for addition of operands 1a and 1b (can be null)*/
/*  a_carry2 : Pointer to dhd_bool_t that will receive carry for addition of operands 2a and 2b (can be null)*/
/* */
/* See Also:*/
/*  dhd_set_color_carry*/
DHD_API_INLINE void dhd_get_color_carry(dhd_handle_t a_device, dhd_bool_t *a_carry1, dhd_bool_t *a_carry2) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1b_0_t *col_argb_op1b = (dhd_reg_col_argb_op1b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1B_0 - (11*64) ];
  dhd_reg_col_argb_op2b_0_t *col_argb_op2b = (dhd_reg_col_argb_op2b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2B_0 - (11*64) ];

  if (a_carry1 != 0) {
    *a_carry1 = col_argb_op1b->m_fields.m_col_rgb_carry1_0;
  }
  if (a_carry2 != 0) {
    *a_carry2 = col_argb_op2b->m_fields.m_col_rgb_carry2_0;
  }
}

/*----------------------------------------------------------------------------------------------------------*/
/* Set alpha addition input carry. Each add operation has an additional carry in bit that enables doing a '+1'.*/
/* This can be used to calculate a two's complement by using the operand source modifier (see dhd_col_modify_t)*/
/* to negate all bits and the carry in to add an additional one.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_carry1 : Carry in for the addition of operands 1a and 1b (can be 0 or 1)*/
/*  a_carry2 : Carry in for the addition of operands 2a and 2b (can be 0 or 1)*/
DHD_API_INLINE void dhd_set_alpha_carry(dhd_handle_t a_device, dhd_bool_t a_carry1, dhd_bool_t a_carry2) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1b_0_t *col_argb_op1b = (dhd_reg_col_argb_op1b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1B_0 - (11*64) ];
  dhd_reg_col_argb_op2b_0_t *col_argb_op2b = (dhd_reg_col_argb_op2b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2B_0 - (11*64) ];

  col_argb_op1b->m_fields.m_col_alpha_carry1_0 = a_carry1;
  col_argb_op2b->m_fields.m_col_alpha_carry2_0 = a_carry2;

  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP1B_0);
  dhd_internal_set_col_busy(device, DHD_REG_COL_ARGB_OP2B_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get alpha addition input carry. Retrieve the currently active rgb carry bit configuration (see */
/* dhd_set_alpha_carry for details). Passing a null pointer indicates that a specific information is not */
/* requested. */
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_carry1 : Pointer to dhd_bool_t that will receive carry for addition of operands 1a and 1b (can be null)*/
/*  a_carry2 : Pointer to dhd_bool_t that will receive carry for addition of operands 2a and 2b (can be null)*/
/* */
/* See Also:*/
/*  dhd_set_alpha_carry*/
DHD_API_INLINE void dhd_get_alpha_carry(dhd_handle_t a_device, dhd_bool_t *a_carry1, dhd_bool_t *a_carry2) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  dhd_reg_col_argb_op1b_0_t *col_argb_op1b = (dhd_reg_col_argb_op1b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1B_0 - (11*64) ];
  dhd_reg_col_argb_op2b_0_t *col_argb_op2b = (dhd_reg_col_argb_op2b_0_t *) &device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2B_0 - (11*64) ];

  if (a_carry1 != 0) {
    *a_carry1 = col_argb_op1b->m_fields.m_col_alpha_carry1_0;
  }
  if (a_carry2 != 0) {
    *a_carry2 = col_argb_op2b->m_fields.m_col_alpha_carry2_0;
  }
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select constant 0 as 1b. Internal function used to effectively disable the addition of operand 1a and 1b*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
DHD_INLINE void dhd_internal_set_zero_1b(dhd_device_data_t *a_device) {   /* PRQA S 3450 *//* $Misra: #INLINE_FUNCTION_NO_DECLARATION $*/
  dhd_reg_col_argb_op1b_0_t *col_argb_op1b = (dhd_reg_col_argb_op1b_0_t *) &a_device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1B_0 - (11*64) ];

  col_argb_op1b->m_fields.m_col_rgb_op1b_input_select_0  = E_DHD_INPUT_ZERO;  /* 1b = ZERO*/
  col_argb_op1b->m_fields.m_col_rgb_op1b_invert_mode_0   = E_DHD_INPUT_DIRECT;
  col_argb_op1b->m_fields.m_col_rgb_op1b_mul2_0          = 0;            
  dhd_internal_set_col_busy(a_device, DHD_REG_COL_ARGB_OP1B_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select constant 1 as 2a. Internal function used to prepare disabling the multiplication.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
DHD_INLINE void dhd_internal_set_one_2a(dhd_device_data_t *a_device) {    /* PRQA S 3450 *//* $Misra: #INLINE_FUNCTION_NO_DECLARATION $*/
  dhd_reg_col_argb_op2a_0_t *col_argb_op2a = (dhd_reg_col_argb_op2a_0_t *) &a_device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2A_0 - (11*64) ];

  col_argb_op2a->m_fields.m_col_rgb_op2a_input_select_0  = E_DHD_INPUT_ZERO;  /* 2a = ONE*/
  col_argb_op2a->m_fields.m_col_rgb_op2a_invert_mode_0   = E_DHD_INPUT_NEG;
  col_argb_op2a->m_fields.m_col_rgb_op2a_mul2_0          = 0;                
  dhd_internal_set_col_busy(a_device, DHD_REG_COL_ARGB_OP2A_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select constant 0 as 2b. Internal function used to effectively disable the addition of operand 2a and 2b*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
DHD_INLINE void dhd_internal_set_zero_2b(dhd_device_data_t *a_device) {   /* PRQA S 3450 *//* $Misra: #INLINE_FUNCTION_NO_DECLARATION $*/
  dhd_reg_col_argb_op2b_0_t *col_argb_op2b = (dhd_reg_col_argb_op2b_0_t *) &a_device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2B_0 - (11*64) ];

  col_argb_op2b->m_fields.m_col_rgb_op2b_input_select_0  = E_DHD_INPUT_ZERO;  /* 2b = ZERO*/
  col_argb_op2b->m_fields.m_col_rgb_op2b_invert_mode_0   = E_DHD_INPUT_DIRECT;
  col_argb_op2b->m_fields.m_col_rgb_op2b_mul2_0          = 0;              
  dhd_internal_set_col_busy(a_device, DHD_REG_COL_ARGB_OP2B_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select constant 0 as 3. Internal function used to effectively disable the addition of operand 3*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
DHD_INLINE void dhd_internal_set_zero_3(dhd_device_data_t *a_device) {    /* PRQA S 3450 *//* $Misra: #INLINE_FUNCTION_NO_DECLARATION $*/
  dhd_reg_col_argb_op3_0_t  *col_argb_op3  = (dhd_reg_col_argb_op3_0_t *)  &a_device->m_col_unit_base[ DHD_REG_COL_ARGB_OP3_0  - (11*64) ];

  col_argb_op3->m_fields.m_col_rgb_op3_input_select_0  = E_DHD_INPUT_ZERO;  /* 2b = ZERO*/
  col_argb_op3->m_fields.m_col_rgb_op3_invert_mode_0   = E_DHD_INPUT_DIRECT;
  col_argb_op3->m_fields.m_col_rgb_op3_mul2_0          = 0;              
  dhd_internal_set_col_busy(a_device, DHD_REG_COL_ARGB_OP3_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select constant 0 as alpha 1b. Internal function used to effectively disable the addition of operand 1a and 1b*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
DHD_INLINE void dhd_internal_set_alpha_zero_1b(dhd_device_data_t *a_device) {   /* PRQA S 3450 *//* $Misra: #INLINE_FUNCTION_NO_DECLARATION $*/
  dhd_reg_col_argb_op1b_0_t *col_argb_op1b = (dhd_reg_col_argb_op1b_0_t *) &a_device->m_col_unit_base[ DHD_REG_COL_ARGB_OP1B_0 - (11*64) ];

  col_argb_op1b->m_fields.m_col_alpha_op1b_input_select_0  = E_DHD_INPUT_ZERO;  /* 1b = ZERO*/
  col_argb_op1b->m_fields.m_col_alpha_op1b_invert_mode_0   = E_DHD_INPUT_DIRECT;
  col_argb_op1b->m_fields.m_col_alpha_op1b_mul2_0          = 0;            
  dhd_internal_set_col_busy(a_device, DHD_REG_COL_ARGB_OP1B_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select constant 1 as alpha 2a. Internal function used to prepare disabling the multiplication.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
DHD_INLINE void dhd_internal_set_alpha_one_2a(dhd_device_data_t *a_device) {    /* PRQA S 3450 *//* $Misra: #INLINE_FUNCTION_NO_DECLARATION $*/
  dhd_reg_col_argb_op2a_0_t *col_argb_op2a = (dhd_reg_col_argb_op2a_0_t *) &a_device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2A_0 - (11*64) ];

  col_argb_op2a->m_fields.m_col_alpha_op2a_input_select_0  = E_DHD_INPUT_ZERO;  /* 2a = ONE*/
  col_argb_op2a->m_fields.m_col_alpha_op2a_invert_mode_0   = E_DHD_INPUT_NEG;
  col_argb_op2a->m_fields.m_col_alpha_op2a_mul2_0          = 0;                
  dhd_internal_set_col_busy(a_device, DHD_REG_COL_ARGB_OP2A_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select constant 0 as alpha 2b. Internal function used to effectively disable the addition of operand 2a and 2b*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
DHD_INLINE void dhd_internal_set_alpha_zero_2b(dhd_device_data_t *a_device) {   /* PRQA S 3450 *//* $Misra: #INLINE_FUNCTION_NO_DECLARATION $*/
  dhd_reg_col_argb_op2b_0_t *col_argb_op2b = (dhd_reg_col_argb_op2b_0_t *) &a_device->m_col_unit_base[ DHD_REG_COL_ARGB_OP2B_0 - (11*64) ];

  col_argb_op2b->m_fields.m_col_alpha_op2b_input_select_0  = E_DHD_INPUT_ZERO;  /* 2b = ZERO*/
  col_argb_op2b->m_fields.m_col_alpha_op2b_invert_mode_0   = E_DHD_INPUT_DIRECT;
  col_argb_op2b->m_fields.m_col_alpha_op2b_mul2_0          = 0;              
  dhd_internal_set_col_busy(a_device, DHD_REG_COL_ARGB_OP2B_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Select constant 0 as alpha 3. Internal function used to effectively disable the addition of operand 3*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
DHD_INLINE void dhd_internal_set_alpha_zero_3(dhd_device_data_t *a_device) {    /* PRQA S 3450 *//* $Misra: #INLINE_FUNCTION_NO_DECLARATION $*/
  dhd_reg_col_argb_op3_0_t  *col_argb_op3  = (dhd_reg_col_argb_op3_0_t *)  &a_device->m_col_unit_base[ DHD_REG_COL_ARGB_OP3_0  - (11*64) ];

  col_argb_op3->m_fields.m_col_alpha_op3_input_select_0  = E_DHD_INPUT_ZERO;  /* 2b = ZERO*/
  col_argb_op3->m_fields.m_col_alpha_op3_invert_mode_0   = E_DHD_INPUT_DIRECT;
  col_argb_op3->m_fields.m_col_alpha_op3_mul2_0          = 0;              
  dhd_internal_set_col_busy(a_device, DHD_REG_COL_ARGB_OP3_0);
}

/*----------------------------------------------------------------------------------------------------------*/
/* Setup color operands for a specific operation. Often the full flexibility of the color unit is not needed.*/
/* In simpler cases setting all 5 operands is unnecessarily verbose. This function can be used to quickly*/
/* set some operands to 0 or 1 in order to simplify the performed operation.*/
/**/
/* Note that dhd_set_color_operation will simply set all operands not required by the requested operation to*/
/* their identity values. If any of these operands is changed after dhd_set_color_operation the color unit */
/* will perform a different operation.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_operation : Select the formula to evaluate (see dhd_color_op_t)*/
DHD_API_INLINE void dhd_set_color_operation(dhd_handle_t a_device, dhd_color_op_t a_operation) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  
  /* no need to set channel select as zero input is equal for all channels*/
  switch (a_operation) {
    case E_DHD_OP_COPY:                           /* output = 1a*/
      dhd_internal_set_zero_1b( device );
      dhd_internal_set_one_2a( device );
      dhd_internal_set_zero_2b( device );
      dhd_internal_set_zero_3( device );
      break;      
    case E_DHD_OP_MUL:                            /* output = 1a * 2a*/
      dhd_internal_set_zero_1b( device );      
      dhd_internal_set_zero_2b( device );
      dhd_internal_set_zero_3( device );
      break;    
    case E_DHD_OP_ADD:                            /* output = 1a + 1b*/
      dhd_internal_set_one_2a( device );
      dhd_internal_set_zero_2b( device );
      dhd_internal_set_zero_3( device );
      break;
    case E_DHD_OP_MULADD:                         /* output = 1a * 2a + 3*/
      dhd_internal_set_zero_1b( device );      
      dhd_internal_set_zero_2b( device );      
      break;
    case E_DHD_OP_LERP:                           /* output = (1a + 1b) * 2a + 3*/
      dhd_internal_set_zero_2b( device );      
      break;
    case E_DHD_OP_PROD:                           /* output = (1a + 1b) * (2a + 2b)*/
      dhd_internal_set_zero_3( device );
      break;
    case E_DHD_OP_FULL:                           /* output = (1a + 1b) * (2a + 2b) + 3*/
      break;  
    default:
      break;
  }
}

/*----------------------------------------------------------------------------------------------------------*/
/* Setup alpha operands for a specific operation. Often the full flexibility of the color unit is not needed.*/
/* In simpler cases setting all 5 operands is unnecessarily verbose. This function can be used to quickly*/
/* set some operands to 0 or 1 in order to simplify the performed operation.*/
/**/
/* Note that dhd_set_alpha_operation will simply set all operands not required by the requested operation to*/
/* their identity values. If any of these operands is changed after dhd_set_alpha_operation the color unit */
/* will perform a different operation.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_operation : Select the formula to evaluate (see dhd_color_op_t)*/
DHD_API_INLINE void dhd_set_alpha_operation(dhd_handle_t a_device, dhd_color_op_t a_operation) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  
  /* no need to set channel select as zero input is equal for all channels*/
  switch (a_operation) {
    case E_DHD_OP_COPY:                           /* output = 1a*/
      dhd_internal_set_alpha_zero_1b( device );
      dhd_internal_set_alpha_one_2a( device );
      dhd_internal_set_alpha_zero_2b( device );
      dhd_internal_set_alpha_zero_3( device );
      break;      
    case E_DHD_OP_MUL:                            /* output = 1a * 2a*/
      dhd_internal_set_alpha_zero_1b( device );      
      dhd_internal_set_alpha_zero_2b( device );
      dhd_internal_set_alpha_zero_3( device );
      break;    
    case E_DHD_OP_ADD:                            /* output = 1a + 1b*/
      dhd_internal_set_alpha_one_2a( device );
      dhd_internal_set_alpha_zero_2b( device );
      dhd_internal_set_alpha_zero_3( device );
      break;
    case E_DHD_OP_MULADD:                         /* output = 1a * 2a + 3*/
      dhd_internal_set_alpha_zero_1b( device );      
      dhd_internal_set_alpha_zero_2b( device );      
      break;
    case E_DHD_OP_LERP:                           /* output = (1a + 1b) * 2a + 3*/
      dhd_internal_set_alpha_zero_2b( device );      
      break;
    case E_DHD_OP_PROD:                           /* output = (1a + 1b) * (2a + 2b)*/
      dhd_internal_set_alpha_zero_3( device );
      break;
    case E_DHD_OP_FULL:                           /* output = (1a + 1b) * (2a + 2b) + 3*/
      break;  
    default:
      break;
  }
}

/*----------------------------------------------------------------------------------------------------------*/
/* Enable color units. All color units from 0 to a_count-1 are enabled all others disabled.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/*  a_count : Number of color units to enable (0..8)*/
/**/
/* See Also:*/
/*  dhd_get_color_unit_count*/
DHD_API_INLINE void dhd_set_color_unit_enable(dhd_handle_t a_device, dhd_uint32_t a_count) {
  dhd_device_data_t *device = dhd_get_device( a_device );

  if (a_count > device->m_description.m_num_col_units) {
    a_count = device->m_description.m_num_col_units;
  }
  
  device->m_shadow_state.m_named.m_col_global.m_fields.m_col_num_passes = a_count;
  device->m_pending_mask.m_named.m_col_global = 1;
}

/*----------------------------------------------------------------------------------------------------------*/
/* Get number of enabled color units. Returns the value last set by dhd_set_color_unit_enable.*/
/**/
/* Parameters:*/
/*  a_device : Driver context handle (as returned by dhd_init)*/
/**/
/* Returns:*/
/*  Number of enabled color units.*/
/**/
/* See Also:*/
/*  dhd_set_color_unit_enable*/
DHD_API_INLINE dhd_uint32_t dhd_get_color_unit_enable(dhd_handle_t a_device) {
  dhd_device_data_t *device = dhd_get_device( a_device );
  
  return device->m_shadow_state.m_named.m_col_global.m_fields.m_col_num_passes;
}

/*----------------------------------------------------------------------------------------------------------*/

#endif /*DAVEHD_DRIVER_COL_INL_H_INCLUDED*/
