#include "Data_AirPressure.h"
#include "Gauges.h"
#include "GaugesInterface.h"
#include "CAN_Communication_Matrix.h"
#include "RTE_ADC.h"

#include "Common_Interface.h"

//断码显示
//-------------------------------------------
//344 - 3.1.2.FRONT/REAR空气计显示
//342 - 3.3.  FRONT／REAR气压指针
//-------------------------------------------

#define     PRI_ID      0
#define     SEC_ID      1

#define     TableLenMax 6u

#define     EnTable     0  // 0 - 计算  1 - 查表

#define     VALID_TIME  100 // 2S

static struct
{
    uint8_t SumCnt ;
    uint32_t Voltage ;
    uint16_t Result ;
} AirPressureFilter[2u] ;

static struct
{
    uint16_t VoltageTable[TableLenMax] ;
    uint16_t kPaTable[TableLenMax] ;
    uint16_t slopeTable[TableLenMax - 1 ] ;// 100倍的
    uint16_t ResultkPa[2u] ; // 100倍的
    uint16_t AvrVoltage[2u] ;
    uint8_t Valid[2u] ;
    uint8_t CurSeg[2u] ;
    uint8_t ValidTimer[2u]  ;
    uint8_t InvalidTimer[2u]  ;
} AirPressureVariable ;

static void Calc_AirPressure_Table(uint16_t Voltage, uint16_t * kPa);
static uint16_t Get_AirPressure_AdC_Filter(uint8_t ID);
static uint8_t CalAirPressure_CurSeg(uint8_t ID, uint8_t Valid, uint16_t Voltage);
/*-------------------------------------------------------------------------
 * Function Name  : AirPressure_KL30_Init
 * Description    :
 * Input          : None
 * Output         : None
 * Return         : None
 * onther         : Table固化 斜率同时固化
 --------------------------------------------------------------------------*/
void AirPressure_KL30_Init(void)
{
    uint8_t i = 0 ;
    uint32_t u32slope = 0 ;
    AirPressureVariable.VoltageTable[ 0 ] = 500 ;
    AirPressureVariable.kPaTable[ 0 ] = 0 ;


    AirPressureVariable.VoltageTable[ 1 ] = 1750    ;
    AirPressureVariable.kPaTable[ 1 ] = 500 ;

    AirPressureVariable.VoltageTable[ 2 ] = 1960    ;
    AirPressureVariable.kPaTable[ 2 ] = 583 ;

    AirPressureVariable.VoltageTable[ 3 ] = 2000    ;
    AirPressureVariable.kPaTable[ 3 ] = 600 ;

    AirPressureVariable.VoltageTable[ 4 ] = 3000    ;
    AirPressureVariable.kPaTable[ 4 ] = 1000 ;

    AirPressureVariable.VoltageTable[ 5 ] = 4250    ;
    AirPressureVariable.kPaTable[ 5 ] = 1500 ;

    for (i = 0; i < (TableLenMax - 1); i ++)
    {
        /*100倍 斜率*/
        u32slope = AirPressureVariable.kPaTable[ i + 1] - AirPressureVariable.kPaTable[ i ];
        u32slope *= 100 ;
        u32slope /= (AirPressureVariable.VoltageTable[ i + 1 ] - AirPressureVariable.VoltageTable[ i ]);
        AirPressureVariable.slopeTable[i] = u32slope ;
    }

    for (i = 0; i < 2; i ++)
    {
        AirPressureFilter[ i ].SumCnt = 0 ;
        AirPressureFilter[ i ].Voltage = 0 ;
        AirPressureFilter[ i ].Result = 0 ;

        AirPressureVariable.ResultkPa[ i ] = 0 ;
        AirPressureVariable.Valid[ i ] = 0 ;
        AirPressureVariable.ValidTimer[ i ] = 0 ;
        AirPressureVariable.InvalidTimer[ i ] = 0 ;
    }



}
void AirPressure_KL15_ON_Init(void)
{

}
void AirPressure_KL15_OFF_Init(void)
{

}
void AirPressure_Wakeup_Init(void)
{

}
void AirPressure_Sleep_Init(void)
{

}
/*-------------------------------------------------------------------------
 * Function Name  : AirPressure_Processing_Service
 * Description    : 主函数周期调用
 * Input          : None
 * Output         : None
 * Return         : None
 * onther         : None
 --------------------------------------------------------------------------*/
void AirPressure_Processing_Service(void)
{
    uint8_t i = 0 ;

    AirPressure_AdcData_Filter();

    for (i = 0; i < 2; i ++)
    {
        AirPressureVariable.AvrVoltage[ i ] = Get_AirPressure_AdC_Filter(i) ;
        if ((AirPressureVariable.AvrVoltage[ i ] < 500) || (AirPressureVariable.AvrVoltage[ i ] > 4250) )
        {
            if (AirPressureVariable.InvalidTimer[ i ] < VALID_TIME)
            {
                //2s
                AirPressureVariable.InvalidTimer[ i ] ++ ;
            }
            else
            {
                AirPressureVariable.ValidTimer[ i ] = 0 ;
                AirPressureVariable.Valid[i] = 0 ;
            }

        }
        else
        {
            if (AirPressureVariable.ValidTimer[ i ] < VALID_TIME)
            {
                AirPressureVariable.ValidTimer[ i ] ++ ;
            }
            else
            {
                AirPressureVariable.InvalidTimer[ i ] = 0 ;
                AirPressureVariable.Valid[i] = 1 ;
            }

        }

        Calc_AirPressure_Table(AirPressureVariable.AvrVoltage[i], (uint16_t *)&AirPressureVariable.ResultkPa[i]);

        AirPressureVariable.CurSeg[ i ] = CalAirPressure_CurSeg(i, AirPressureVariable.Valid[i], AirPressureVariable.AvrVoltage[ i ]);
    }
}
/*-------------------------------------------------------------------------
 * Function Name  : Calc_AirPressure_Table
 * Description    : 电压 转为 压力
 * Input          : None
 * Output         : None
 * Return         : None
 * onther         : None
 --------------------------------------------------------------------------*/
static void Calc_AirPressure_Table(uint16_t Voltage, uint16_t * kPa)
{
    uint8_t i = 0 ;
    uint8_t Index = 0 ;
    uint32_t u32Result = 0 ;

#if EnTable
    if (Voltage <= AirPressureVariable.VoltageTable[ 0 ])
    {
        u32Result = AirPressureVariable.kPaTable[ 0 ] ;
    }
    else if (Voltage >= AirPressureVariable.VoltageTable[ TableLenMax - 1 ])
    {
        u32Result = AirPressureVariable.kPaTable[ TableLenMax - 1 ] ;
    }
    else
    {
        for (i = 0; i < TableLenMax; i ++ )
        {
            if (Voltage < AirPressureVariable.VoltageTable[ i ])
            {
                Index = i ;
                break ;
            }
        }
        if ((Index < 1) || (Index > (TableLenMax - 1)) )
        {
            //err
            while (1)
            {
                ;
            }
        }

        //y=kx + b
        Voltage -= AirPressureVariable.VoltageTable[Index - 1] ;
        u32Result = AirPressureVariable.slopeTable[Index - 1] * Voltage  ;
        u32Result += 50 ;
        u32Result /= 100 ;
        u32Result += AirPressureVariable.kPaTable[ Index - 1 ]  ;

    }
#else
    if (Voltage <= AirPressureVariable.VoltageTable[ 0 ])
    {
        u32Result = AirPressureVariable.kPaTable[ 0 ] ;
    }
    else if (Voltage >= AirPressureVariable.VoltageTable[ TableLenMax - 1 ])
    {
        u32Result = AirPressureVariable.kPaTable[ TableLenMax - 1 ] ;
    }
    else
    {
        //指示刻度 = (输入电压– 0.5) ／ 0.0025
        if (Voltage >= 500)
        {
            Voltage -= 500 ;
        }

        u32Result = Voltage ;
        u32Result *= 10 ;
        u32Result /= 25 ;
    }
#endif

    *kPa = (uint16_t)u32Result ;
}
/*-------------------------------------------------------------------------
 * Function Name  : AirPressure_AdcData_Filter
 * Description    : 主负气压输入数据滤波 - 主函数周期调用
 * Input          : None
 * Output         : None
 * Return         : None
 * onther         : None
 --------------------------------------------------------------------------*/
void AirPressure_AdcData_Filter(void)
{
    uint8_t u8PriValid = 0 ;
    uint8_t u8SecValid = 0 ;
    uint16_t PriVoltage = 0 ;
    uint16_t SecVoltage = 0 ;

    if ((K_Line_Set.K_Line_LID42 == 0x00) || (K_Line_Set.K_Line_LID42 == 0x02))
    {
        u8PriValid = RTE_Read_PRIMARY_AIR_Valid();
        u8SecValid = RTE_Read_SECONDARY_AIR_Valid();
    }
    else if ((K_Line_Set.K_Line_LID42 == 0x01) || (K_Line_Set.K_Line_LID42 == 0x03))
    {
        u8PriValid = RTE_Read_SECONDARY_AIR_Valid();
        u8SecValid = RTE_Read_PRIMARY_AIR_Valid();
    }

    if (u8PriValid)
    {
        if (AirPressureFilter[PRI_ID].SumCnt < 10)
        {
            AirPressureFilter[PRI_ID].SumCnt ++ ;
            PriVoltage = RTE_Read_PRIMARY_AIR_Voltage() ;
            AirPressureFilter[PRI_ID].Voltage += PriVoltage ;
            PriVoltage = (uint16_t)(AirPressureFilter[PRI_ID].Voltage / AirPressureFilter[PRI_ID].SumCnt) ;
        }
        else
        {
            AirPressureFilter[PRI_ID].SumCnt = 0 ;
            AirPressureFilter[PRI_ID].Voltage = 0 ;
            PriVoltage = RTE_Read_PRIMARY_AIR_Voltage() ;
            AirPressureFilter[PRI_ID].SumCnt ++ ;
            AirPressureFilter[PRI_ID].Voltage += PriVoltage ;
            PriVoltage = (uint16_t)(AirPressureFilter[PRI_ID].Voltage / AirPressureFilter[PRI_ID].SumCnt) ;
        }

        AirPressureFilter[PRI_ID].Result = PriVoltage ;
    }
    else
    {
        AirPressureFilter[PRI_ID].SumCnt = 0 ;
        AirPressureFilter[PRI_ID].Voltage = 0 ;
        PriVoltage = 0 ;
    }

    if (u8SecValid)
    {
        if (AirPressureFilter[SEC_ID].SumCnt < 10)
        {
            AirPressureFilter[SEC_ID].SumCnt ++ ;
            SecVoltage = RTE_Read_SECONDARY_AIR_Voltage() ;
            AirPressureFilter[SEC_ID].Voltage += SecVoltage ;
            SecVoltage = (uint16_t)(AirPressureFilter[SEC_ID].Voltage / AirPressureFilter[SEC_ID].SumCnt) ;
        }
        else
        {
            AirPressureFilter[SEC_ID].SumCnt = 0 ;
            AirPressureFilter[SEC_ID].Voltage = 0 ;
            SecVoltage = RTE_Read_SECONDARY_AIR_Voltage() ;
            AirPressureFilter[SEC_ID].SumCnt ++ ;
            AirPressureFilter[SEC_ID].Voltage += SecVoltage ;
            SecVoltage = (uint16_t)(AirPressureFilter[SEC_ID].Voltage / AirPressureFilter[SEC_ID].SumCnt) ;
        }
        AirPressureFilter[SEC_ID].Result = SecVoltage ;
    }
    else
    {
        AirPressureFilter[SEC_ID].SumCnt = 0 ;
        AirPressureFilter[SEC_ID].Voltage = 0 ;
        SecVoltage = 0 ;
    }
}
/*-------------------------------------------------------------------------
 * Function Name  : CalAirPressure_CurSeg
 * Description    : 计算主负气压显示段
 * Input          : None
 * Output         : None
 * Return         : None
 * onther         : None
 --------------------------------------------------------------------------*/
static uint8_t CalAirPressure_CurSeg(uint8_t ID, uint8_t Valid, uint16_t Voltage)
{
    uint8_t u8Result = 0 ;
    if (Valid)
    {

    }
    else
    {

    }

    if (ID == PRI_ID)
    {
        SetGaugesPara(PRIGasGauges, Voltage);
        u8Result = GetGaugesCurrentPos(PRIGasGauges) ;
    }
    else
    {
        SetGaugesPara(SECGasGauges, Voltage);
        u8Result = GetGaugesCurrentPos(SECGasGauges) ;
    }

    return u8Result ;
}
/*-------------------------------------------------------------------------
 * Function Name  : Get_AirPressure_AdC_Filter
 * Description    : 主负气压滤波后结果
 * Input          : None
 * Output         : None
 * Return         : None
 * onther         : None
 --------------------------------------------------------------------------*/
static uint16_t Get_AirPressure_AdC_Filter(uint8_t ID)
{
    uint16_t u16Result ;
    u16Result = AirPressureFilter[ID].Result ;
    return u16Result ;
}
/*-------------------------------------------------------------------------
 * Function Name  : AirPressure_Get_kPa
 * Description    : 压力值 kPa
 * Input          : None
 * Output         : None
 * Return         : None
 * onther         : None
 --------------------------------------------------------------------------*/
uint16_t AirPressure_Get_kPa(uint8_t ID)
{
    uint16_t u16Result = 0u;
    u16Result = AirPressureVariable.ResultkPa[ID];
    return u16Result;
}
/*-------------------------------------------------------------------------
 * Function Name  : AirPressure_Get_Valid
 * Description    :
 * Input          : None
 * Output         : None
 * Return         : None
 * onther         : None
 --------------------------------------------------------------------------*/
uint8_t AirPressure_Get_Valid(uint8_t ID)
{
    uint8_t u8Result = 0u;
    u8Result = AirPressureVariable.Valid[ID];
    return u8Result ;
}
/*-------------------------------------------------------------------------
 * Function Name  : AirPressure_Get_CurSeg
 * Description    : 当前显示段
 * Input          : None
 * Output         : None
 * Return         : None
 * onther         : None
 --------------------------------------------------------------------------*/
uint8_t AirPressure_Get_CurSeg(uint8_t ID)
{
    return AirPressureVariable.CurSeg[ ID ] ;
}



