#include "CAN_CH0_CAN_Communication_Matrix.h"
#include "Common_Interface.h"
#include "Data_VSpeed.h"
#include "RTE_CAN.h"
#include "DisplaySch.h"
#include "Configuration.h"
#include "FreIn.h"
#include "EOL_Para.h"

static uint32_t VSpeedCAFD;    // 给燃油库下电后的车速实际值
static uint32_t DataVSpeedActual;
static uint32_t DataVSpeedActual_ignoreError;
static uint32_t DataVSpeedDisp;
static uint32_t DataVSpeedHysteresis;
static uint8_t  DataVSpeedValid;


static DataVSpeedSamplerStruct DataVSpeedSampler;
static DataVSPeedDampingStruct DataVSPeedDamping;

#define DATA_APPR_DIR_INC 0x01
#define DATA_APPR_DIR_DEC 0x02

uint32_t DANA_Vspeed_Process(uint32_t Vspeed);

/******************************************************************************
函数名：Data_Vehicle_Speed_KL30_Init
功  能：车速数据KL30初始化
参  数：无
返回值：无
******************************************************************************
注  意：该函数KL30初始化被调用一次
******************************************************************************/
void Data_Vehicle_Speed_KL30_Init(void)
{
    DataVSpeedActual             = 0;
    DataVSpeedActual_ignoreError = 0;
    DataVSpeedDisp               = 0;
    DataVSpeedHysteresis         = 0;
    DataVSpeedValid              = 0;
    DataVSpeedSampler.Cnt        = 0;
    DataVSPeedDamping.Speed      = 0;
    DataVSPeedDamping.Delta      = 0;
    DataVSPeedDamping.Dir        = DATA_APPR_DIR_INC;
}

/******************************************************************************
函数名：Data_Vehicle_Speed_Wakeup_Init
功  能：车速数据唤醒初始化
参  数：无
返回值：无
******************************************************************************
注  意：该函数唤醒初始化被调用一次
******************************************************************************/
void Data_Vehicle_Speed_Wakeup_Init(void)
{
    DataVSpeedDisp          = 0;
    DataVSpeedHysteresis    = 0;
    DataVSPeedDamping.Speed = 0;
    DataVSPeedDamping.Delta = 0;
    DataVSPeedDamping.Dir   = DATA_APPR_DIR_INC;
}

/******************************************************************************
函数名：Data_Vehicle_Speed_Processing_Service
功  能：车速数据处理函数
参  数：无
返回值：无
******************************************************************************
注  意：该函数必须每20ms被调用一次
******************************************************************************/
void Data_Vehicle_Speed_Processing_Service(void)
{
    uint8_t  i         = 0U;
    uint16_t Delta     = 0U;
    uint32_t VSpeed    = 0U;
    uint32_t VSpeedCal = 0U;
    uint32_t VSpeed32  = 0U;
    uint16_t ErrorValue = VSpeed_Error_Value(); //误差系数
    uint32_t KValue    = K_Value(); //K值
    uint32_t EffectiveRange = 0;
    static uint8_t time = 0;
    static uint32_t MeanValue = 0;
    static uint32_t MeanValueA = 0;

    if(KValue == 0) //k值最小为1
    {
        KValue = 1;
    }

    //变速箱类型为ZF或者徐工自制变速箱
    if((Val_EOL_0300_Transmission_Type_Configuration() == 1) || (Val_EOL_0300_Transmission_Type_Configuration() == 3))
    {
        // 第1步：获取实际车速值及车速有效性
        if ( Common_Get_IG_Sts( ) == COMMON_POWER_ON )
        {
            if((Get_CAN_CH0_ID_0CF00203_Sig_TorqueConverterOutputShaftSpeed() > 0xFAFF) || 
              (CAN_MSG_Status(&CAN_CH0_CanMsgOp, CAN_CH0_ID_Rx_TC2_Msg_Count) == CAN_SIG_LOST))
            {
                DataVSpeedValid              = 0;
                DataVSpeedActual             = 0;
                DataVSpeedActual_ignoreError = 0;
                DataVSpeedSampler.Cnt        = 0;
            }
            else
            {
                if ( DataVSpeedValid == 0 )
                {
                    DataVSpeedValid = 1;
                    VSpeed = Get_CAN_CH0_ID_0CF00203_Sig_TorqueConverterOutputShaftSpeed( );
                    VSpeed *= 1000;
                    VSpeed /= 8;
                    VSpeed *= 36;
                    VSpeed /= KValue;
                    DataVSpeedActual_ignoreError = ( uint16_t )VSpeed;
                    if ( VSpeed > 1400 )
                    {
                        VSpeed = 1400;
                    }
                    DataVSpeedActual = ( uint16_t )VSpeed;
                }
                else
                {
                    DataVSpeedSampler.Buffer [ DataVSpeedSampler.Cnt ] = Get_CAN_CH0_ID_0CF00203_Sig_TorqueConverterOutputShaftSpeed( );
                    i                                                  = DataVSpeedSampler.Cnt;
                    while ( (i > 0) && (DataVSpeedSampler.Buffer [ i ] < DataVSpeedSampler.Buffer [ i - 1 ]) )
                    {
                        VSpeed                             = DataVSpeedSampler.Buffer [ i ];
                        DataVSpeedSampler.Buffer [ i ]     = DataVSpeedSampler.Buffer [ i - 1 ];
                        DataVSpeedSampler.Buffer [ i - 1 ] = ( uint16_t )VSpeed;
                        i--;
                    }

                    DataVSpeedSampler.Cnt++;
                    if ( DataVSpeedSampler.Cnt >= 3 )
                    {
                        DataVSpeedSampler.Cnt = 0;
                        VSpeed                = DataVSpeedSampler.Buffer [ 1 ];
                        if(VSpeed >= 0xFAFF)
                        {
                            VSpeed = 0;
                        }
                        VSpeed *= 1000;
                        VSpeed /= 8;

                        VSpeed *= 36;
                        VSpeed /= KValue;
                        DataVSpeedActual_ignoreError = ( uint16_t )VSpeed;
                        if ( VSpeed > 1400 )
                        {
                            VSpeed = 1400;
                        }
                        DataVSpeedActual = ( uint16_t )VSpeed;
                    }
                }
            }
        }
        else
        {
            if ( CAN_MSG_Status(&CAN_CH0_CanMsgOp, CAN_CH0_ID_Rx_TC2_Msg_Count) == CAN_SIG_LOST )    // 判掉线
            {
                VSpeedCAFD = 0;
            }
            else if ( (Get_CAN_CH0_ID_0CF00203_Sig_TorqueConverterOutputShaftSpeed( ) == 0xFFFF) )    // 车速异常
            {
                VSpeedCAFD = 0;
            }
            else
            {
                VSpeed32 = Get_CAN_CH0_ID_0CF00203_Sig_TorqueConverterOutputShaftSpeed( );
                if(VSpeed32 > 0xFAFF)
                {
                    VSpeed32 = 0;
                }
                VSpeed32 *= 1000;
                VSpeed32 /= 8;

                VSpeed32 *= 36;
                VSpeed32 /= KValue;
                DataVSpeedActual_ignoreError = ( uint16_t )VSpeed32;
                // 最大车速限制
                if ( VSpeed32 > 1400 )
                {
                    VSpeed32 = 1400;
                }

                VSpeedCAFD = ( uint16_t )VSpeed32;
            }

            DataVSpeedValid                = 0;
            DataVSpeedActual               = 0;
            DataVSpeedActual_ignoreError   = 0;
            DataVSpeedSampler.Buffer [ 0 ] = 0;
            DataVSpeedSampler.Buffer [ 1 ] = 0;
            DataVSpeedSampler.Buffer [ 2 ] = 0;
            DataVSpeedSampler.Cnt          = 0;
        }

    }

    //变速箱类型为机械变速箱
    else if(Val_EOL_0300_Transmission_Type_Configuration() == 0)
    {

        // 第1步：获取实际车速值及车速有效性
        if ( Common_Get_IG_Sts( ) == COMMON_POWER_ON )
        {
            time++;
            MeanValue += Fre_In_Get_Channel_HZ(FRE_VEHICLE_0);
            if(time == 5)
            {
                MeanValue   /= 5;
                MeanValueA  = MeanValue;
                MeanValue   = 0;
                time = 0;
            }

            EffectiveRange = (2250 * KValue) / 3600;

            if((MeanValueA > EffectiveRange) || (Fre_In_Get_Channel_Valid(FRE_VEHICLE_0) == 0))
            {
                DataVSpeedValid              = 0;
                DataVSpeedActual             = 0;
                DataVSpeedActual_ignoreError = 0;
                DataVSpeedSampler.Cnt        = 0;
            }
            else
            {
                if ( DataVSpeedValid == 0 )
                {
                    DataVSpeedValid = 1;
                    VSpeed = Fre_In_Get_Channel_HZ(FRE_VEHICLE_0);


                    VSpeed *= 3600;
                    VSpeed /= KValue;
                    DataVSpeedActual_ignoreError = ( uint16_t )VSpeed;
                    if ( VSpeed > 1400 )
                    {
                        VSpeed = 1400;
                    }
                    DataVSpeedActual = ( uint16_t )VSpeed;
                }
                else
                {
                    DataVSpeedSampler.Buffer [ DataVSpeedSampler.Cnt ] = Fre_In_Get_Channel_HZ(FRE_VEHICLE_0);
                    i                                                  = DataVSpeedSampler.Cnt;
                    while ( (i > 0) && (DataVSpeedSampler.Buffer [ i ] < DataVSpeedSampler.Buffer [ i - 1 ]) )
                    {
                        VSpeed                             = DataVSpeedSampler.Buffer [ i ];
                        DataVSpeedSampler.Buffer [ i ]     = DataVSpeedSampler.Buffer [ i - 1 ];
                        DataVSpeedSampler.Buffer [ i - 1 ] = ( uint16_t )VSpeed;
                        i--;
                    }

                    DataVSpeedSampler.Cnt++;
                    if ( DataVSpeedSampler.Cnt >= 3 )
                    {
                        DataVSpeedSampler.Cnt = 0;
                        VSpeed                = DataVSpeedSampler.Buffer [ 1 ];
                        if(VSpeed > 8000)
                        {
                            VSpeed = 0;
                        }
                        VSpeed *= 3600;
                        VSpeed /= KValue;
                        DataVSpeedActual_ignoreError = ( uint16_t )VSpeed;
                        if ( VSpeed > 1400 )
                        {
                            VSpeed = 1400;
                        }
                        DataVSpeedActual = ( uint16_t )VSpeed;
                    }
                }
            }
        }
        else
        {
            if ( Fre_In_Get_Channel_Valid(FRE_VEHICLE_0) == 0 )    // 信号无效
            {
                VSpeedCAFD = 0;
            }
            else
            {
                VSpeed32 = Fre_In_Get_Channel_HZ(FRE_VEHICLE_0);
                if(VSpeed32 > 8000)
                {
                    VSpeed32 = 0;
                }
                VSpeed32 *= 3600;
                VSpeed32 /= KValue;
                DataVSpeedActual_ignoreError = ( uint16_t )VSpeed32;
                // 最大车速限制
                if ( VSpeed32 > 1400 )
                {
                    VSpeed32 = 1400;
                }

                VSpeedCAFD = ( uint16_t )VSpeed32;
            }

            DataVSpeedValid                = 0;
            DataVSpeedActual               = 0;
            DataVSpeedActual_ignoreError   = 0;
            DataVSpeedSampler.Buffer [ 0 ] = 0;
            DataVSpeedSampler.Buffer [ 1 ] = 0;
            DataVSpeedSampler.Buffer [ 2 ] = 0;
            DataVSpeedSampler.Cnt          = 0;
        }
        
    }

    //变速箱类型为DANA变速箱
    else
    {
        // 第1步：获取实际车速值及车速有效性
        if ( Common_Get_IG_Sts( ) == COMMON_POWER_ON )
        {
            if((Get_CAN_CH0_ID_0CFF3003_Sig_VehicleSpeed() > 250) || 
               (CAN_MSG_Status(&CAN_CH0_CanMsgOp, CAN_CH0_ID_Rx_DANA_Msg_Count) == CAN_SIG_LOST))
            {
                DataVSpeedValid              = 0;
                DataVSpeedActual             = 0;
                DataVSpeedActual_ignoreError = 0;
                DataVSpeedSampler.Cnt        = 0;
            }
            else
            {
                if ( DataVSpeedValid == 0 )
                {
                    DataVSpeedValid = 1;
                    VSpeed = Get_CAN_CH0_ID_0CFF3003_Sig_VehicleSpeed();
                    VSpeed = DANA_Vspeed_Process(VSpeed);
                    DataVSpeedActual_ignoreError = ( uint16_t )VSpeed;
                    if ( VSpeed > 1400 )
                    {
                        VSpeed = 1400;
                    }
                    DataVSpeedActual = ( uint16_t )VSpeed;
                }
                else
                {
                    DataVSpeedSampler.Buffer [ DataVSpeedSampler.Cnt ] = Get_CAN_CH0_ID_0CFF3003_Sig_VehicleSpeed( );
                    i                                                  = DataVSpeedSampler.Cnt;
                    while ( (i > 0) && (DataVSpeedSampler.Buffer [ i ] < DataVSpeedSampler.Buffer [ i - 1 ]) )
                    {
                        VSpeed                             = DataVSpeedSampler.Buffer [ i ];
                        DataVSpeedSampler.Buffer [ i ]     = DataVSpeedSampler.Buffer [ i - 1 ];
                        DataVSpeedSampler.Buffer [ i - 1 ] = ( uint16_t )VSpeed;
                        i--;
                    }

                    DataVSpeedSampler.Cnt++;
                    if ( DataVSpeedSampler.Cnt >= 3 )
                    {
                        DataVSpeedSampler.Cnt = 0;
                        VSpeed                = DataVSpeedSampler.Buffer [ 1 ];
                        if(VSpeed > 250)
                        {
                            VSpeed = 0;
                        }
                        VSpeed = DANA_Vspeed_Process(VSpeed);
                        DataVSpeedActual_ignoreError = ( uint16_t )VSpeed;
                        if ( VSpeed > 1400 )
                        {
                            VSpeed = 1400;
                        }
                        DataVSpeedActual = ( uint16_t )VSpeed;
                    }
                }
            }
        }
        else
        {
            if ( CAN_MSG_Status(&CAN_CH0_CanMsgOp, CAN_CH0_ID_Rx_DANA_Msg_Count) == CAN_SIG_LOST )    // 判掉线
            {
                VSpeedCAFD = 0;
            }
            else if (Get_CAN_CH0_ID_0CFF3003_Sig_VehicleSpeed( ) > 250)    // 车速异常
            {
                VSpeedCAFD = 0;
            }
            else
            {
                VSpeed32 = Get_CAN_CH0_ID_0CFF3003_Sig_VehicleSpeed();
                VSpeed32 = DANA_Vspeed_Process(VSpeed32);
                DataVSpeedActual_ignoreError = ( uint16_t )VSpeed32;
                // 最大车速限制
                if ( VSpeed32 > 1400 )
                {
                    VSpeed32 = 1400;
                }

                VSpeedCAFD = ( uint16_t )VSpeed32;
            }

            DataVSpeedValid                = 0;
            DataVSpeedActual               = 0;
            DataVSpeedActual_ignoreError   = 0;
            DataVSpeedSampler.Buffer [ 0 ] = 0;
            DataVSpeedSampler.Buffer [ 1 ] = 0;
            DataVSpeedSampler.Buffer [ 2 ] = 0;
            DataVSpeedSampler.Cnt          = 0;
        }

    }
    

    // 第2步：车速的阻尼处理
    VSpeed = DataVSpeedActual;
    if(Get_HMI_SelfCheck_Timer() > 3000)
    {    
        if ( VSpeed > DataVSPeedDamping.Speed )    // 实际值比当前显示值大时
        {
            Delta = ( uint16_t )VSpeed - DataVSPeedDamping.Speed;

            if ( DataVSPeedDamping.Dir == DATA_APPR_DIR_INC )    // 显示值正在向实际值增加,则维持当前方向调节增加速度
            {
                if ( DataVSPeedDamping.Delta < Delta )    // 当前增量小于显示值与实际值的差值,则提升增加的速度
                {
                    DataVSPeedDamping.Delta += DATA_VSPEED_INC_STEP;
                    if ( DataVSPeedDamping.Delta > Delta )
                    {
                        DataVSPeedDamping.Delta = Delta;
                    }
                }
                else    // 当前增量大于等于显示值与实际值的差值,保持增量与当前差值同步,增加的速度即逐步减小
                {
                    DataVSPeedDamping.Delta = Delta;
                }

                Delta = DataVSPeedDamping.Delta;
                Delta /= DATA_VSPEED_DAMPING_FACTOR;

                if ( Delta < DATA_VSPEED_APPR_SPEED_MIN )
                {
                    Delta = DATA_VSPEED_APPR_SPEED_MIN;
                }

                DataVSPeedDamping.Speed += Delta;
                if ( DataVSPeedDamping.Speed > VSpeed )
                {
                    DataVSPeedDamping.Speed = ( uint16_t )VSpeed;
                }
            }
            else    // 显示值正在减小,则尽快减速至速度为最小时更换方向
            {
                if ( DataVSPeedDamping.Delta > DATA_VSPEED_DEC_STEP )
                {
                    DataVSPeedDamping.Delta -= DATA_VSPEED_DEC_STEP;
                }
                else
                {
                    DataVSPeedDamping.Delta = 0;
                }

                Delta = DataVSPeedDamping.Delta;
                Delta /= DATA_VSPEED_DAMPING_FACTOR;

                if ( Delta < DATA_VSPEED_APPR_SPEED_MIN )    // 已减速至最小速度
                {
                    DataVSPeedDamping.Dir = DATA_APPR_DIR_INC;
                }    // 更换方向
                else
                {
                    if ( DataVSPeedDamping.Speed > Delta )
                    {
                        DataVSPeedDamping.Speed -= Delta;
                    }
                    else
                    {
                        DataVSPeedDamping.Speed = 0;
                    }
                }
            }
        }
        else if ( VSpeed < DataVSPeedDamping.Speed )    // 实际值比当前显示值小时
        {
            Delta = DataVSPeedDamping.Speed - ( uint16_t )VSpeed;

            if ( DataVSPeedDamping.Dir == DATA_APPR_DIR_INC )    // 显示值仍在增加,则尽快减速至速度为最小时更换方向
            {
                if ( DataVSPeedDamping.Delta > DATA_VSPEED_DEC_STEP )
                {
                    DataVSPeedDamping.Delta -= DATA_VSPEED_DEC_STEP;
                }
                else
                {
                    DataVSPeedDamping.Delta = 0;
                }

                Delta = DataVSPeedDamping.Delta;
                Delta /= DATA_VSPEED_DAMPING_FACTOR;

                if ( Delta < DATA_VSPEED_APPR_SPEED_MIN )    // 已减速至最小速度
                {
                    DataVSPeedDamping.Dir = DATA_APPR_DIR_DEC;
                }    // 更换方向
                else
                {
                    DataVSPeedDamping.Speed += Delta;
                    if ( DataVSPeedDamping.Speed > 1400 )
                    {
                        DataVSPeedDamping.Speed = 1400;
                    }
                }
            }
            else    // 显示值正在向实际值减小,则维持当前方向调节增加速度
            {
                if ( DataVSPeedDamping.Delta < Delta )    // 当前(负)增量小于显示值与实际值的差值,则提升减小的速度
                {
                    DataVSPeedDamping.Delta += DATA_VSPEED_INC_STEP;
                    if ( DataVSPeedDamping.Delta > Delta )
                    {
                        DataVSPeedDamping.Delta = Delta;
                    }
                }
                else    // 当前(负)增量大于等于显示值与实际值的差值,保持(负)增量与当前差值同步,减小的速度即逐步减小
                {
                    DataVSPeedDamping.Delta = Delta;
                }

                Delta = DataVSPeedDamping.Delta;
                Delta /= DATA_VSPEED_DAMPING_FACTOR;

                if ( Delta < DATA_VSPEED_APPR_SPEED_MIN )
                {
                    Delta = DATA_VSPEED_APPR_SPEED_MIN;
                }

                if ( DataVSPeedDamping.Speed < VSpeed + Delta )
                {
                    DataVSPeedDamping.Speed = ( uint16_t )VSpeed;
                }
                else
                {
                    DataVSPeedDamping.Speed -= Delta;
                }
            }
        }
        else    // 实际值与当前显示值相等时
        {
            Delta = DataVSPeedDamping.Delta;
            Delta /= DATA_VSPEED_DAMPING_FACTOR;

            if ( Delta > DATA_VSPEED_APPR_SPEED_MIN )    // 当前的速度不是最小,说明数值正在增加/减小中,则继续原过程
            {
                if ( DataVSPeedDamping.Delta > DATA_VSPEED_DEC_STEP )    // 显示值越过了实际值,必然要先减速至最小速度,再改变方向返回实际值
                {
                    DataVSPeedDamping.Delta -= DATA_VSPEED_DEC_STEP;
                }
                else
                {
                    DataVSPeedDamping.Delta = 0;
                }
                Delta = DataVSPeedDamping.Delta;
                Delta /= DATA_VSPEED_DAMPING_FACTOR;

                if ( DataVSPeedDamping.Dir == DATA_APPR_DIR_INC )    // 显示值当前是增加方向
                {
                    if ( Delta < DATA_VSPEED_APPR_SPEED_MIN )    // 已减速至最小速度
                    {
                        DataVSPeedDamping.Dir = DATA_APPR_DIR_DEC;
                    }    // 更换方向
                    else
                    {
                        DataVSPeedDamping.Speed += Delta;
                        if ( DataVSPeedDamping.Speed > 1400 )
                        {
                            DataVSPeedDamping.Speed = 1400;
                        }
                    }
                }
                else    // 显示值当前是减小方向
                {
                    if ( Delta < DATA_VSPEED_APPR_SPEED_MIN )         // 已减速至最小速度
                        DataVSPeedDamping.Dir = DATA_APPR_DIR_INC;    // 更换方向
                    else
                    {
                        if ( DataVSPeedDamping.Speed > Delta )
                        {
                            DataVSPeedDamping.Speed -= Delta;
                        }
                        else
                        {
                            DataVSPeedDamping.Speed = 0;
                        }
                    }
                }
            }
        }
    }
    else 
    {
        DataVSPeedDamping.Speed = 0;
        DataVSPeedDamping.Delta = 0;
        DataVSPeedDamping.Dir   = DATA_APPR_DIR_INC;
    }

    // 第3步：生成显示车速
    if ( (DataVSPeedDamping.Speed >= DataVSpeedHysteresis) || (DataVSPeedDamping.Speed < 10) )
    {
        DataVSpeedHysteresis = DataVSPeedDamping.Speed;
    }
    else
    {
        if ( DataVSpeedHysteresis - DataVSPeedDamping.Speed >= DATA_VSPEED_HYSTERESIS )
        {
            DataVSpeedHysteresis = DataVSPeedDamping.Speed;
        }
    }
    if(DataVSpeedValid == 0)
    {
        DataVSpeedDisp = 0xFFFF;
    }
    else
    {
        DataVSpeedDisp = ( uint32_t )DataVSpeedHysteresis;
        DataVSpeedDisp *= ErrorValue;
        DataVSpeedDisp /= 100;
        if ( DataVSpeedDisp > 1400 )
        {
            DataVSpeedDisp = 1400;
        }
    }
    
}

uint32_t DANA_Vspeed_Process(uint32_t Vspeed)
{
    Vspeed *= 10;
    if(Vspeed <= 1000)
    {
        Vspeed /= 10;
    }
    else if(Vspeed <= 2000)
    {
        Vspeed -= 1000;
        Vspeed /= 5;
        Vspeed += 100;
    }
    else
    {
        Vspeed -= 2000;
        Vspeed += 300;
    }

    return Vspeed;
}

/*车速有效位：有效=1，无效=0*/
uint8_t Get_VechileSpeedValid(void)
{
    return DataVSpeedValid;
}
/*真实的车速，精度10倍,*/
uint16_t Get_ActualVechileSpeed(void)
{
    return DataVSpeedActual;
}
/*显示车速，精度10倍，取整，舍去小数点*/
uint16_t Get_DispVechileSpeed(void)
{
    return DataVSpeedDisp;
}

uint16_t Get_VSpeedCAFD(void)
{
    return VSpeedCAFD;
}
uint16_t Get_DataVSpeedActual_ignoreError(void)
{
    return DataVSpeedActual_ignoreError;
}
