#include "CAN_CH0_CAN_Communication_Matrix.h"
#include "Common_Interface.h"
#include "Fuel_Level_Table.h"
#include "Analog_Signals.h"

static uint8_t                g_FuelLevelSegNum_Disp;
static uint8_t                g_FuelLevelValid;          // -0 无效 -1 有效
static DataFuelLevelCalcStruct g_DataFuelLevelTempCalc;

/*** 燃油液位上升表格 ***/
static const uint16_t DataFuelLevelTable_Up [ DATA_FuelLevel_TEMP_TOTAL_SEG ] =
    /*显示段：|    0     |    1     |    2     |    3     |    4     |*/
    {33, 57, 91, 124, 176};    // 上升时小于等于表值则熄灭相应段

/*** 燃油液位下降表格 ***/
static const uint16_t DataFuelLevelTable_Dn [ DATA_FuelLevel_TEMP_TOTAL_SEG ] =
    /*显示段：|    0     |    1     |    2     |    3     |    4     |*/
    {29, 53, 87, 120, 172};    // 下降时小于等于表值则熄灭相应段

void Data_FuelLevel_TempPara_KL30_Init(void)
{
    g_DataFuelLevelTempCalc.FastMode         = 0u;
    g_FuelLevelSegNum_Disp                   = 0u;
    g_FuelLevelValid                         = 1u;
    g_DataFuelLevelTempCalc.Last             = 0u;
    g_DataFuelLevelTempCalc.InvalidCnt       = 0u;
    g_DataFuelLevelTempCalc.DstSeg           = 0u;
    g_DataFuelLevelTempCalc.State            = DATA_FuelLevel_TEMP_STATE_DELAY;
    g_DataFuelLevelTempCalc.Wait             = DATA_FuelLevel_TEMP_WAIT_SEC * 10;
    g_DataFuelLevelTempCalc.Prescaler        = 0u;
    g_DataFuelLevelTempCalc.OutputEnable     = 0u;
    g_DataFuelLevelTempCalc.DispTimer        = 0u;
    g_DataFuelLevelTempCalc.EngineStartTimer = 0u;
    g_DataFuelLevelTempCalc.Backup           = 0u;
}

void Data_FuelLevel_TempPara_Wakeup_Init(void)
{
    g_FuelLevelSegNum_Disp                   = 0u;
    g_FuelLevelValid                         = 1u;
    g_DataFuelLevelTempCalc.Last             = 0u;
    g_DataFuelLevelTempCalc.InvalidCnt       = 0u;
    g_DataFuelLevelTempCalc.DstSeg           = 0u;
    g_DataFuelLevelTempCalc.State            = DATA_FuelLevel_TEMP_STATE_DELAY;
    g_DataFuelLevelTempCalc.Wait             = DATA_FuelLevel_TEMP_WAIT_SEC * 10;
    g_DataFuelLevelTempCalc.Prescaler        = 0u;
    g_DataFuelLevelTempCalc.OutputEnable     = 0u;
    g_DataFuelLevelTempCalc.DispTimer        = 0u;
    g_DataFuelLevelTempCalc.EngineStartTimer = 0u;
}

/******************************************************************************
函数名：Data_FuelLevel_Temp_Processing_Service
功  能：SOC数据处理函数
参  数：无
返回值：无
*******************************************************************************
注  意：该函数必须每20ms被调用一次
******************************************************************************/
void Data_FuelLevel_Temp_Processing_Service(void)
{
    uint16_t  Data_Fuel_Level    = 0u;
    uint8_t  Delta             = 0u;
    uint16_t DataFuelLevel_Temp = 0u;

    if ( Common_Get_IG_Sts( ) == COMMON_POWER_ON )
    {
    
       Data_Fuel_Level = ADC_Read_Signal(ADC_CH_FUEL_LEVEL) / 10;
       if ( Data_Fuel_Level > 300 )
       {
           g_FuelLevelValid = 0u;
           Data_Fuel_Level  = 0;
       }
       else
       {
           DataFuelLevel_Temp = Data_Fuel_Level;
           g_FuelLevelValid = 1u;
       }

        // 第2步：获取燃油液位格数
        if ( g_FuelLevelValid )
        {
            if ( DataFuelLevel_Temp > g_DataFuelLevelTempCalc.Last )    // 新燃油液位高于前次燃油液位,查上升表
            {
                while ( g_DataFuelLevelTempCalc.DstSeg < DATA_FuelLevel_TEMP_TOTAL_SEG )
                {
                    if ( DataFuelLevel_Temp < DataFuelLevelTable_Up [ g_DataFuelLevelTempCalc.DstSeg ] )
                        break;
                    g_DataFuelLevelTempCalc.DstSeg++;
                }
            }
            else if ( DataFuelLevel_Temp < g_DataFuelLevelTempCalc.Last )    // 新燃油液位低于前次燃油液位,查下降表
            {
                while ( g_DataFuelLevelTempCalc.DstSeg )
                {
                    if ( DataFuelLevel_Temp > DataFuelLevelTable_Dn [ g_DataFuelLevelTempCalc.DstSeg - 1 ] )
                        break;
                    g_DataFuelLevelTempCalc.DstSeg--;
                }
            }

            g_DataFuelLevelTempCalc.Last = DataFuelLevel_Temp;
        }

        // 第3步：状态转换,每100ms执行一次
        g_DataFuelLevelTempCalc.Prescaler++;
        if ( g_DataFuelLevelTempCalc.Prescaler >= 5 )
        {
            g_DataFuelLevelTempCalc.Prescaler = 0;

            switch ( g_DataFuelLevelTempCalc.State )
            {
                case DATA_FuelLevel_TEMP_STATE_DELAY:
                    if ( g_DataFuelLevelTempCalc.Wait )
                        g_DataFuelLevelTempCalc.Wait--;
                    else
                        g_DataFuelLevelTempCalc.State = DATA_FuelLevel_TEMP_STATE_INIT;

                    break;

                case DATA_FuelLevel_TEMP_STATE_INIT:
                    if ( g_FuelLevelValid )
                    {
                        if ( g_DataFuelLevelTempCalc.DstSeg )
                        {
                            g_DataFuelLevelTempCalc.OutputEnable = 1;
                            g_DataFuelLevelTempCalc.FastMode     = 1;
                            g_DataFuelLevelTempCalc.State        = DATA_FuelLevel_TEMP_STATE_FAST;
                        }
                    }
                    else
                    {
                        g_DataFuelLevelTempCalc.OutputEnable = 1;
                        g_DataFuelLevelTempCalc.FastMode     = 1;
                        g_DataFuelLevelTempCalc.Last         = 0;
                        g_DataFuelLevelTempCalc.DstSeg       = 0;

                        g_DataFuelLevelTempCalc.State = DATA_FuelLevel_TEMP_STATE_FAULT;
                    }

                    break;

                case DATA_FuelLevel_TEMP_STATE_FAST:
                    if ( g_FuelLevelValid )
                    {
                        if ( g_DataFuelLevelTempCalc.FastMode == 0 )
                            g_DataFuelLevelTempCalc.State = DATA_FuelLevel_TEMP_STATE_NORMAL;
                    }
                    else
                    {
                        g_DataFuelLevelTempCalc.OutputEnable = 1;
                        g_DataFuelLevelTempCalc.FastMode     = 1;
                        g_DataFuelLevelTempCalc.Last         = 0;
                        g_DataFuelLevelTempCalc.DstSeg       = 0;

                        g_DataFuelLevelTempCalc.State = DATA_FuelLevel_TEMP_STATE_FAULT;
                    }

                    break;

                case DATA_FuelLevel_TEMP_STATE_NORMAL:
                    if ( g_FuelLevelValid == 0 )
                    {
                        g_DataFuelLevelTempCalc.OutputEnable = 1;
                        g_DataFuelLevelTempCalc.FastMode     = 1;
                        g_DataFuelLevelTempCalc.Last         = 0;
                        g_DataFuelLevelTempCalc.DstSeg       = 0;

                        g_DataFuelLevelTempCalc.State = DATA_FuelLevel_TEMP_STATE_FAULT;
                    }

                    break;

                case DATA_FuelLevel_TEMP_STATE_FAULT:
                    if ( g_FuelLevelValid )
                        g_DataFuelLevelTempCalc.State = DATA_FuelLevel_TEMP_STATE_INIT;

                    break;

                default:
                    g_FuelLevelSegNum_Disp               = 0;
                    DataFuelLevel_Temp                   = 0;
                    g_FuelLevelValid                     = 0;
                    g_DataFuelLevelTempCalc.OutputEnable = 1;
                    g_DataFuelLevelTempCalc.FastMode     = 1;
                    g_DataFuelLevelTempCalc.Last         = 0;
                    g_DataFuelLevelTempCalc.DstSeg       = 0;

                    g_DataFuelLevelTempCalc.State = DATA_FuelLevel_TEMP_STATE_FAULT;

                    break;
            }
        }

        // 第4步：显示输出格数
        if ( g_DataFuelLevelTempCalc.OutputEnable )
        {
            if ( g_FuelLevelSegNum_Disp < g_DataFuelLevelTempCalc.DstSeg )    // 需要增加段数时
            {
                if ( g_DataFuelLevelTempCalc.FastMode )
                    Delta = ( uint8_t )(DATA_FuelLevel_TEMP_SEG_STEP_FAST / 20);
                else
                    Delta = ( uint8_t )(DATA_FuelLevel_TEMP_SEG_STEP_SLOW / 20);

                g_DataFuelLevelTempCalc.DispTimer++;
                if ( g_DataFuelLevelTempCalc.DispTimer >= Delta )
                {
                    g_DataFuelLevelTempCalc.DispTimer = 0;
                    g_FuelLevelSegNum_Disp++;
                }
            }
            else if ( g_FuelLevelSegNum_Disp > g_DataFuelLevelTempCalc.DstSeg )    // 需要减少段数时
            {
                if ( g_DataFuelLevelTempCalc.FastMode )
                    Delta = ( uint8_t )(DATA_FuelLevel_TEMP_SEG_STEP_FAST / 20);
                else
                    Delta = ( uint8_t )(DATA_FuelLevel_TEMP_SEG_STEP_SLOW / 20);

                g_DataFuelLevelTempCalc.DispTimer++;
                if ( g_DataFuelLevelTempCalc.DispTimer >= Delta )
                {
                    g_DataFuelLevelTempCalc.DispTimer = 0;
                    g_FuelLevelSegNum_Disp--;
                }
            }
            else
            {
                g_DataFuelLevelTempCalc.DispTimer = 0;
                g_DataFuelLevelTempCalc.FastMode  = 0;
            }
        }
    }
    else
    {
        g_DataFuelLevelTempCalc.EngineStartTimer = 0;
        g_DataFuelLevelTempCalc.Backup           = DataFuelLevel_Temp;
        // if ( DataFuelLevelFristPower == 1 )
        // {
        //     DataFuelLevelFristPower = 0;
        // }
    }
}

uint8_t Get_Fuel_Level_Valid(void)
{
    return g_FuelLevelValid;
}
uint8_t Get_Fuel_Level_Display(void)
{
    return g_FuelLevelSegNum_Disp;
}
