#include "UDS_Common.h"
#include "RTE_Memory.h"
#include "Analog_Signals.h"
#include "CAN_CFG.h"
#include "RTE_WDT.h"


uint8_t UDS_22_ButtonStatus;
ReadOnlayStructure *DID_Data;
uint8_t SessionType = DefaultSession;

uint16_t S3_ServerCnt     = S3_SERVER;
uint8_t  S3_ServerClear   = 0u;
uint8_t  S3_ServerTimeOut = 1u;

uint8_t UDSKL15Status  = 0u;
uint8_t UDSSleepStatus = 0u;

uint16_t UDSTimer10ms = 0u;

uint8_t S31DID0203Status = 0u;

uint8_t  isDTCOpen      = 1u;
uint8_t  isReceive      = 1u;
uint8_t  isEnable       = 0u;
uint16_t enableTimer    = 0u;
uint16_t enableMaxTimer = KL15DTCEnableTimer;

DiagMSGUnion           DiagMSG;
DiagFlagUnion          DiagFlag;
Ser27_FlowCtrlUnion    Ser27_FlowCtrl;
Ser27_FlowCtrlCntUnion Ser27_FlowCtrlCnt;
EOL0300ToDFlashInfoUnion   EOL0300ToDFlashInfo;
EOL0301ToDFlashInfoUnion   EOL0301ToDFlashInfo;
DTCToDFlashInfoUnion   DTCToDFlashInfo;

#pragma ghs startdata
#pragma ghs section rodata = ".ROM_BOOT_VER_STRINGS"
typedef struct
{
    uint8_t     ProjecName[16];// 项目名*
    uint8_t     PartNumber[16];// 零件号*
    uint8_t     PlatForm[16];  // 平台*
    struct 
    {
      uint8_t   type[16];       //标准*
      uint8_t   ch[16];         //通道*
      uint8_t   baudrate[16];   //速率*
      uint32_t  diagID[4];      //诊断ID*
    }DiagCanCfg[1];             //(诊断CAN)通道配置*
    uint8_t     build_number[16];//内部版本号
    
}buidInfoStruct;
const buidInfoStruct buidInfo[1] = 
{
        "XuGong",
        "XCMG-Loader-B04",
        "RH850F1KM16843",        
        {"CAN_EXT", //"CAN_EXT" "CANFD"
         "CAN0_CH4",//"CAN0_CH4"
         "25000",  //"25000" "125000" "5000002000000"
         {DIAG_ID_Tx, DIAG_ID_Rx_PHY, DIAG_ID_Rx_FUN,0x00}}, 
        {"BV1.01"}, 
};
#pragma ghs section rodata = default
#pragma ghs enddata

/**
  *设置会话类型
  *@param status 会话类型
  */
void setSessionType(uint8_t status)
{
    if ( (status == DefaultSession) || (status == ProgrammingSession) || (status == ExtendedDiagnosticSession) )
    {
        SessionType = status;
    }
    else
    {
        SessionType = DefaultSession;
    }
}

/**
  *获取会话类型
  *@return 当前会话类型
  */
uint8_t getSessionType(void)
{
    return SessionType;
}

/**
  *设置诊断信息
  *@param Identifier 收到的诊断报文ID
  *@param dlc 收到的诊断报文长度
  *@param Data 收到的诊断报文数据
  */
void setDiagMSG(uint32_t Identifier, uint16_t dlc, uint8_t Data [])
{
    uint16_t i;
    DiagMSG.msgID  = Identifier;
    DiagMSG.msgDLC = dlc;
    for ( i = 0u; i < dlc; ++i )
    {
        DiagMSG.msgData.Data [ i ] = Data [ i ];
    }
}

/**
  *清除诊断信息
  */
void clearDiagMSG(void)
{
    uint16_t i;
    DiagMSG.msgID  = 0u;
    DiagMSG.msgDLC = 0u;
    for ( i = 0u; i < DIM(DiagMSG.msgData.Data); ++i )
    {
        DiagMSG.msgData.Data [ i ] = 0u;
    }
}

/**
  *获取当前车速
  *@return 返回当前车速
  */
uint16_t getSpeed(void)
{
    uint16_t VSpeed = Common_Get_Act_V_Speed( );
    if ( (VSpeed == 0xFFFFu) || (VSpeed <= 20) )
    {
        VSpeed = 0u;
    }
    return (VSpeed / 10u);
}

/**
  *获取当前转速
  *@return 返回当前转速
  */
uint16_t getESpeed(void)
{
    uint16_t ESpeed = 0;//= Common_Get_Act_E_Speed( );
    if ( (ESpeed == 0xFFFFu) || (ESpeed < 500) )
    {
        ESpeed = 0u;
    }
    return ESpeed;
}

/**
  *获取当前30电压
  *@return 返回当前30电压
  */
uint16_t getKL30Voltage(void)
{
    return ADC_Read_Signal(ADC_CH_KL30_VOLTAGE);
}

/**
  *获取15电状态
  *@return 15点状态 0为IGN OFF,1为IGN ON
  */
uint8_t getKL15Status(void)
{
    return Common_Get_IG_Sts( );
}

/**
  *获取当前安全级别
  *@return 返回当前安全级别
  */
uint8_t getSecurityLevel(void)
{
    uint8_t Status = NormalKeyLock;
    if ( Ser27_FlowCtrl.LV1SafetyStatus == NormalKeyUnlock )
    {
        Status = NormalKeyUnlock;
    }
    if ( Ser27_FlowCtrl.LV11SafetyStatus == NormalKeyUnlock )
    {
        Status = BootKeyUnlock;
    }
    return Status;
}

/**
  *重置安全等级
  */
void Ser27_ChangeSessionHandle(void)
{
    Ser27_FlowCtrl.LV1Ctrl          = SER27_CLEAR_REQ_SEEDED;
    Ser27_FlowCtrl.LV1Key [ 0 ]     = 0x00u;
    Ser27_FlowCtrl.LV1Key [ 1 ]     = 0x00u;
    Ser27_FlowCtrl.LV1Key [ 2 ]     = 0x00u;
    Ser27_FlowCtrl.LV1Key [ 3 ]     = 0x00u;
    Ser27_FlowCtrl.LV1SafetyStatus  = NormalKeyLock;
    Ser27_FlowCtrl.LV1Seed [ 0 ]    = 0x00u;
    Ser27_FlowCtrl.LV1Seed [ 1 ]    = 0x00u;
    Ser27_FlowCtrl.LV1Seed [ 2 ]    = 0x00u;
    Ser27_FlowCtrl.LV1Seed [ 3 ]    = 0x00u;
    Ser27_FlowCtrl.LV11Ctrl         = SER27_CLEAR_REQ_SEEDED;
    Ser27_FlowCtrl.LV11Key [ 0 ]    = 0x00u;
    Ser27_FlowCtrl.LV11Key [ 1 ]    = 0x00u;
    Ser27_FlowCtrl.LV11Key [ 2 ]    = 0x00u;
    Ser27_FlowCtrl.LV11Key [ 3 ]    = 0x00u;
    Ser27_FlowCtrl.LV11SafetyStatus = NormalKeyLock;
    Ser27_FlowCtrl.LV11Seed [ 0 ]   = 0x00u;
    Ser27_FlowCtrl.LV11Seed [ 1 ]   = 0x00u;
    Ser27_FlowCtrl.LV11Seed [ 2 ]   = 0x00u;
    Ser27_FlowCtrl.LV11Seed [ 3 ]   = 0x00u;
}

/**
  *复位函数
  */
void UDS_RESET(void)
{
    ControlMessageCommunication(networkManagementMsgAndnormalMsg, disableRxAndTx);
    UDS_Delay(30000u);
    hardware_reset( );
}

/**
  *诊断延时
  *@param count 延时次数
  */
void UDS_Delay(uint16_t count)
{
    uint16_t tmpcnt;
    tmpcnt = count;
    while ( tmpcnt-- )
    {
        WDT_Clear( );
        nop( );
    }
}

/**
  *发送诊断11服务正反馈
  */
void UDS_S11_Sand(uint8_t subid)
{
    _CAN_Msg canMag;
    canMag.MsgID     = DIAG_ID_Tx;
    canMag.Msg [ 0 ] = 0x02u;
    canMag.Msg [ 1 ] = 0x51u;
    canMag.Msg [ 2 ] = subid;
    canMag.Msg [ 3 ] = FILLER_BYTE;
    canMag.Msg [ 4 ] = FILLER_BYTE;
    canMag.Msg [ 5 ] = FILLER_BYTE;
    canMag.Msg [ 6 ] = FILLER_BYTE;
    canMag.Msg [ 7 ] = FILLER_BYTE;
    canMag.MsgDLC    = 8u;
    canMag.MsgPro    = 25u;
    Can_Write(&canMag);
}

/**
  *诊断11服务正反馈确认
  */
void UDS_S11_Confirm(void)
{
    UDS_RESET( );
}

/**
  *发送诊断78负反馈
  */
void UDS_NRC78_Sand(uint8_t sid)
{
    _CAN_Msg canMag;
    canMag.MsgID             = DIAG_ID_Tx;
    canMag.Msg [ 0 ]         = 0x03u;
    canMag.Msg [ 1 ]         = 0x7Fu;
    canMag.Msg [ 2 ]         = sid;
    canMag.Msg [ 3 ]         = 0x78u;
    canMag.Msg [ 4 ]         = FILLER_BYTE;
    canMag.Msg [ 5 ]         = FILLER_BYTE;
    canMag.Msg [ 6 ]         = FILLER_BYTE;
    canMag.Msg [ 7 ]         = FILLER_BYTE;
    canMag.MsgDLC            = 8u;
    canMag.MsgPro            = 0u;
    TpTimingCtrl.pendingFlag = 1;
    Can_Write(&canMag);
}

/**
  *诊断50us任务
  */
void UDS_50us_Service(void)
{
    //if(UDSSleepStatus == 1u)
    {
        ++UDSTimer10ms;
    }
}

/**
  *诊断10ms任务
  */
void UDS_10ms_Service(void)
{
    uint16_t count;
    UDS_KL15_init( );
    if ( UDSTimer10ms >= 200u )
    {
        count = UDSTimer10ms / 200u;
        UDSTimer10ms %= 200u;
        //enableDTCTimer(count * 10u);
        //MsgLostTimer(count * 10u);
        //TestDTC(count * 10u);
        S3_Service(count * 10u);
        Ser27_ErrDelayTimerService(count * 10u);
    }
}

/**
  *诊断KL15init
  */
uint16_t KL15_4s_Timer = 0; 
void UDS_KL15_init(void)
{
    uint8_t KL15Status = getKL15Status( );
    if ( UDSKL15Status == 0u )
    {
        if ( KL15Status == 1u )
        {
            UDSKL15Status  = 1u;
            UDSSleepStatus = 1u;
            setDTCEnableTimer(KL15DTCEnableTimer);
            KL15_4s_Timer = 0;
            //setDTCEnableTimer(KL15DTCEnableTimer);
            S3_ServiceTimeOut( );
            DTCAgeing( );
        }
    }
    else
    {
        if ( KL15Status == 0u )
        {
            UDSKL15Status = 0u;
            KL15_4s_Timer = 0;
            setDTCEnableTimer(KL15DTCEnableTimer);
        }
    }
}

void UDS_Slepp(void)
{
    UDSSleepStatus = 0u;
    UDSTimer10ms   = 0u;
}

/**
  *报文掉线计时器
  *@param timer 计时器增加的时间，单位：ms
  */
void MsgLostTimer(uint16_t timer)
{
    uint8_t i;
    uint8_t BUS_OFF_Status = getBusoffStatus( );
    if ( (isEnable == 1u) && (isReceive == 1u) && (UDSKL15Status == 1u) && (BUS_OFF_Status == 0u) && (HVoltageStatus == 0u) && (LVoltageStatus == 0u) )
    {
        for ( i = 0u; i < (sizeof(msgList) / sizeof(msgList [ 0u ])); ++i )
        {
            if ( msgList [ i ][ 2u ] < 200u )
            {
                if ( msgList [ i ][ 3u ] < 200u * OfflineLoopMultiple )
                {
                    msgList [ i ][ 3u ] += timer;
                }
            }
            else
            {
                if ( msgList [ i ][ 3u ] < msgList [ i ][ 2u ] * OfflineLoopMultiple )
                {
                    msgList [ i ][ 3u ] += timer;
                }
            }
        }
    }
}

/**
  *收到报文清除计时器
  *@param MsgID 报文ID
  *@param DLC   报文长度
  */
void ReceivedMsg(uint32_t MsgID, uint8_t DLC)
{
    uint8_t i;
    if ( isReceive == 1u )
    {
        for ( i = 0u; i < (sizeof(msgList) / sizeof(msgList [ 0u ])); ++i )
        {
            if ( (MsgID == msgList [ i ][ 0u ]) )//&& (DLC == msgList [ i ][ 1u ]) )
            {
                msgList [ i ][ 3 ] = 0u;
                break;
            }
        }
    }
}

/**
  *S3服务
  */
void S3_Service(uint16_t timer)
{
    if ( S3_ServerClear == 1u )
    {
        S3_ServerClear   = 0u;
        S3_ServerCnt     = 0u;
        S3_ServerTimeOut = 0u;
    }
    else
    {
        if ( S3_ServerCnt < S3_SERVER )
        {
            S3_ServerCnt += timer;
        }
        else
        {
            if ( S3_ServerTimeOut == 0u )
            {
                S3_ServerTimeOut = 1u;
                S3_ServiceTimeOut( );
            }
        }
    }
}

/**
  *S3重置服务
  */
void S3_ServiceClear(void)
{
    S3_ServerClear = 1u;
}

/**
  *S3等待服务
  */
void S3_ServiceWait(void)
{
    S3_ServerClear   = 0u;
    S3_ServerCnt     = S3_SERVER;
    S3_ServerTimeOut = 1u;
}

/**
  *S3超时服务
  */
void S3_ServiceTimeOut(void)
{
    /*10服务重置*/
    setSessionType(DefaultSession);
    /*27服务重置*/
    Ser27_ChangeSessionHandle( );
    /*28服务重置*/
    ControlMessageCommunication(networkManagementMsgAndnormalMsg, enableRxAndTx);
    /*2F服务重置*/
    UDS_10Service_S2FInit( );
    /*85服务重置*/
    OpenDTCTest( );
    /*诊断发送重置*/
    clearDiagMSG( );
    /*重置OTA状态*/
}

/**
  *27服务10s延时
  */
void Ser27_ErrDelayTimerService(uint16_t timer)
{
    if ( Ser27_FlowCtrl.LV1DelayCtrl == SER27_ERR_DELAY_TIMER_EN )
    {
        if ( Ser27_FlowCtrl.LV1DelayTimer < 0xfff0u )
        {
            Ser27_FlowCtrl.LV1DelayTimer += timer;
            if ( Ser27_FlowCtrl.LV1DelayTimer >= SER27_ERR_DELAY_TIMEOUT )
            {
                if ( Ser27_FlowCtrlCnt.LV1ReqSeedCnt >= SER27_MAX_REQ_SEED_NUM )
                {
                    Ser27_FlowCtrlCnt.LV1ReqSeedCnt = SER27_MAX_REQ_SEED_NUM - 1u;
                }
                if ( Ser27_FlowCtrlCnt.LV11ReqSeedCnt >= SER27_MAX_REQ_SEED_NUM )
                {
                    Ser27_FlowCtrlCnt.LV11ReqSeedCnt = SER27_MAX_REQ_SEED_NUM - 1u;
                }
                Ser27_FlowCtrl.LV1Ctrl       = SER27_CLEAR_REQ_SEEDED;
                Ser27_FlowCtrl.LV1DelayTimer = 0u;
                Ser27_FlowCtrl.LV1DelayCtrl  = 0u;
                /*调用向DFlash写入函数。*/
                ( void )WriteDFlashData(WriteSer27Addr, ( uint32_t * )&Ser27_FlowCtrlCnt.Flag, sizeof(Ser27_FlowCtrlCnt), NoNeedWait,0xFD);
            }
        }
    }
}

/**
  *控制报文通信
  *@param Type 控制报文的类型
  *@param Status 控制的标志位
  */
void ControlMessageCommunication(uint8_t Type, uint8_t Status)
{
    if ( ( uint8_t )(Type & normalMsg) == normalMsg )
    {
        if ( ( uint8_t )(Status & enableRxAndDisableTx) == enableRxAndDisableTx )
        {
            CAN_SET_APP_SEND(1u);
        }
        else
        {
            CAN_SET_APP_SEND(0u);
        }
        if ( ( uint8_t )(Status & disableRxAndEnableTx) == disableRxAndEnableTx )
        {
            CAN_RX_DiagCtrlEnable(&CAN_CH0_CanMsgOp, CAN_N_RX_Disable);
            isReceive = 0u;
        }
        else
        {
            isReceive = 1u;
            CAN_RX_DiagCtrlEnable(&CAN_CH0_CanMsgOp, CAN_N_RX_Enable);
        }
    }
    if ( ( uint8_t )(Type & networkManagementMsg) == networkManagementMsg )
    {
        if ( ( uint8_t )(Status & enableRxAndDisableTx) == enableRxAndDisableTx )
        {
            CAN_SET_NM_SEND(1u);
        }
        else
        {
            CAN_SET_NM_SEND(0u);
        }
        if ( ( uint8_t )(Status & disableRxAndEnableTx) == disableRxAndEnableTx )
        {
            CAN_SET_NM_REC(1u);
        }
        else
        {
            CAN_SET_NM_REC(0u);
        }
    }
}

/**
  *DTC使能定时
  */
void enableDTCTimer(uint16_t Timer)
{
    uint8_t BUS_OFF_Status = getBusoffStatus( );
    if ( (UDSKL15Status == 1u) && (BUS_OFF_Status == 0u) && (HVoltageStatus == 0u) && (LVoltageStatus == 0u) )
    {
        if ( isEnable == 0u )
        {
            if ( enableTimer >= enableMaxTimer )
            {
                isEnable    = 1u;
                enableTimer = 0u;
            }
            else
            {
                enableTimer += Timer;
            }
        }
    }
    if((UDSKL15Status == 1u) && (HVoltageStatus == 0u) && (LVoltageStatus == 0u))
    {
        if(KL15_4s_Timer < 4000)
        {
            KL15_4s_Timer = KL15_4s_Timer + Timer;
        }
    }
    
}

/**
  *DTC是否使能
  *@return DTC使能状态 0为未使能,1为使能
  */
uint8_t isDTCEnable(void)
{
    return isEnable;
}

/**
  *设置DTC使能时间
  */
void setDTCEnableTimer(uint16_t Timer)
{
    uint8_t i;
    if ( isEnable == 1u )
    {
        isEnable       = 0u;
        enableTimer    = 0u;
        enableMaxTimer = Timer;
        for ( i = 0u; i < (sizeof(msgList) / sizeof(msgList [ 0u ])); ++i )
        {
            msgList [ i ][ 3 ] = 0u;
        }
    }
    else
    {
        if ( enableMaxTimer == Timer )
        {
            enableTimer = 0u;
        }
        else
        {
            if ( enableMaxTimer < enableTimer + Timer )
            {
                enableMaxTimer = enableTimer + Timer;
            }
        }
    }
}

uint8_t IGNON_4S(void)
{
    uint8_t EnableFlag = 0;
    if(KL15_4s_Timer >= 4000)
    {
        EnableFlag = 1;
    }
    return EnableFlag;
}

/**
  *开启DTC检测
  */
void OpenDTCTest(void)
{
    isDTCOpen = 1u;
}

/**
  *关闭DTC检测
  */
void CloseDTCTest(void)
{
    isDTCOpen = 0u;
}

/**
  *DTC是否开启
  *@return DTC状态 0关闭,1开启
  */
uint8_t DTCTestIsOpen(void)
{
    return isDTCOpen;
}

/**
  *DTC老化
  */
void DTCAgeing(void)
{
    uint8_t i;
    for ( i = 0u; i < sizeof(DTCToDFlashInfo.DTC) / sizeof(DTCUnion); ++i )
    {
        DTCToDFlashInfo.DTC [ i ].DTCStatus |= (DTCNoTestThisCycle);
        if ( ( uint8_t )(DTCToDFlashInfo.DTC [ i ].DTCStatus & DTCErrorThisCycle) != DTCErrorThisCycle )
        {
            DTCToDFlashInfo.DTC [ i ].DTCStatus       = ( uint8_t )(DTCToDFlashInfo.DTC [ i ].DTCStatus & (~DTCPending));
            DTCToDFlashInfo.DTC [ i ].DTCPendingCount = 0u;
            if ( ( uint8_t )(DTCToDFlashInfo.DTC [ i ].DTCStatus & DTCConfirmed) == DTCConfirmed )
            {
                DTCToDFlashInfo.DTC [ i ].DTCAgeing++;
                if ( DTCToDFlashInfo.DTC [ i ].DTCAgeing >= AgeingCount )
                {
                    DTCToDFlashInfo.DTC [ i ].DTCStatus = ( uint8_t )(DTCToDFlashInfo.DTC [ i ].DTCStatus & (~DTCConfirmed));
                }
            }
        }
        else
        {
            DTCToDFlashInfo.DTC [ i ].DTCStatus = ( uint8_t )(DTCToDFlashInfo.DTC [ i ].DTCStatus & (~DTCErrorThisCycle));
        }
        DTCToDFlashInfo.DTC [ i ].DTCStatus = ( uint8_t )(DTCToDFlashInfo.DTC [ i ].DTCStatus & DTCNoError);
    }
}

/**
  *读取所有的诊断信息
  */               
void ReadAllDFlashData(void)
{
    //uint32_t i = 0;
    //DID_Data= (ReadOnlayStructure *)((DID_INFO_ADDR+16));
    ( void )ReadDFlashData(WriteProgramValidFlagAddr, ( uint32_t * )&DiagFlag.Flag, sizeof(DiagFlag), AllFlagInit, 0xFD);
    ( void )ReadDFlashData(WriteSer27Addr,     ( uint32_t * )&Ser27_FlowCtrlCnt.Flag, sizeof(Ser27_FlowCtrlCnt), Ser27ToDFlashInfoInit, 0xFD);
    //( void )ReadDFlashData(WriteBootDataAddr,  ( uint32_t * )&BootToDFlashInfo.Flag, sizeof(BootToDFlashInfo), BootToDFlashInfoInit, 0xFD);
    //( void )ReadDFlashData(WriteSer2EDataAddr, ( uint32_t * )&Ser2EToDFlashInfo.Flag, sizeof(Ser2EToDFlashInfo), Ser2EToDFlashInfoInit, 0xFD);
    ( void )ReadDFlashData(WriteEOL0300DataAddr,   ( uint32_t * )&EOL0300ToDFlashInfo.Flag, sizeof(EOL0300ToDFlashInfo), EOL_0300_InitVal, 0xFD);
    ( void )ReadDFlashData(WriteEOL0301DataAddr,   ( uint32_t * )&EOL0301ToDFlashInfo.Flag, sizeof(EOL0301ToDFlashInfo), EOL_0301_InitVal, 0xFD);

    //( void )ReadDFlashData(WriteDTCAddr, ( uint32_t * )&DTCToDFlashInfo.Flag, sizeof(DTCToDFlashInfo), DTCToDFlashInfoInit, 0xEE);

    if ( Ser27_FlowCtrlCnt.LV1ReqSeedCnt >= SER27_MAX_REQ_SEED_NUM )
    {
        Ser27_FlowCtrl.LV1DelayCtrl = SER27_ERR_DELAY_TIMER_EN;
    }
    if ( Ser27_FlowCtrlCnt.LV11ReqSeedCnt >= SER27_MAX_REQ_SEED_NUM )
    {
        Ser27_FlowCtrl.LV11DelayCtrl = SER27_ERR_DELAY_TIMER_EN;
    }

    EOL_Set0300(EOL0300ToDFlashInfo.DID_0300, 90u);
    EOL_Set0301(EOL0301ToDFlashInfo.DID_0301, 224u);

    //DTCConfigurationInit( );

    /*
    if (Check_Infordata_CRC_32(DID_Data->D_F189,DID_Data->D_F189[1]+4)==1u)
    {
        if (memcmp(BootToDFlashInfo.DID_F189,&DID_Data->D_F189[4],DID_Data->D_F189[1])!=0)
        {
            ( void )memcpy(BootToDFlashInfo.DID_F189,&DID_Data->D_F189[4],DID_Data->D_F189[1]);
            WriteDFlashData(WriteBootDataBlockSize, ( uint32_t * )&BootToDFlashInfo.Flag, sizeof(BootToDFlashInfo), NeedWait, 0xFD);
        }
    }
    */

}

/**
  *读取信息
  */
uint8_t ReadDFlashData(uint16_t u16BlockNum, uint32_t u32Data [], uint16_t u16Len, InitFunction initFunction, uint8_t ReadType)
{
    uint8_t enResult = OK;
    enResult         = DFlash_ReadData(u16BlockNum, ( uint32_t * )u32Data, u16Len, ReadType);
    if ( enResult == OK )
    {
        //u32Data = (uint32_t *)u8Data;  
        if ( *u32Data != 0x7AA7A55Au )
        {
            initFunction( );
            enResult = WriteDFlashData(u16BlockNum, u32Data, u16Len, NeedWait, ReadType);
        }
    }
    else
    {
        initFunction( );
        enResult = WriteDFlashData(u16BlockNum, u32Data, u16Len, NeedWait, ReadType);
    }
    return enResult;
}

/**
  *写入信息
  */
uint8_t WriteDFlashData(uint16_t u16BlockNum, uint32_t u32Data [], uint16_t u16Len, uint8_t isWait, uint8_t WriteType)
{
    uint8_t enResult = OK;
    ( void )isWait;
    enResult = DFlash_WriteData(u16BlockNum, ( uint32_t * )u32Data, u16Len, WriteType);
    return enResult;
}

/**
  *所有标志位初始化
  */
void AllFlagInit(void)
{
    DiagFlag.Flag           = 0x7AA7A55Au;
    DiagFlag.JumpToBootFlag = 0x00000000u;
    DiagFlag.AppValidFlag   = 0x00000000u;
    DiagFlag.S11Flag        = 0x00000000u;
}

/**
  *27信息初始化
  */
void Ser27ToDFlashInfoInit(void)
{
    Ser27_FlowCtrlCnt.Flag             = 0x7AA7A55Au;
    Ser27_FlowCtrlCnt.LV11InvaidKyeCnt = 0u;
    Ser27_FlowCtrlCnt.LV11ReqSeedCnt   = 0u;
    Ser27_FlowCtrlCnt.LV1InvaidKyeCnt  = 0u;
    Ser27_FlowCtrlCnt.LV1ReqSeedCnt    = 0u;
    Ser27_FlowCtrlCnt.RandNum          = 0u;
}


/**
  *DTC信息初始化
  */
void DTCToDFlashInfoInit(void)
{
    uint8_t i;
    DTCToDFlashInfo.Flag = 0x7AA7A55Au;
    for ( i = 0u; i < sizeof(DTCToDFlashInfo.DTC) / sizeof(DTCUnion); ++i )
    {
        DTCToDFlashInfo.DTC [ i ].DTCStatus                    = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC [ i ].DTCAgeing                    = 0x00u;
        DTCToDFlashInfo.DTC [ i ].DTCCount                     = 0x00u;
        DTCToDFlashInfo.DTC [ i ].DTCPendingCount              = 0x00u;
        //DTCToDFlashInfo.DTC [ i ].Snapshot.OdometerValue       = 0x00000000u;
        //DTCToDFlashInfo.DTC [ i ].Snapshot.SystemSupplyVoltage = 0x00u;
        //DTCToDFlashInfo.DTC [ i ].Snapshot.VehicleSpeed        = 0x0000u;
        //DTCToDFlashInfo.DTC [ i ].Snapshot.DTCLocalTime_second = 0x00u;
        //DTCToDFlashInfo.DTC [ i ].Snapshot.DTCLocalTime_minute = 0x00u;
        //DTCToDFlashInfo.DTC [ i ].Snapshot.DTCLocalTime_hour   = 0x00u;
        //DTCToDFlashInfo.DTC [ i ].Snapshot.DTCLocalTime_day    = 0x00u;
        //DTCToDFlashInfo.DTC [ i ].Snapshot.DTCLocalTime_month  = 0x00u;
        //DTCToDFlashInfo.DTC [ i ].Snapshot.DTCLocalTime_year   = 0x00u;
    }
    for ( i = 0u; i < (sizeof(msgList) / sizeof(msgList [ 0u ])); ++i )
    {
        msgList [ i ][ 3 ] = 0u;
    }
    for ( i = 0u; i < (sizeof(DTCTimerList) / sizeof(DTCTimerList [ 0u ])); ++i )
    {
        DTCTimerList [ i ] = 0u;
    }
    //DTCUpdataStatusList [ i ]                              = 1u;
}


/**
  *DTC信息配置初始化
  */
void DTCConfigurationInit(void)
{
    #if 0
    //if((Val_EOL_F1F8_Radar( ) != RADAR_R4) && (Val_EOL_F1F8_Radar( ) != RADAR_F2R4))
    {
        /*2E0*/
        DTCToDFlashInfo.DTC [ cnU108A87 ].DTCStatus                    = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC [ cnU108A87 ].DTCAgeing                    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].DTCCount                     = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].DTCPendingCount              = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].Snapshot.OdometerValue       = 0x00000000u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].Snapshot.VehicleSpeed        = 0x0000u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].Snapshot.DTCLocalTime_second = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].Snapshot.DTCLocalTime_minute = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].Snapshot.DTCLocalTime_hour   = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].Snapshot.DTCLocalTime_day    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].Snapshot.DTCLocalTime_month  = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108A87 ].Snapshot.DTCLocalTime_year   = 0x00u;
    }
    //if(Val_EOL_F1F8_Radar() != RADAR_APA)/*3.功能配置有APA功能*/
    {
        /*26C*/
        DTCToDFlashInfo.DTC [ cnU109581 ].DTCStatus                    = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC [ cnU109581 ].DTCAgeing                    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109581 ].DTCCount                     = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109581 ].DTCPendingCount              = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109581 ].Snapshot.OdometerValue       = 0x00000000u;
        DTCToDFlashInfo.DTC [ cnU109581 ].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109581 ].Snapshot.VehicleSpeed        = 0x0000u;
        DTCToDFlashInfo.DTC [ cnU109581 ].Snapshot.DTCLocalTime_second = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109581 ].Snapshot.DTCLocalTime_minute = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109581 ].Snapshot.DTCLocalTime_hour   = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109581 ].Snapshot.DTCLocalTime_day    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109581 ].Snapshot.DTCLocalTime_month  = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109581 ].Snapshot.DTCLocalTime_year   = 0x00u;
    }
    //if((Val_EOL_F1F8_CruiseControl() != CC_ACC) && (Val_EOL_F1F8_CruiseControl() != CC_IACC))/*3.功能配置有ACC功能*/
    {
        /*35E*/
        DTCToDFlashInfo.DTC [ cnU109681 ].DTCStatus                    = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC [ cnU109681 ].DTCAgeing                    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109681 ].DTCCount                     = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109681 ].DTCPendingCount              = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109681 ].Snapshot.OdometerValue       = 0x00000000u;
        DTCToDFlashInfo.DTC [ cnU109681 ].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109681 ].Snapshot.VehicleSpeed        = 0x0000u;
        DTCToDFlashInfo.DTC [ cnU109681 ].Snapshot.DTCLocalTime_second = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109681 ].Snapshot.DTCLocalTime_minute = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109681 ].Snapshot.DTCLocalTime_hour   = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109681 ].Snapshot.DTCLocalTime_day    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109681 ].Snapshot.DTCLocalTime_month  = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109681 ].Snapshot.DTCLocalTime_year   = 0x00u;
    }
    //if(!Val_EOL_F1F8_TSR_DisplayMode())/*3.����������ADS����*/
    {
        /*342*/
        DTCToDFlashInfo.DTC [ cnU10EE87 ].DTCStatus                    = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].DTCAgeing                    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].DTCCount                     = 0x00u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].DTCPendingCount              = 0x00u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].Snapshot.OdometerValue       = 0x00000000u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].Snapshot.VehicleSpeed        = 0x0000u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].Snapshot.DTCLocalTime_second = 0x00u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].Snapshot.DTCLocalTime_minute = 0x00u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].Snapshot.DTCLocalTime_hour   = 0x00u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].Snapshot.DTCLocalTime_day    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].Snapshot.DTCLocalTime_month  = 0x00u;
        DTCToDFlashInfo.DTC [ cnU10EE87 ].Snapshot.DTCLocalTime_year   = 0x00u;

        /*332*/
        DTCToDFlashInfo.DTC [ cnU109781 ].DTCStatus                    = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC [ cnU109781 ].DTCAgeing                    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109781 ].DTCCount                     = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109781 ].DTCPendingCount              = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109781 ].Snapshot.OdometerValue       = 0x00000000u;
        DTCToDFlashInfo.DTC [ cnU109781 ].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109781 ].Snapshot.VehicleSpeed        = 0x0000u;
        DTCToDFlashInfo.DTC [ cnU109781 ].Snapshot.DTCLocalTime_second = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109781 ].Snapshot.DTCLocalTime_minute = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109781 ].Snapshot.DTCLocalTime_hour   = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109781 ].Snapshot.DTCLocalTime_day    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109781 ].Snapshot.DTCLocalTime_month  = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109781 ].Snapshot.DTCLocalTime_year   = 0x00u;
    }
    //if(!Val_EOL_F1F8_LCDA_DisplayMode())/*3.����������LCDAR����*/
    {
        /*2AE*/
        DTCToDFlashInfo.DTC [ cnU109881 ].DTCStatus                    = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC [ cnU109881 ].DTCAgeing                    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109881 ].DTCCount                     = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109881 ].DTCPendingCount              = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109881 ].Snapshot.OdometerValue       = 0x00000000u;
        DTCToDFlashInfo.DTC [ cnU109881 ].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109881 ].Snapshot.VehicleSpeed        = 0x0000u;
        DTCToDFlashInfo.DTC [ cnU109881 ].Snapshot.DTCLocalTime_second = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109881 ].Snapshot.DTCLocalTime_minute = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109881 ].Snapshot.DTCLocalTime_hour   = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109881 ].Snapshot.DTCLocalTime_day    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109881 ].Snapshot.DTCLocalTime_month  = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109881 ].Snapshot.DTCLocalTime_year   = 0x00u;
        /*2AF*/
        DTCToDFlashInfo.DTC [ cnU109981 ].DTCStatus                    = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC [ cnU109981 ].DTCAgeing                    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109981 ].DTCCount                     = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109981 ].DTCPendingCount              = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109981 ].Snapshot.OdometerValue       = 0x00000000u;
        DTCToDFlashInfo.DTC [ cnU109981 ].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109981 ].Snapshot.VehicleSpeed        = 0x0000u;
        DTCToDFlashInfo.DTC [ cnU109981 ].Snapshot.DTCLocalTime_second = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109981 ].Snapshot.DTCLocalTime_minute = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109981 ].Snapshot.DTCLocalTime_hour   = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109981 ].Snapshot.DTCLocalTime_day    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109981 ].Snapshot.DTCLocalTime_month  = 0x00u;
        DTCToDFlashInfo.DTC [ cnU109981 ].Snapshot.DTCLocalTime_year   = 0x00u;
    }
    //if(!Val_EOL_F1F8_IdleStartAndStop())/*3.����������ACM����*/
    {
        /*249*/
        DTCToDFlashInfo.DTC [ cnU108C87 ].DTCStatus                    = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC [ cnU108C87 ].DTCAgeing                    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].DTCCount                     = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].DTCPendingCount              = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].Snapshot.OdometerValue       = 0x00000000u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].Snapshot.VehicleSpeed        = 0x0000u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].Snapshot.DTCLocalTime_second = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].Snapshot.DTCLocalTime_minute = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].Snapshot.DTCLocalTime_hour   = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].Snapshot.DTCLocalTime_day    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].Snapshot.DTCLocalTime_month  = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108C87 ].Snapshot.DTCLocalTime_year   = 0x00u;
    }
    //if(!Val_EOL_F1F8_DSM_DisplayMode())/*3.����������ACM����*/
    {
        /*352*/
        DTCToDFlashInfo.DTC [ cnU108D87 ].DTCStatus                    = DTCNoTestLastClear | DTCNoTestThisCycle;
        DTCToDFlashInfo.DTC [ cnU108D87 ].DTCAgeing                    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].DTCCount                     = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].DTCPendingCount              = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].Snapshot.OdometerValue       = 0x00000000u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].Snapshot.SystemSupplyVoltage = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].Snapshot.VehicleSpeed        = 0x0000u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].Snapshot.DTCLocalTime_second = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].Snapshot.DTCLocalTime_minute = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].Snapshot.DTCLocalTime_hour   = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].Snapshot.DTCLocalTime_day    = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].Snapshot.DTCLocalTime_month  = 0x00u;
        DTCToDFlashInfo.DTC [ cnU108D87 ].Snapshot.DTCLocalTime_year   = 0x00u;
    }
    #endif
}

const uint32_t CRC32_Table[256] =
{
    0x00000000ul, 0x77073096ul, 0xEE0E612Cul, 0x990951BAul,
    0x076DC419ul, 0x706AF48Ful, 0xE963A535ul, 0x9E6495A3ul,
    0x0EDB8832ul, 0x79DCB8A4ul, 0xE0D5E91Eul, 0x97D2D988ul,
    0x09B64C2Bul, 0x7EB17CBDul, 0xE7B82D07ul, 0x90BF1D91ul,
    0x1DB71064ul, 0x6AB020F2ul, 0xF3B97148ul, 0x84BE41DEul,
    0x1ADAD47Dul, 0x6DDDE4EBul, 0xF4D4B551ul, 0x83D385C7ul,
    0x136C9856ul, 0x646BA8C0ul, 0xFD62F97Aul, 0x8A65C9ECul,
    0x14015C4Ful, 0x63066CD9ul, 0xFA0F3D63ul, 0x8D080DF5ul,
    0x3B6E20C8ul, 0x4C69105Eul, 0xD56041E4ul, 0xA2677172ul,
    0x3C03E4D1ul, 0x4B04D447ul, 0xD20D85FDul, 0xA50AB56Bul,
    0x35B5A8FAul, 0x42B2986Cul, 0xDBBBC9D6ul, 0xACBCF940ul,
    0x32D86CE3ul, 0x45DF5C75ul, 0xDCD60DCFul, 0xABD13D59ul,
    0x26D930ACul, 0x51DE003Aul, 0xC8D75180ul, 0xBFD06116ul,
    0x21B4F4B5ul, 0x56B3C423ul, 0xCFBA9599ul, 0xB8BDA50Ful,
    0x2802B89Eul, 0x5F058808ul, 0xC60CD9B2ul, 0xB10BE924ul,
    0x2F6F7C87ul, 0x58684C11ul, 0xC1611DABul, 0xB6662D3Dul,
    0x76DC4190ul, 0x01DB7106ul, 0x98D220BCul, 0xEFD5102Aul,
    0x71B18589ul, 0x06B6B51Ful, 0x9FBFE4A5ul, 0xE8B8D433ul,
    0x7807C9A2ul, 0x0F00F934ul, 0x9609A88Eul, 0xE10E9818ul,
    0x7F6A0DBBul, 0x086D3D2Dul, 0x91646C97ul, 0xE6635C01ul,
    0x6B6B51F4ul, 0x1C6C6162ul, 0x856530D8ul, 0xF262004Eul,
    0x6C0695EDul, 0x1B01A57Bul, 0x8208F4C1ul, 0xF50FC457ul,
    0x65B0D9C6ul, 0x12B7E950ul, 0x8BBEB8EAul, 0xFCB9887Cul,
    0x62DD1DDFul, 0x15DA2D49ul, 0x8CD37CF3ul, 0xFBD44C65ul,
    0x4DB26158ul, 0x3AB551CEul, 0xA3BC0074ul, 0xD4BB30E2ul,
    0x4ADFA541ul, 0x3DD895D7ul, 0xA4D1C46Dul, 0xD3D6F4FBul,
    0x4369E96Aul, 0x346ED9FCul, 0xAD678846ul, 0xDA60B8D0ul,
    0x44042D73ul, 0x33031DE5ul, 0xAA0A4C5Ful, 0xDD0D7CC9ul,
    0x5005713Cul, 0x270241AAul, 0xBE0B1010ul, 0xC90C2086ul,
    0x5768B525ul, 0x206F85B3ul, 0xB966D409ul, 0xCE61E49Ful,
    0x5EDEF90Eul, 0x29D9C998ul, 0xB0D09822ul, 0xC7D7A8B4ul,
    0x59B33D17ul, 0x2EB40D81ul, 0xB7BD5C3Bul, 0xC0BA6CADul,
    0xEDB88320ul, 0x9ABFB3B6ul, 0x03B6E20Cul, 0x74B1D29Aul,
    0xEAD54739ul, 0x9DD277AFul, 0x04DB2615ul, 0x73DC1683ul,
    0xE3630B12ul, 0x94643B84ul, 0x0D6D6A3Eul, 0x7A6A5AA8ul,
    0xE40ECF0Bul, 0x9309FF9Dul, 0x0A00AE27ul, 0x7D079EB1ul,
    0xF00F9344ul, 0x8708A3D2ul, 0x1E01F268ul, 0x6906C2FEul,
    0xF762575Dul, 0x806567CBul, 0x196C3671ul, 0x6E6B06E7ul,
    0xFED41B76ul, 0x89D32BE0ul, 0x10DA7A5Aul, 0x67DD4ACCul,
    0xF9B9DF6Ful, 0x8EBEEFF9ul, 0x17B7BE43ul, 0x60B08ED5ul,
    0xD6D6A3E8ul, 0xA1D1937Eul, 0x38D8C2C4ul, 0x4FDFF252ul,
    0xD1BB67F1ul, 0xA6BC5767ul, 0x3FB506DDul, 0x48B2364Bul,
    0xD80D2BDAul, 0xAF0A1B4Cul, 0x36034AF6ul, 0x41047A60ul,
    0xDF60EFC3ul, 0xA867DF55ul, 0x316E8EEFul, 0x4669BE79ul,
    0xCB61B38Cul, 0xBC66831Aul, 0x256FD2A0ul, 0x5268E236ul,
    0xCC0C7795ul, 0xBB0B4703ul, 0x220216B9ul, 0x5505262Ful,
    0xC5BA3BBEul, 0xB2BD0B28ul, 0x2BB45A92ul, 0x5CB36A04ul,
    0xC2D7FFA7ul, 0xB5D0CF31ul, 0x2CD99E8Bul, 0x5BDEAE1Dul,
    0x9B64C2B0ul, 0xEC63F226ul, 0x756AA39Cul, 0x026D930Aul,
    0x9C0906A9ul, 0xEB0E363Ful, 0x72076785ul, 0x05005713ul,
    0x95BF4A82ul, 0xE2B87A14ul, 0x7BB12BAEul, 0x0CB61B38ul,
    0x92D28E9Bul, 0xE5D5BE0Dul, 0x7CDCEFB7ul, 0x0BDBDF21ul,
    0x86D3D2D4ul, 0xF1D4E242ul, 0x68DDB3F8ul, 0x1FDA836Eul,
    0x81BE16CDul, 0xF6B9265Bul, 0x6FB077E1ul, 0x18B74777ul,
    0x88085AE6ul, 0xFF0F6A70ul, 0x66063BCAul, 0x11010B5Cul,
    0x8F659EFFul, 0xF862AE69ul, 0x616BFFD3ul, 0x166CCF45ul,
    0xA00AE278ul, 0xD70DD2EEul, 0x4E048354ul, 0x3903B3C2ul,
    0xA7672661ul, 0xD06016F7ul, 0x4969474Dul, 0x3E6E77DBul,
    0xAED16A4Aul, 0xD9D65ADCul, 0x40DF0B66ul, 0x37D83BF0ul,
    0xA9BCAE53ul, 0xDEBB9EC5ul, 0x47B2CF7Ful, 0x30B5FFE9ul,
    0xBDBDF21Cul, 0xCABAC28Aul, 0x53B39330ul, 0x24B4A3A6ul,
    0xBAD03605ul, 0xCDD70693ul, 0x54DE5729ul, 0x23D967BFul,
    0xB3667A2Eul, 0xC4614AB8ul, 0x5D681B02ul, 0x2A6F2B94ul,
    0xB40BBE37ul, 0xC30C8EA1ul, 0x5A05DF1Bul, 0x2D02EF8Dul
};


uint32_t Check_Infordata_CRC_32(uint8_t p[],uint32_t len)
{
    uint32_t i=0U;
    uint32_t m32=0u;
    uint32_t FlashCrcValue=0xFFFFFFFF;
    uint32_t result=0;
    if (len<0xff)
    {
        for ( i = 0; i < len; i++)
        {
        FlashCrcValue  = CRC32_Table[(FlashCrcValue  ^ p[i]) & 0xFF] ^ ((FlashCrcValue  >> 8) & 0x00ffffff);
        }

        FlashCrcValue ^= 0xffffffff;
    }

    m32 = (((uint32_t)p[i] << 24) & 0xff000000) | (((uint32_t)p[i+1] << 16) & 0x00ff0000) | (((uint32_t)p[i+2] << 8) & 0x0000ff00) | (((uint32_t)p[i+3] ) & 0xff0000ff) ;

    if ((m32==FlashCrcValue)&&(FlashCrcValue!=0xFFFFFFFF))
    {
        result=1;
    }

    return result;
}
