#include "CAN_CH0_CAN_Communication_Matrix.h"
#include "Common_Interface.h"
#include "Brake_Pressure_Table.h"
#include "Analog_Signals.h"

static uint8_t                g_BrakePressureSegNum_Disp;
static uint8_t                g_BrakePressureValid;          //  -0 无效 -1 有效
static DataBrakePressureCalcStruct g_DataBrakePressureTempCalc;

/*** 制动气压上升表格 ***/
static const uint16_t DataBrakePressureTable_Up [ DATA_BrakePressure_TEMP_TOTAL_SEG ] =
    /*显示段：|    0     |    1     |    2     |    3     |*/
    {480, 900, 1240, 1560, 1840};    // 上升时小于等于表值则熄灭相应段

/*** 制动气压下降表格 ***/
static const uint16_t DataBrakePressureTable_Dn [ DATA_BrakePressure_TEMP_TOTAL_SEG ] =
    /*显示段：|    0     |    1     |    2     |    3     |*/
    {430, 860, 1180, 1410, 1640};    // 下降时小于等于表值则熄灭相应段

void Data_BrakePressure_TempPara_KL30_Init(void)
{
    g_DataBrakePressureTempCalc.FastMode         = 0u;
    g_BrakePressureSegNum_Disp                   = 0u;
    g_BrakePressureValid                         = 1u;
    g_DataBrakePressureTempCalc.Last             = 0u;
    g_DataBrakePressureTempCalc.InvalidCnt       = 0u;
    g_DataBrakePressureTempCalc.DstSeg           = 0u;
    g_DataBrakePressureTempCalc.State            = DATA_BrakePressure_TEMP_STATE_DELAY;
    g_DataBrakePressureTempCalc.Wait             = DATA_BrakePressure_TEMP_WAIT_SEC * 10;
    g_DataBrakePressureTempCalc.Prescaler        = 0u;
    g_DataBrakePressureTempCalc.OutputEnable     = 0u;
    g_DataBrakePressureTempCalc.DispTimer        = 0u;
    g_DataBrakePressureTempCalc.EngineStartTimer = 0u;
    g_DataBrakePressureTempCalc.Backup           = 0u;
}

void Data_BrakePressure_TempPara_Wakeup_Init(void)
{
    g_BrakePressureSegNum_Disp                   = 0u;
    g_BrakePressureValid                         = 1u;
    g_DataBrakePressureTempCalc.Last             = 0u;
    g_DataBrakePressureTempCalc.InvalidCnt       = 0u;
    g_DataBrakePressureTempCalc.DstSeg           = 0u;
    g_DataBrakePressureTempCalc.State            = DATA_BrakePressure_TEMP_STATE_DELAY;
    g_DataBrakePressureTempCalc.Wait             = DATA_BrakePressure_TEMP_WAIT_SEC * 10;
    g_DataBrakePressureTempCalc.Prescaler        = 0u;
    g_DataBrakePressureTempCalc.OutputEnable     = 0u;
    g_DataBrakePressureTempCalc.DispTimer        = 0u;
    g_DataBrakePressureTempCalc.EngineStartTimer = 0u;
}

/******************************************************************************
函数名：Data_BrakePressure_Temp_Processing_Service
功  能：SOC数据处理函数
参  数：无
返回值：无
*******************************************************************************
注  意：该函数必须每20ms被调用一次
******************************************************************************/
void Data_BrakePressure_Temp_Processing_Service(void)
{
    uint16_t  Data_Brake_Pressure    = 0u;
    uint8_t  Delta                   = 0u;
    uint16_t DataBrakePressure_Temp  = 0u;
    static uint8_t Hysteresis        = 0u;

    Data_Brake_Pressure = ADC_Read_Signal(ADC_CH_BRAKE_AIRPRESSURE);

    if ( Common_Get_IG_Sts( ) == COMMON_POWER_ON )
    {
        if((Data_Brake_Pressure >=80) && (Data_Brake_Pressure <=3000) && (Hysteresis == 1))
        {
            Hysteresis = 0;
        }
        if ( Data_Brake_Pressure < 20 )
        {
            g_BrakePressureValid   = 0u;
            DataBrakePressure_Temp = 0u;
            Hysteresis = 1;
        }
        else
        {
            if ( Data_Brake_Pressure > 3500 )
            {
                g_BrakePressureValid = 0;
                Data_Brake_Pressure  = 0;
                Hysteresis = 1;
            }
            else
            {
                if(Hysteresis == 1)
                {
                    g_BrakePressureValid   = 0u;
                    DataBrakePressure_Temp = 0u;
                }
                else
                {
                    DataBrakePressure_Temp = Data_Brake_Pressure;
                    g_BrakePressureValid = 1u;
                }
                
            }
        }

        // 第2步：获取制动气压格数
        if ( g_BrakePressureValid )
        {
            if ( DataBrakePressure_Temp > g_DataBrakePressureTempCalc.Last )    // 新制动气压高于前次制动气压,查上升表
            {
                while ( g_DataBrakePressureTempCalc.DstSeg < DATA_BrakePressure_TEMP_TOTAL_SEG )
                {
                    if ( DataBrakePressure_Temp < DataBrakePressureTable_Up [ g_DataBrakePressureTempCalc.DstSeg ] )
                        break;
                    g_DataBrakePressureTempCalc.DstSeg++;
                }
            }
            else if ( DataBrakePressure_Temp < g_DataBrakePressureTempCalc.Last )    // 新制动气压低于前次制动气压,查下降表
            {
                while ( g_DataBrakePressureTempCalc.DstSeg )
                {
                    if ( DataBrakePressure_Temp > DataBrakePressureTable_Dn [ g_DataBrakePressureTempCalc.DstSeg - 1 ] )
                        break;
                    g_DataBrakePressureTempCalc.DstSeg--;
                }
            }

            g_DataBrakePressureTempCalc.Last = DataBrakePressure_Temp;
        }

        // 第3步：状态转换,每100ms执行一次
        g_DataBrakePressureTempCalc.Prescaler++;
        if ( g_DataBrakePressureTempCalc.Prescaler >= 5 )
        {
            g_DataBrakePressureTempCalc.Prescaler = 0;

            switch ( g_DataBrakePressureTempCalc.State )
            {
                case DATA_BrakePressure_TEMP_STATE_DELAY:
                    if ( g_DataBrakePressureTempCalc.Wait )
                        g_DataBrakePressureTempCalc.Wait--;
                    else
                        g_DataBrakePressureTempCalc.State = DATA_BrakePressure_TEMP_STATE_INIT;

                    break;

                case DATA_BrakePressure_TEMP_STATE_INIT:
                    if ( g_BrakePressureValid )
                    {
                        if ( g_DataBrakePressureTempCalc.DstSeg )
                        {
                            g_DataBrakePressureTempCalc.OutputEnable = 1;
                            g_DataBrakePressureTempCalc.FastMode     = 1;
                            g_DataBrakePressureTempCalc.State        = DATA_BrakePressure_TEMP_STATE_FAST;
                        }
                    }
                    else
                    {
                        g_DataBrakePressureTempCalc.OutputEnable = 1;
                        g_DataBrakePressureTempCalc.FastMode     = 1;
                        g_DataBrakePressureTempCalc.Last         = 0;
                        g_DataBrakePressureTempCalc.DstSeg       = 0;

                        g_DataBrakePressureTempCalc.State = DATA_BrakePressure_TEMP_STATE_FAULT;
                    }

                    break;

                case DATA_BrakePressure_TEMP_STATE_FAST:
                    if ( g_BrakePressureValid )
                    {
                        if ( g_DataBrakePressureTempCalc.FastMode == 0 )
                            g_DataBrakePressureTempCalc.State = DATA_BrakePressure_TEMP_STATE_NORMAL;
                    }
                    else
                    {
                        g_DataBrakePressureTempCalc.OutputEnable = 1;
                        g_DataBrakePressureTempCalc.FastMode     = 1;
                        g_DataBrakePressureTempCalc.Last         = 0;
                        g_DataBrakePressureTempCalc.DstSeg       = 0;

                        g_DataBrakePressureTempCalc.State = DATA_BrakePressure_TEMP_STATE_FAULT;
                    }

                    break;

                case DATA_BrakePressure_TEMP_STATE_NORMAL:
                    if ( g_BrakePressureValid == 0 )
                    {
                        g_DataBrakePressureTempCalc.OutputEnable = 1;
                        g_DataBrakePressureTempCalc.FastMode     = 1;
                        g_DataBrakePressureTempCalc.Last         = 0;
                        g_DataBrakePressureTempCalc.DstSeg       = 0;

                        g_DataBrakePressureTempCalc.State = DATA_BrakePressure_TEMP_STATE_FAULT;
                    }

                    break;

                case DATA_BrakePressure_TEMP_STATE_FAULT:
                    if ( g_BrakePressureValid )
                        g_DataBrakePressureTempCalc.State = DATA_BrakePressure_TEMP_STATE_INIT;

                    break;

                default:
                    g_BrakePressureSegNum_Disp               = 0;
                    DataBrakePressure_Temp                   = 0;
                    g_BrakePressureValid                     = 0;
                    g_DataBrakePressureTempCalc.OutputEnable = 1;
                    g_DataBrakePressureTempCalc.FastMode     = 1;
                    g_DataBrakePressureTempCalc.Last         = 0;
                    g_DataBrakePressureTempCalc.DstSeg       = 0;

                    g_DataBrakePressureTempCalc.State = DATA_BrakePressure_TEMP_STATE_FAULT;

                    break;
            }
        }

        // 第4步：显示输出格数
        if ( g_DataBrakePressureTempCalc.OutputEnable )
        {
            if ( g_BrakePressureSegNum_Disp < g_DataBrakePressureTempCalc.DstSeg )    // 需要增加段数时
            {
                if ( g_DataBrakePressureTempCalc.FastMode )
                    Delta = ( uint8_t )(DATA_BrakePressure_TEMP_SEG_STEP_FAST / 20);
                else
                    Delta = ( uint8_t )(DATA_BrakePressure_TEMP_SEG_STEP_SLOW / 20);

                g_DataBrakePressureTempCalc.DispTimer++;
                if ( g_DataBrakePressureTempCalc.DispTimer >= Delta )
                {
                    g_DataBrakePressureTempCalc.DispTimer = 0;
                    g_BrakePressureSegNum_Disp++;
                }
            }
            else if ( g_BrakePressureSegNum_Disp > g_DataBrakePressureTempCalc.DstSeg )    // 需要减少段数时
            {
                if ( g_DataBrakePressureTempCalc.FastMode )
                    Delta = ( uint8_t )(DATA_BrakePressure_TEMP_SEG_STEP_FAST / 20);
                else
                    Delta = ( uint8_t )(DATA_BrakePressure_TEMP_SEG_STEP_SLOW / 20);

                g_DataBrakePressureTempCalc.DispTimer++;
                if ( g_DataBrakePressureTempCalc.DispTimer >= Delta )
                {
                    g_DataBrakePressureTempCalc.DispTimer = 0;
                    g_BrakePressureSegNum_Disp--;
                }
            }
            else
            {
                g_DataBrakePressureTempCalc.DispTimer = 0;
                g_DataBrakePressureTempCalc.FastMode  = 0;
            }
        }
    }
    else
    {
        g_DataBrakePressureTempCalc.EngineStartTimer = 0;
        g_DataBrakePressureTempCalc.Backup           = DataBrakePressure_Temp;
    }
}

uint8_t Get_Brake_Pressure_Valid(void)
{
    return g_BrakePressureValid;
}
uint8_t Get_Brake_Pressure_Display(void)
{
    return g_BrakePressureSegNum_Disp;
}
