#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include "../lib/sock_msg.h"

// 一个简单的回调函数示例
void simple_callback(int fd, uint32_t events)
{
    char buffer[BUFFER_SIZE];
    if (events & POLLIN)
    {
        ssize_t bytes_read = read(fd, buffer, BUFFER_SIZE);
        if (bytes_read > 0)
        {
            // 处理读取到的数据
            printf("Received from client: %s\n", buffer);

            // 在接收到的数据后面加上" - echo"
            const char *echo_str = "- echo";
            size_t echo_str_len = strlen(echo_str);
            // 确保不会超出buffer的大小
            if (bytes_read + echo_str_len < BUFFER_SIZE)
            {
                strcat(buffer, echo_str);   // 连接字符串
                bytes_read += echo_str_len; // 更新bytes_read以包含新加入的字符
            }
            write(fd, buffer, bytes_read); // 回显数据
        }
        else
        {
            // 客户端断开连接或出错
            sock_ev_unregister(fd); // 注销事件监听
            close(fd);
        }
    }
    // 可以根据需要添加对POLLHUP、POLLERR等事件的处理
}

int main(int argc, char *argv[])
{
    int listen_fd, port;
    struct sockaddr_in server_addr;
    struct pollfd fds[MAX_CLIENTS + 1];

    if (argc < 2)
    {
        fprintf(stderr, "Usage: %s <port>\n", argv[0]);
        exit(EXIT_FAILURE);
    }

    port = atoi(argv[1]);

    listen_fd = socket(AF_INET, SOCK_STREAM, 0);
    if (listen_fd < 0)
    {
        perror("socket failed");
        exit(EXIT_FAILURE);
    }

    memset(&server_addr, 0, sizeof(server_addr));
    server_addr.sin_family = AF_INET;
    server_addr.sin_addr.s_addr = INADDR_ANY;
    server_addr.sin_port = htons(port);

    if (bind(listen_fd, (struct sockaddr *)&server_addr, sizeof(server_addr)) < 0)
    {
        perror("bind failed");
        exit(EXIT_FAILURE);
    }

    if (listen(listen_fd, MAX_CLIENTS) < 0)
    {
        perror("listen failed");
        exit(EXIT_FAILURE);
    }

    // 初始化pollfd结构和sockets数组
    memset(fds, 0, sizeof(fds));
    memset(sockets, 0, sizeof(sockets));
    fds[0].fd = listen_fd;
    fds[0].events = POLLIN;
    sockets[0].fd = listen_fd;
    sockets[0].callback = simple_callback;
    sockets[0].mode = SOCK_READ;

    while (1)
    {
        int nfds = poll(fds, MAX_CLIENTS + 1, -1);
        if (nfds < 0)
        {
            perror("poll failed");
            exit(EXIT_FAILURE);
        }

        if (fds[0].revents & POLLIN)
        {
            struct sockaddr_in client_addr;
            socklen_t client_len = sizeof(client_addr);
            int client_fd = accept(listen_fd, (struct sockaddr *)&client_addr, &client_len);
            if (client_fd < 0)
            {
                perror("accept failed");
                continue;
            }

            if (sock_ev_register(client_fd, simple_callback, SOCK_READ) < 0)
            {
                fprintf(stderr, "Failed to register new client socket.\n");
                close(client_fd);
            }
        }

        for (int i = 1; i < MAX_CLIENTS + 1; i++)
        {
            if (sockets[i].fd > 0)
            {
                fds[i].fd = sockets[i].fd;
                fds[i].events = 0;
                if (sockets[i].mode & SOCK_READ)
                {
                    fds[i].events |= POLLIN;
                }
                if (sockets[i].mode & SOCK_WRITE)
                {
                    fds[i].events |= POLLOUT;
                }
            }
        }

        for (int i = 1; i < MAX_CLIENTS + 1; i++)
        {
            if (sockets[i].fd > 0 && fds[i].revents & (POLLIN | POLLOUT))
            {
                sockets[i].callback(sockets[i].fd, fds[i].revents);
            }
        }
    }

    close(listen_fd);
    return 0;
}