
#include <stdlib.h>
#include "main.h"
#include "RTE.h"

static GPIO_TypeDef *g_pstRTEGPIOPortList[5] =
{
    GPIOA,
    GPIOB,
    GPIOC,
    GPIOF,
};

static uint32_t g_u32RTEGPIOEXTIPortList[5] =
{
    EXTI_GPIOA,
    EXTI_GPIOB,
    EXTI_GPIOC,
    EXTI_GPIOF,
};

RTE_GPIO_IRQ_Desc_st_t *g_pstRTEGPIOIRQDesc[16] = 
{
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
};

void EXTI0_1_IRQHandler(void);
void EXTI2_3_IRQHandler(void);
void EXTI4_15_IRQHandler(void);

/**
 * @brief 初始化GPIO的工作条件
 * @warning None
 *
 * 示例
 @code
    RTE_GPIO_Init();
 @endcode
 *
 * @since 1.0.0
 */
void RTE_GPIO_Init(void)
{
    uint32_t i;

    /* 启动GPIO外设时钟 */
    __HAL_RCC_GPIOA_CLK_ENABLE();
    __HAL_RCC_GPIOB_CLK_ENABLE();
    __HAL_RCC_GPIOC_CLK_ENABLE(); 
    __HAL_RCC_GPIOF_CLK_ENABLE();
    

    /* 禁用所有GPIO中断 */
    //EXTI_DeInit();

    NVIC_DisableIRQ(EXTI0_1_IRQn);
    NVIC_DisableIRQ(EXTI2_3_IRQn);
    NVIC_DisableIRQ(EXTI4_15_IRQn);


    /* 清除所有GPIO中断回调函数 */
    for (i = 0UL; i < 16UL; i++)
    {
        g_pstRTEGPIOIRQDesc[i] = NULL;
    }
}

/**
 * @brief 复位GPIO至默认状态
 * @warning None
 *
 * 示例
 @code
    RTE_GPIO_DeInit();
 @endcode
 *
 * @since 1.0.0
 */
void RTE_GPIO_DeInit(void)
{
    uint32_t i;
    
    /* 禁用所有GPIO中断 */
    //EXTI_DeInit();
    
    NVIC_DisableIRQ(EXTI0_1_IRQn);
    NVIC_DisableIRQ(EXTI2_3_IRQn);
    NVIC_DisableIRQ(EXTI4_15_IRQn);

    /* 清除所有GPIO中断回调函数 */
    for (i = 0UL; i < 16UL; i++)
    {
        g_pstRTEGPIOIRQDesc[i] = NULL;
    }

    /* GPIO恢复至默认状态 */
    HAL_GPIO_DeInit(GPIOA, GPIO_PIN_All);
    HAL_GPIO_DeInit(GPIOB, GPIO_PIN_All);
    HAL_GPIO_DeInit(GPIOC, GPIO_PIN_All);
    HAL_GPIO_DeInit(GPIOF, GPIO_PIN_All);

    /* 停止GPIO外设时钟 */
    __HAL_RCC_GPIOA_CLK_DISABLE();
    __HAL_RCC_GPIOB_CLK_DISABLE(); 
    __HAL_RCC_GPIOC_CLK_DISABLE(); 
    __HAL_RCC_GPIOF_CLK_DISABLE(); 
}

/**
 * @brief 使端口工作在设定的GPIO模式下
 * @param[in] u16Pin 端口编号
 * @param[in] u8Mode GPIO工作模式
 * @return 设置结果, 0为设置成功, -1为设置失败
 * @warning None
 *
 * 示例
 @code
    RTE_GPIO_Config(RTE_GPIO_PORTxx_PINyy, RTE_GPIO_DIR_OUT | RTE_GPIO_LEVEL_HIGH);
 @endcode
 *
 * @since 1.0.0
 */
int32_t RTE_GPIO_Config(uint32_t u32Pin, uint8_t u8Mode)
{
    int32_t  i32Result;
    uint16_t u16PortIndex;
    uint16_t u16PinIndex;
    
    GPIO_TypeDef *pPort;
    GPIO_InitTypeDef stInit;

    i32Result = -1;
    u16PinIndex  = u32Pin & 0xFFFFU;
    u16PortIndex = (u32Pin >> 16) & 0xFFFFU;
    if ((u16PortIndex < 5U) && (u16PinIndex < 16U))
    {
        pPort = g_pstRTEGPIOPortList[u16PortIndex];

        stInit.Pin = 1U << u16PinIndex;
            
        if ((u8Mode & RTE_GPIO_DIR_MASK) == RTE_GPIO_DIR_IN )
        {
            stInit.Mode = GPIO_MODE_INPUT;
            if ((u8Mode & RTE_GPIO_PULL_MASK) == RTE_GPIO_PULL_UP)
            {
                
                stInit.Pull = GPIO_PULLUP;  
            }
            else if ((u8Mode & RTE_GPIO_PULL_MASK) == RTE_GPIO_PULL_DOWN)
            {
                stInit.Pull = GPIO_PULLDOWN;
            }
            else
            {
                stInit.Pull = GPIO_NOPULL;
            }

            HAL_GPIO_Init(pPort, &stInit);
        }
        else if ((u8Mode & RTE_GPIO_DIR_MASK) == RTE_GPIO_DIR_OUT)
        {
            if ((u8Mode & RTE_GPIO_SPEED_MASK) == RTE_GPIO_SPEED_0)
            {
                 stInit.Speed = GPIO_SPEED_FREQ_VERY_HIGH; 
            }
            else if ((u8Mode & RTE_GPIO_SPEED_MASK) == RTE_GPIO_SPEED_1)
            {
                 stInit.Speed = GPIO_SPEED_FREQ_HIGH; 
            }
            else if ((u8Mode & RTE_GPIO_SPEED_MASK) == RTE_GPIO_SPEED_2)
            {
                 stInit.Speed = GPIO_SPEED_FREQ_MEDIUM; 
            }
            else if ((u8Mode & RTE_GPIO_SPEED_MASK) == RTE_GPIO_SPEED_3)
            {
                 stInit.Speed = GPIO_SPEED_FREQ_LOW; 
            }
            else
            {
                stInit.Speed = GPIO_SPEED_FREQ_LOW;
            }
            
            if ((u8Mode & RTE_GPIO_OUTPUT_TYPE_MASK) == RTE_GPIO_OUTPUT_PP)
            {
                stInit.Mode = GPIO_MODE_OUTPUT_PP;
            }
            else
            {
                stInit.Mode = GPIO_MODE_OUTPUT_OD;
            }
            HAL_GPIO_Init(pPort, &stInit);
            HAL_GPIO_WritePin(pPort, stInit.Pin, (GPIO_PinState)(u8Mode & RTE_GPIO_LEVEL_MASK));
        }

        i32Result = 0;
    }
    
    return i32Result;
}


/**
 * @brief GPIO端口批量初始化
 * @param[in] pstGPIOPinList 指向GPIO端口配置列表的指针
 * @param[in] u32Num 列表中的GPIO端口数量
 * @return  设置结果, 0为设置成功, -1为设置失败
 * @warning None
 *
 * 示例
 @code
    RTE_GPIO_Bulk_Config(&stGPIOPinList, (sizeof(stGPIOPinList) / sizeof(RTE_GPIO_Config_st_t)));
 @endcode
 *
 * @since 1.0.0
 */
int32_t RTE_GPIO_Bulk_Config(RTE_GPIO_Config_st_t *pstGPIOPinList, uint32_t u32Num)
{
    uint32_t i;
    uint32_t u32Pin;
    uint8_t  u8Mode;
    int32_t  i32Result;
    
    i32Result = -1;
    if ((pstGPIOPinList != NULL) && (u32Num != 0))
    {
        for(i = 0UL; i < u32Num; i++)
        {
            u32Pin = pstGPIOPinList[i].u32PinNum;
            u8Mode = (uint8_t)(pstGPIOPinList[i].u16PinMode & 0x00FFU);
            RTE_GPIO_Config(u32Pin, u8Mode);
        }

        i32Result = 0;
    }
    
    return i32Result;
}

/**
 * @brief 设置指定的端口的输出电平
 * @param[in] u16Pin 端口编号
 * @param[in] u8Level 输出电平,0为低电平,1为高电平
 * @warning None
 *
 * 示例
 @code
    RTE_GPIO_Set_Level(RTE_GPIO_PORTxx_PINyy, 0U);
 @endcode
 *
 * @since 1.0.0
 */
int32_t RTE_GPIO_Set_Level(uint32_t u32Pin, uint8_t u8Level)
{
    uint16_t u16PortIndex;
    uint16_t u16PinIndex;
    uint16_t u16PinBitMask;
    int32_t  i32Result;
    
    GPIO_TypeDef *pPort;
	
    i32Result = -1;
    u16PinIndex  = u32Pin & 0xFFFFU;
    u16PortIndex = (u32Pin >> 16) & 0xFFFFU;
    if ((u16PortIndex < 5U) && (u16PinIndex < 16U))
	{
		pPort = g_pstRTEGPIOPortList[u16PortIndex];
		u16PinBitMask = 1U << u16PinIndex;
		
		HAL_GPIO_WritePin(pPort, u16PinBitMask, (GPIO_PinState)u8Level);
		
		i32Result = 0;
	}
	return i32Result;
}

/**
 * @brief 读取设置指定的端口的输入电平
 * @param[in] u16Pin 端口编号
 * @return  端口输入电平,0为低电平,1为高电平
 * @warning None
 *
 * 示例
 @code
    uint8_t u8Level = RTE_GPIO_Get_Level(RTE_GPIO_PORTxx_PINyy);
 @endcode
 *
 * @since 1.0.0
 */
uint8_t RTE_GPIO_Get_Level(uint32_t u32Pin)
{
    uint16_t u16PortIndex;
    uint16_t u16PinIndex;
    uint16_t u16PinBitMask;
    uint8_t  u8Level;
    
    GPIO_TypeDef *pPort;
	
    u16PinIndex  = u32Pin & 0xFFFFU;
    u16PortIndex = (u32Pin >> 16) & 0xFFFFU;
    u8Level = 0U;
    if ((u16PortIndex < 5U) && (u16PinIndex < 16U))
	{
		pPort = g_pstRTEGPIOPortList[u16PortIndex];
		u16PinBitMask = 1U << u16PinIndex;
		
		u8Level = HAL_GPIO_ReadPin(pPort, u16PinBitMask);
	}
	return u8Level;
}

/**
 * @brief 设置指定的一组端口的输出电平
 * @param[in] u8Port 端口组编号
 * @param[in] u32Value 整组端口的输出电平
 * @warning None
 *
 * 示例
 @code
    RTE_GPIO_Set_Port(RTE_GPIO_PORT_GROUP_x, 0xAABBCCDDUL);
 @endcode
 *
 * @since 1.0.0
 */
int32_t RTE_GPIO_Set_Port(uint8_t u8Port, uint32_t u32Value)
{
    int32_t  i32Result;
    GPIO_TypeDef *pPort;

    i32Result = -1;
    if (u8Port < 5U)
	{
        pPort = g_pstRTEGPIOPortList[u8Port];
        HAL_GPIO_WritePin(pPort, GPIO_PIN_All, (GPIO_PinState)u32Value);
        i32Result = 0;
    }
    return i32Result;
}

/**
 * @brief 读取设置指定的一组端口的输入电平
 * @param[in] u8Port 端口组编号
 * @return  整组端口输入电平
 * @warning None
 *
 * 示例
 @code
    uint32_t u32Value = RTE_GPIO_Get_Port(RTE_GPIO_PORT_GROUP_x);
 @endcode
 *
 * @since 1.0.0
 */
uint32_t RTE_GPIO_Get_Port(uint8_t u8Port)
{
    uint32_t u32Value;
    GPIO_TypeDef *pPort;

    u32Value = 0UL;
    if (u8Port < 5U)
	{
        pPort = g_pstRTEGPIOPortList[u8Port];
        u32Value = (uint16_t)(pPort->IDR);
    }
    return u32Value;
}

/**
 * @brief 将指定端口中断方式注册到中断描述符,所有的中断操作都是基于中断描述符的
 * @param[in] pstIRQDesc 中断描述符
 * @param[in] u16Pin 端口编号
 * @param[in] enType 中断类型(上升沿中断,下降沿中断...)
 * @param[in] pfnHandler 中断回调函数,不需要中断回调函数时可以填NULL
 * @return 注册结果,0为注册成功，-1为注册失败 
 * @warning 1.禁止直接改写中断描述符中的数据,更改中断描述符必须通过此函数 \
            2.已使能的中断的描述符不可更改,如需更改应先禁止这一中断 \
            3.受单片机特性限制,并非所有的中断类型都会被支持,不支持的中断类型将返回注册失败
 *
 * 示例
 @code
    int32_t i32Result = RTE_GPIO_Interrupt_Register(&IRQDesc, RTE_GPIO_PORT_GROUP_x, RTE_GPIO_INT_FALLING_EDGE, NULL);
 @endcode
 *
 * @since 1.0.0
 */
int32_t RTE_GPIO_Interrupt_Register(RTE_GPIO_IRQ_Desc_st_t *pstIRQDesc,
                                    uint32_t u32Pin, 
                                    RTE_GPIO_Interrupt_Type_en_t enType,
                                    RTE_GPIO_Int_Handler_Func_ptr_t pfnHandler)
{
    uint32_t i;
    uint16_t u16PortIndex;
    uint16_t u16PinIndex;
    uint32_t u32Available;
    int32_t  i32Result;
    uint8_t u8EXTIType;

    i32Result = -1;
    if (pstIRQDesc != NULL)
    {
        u16PinIndex  = u32Pin & 0xFFFFU;
        u16PortIndex = (u32Pin >> 16) & 0xFFFFU;
        if ((u16PortIndex < 5U) && (u16PinIndex < 16U))
        {
            /* 确定是否支持用户请求的中断类型 */
            if (enType == RTE_GPIO_INT_FALLING_EDGE)
            {
                u8EXTIType = EXTI_TRIGGER_FALLING;
                i32Result  = 0;
            }
            else if (enType == RTE_GPIO_INT_RISING_EDGE)
            {
                u8EXTIType = EXTI_TRIGGER_RISING;
                i32Result  = 0;
            }
            else if (enType == RTE_GPIO_INT_DUAL_EDGE)
            {
                u8EXTIType = EXTI_TRIGGER_RISING_FALLING;
                i32Result  = 0;
            }
            else
            {

            }

            if (i32Result == 0)
            {
                /* 确定用户请求的中断描述符没有被其他中断占用 */
                i = 0UL;
                u32Available = 1UL;
                while ((i < 16UL) && (u32Available != 0UL))
                {
                    if (pstIRQDesc == g_pstRTEGPIOIRQDesc[i])
                    {
                        u32Available = 0UL;
                    }
                    i++;
                }
                
                if (u32Available)
                {
                    /* 用户请求有效,将中断参数注册到当前描述符 */
                    g_pstRTEGPIOIRQDesc[u16PinIndex] = pstIRQDesc;
                    g_pstRTEGPIOIRQDesc[u16PinIndex]->u32Pin    = u32Pin;
                    g_pstRTEGPIOIRQDesc[u16PinIndex]->u16Type   = (uint16_t)u8EXTIType;
                    g_pstRTEGPIOIRQDesc[u16PinIndex]->i32Flag   = 0;
                    g_pstRTEGPIOIRQDesc[u16PinIndex]->pvHandler = (void *)pfnHandler;
                    
                }
                else
                {
                    i32Result = -1;
                }
            }
        }
    }

    return i32Result;
}

/**
 * @brief 使能中断
 * @param[in] pstIRQDesc 中断描述符
 * @return 中断使能结果,0为中断成功使能，-1为中断使能失败，需检查描述符是否正确注册 
 * @warning None
 *
 * 示例
 @code
    int32_t i32Result = RTE_GPIO_Interrupt_Enable(&IRQDesc);
 @endcode
 *
 * @since 1.0.0
 */
int32_t RTE_GPIO_Interrupt_Enable(RTE_GPIO_IRQ_Desc_st_t *pstIRQDesc)
{
    int32_t  i32Result;
    uint16_t u16PortIndex;
    uint16_t u16PinIndex;
   
    IRQn_Type em_IRQnTYPE;
    EXTI_HandleTypeDef stEXTIhandleDef = {0};
    EXTI_ConfigTypeDef stExtiConfigDef = {0};
    
    i32Result = -1;
    if (pstIRQDesc != NULL)
    {
        u16PinIndex  = pstIRQDesc->u32Pin & 0xFFFFU;
        u16PortIndex = (pstIRQDesc->u32Pin >> 16) & 0xFFFFU;
        if ((u16PortIndex < 5U) && (u16PinIndex < 16U))
        {
            if (g_pstRTEGPIOIRQDesc[u16PinIndex] != NULL)    /* 这一路中断是未被使能的 */
            {
                stExtiConfigDef.Line = (EXTI_GPIO | EXTI_REG1 | u16PinIndex);
                stExtiConfigDef.Mode = EXTI_MODE_INTERRUPT;
                stExtiConfigDef.Trigger = g_pstRTEGPIOIRQDesc[u16PinIndex]->u16Type;
                stExtiConfigDef.GPIOSel = g_u32RTEGPIOEXTIPortList[u16PortIndex];
                HAL_EXTI_SetConfigLine(&stEXTIhandleDef, &stExtiConfigDef);
                
                if ((u16PinIndex >= 0U) && (u16PinIndex <= 1U))
                {
                    em_IRQnTYPE =   EXTI0_1_IRQn;         
                }
                else if ((u16PinIndex >= 2U) && (u16PinIndex <= 3U))
                {
                    em_IRQnTYPE =   EXTI2_3_IRQn;              
                }
                else if ((u16PinIndex >= 4U) && (u16PinIndex <= 15U))
                {
                    em_IRQnTYPE =   EXTI4_15_IRQn;             
                }

               /* Enable EXTI interrupt */
                HAL_NVIC_EnableIRQ(em_IRQnTYPE);
               /* Configure interrupt priority */
                HAL_NVIC_SetPriority(em_IRQnTYPE, 0, 0);
                i32Result = 0;
            }
        }
    }

    return i32Result;
}

/**
 * @brief 禁止中断
 * @param[in] pstIRQDesc 中断描述符
 * @return 中断禁止结果,0为中断成功禁止，-1为中断禁止失败，需检查描述符是否正确注册 
 * @warning None
 *
 * 示例
 @code
    int32_t i32Result = RTE_GPIO_Interrupt_Disable(&IRQDesc);
 @endcode
 *
 * @since 1.0.0
 */
int32_t RTE_GPIO_Interrupt_Disable(RTE_GPIO_IRQ_Desc_st_t *pstIRQDesc)
{
    int32_t  i32Result;
    uint16_t u16PortIndex;
    uint16_t u16PinIndex;
    uint16_t u16PinBitMask;
    IRQn_Type em_IRQnTYPE;
    GPIO_TypeDef *pPort;
    
  
    i32Result = -1;
    if (pstIRQDesc != NULL)
    {
        u16PinIndex  = pstIRQDesc->u32Pin & 0xFFFFU;
        u16PortIndex = (pstIRQDesc->u32Pin >> 16) & 0xFFFFU;
        if ((u16PortIndex < 5U) && (u16PinIndex < 16U))
        {
            if (g_pstRTEGPIOIRQDesc[u16PinIndex] == pstIRQDesc)    /* 与当前已使能的中断的描述符相同 */
            {
                g_pstRTEGPIOIRQDesc[u16PinIndex] = NULL;
                pPort = g_pstRTEGPIOPortList[u16PortIndex];
		        u16PinBitMask = 1U << u16PinIndex;
                HAL_GPIO_DeInit(pPort, u16PinBitMask);
                
                if ((u16PinIndex >= 0U) && (u16PinIndex <= 1U))
                {
                    em_IRQnTYPE =   EXTI0_1_IRQn;         
                }
                else if ((u16PinIndex >= 2U) && (u16PinIndex <= 3U))
                {
                    em_IRQnTYPE =   EXTI2_3_IRQn;              
                }
                else if ((u16PinIndex >= 4U) && (u16PinIndex <= 15U))
                {
                    em_IRQnTYPE =   EXTI4_15_IRQn;             
                }
                
                NVIC_DisableIRQ(em_IRQnTYPE);
                i32Result = 0;
            }
        }
    }
    
    return i32Result;
}

/**
 * @brief 获取中断标志
 * @param[in] pstIRQDesc 中断描述符
 * @return 中断标志,0为无中断, 1为中断已发生, -1为获取中断标志失败, 需检查描述符是否正确注册 
 * @warning 1.使用形如if(RTE_GPIO_Get_Interrupt_Flag(&IRQDesc))的方式判断中断是否发生时需注意 \
              如果描述符未正确注册, 函数将返回-1, 也会被判定为真 \
            2.获取中断标志后并不会清除中断标志, 如需清除应调用RTE_GPIO_Clear_Interrupt_Flag函数
 *
 * 示例
 @code
    int32_t i32Flag = RTE_GPIO_Get_Interrupt_Flag(&IRQDesc);
 @endcode
 *
 * @since 1.0.0
 */
int32_t RTE_GPIO_Get_Interrupt_Flag(RTE_GPIO_IRQ_Desc_st_t *pstIRQDesc)
{
    int32_t  i32Result;

    i32Result = -1;
    if (pstIRQDesc != NULL)
    {
        i32Result = pstIRQDesc->i32Flag;
    }

    return i32Result;
}

/**
 * @brief 清除中断标志
 * @param[in] pstIRQDesc 中断描述符
 * @return 中断标志清除结果, 0为中断标志成功清除，-1为中断标志清除失败，需检查描述符是否正确注册 
 * @warning None
 *
 * 示例
 @code
    int32_t i32Result = RTE_GPIO_Clear_Interrupt_Flag(&IRQDesc);
 @endcode
 *
 * @since 1.0.0
 */
int32_t RTE_GPIO_Clear_Interrupt_Flag(RTE_GPIO_IRQ_Desc_st_t *pstIRQDesc)
{
    int32_t  i32Result;

    i32Result = -1;
    if (pstIRQDesc != NULL)
    {
        pstIRQDesc->i32Flag = 0;
        i32Result = 0;
    }

    return i32Result;
}

/**
 * @brief 外部中断0-1中断服务
 * @warning None
 * @since 1.0.0
 */
void EXTI0_1_IRQHandler(void)
{
    uint8_t  u8Level[2];
    uint32_t u32IntFlag;
    uint32_t u32IntMask;
    uint32_t u32Index;

    RTE_GPIO_IRQ_Desc_st_t         *pstDesc;
    RTE_GPIO_Int_Handler_Func_ptr_t pfnIRQHandler;

    /* 清除中断标志 */
    
   u32IntFlag  = EXTI->PR;
   u32IntFlag &= 0x00000003UL;
   EXTI->PR  = u32IntFlag;

   
    /* 读取中断发生时端口的电平 */
    u32Index = 0UL;
    u32IntMask = 0x00000001UL;
    while(u32Index < 2UL)
    {
        if (((u32IntFlag & u32IntMask) != 0UL) && (g_pstRTEGPIOIRQDesc[u32Index] != NULL))
        {
            pstDesc = g_pstRTEGPIOIRQDesc[u32Index];
            u8Level[u32Index] = RTE_GPIO_Get_Level(pstDesc->u32Pin);
            pstDesc->i32Flag = 1;
        }

        u32Index++;
        u32IntMask <<= 1;
    }

    /* 执行注册的回调函数 */
 
    u32Index = 0UL;
    u32IntMask = 0x00000001UL;
    while(u32Index < 2UL)
    {
        if (((u32IntFlag & u32IntMask) != 0UL) && (g_pstRTEGPIOIRQDesc[u32Index] != NULL))
        {
            pstDesc = g_pstRTEGPIOIRQDesc[u32Index];
            if (pstDesc->pvHandler != NULL)
            {
                pfnIRQHandler = (RTE_GPIO_Int_Handler_Func_ptr_t)(pstDesc->pvHandler);
                pfnIRQHandler(pstDesc->u32Pin, u8Level[u32Index]);
            }
        }

        u32Index++;
        u32IntMask <<= 1;
    }
}


/**
 * @brief 外部中断2-3中断服务
 * @warning None
 * @since 1.0.0
 */
void EXTI2_3_IRQHandler(void)
{
    uint8_t  u8Level[2];
    uint32_t u32IntFlag;
    uint32_t u32IntMask;
    uint32_t u32Index;

    RTE_GPIO_IRQ_Desc_st_t         *pstDesc;
    RTE_GPIO_Int_Handler_Func_ptr_t pfnIRQHandler;

    /* 清除中断标志 */
    
    u32IntFlag  = EXTI->PR;
    u32IntFlag &= 0x0000000CUL;
    EXTI->PR  = u32IntFlag;

    /* 读取中断发生时端口的电平 */
    u32Index = 2UL;
    u32IntMask = 0x00000004UL;
    while(u32Index < 4UL)
    {
        if (((u32IntFlag & u32IntMask) != 0UL) && (g_pstRTEGPIOIRQDesc[u32Index] != NULL))
        {
            pstDesc = g_pstRTEGPIOIRQDesc[u32Index];
            u8Level[u32Index] = RTE_GPIO_Get_Level(pstDesc->u32Pin);
            pstDesc->i32Flag = 1;
        }

        u32Index++;
        u32IntMask <<= 1;
    }

    /* 执行注册的回调函数 */
    u32Index = 2UL;
    u32IntMask = 0x00000004UL;
    while(u32Index < 4UL)
    {
        if (((u32IntFlag & u32IntMask) != 0UL) && (g_pstRTEGPIOIRQDesc[u32Index] != NULL))
        {
            pstDesc = g_pstRTEGPIOIRQDesc[u32Index];
            if (pstDesc->pvHandler != NULL)
            {
                pfnIRQHandler = (RTE_GPIO_Int_Handler_Func_ptr_t)(pstDesc->pvHandler);
                pfnIRQHandler(pstDesc->u32Pin, u8Level[u32Index - 2UL]);
            }
        }
        u32Index++;
        u32IntMask <<= 1;
    }
}

/**
 * @brief 外部中断4-15中断服务
 * @warning None
 * @since 1.0.0
 */
void EXTI4_15_IRQHandler(void)
{
    uint8_t  u8Level[12];
    uint32_t u32IntFlag;
    uint32_t u32IntMask;
    uint32_t u32Index;

    RTE_GPIO_IRQ_Desc_st_t         *pstDesc;
    RTE_GPIO_Int_Handler_Func_ptr_t pfnIRQHandler;

    /* 清除中断标志 */
    
    u32IntFlag  = EXTI->PR;
    u32IntFlag &= 0x0000FFF0UL;
    EXTI->PR  = u32IntFlag;

    /* 读取中断发生时端口的电平 */
    u32Index = 4UL;
    u32IntMask = 0x00000010UL;
    while(u32Index < 16UL)
    {
        if (((u32IntFlag & u32IntMask) != 0UL) && (g_pstRTEGPIOIRQDesc[u32Index] != NULL))
        {
            pstDesc = g_pstRTEGPIOIRQDesc[u32Index];
            u8Level[u32Index] = RTE_GPIO_Get_Level(pstDesc->u32Pin);
            pstDesc->i32Flag = 1;
        }

        u32Index++;
        u32IntMask <<= 1;
    }

    /* 执行注册的回调函数 */
    u32Index = 4UL;
    u32IntMask = 0x00000010UL;
    while(u32Index < 16UL)

    {
        if (((u32IntFlag & u32IntMask) != 0UL) && (g_pstRTEGPIOIRQDesc[u32Index] != NULL))
        {
            pstDesc = g_pstRTEGPIOIRQDesc[u32Index];
            if (pstDesc->pvHandler != NULL)
            {
                pfnIRQHandler = (RTE_GPIO_Int_Handler_Func_ptr_t)(pstDesc->pvHandler);
                pfnIRQHandler(pstDesc->u32Pin, u8Level[u32Index - 4UL]);
            }
        }
        u32Index++;
        u32IntMask <<= 1;
    }
}
