#include <stdio.h>
#include <string.h>
#include "nvs_flash.h"
#include "esp_event.h"
#include "esp_mac.h"
#include "esp_netif.h"
#include "esp_log.h"
#include "softap.h"
#include "wifi_service.h"

static const char *TAG = "softAP";
esp_netif_t *softap_netif = NULL;
wifi_sta_list_t softap_sta_list;
unsigned char softap_sta_connect = STA_CONNECTED_NEVER;

static void softap_event_handler(void *event_handler_arg, esp_event_base_t event_base, int32_t event_id, void *event_data)
{
    if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_AP_STACONNECTED)
    {
        wifi_event_ap_staconnected_t *event = (wifi_event_ap_staconnected_t *)event_data;
        ESP_LOGI(TAG, "---- 设备加入网络, MAC地址 :" MACSTR ", 编号(AID) = %d.", MAC2STR(event->mac), event->aid);
        softap_sta_connect = STA_CONNECTED_ONCE;
        connect_handler(event_handler_arg);
        wifi_service_stop_timer_clear();
    }
    if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_AP_STADISCONNECTED)
    {
        wifi_event_ap_stadisconnected_t *event = (wifi_event_ap_stadisconnected_t *)event_data;
        ESP_LOGI(TAG, "---- 设备退出网络, MAC地址 :" MACSTR ", 编号(AID) = %d.", MAC2STR(event->mac), event->aid);
        disconnect_handler(event_handler_arg);
        wifi_service_stop_timer_clear();
    }
}

void softap_service_start(void)
{
    // NVS分区初始化
    ESP_LOGI(TAG, "---- nvs分区初始化...");
    esp_err_t ret = nvs_flash_init();
    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND)
    {
        ESP_ERROR_CHECK(nvs_flash_erase());
        ret = nvs_flash_init();
    }
    ESP_ERROR_CHECK(ret);

    // 创建系统事件循环
    ESP_LOGI(TAG, "---- 创建系统事件循环...");
    ESP_ERROR_CHECK(esp_event_loop_create_default());

    // 将WiFi的所有事件作为实例注册进系统事件循环中
    ESP_LOGI(TAG, "---- 注册WiFi事件实例...");
    ESP_ERROR_CHECK(esp_event_handler_register(WIFI_EVENT, ESP_EVENT_ANY_ID, softap_event_handler, &http_server));

    // 初始化网卡相关的底层配置
    ESP_LOGI(TAG, "---- 网卡初始化...");
    ESP_ERROR_CHECK(esp_netif_init());

    // 以默认的方式创建一个ap类型的网卡
    ESP_LOGI(TAG, "---- 创建AP类型网卡...");
    softap_netif = esp_netif_create_default_wifi_ap();

    // 初始化WiFi底层配置
    ESP_LOGI(TAG, "---- WiFi初始化...");
    wifi_init_config_t cfg = WIFI_INIT_CONFIG_DEFAULT();
    ESP_ERROR_CHECK(esp_wifi_init(&cfg));

    // 将WiFi设置成AP模式
    ESP_LOGI(TAG, "---- 配置WiFi为AP模式...");
    ESP_ERROR_CHECK(esp_wifi_set_mode(WIFI_MODE_AP));

    // 创建AP配置
    ESP_LOGI(TAG, "---- 创建AP配置...");
    wifi_config_t wifi_config = {
        .ap = {
            .ssid = ESP_WIFI_SSID,
            .ssid_len = strlen(ESP_WIFI_SSID),
            .password = ESP_WIFI_PASSWD,
            .max_connection = 1,
            .authmode = WIFI_AUTH_WPA_WPA2_PSK}};
    // 获取当前AP的MAC地址
    unsigned char softap_mac[6];
    esp_wifi_get_mac(WIFI_IF_AP, softap_mac);
    // 将MAC地址后两个字节转为字符串
    char softap_temp[5];
    //sprintf(softap_temp, "%02X%02X", softap_mac[4], softap_mac[5]);
    // 将MAC地址拼接到 ESP_WIFI_SSID 后面
    char softap_ssid[32] = ESP_WIFI_SSID;
    strcat(softap_ssid, softap_temp);
    // 重新配置 ssid 和 ssid_len
    memcpy(wifi_config.ap.ssid, softap_ssid, sizeof(softap_ssid));
    wifi_config.ap.ssid_len = strlen(softap_ssid);
    // 配置AP的属性
    ESP_ERROR_CHECK(esp_wifi_set_config(WIFI_IF_AP, &wifi_config));

    // 开启WiFi
    ESP_LOGI(TAG, "---- 开启WiFi, 名称 : %s 密码 : %s", wifi_config.ap.ssid, wifi_config.ap.password);
    ESP_ERROR_CHECK(esp_wifi_start());

    // 首次开启http_server
    ESP_LOGI(TAG, "---- 首次开启HTTP服务器...");
    http_server = http_service();

    // 启动服务关闭计时器
    ESP_LOGI(TAG, "---- 启动服务关闭计时器...");
    wifi_service_stop_timer_clear();
}

void softap_service_stop(void)
{
    // 关闭http_server
    ESP_LOGI(TAG, "---- 关闭HTTP服务器...");
    disconnect_handler(&http_server);

    // 关闭WiFi
    ESP_LOGI(TAG, "---- 关闭WiFi...");
    ESP_ERROR_CHECK(esp_wifi_stop());

    // WiFi反初始化
    ESP_LOGI(TAG, "---- WiFi反初始化...");
    ESP_ERROR_CHECK(esp_wifi_deinit());

    // 销毁网卡实例
    ESP_LOGI(TAG, "---- 销毁网卡实例...");
    esp_netif_destroy_default_wifi(softap_netif);

    // 网卡反初始化 - 官方未实现的功能
    // ESP_LOGI(TAG, "---- 网卡反初始化...");
    // ESP_ERROR_CHECK(esp_netif_deinit());

    // 注销WiFi事件实例
    ESP_LOGI(TAG, "---- 注销WiFi事件实例...");
    ESP_ERROR_CHECK(esp_event_handler_unregister(WIFI_EVENT, ESP_EVENT_ANY_ID, softap_event_handler));

    // 删除系统事件循环
    ESP_LOGI(TAG, "---- 删除系统事件循环...");
    ESP_ERROR_CHECK(esp_event_loop_delete_default());

    // NVS分区反初始化
    ESP_LOGI(TAG, "---- NVS分区反初始化...");
    ESP_ERROR_CHECK(nvs_flash_deinit());

    ESP_LOGI(TAG, "---- WiFi服务已关闭, 重置WiFi参数");
    wifi_service_init();
}
