package com.tyw.colortest.tools;


import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;

import com.tyw.colortest.bean.UsbConfig;
import com.tyw.colortest.bean.UsbConfigXlsxBean;
import com.tyw.colortest.constance.AppConfig; // 【新增】import
import com.tyw.colortest.constance.LHStyle; // 【新增】import

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class SpUtils {
    public static final String KEY_APP_CONFIG = "app_config";
    public static final String KEY_LH_STYLE = "lh_style";
    public static final String KEY_LH_TIME = "lh_time";
    private static final String prefsName="my_preferences";
    private static final String PREFS_APP_GENERAL = "AppGeneralPrefs";

    // --- 旧的 Key ---
    public static final String KEY_PRODUCTION_VERSION = "version";
    public static final String KEY_PRODUCTION_ID = "报文ID";

    // --- 【新增】为每个图片目录定义的 Key ---
    public static final String KEY_SPEC_PATHS = "spec_image_paths";
    public static final String KEY_LAOHUA_PATHS = "laohua_image_paths";
    public static final String KEY_JIANCETAI_PATHS = "jiancetai_image_paths";
    public static final String KEY_GAMMA_PATHS = "gamma_image_paths";
    public static final String KEY_FLIKER_PATHS = "fliker_image_paths";
    public static final String KEY_BW_PATHS = "black_white_image_paths";
    public static final String KEY_COLOR_PATHS = "color_image_paths";
    public static final String KEY_CR_PATHS = "cr_image_paths";

    public static final String KEY_TIEHE_PATHS = "tiehe_image_paths"; // 【新增】


    private static SharedPreferences getPrefs(Context context) {
        return context.getSharedPreferences(PREFS_APP_GENERAL, Context.MODE_PRIVATE);
    }

    // --- 旧的方法 (保持不变) ---
    public static void setProductionVersion(Context context, String version) {
        getPrefs(context).edit().putString(KEY_PRODUCTION_VERSION, version).apply();
    }

    public static String getProductionVersion(Context context, String defaultValue) {
        return getPrefs(context).getString(KEY_PRODUCTION_VERSION, defaultValue);
    }

    public static void setProductionID(Context context, String version) {
        getPrefs(context).edit().putString(KEY_PRODUCTION_ID, version).apply();
    }

    public static String getProductionID(Context context, String defaultValue) {
        return getPrefs(context).getString(KEY_PRODUCTION_ID, defaultValue);
    }

    public static void setAppConfig(Context context, AppConfig config) {
        if (config == null) return;
        getPrefs(context).edit().putString(KEY_APP_CONFIG, config.name()).apply();
    }

    /**
     * 读取 AppConfig (工装类型)
     * @param defaultValue 如果SP中没有，则返回此默认值
     */
    public static AppConfig getAppConfig(Context context, AppConfig defaultValue) {
        String name = getPrefs(context).getString(KEY_APP_CONFIG, null);
        if (name == null) {
            return defaultValue;
        }
        try {
            return AppConfig.valueOf(name);
        } catch (IllegalArgumentException e) {
            // 如果SP中存的值无效 (例如，枚举名已更改)，返回默认值
            return defaultValue;
        }
    }

    /**
     * 保存 LHStyle (老化类型)
     */
    public static void setLhStyle(Context context, LHStyle style) {
        if (style == null) return;
        getPrefs(context).edit().putString(KEY_LH_STYLE, style.name()).apply();
    }

    /**

     * 读取 LHStyle (老化类型)
     */
    public static LHStyle getLhStyle(Context context, LHStyle defaultValue) {
        String name = getPrefs(context).getString(KEY_LH_STYLE, null);
        if (name == null) {
            return defaultValue;
        }
        try {
            return LHStyle.valueOf(name);
        } catch (IllegalArgumentException e) {
            return defaultValue;
        }
    }

    /**
     * 保存老化时间
     */
    public static void setLhTime(Context context, int time) {
        getPrefs(context).edit().putInt(KEY_LH_TIME, time).apply();
    }

    /**
     * 读取老化时间
     */
    public static int getLhTime(Context context, int defaultValue) {
        return getPrefs(context).getInt(KEY_LH_TIME, defaultValue);
    }

    // --- 【新增】缓存 UsbConfig 对象的 Key ---
    public static final String KEY_USB_CONFIG_CACHE = "usb_config_cache";

    /**
     * 【新增】将 UsbConfig 对象序列化为 JSON 字符串并保存到 SharedPreferences。
     * @param context Context
     * @param config  要缓存的 UsbConfig 对象。如果为 null，则会清除缓存。
     */
    public static void saveUsbConfig(Context context, UsbConfigXlsxBean config) {
        SharedPreferences.Editor editor = getPrefs(context).edit();
        if (config == null) {
            // 如果传入 null，则移除缓存
            editor.remove(KEY_USB_CONFIG_CACHE).apply();
            Log.d("SpUtils", "UsbConfig cache cleared.");
        } else {
            // 使用 Gson 将对象转换为 JSON 字符串
            com.google.gson.Gson gson = new com.google.gson.Gson();
            String jsonString = gson.toJson(config);
            editor.putString(KEY_USB_CONFIG_CACHE, jsonString).apply();
            Log.d("SpUtils", "UsbConfig cached successfully.");
        }
    }

    /**
     * 【新增】从 SharedPreferences 读取 JSON 字符串并反序列化为 UsbConfig 对象。
     * @param context Context
     * @return 返回缓存的 UsbConfig 对象。如果没有缓存或解析失败，则返回 null。
     */
    public static UsbConfigXlsxBean getUsbConfig(Context context) {
        String jsonString = getPrefs(context).getString(KEY_USB_CONFIG_CACHE, null);
        if (jsonString == null || jsonString.isEmpty()) {
            Log.d("SpUtils", "No UsbConfig cache found.");
            return null; // 没有缓存
        }

        try {
            // 使用 Gson 将 JSON 字符串转换回对象
            com.google.gson.Gson gson = new com.google.gson.Gson();
            UsbConfigXlsxBean config = gson.fromJson(jsonString, UsbConfigXlsxBean.class);
            Log.d("SpUtils", "UsbConfig loaded from cache successfully.");
            return config;
        } catch (com.google.gson.JsonSyntaxException e) {
            Log.e("SpUtils", "Failed to parse cached UsbConfig JSON.", e);
            // 如果JSON损坏，清除错误的缓存
            getPrefs(context).edit().remove(KEY_USB_CONFIG_CACHE).apply();
            return null; // 解析失败
        }
    }
    // 【新增】定义一个不可能出现在文件名中的特殊分隔符
    private static final String PATH_SEPARATOR = "##_##";

    /**
     * 【已重构】将一个字符串数组（图片路径）保存到 SharedPreferences。
     * 这个版本会先对数组进行排序，然后拼接成一个字符串来保证顺序。
     * @param context Context
     * @param key     用于保存的 Key
     * @param paths   要保存的图片路径数组
     */
    public static void saveImagePathSet(Context context, String key, String[] paths) {
        SharedPreferences.Editor editor = getPrefs(context).edit();
        if (paths == null || paths.length == 0) {
            editor.remove(key).apply();
            return;
        }

        // --- 核心修改 ---

        // 1. 将数组转换为可排序的列表
        List<String> pathList = new ArrayList<>(Arrays.asList(paths));

        // 2. 定义一个能提取数字并进行排序的比较器
        Comparator<String> numericalComparator = new Comparator<String>() {
            private final Pattern pattern = Pattern.compile("(\\d+)"); // 正则表达式，用于找到文件名中的数字

            @Override
            public int compare(String s1, String s2) {
                return Integer.compare(extractNumber(s1), extractNumber(s2));
            }

            private int extractNumber(String s) {
                // 从完整路径中提取文件名
                String fileName = new File(s).getName();
                Matcher matcher = pattern.matcher(fileName);
                if (matcher.find()) {
                    try {
                        // 返回找到的第一个数字序列
                        return Integer.parseInt(matcher.group(1));
                    } catch (NumberFormatException e) {
                        // 忽略解析失败的情况
                    }
                }
                // 如果文件名中没有数字，则返回0，或者可以根据需要进行其他处理
                return 0;
            }
        };

        // 3. 对列表进行排序
        Collections.sort(pathList, numericalComparator);

        // 4. 使用特殊分隔符将排序后的列表拼接成一个字符串
        // 在现代Java (API 26+) 中，可以使用 String.join
        // 为了更好的兼容性，我们使用 StringBuilder
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < pathList.size(); i++) {
            sb.append(pathList.get(i));
            if (i < pathList.size() - 1) {
                sb.append(PATH_SEPARATOR);
            }
        }

        // 5. 将拼接好的字符串存入 SharedPreferences
        editor.putString(key, sb.toString()).apply();
    }

    /**
     * 【已重构】从 SharedPreferences 中读取图片路径数组。
     * 这个版本会读取拼接的字符串，并按分隔符拆分回数组，从而恢复顺序。
     * @param context Context
     * @param key     用于读取的 Key
     * @return 返回一个已排序的字符串数组。如果未找到，则返回空数组。
     */
    public static String[] getImagePathArray(Context context, String key) {
        String concatenatedPaths = getPrefs(context).getString(key, null);
        if (concatenatedPaths == null || concatenatedPaths.isEmpty()) {
            return new String[0];
        }

        // 使用相同的分隔符将字符串拆分回数组
        return concatenatedPaths.split(PATH_SEPARATOR);
    }
}