def calculate_crc8(data: bytes) -> int:
    """
    根据给定的参数计算CRC-8/DVB-S2校验和。
    - Polynomial: 0x1D
    - Initial Value: 0xFF
    - No Refin, No Refout, No Xorout
    """
    crc_value = 0xFF
    crc_poly = 0x1D
    for byte in data:
        crc_value ^= byte
        for _ in range(8):
            if (crc_value & 0x80) != 0:
                # 如果最高位是1，左移一位并与多项式异或
                crc_value = ((crc_value << 1) ^ crc_poly) & 0xFF
            else:
                # 如果最高位是0，仅左移一位
                crc_value = (crc_value << 1) & 0xFF
    return crc_value

def parse_hex_string(hex_string: str) -> bytes or None:
    """
    将空格分隔的十六进制字符串（如 '0x41 0x23'）解析为bytes对象。
    """
    try:
        # 去除首尾空格，并按空格分割
        parts = hex_string.strip().split()
        # 将每个部分从16进制字符串转换为整数，然后创建bytes对象
        return bytes([int(part, 16) for part in parts])
    except (ValueError, TypeError):
        print(f"错误: 无法解析输入的十六进制字符串。请确保格式正确。")
        return None

def verify_data_with_crc(full_data: bytes) -> bool:
    """
    验证一个完整数据帧的CRC校验和是否正确。
    数据帧的最后一位被视为期望的CRC值。
    """
    if not full_data or len(full_data) < 2:
        print("错误: 数据太短，无法进行CRC校验。")
        return False

    # 分离数据和期望的CRC
    data_to_check = full_data[:-1]  # 所有字节，除了最后一个
    expected_crc = full_data[-1]   # 最后一个字节

    # 计算CRC
    calculated_crc = calculate_crc8(data_to_check)

    # 打印验证过程信息
    print("-" * 40)
    print(f"待计算的数据 ({len(data_to_check)} bytes):")
    # 为了方便查看，每16个字节换一行
    for i in range(0, len(data_to_check), 16):
         print(" ".join(f"{b:02X}" for b in data_to_check[i:i+16]))

    print("\n--- 校验结果 ---")
    print(f"期望的 CRC 值: 0x{expected_crc:02X}")
    print(f"计算出的 CRC 值: 0x{calculated_crc:02X}")
    print("-" * 18)

    # 比较并返回结果
    if calculated_crc == expected_crc:
        print("✅ CRC 校验通过! 数据正确。")
        return True
    else:
        print("❌ CRC 校验失败! 数据可能已损坏。")
        return False

# 主程序入口
if __name__ == '__main__':
    # 您可以直接在这里粘贴从设备或日志中复制的完整数据字符串
    input_hex_string = "0x41 0x23 0x00 0x05 0x01 0x40 0x03 0xCA 0x02 0xBB 0x02 0x40 0x02 0x84 0x01 0x4A 0x03 0x40 0x02 0xE6 0x02 0x6D 0x04 0x40 0x04 0xC4 0x01 0xBC 0x05 0x40 0x02 0x94 0x02 0x0A 0xBA"

    # 1. 解析字符串为字节数据
    full_data_bytes = parse_hex_string(input_hex_string)

    # 2. 如果解析成功，则进行CRC校验
    if full_data_bytes:
        verify_data_with_crc(full_data_bytes)