package com.tyw.colortest.tools;

import java.util.Arrays;

public class ProtocolHelper {

    /**
     * 【最终正确版】构建 "MFD backlight mode" (0x83) 的指令帧。
     * 此版本基于已验证的实际数据编写，正确实现了此命令独特的Length定义和帧结构。
     *
     * @param aliveCount  那个0-15循环的计数器，它是data载荷的一部分
     * @param backlightOn 背光是否开启 (true为开, false为关)
     * @param brightness  背光亮度 (0-100)
     * @return 完整的、正确的10字节指令数组。
     */
    public static byte[] buildMfdBacklightModeCommand ( int brightness) {

        // --- 1. 根据验证过的数据，定义完整的10字节帧结构 ---
        final int TOTAL_FRAME_LENGTH = 10;
        byte[] commandFrame = new byte[TOTAL_FRAME_LENGTH];

        // Byte 0: 命令ID
        commandFrame[0] = (byte) 0x83;

        // Byte 1: Length (值为整个帧的总长度)
        commandFrame[1] = (byte) TOTAL_FRAME_LENGTH; // 即 0x0A

        // Byte 2: 通用协议中定义的、固定的Alive Counter预留位
        commandFrame[2] = 0x00;

        // Bytes 3-8: 0x83命令专属的6字节data载荷
        commandFrame[3] =0x00; // data[0]: 0-15循环的Alive_Count
        commandFrame[4] = 0x00;                       // data[1]: E2E状态
        commandFrame[5] = 0x01;
        commandFrame[6] = (byte) brightness;          // data[3]: 亮度
        commandFrame[7] = 0x00;                       // data[4]: reserved
        // 注意：根据验证数据，此预留位是0x10，而不是文档中的0x00
        commandFrame[8] = 0x10;                       // data[5]: reserved

        // --- 2. 计算CRC ---
        // 对除最后一个CRC字节外的所有字节进行计算
        byte[] dataForCrc = Arrays.copyOf(commandFrame, TOTAL_FRAME_LENGTH - 1);
        byte crc8 = calculateCrc8(dataForCrc);

        // --- 3. 将CRC值放入最后一个字节 ---
        commandFrame[9] = crc8;

        return commandFrame;
    }

    /**
     * 根据协议文档，为 "CSC startup done" (0x80) 命令构建完整的指令帧。
     * @param brightness     Backlight brightness (0-100)
     * @return 完整的指令字节数组，可以直接在I2C总线上发送 (不含地址)。
     */
    public static byte[] buildCscStartupDoneCommand( int brightness) {
        // --- 1. 定义固定参数和载荷长度 ---
        final byte CMD_ID = (byte) 0x80;
        final byte ALIVE_COUNTER = 0x00;
        final int DATA_PAYLOAD_LENGTH = 8; // data[n] 的长度为8字节

        // --- 2. 计算Length字段 ---
        // Length = Alive Counter的长度(1) + data的长度(8) = 9
        final byte LENGTH = 0x0B;

        // --- 3. 构建 data[n] 载荷 ---
        byte[] dataPayload = new byte[DATA_PAYLOAD_LENGTH];
        dataPayload[0] = 0x01;    // Byte1: CSC startup result
        dataPayload[1] = 0x01;         // Byte2: MFD backlight work mode
        dataPayload[2] = (byte) brightness;       // Byte3: Backlight brightness
        dataPayload[3] = 0x00;   // Byte4: Rear screen info
        dataPayload[4] = 0x00;                    // Byte5: reserved
        dataPayload[5] = 0x00;                    // Byte6: reserved
        dataPayload[6] = 0x00;                    // Byte7: reserved
        dataPayload[7] = 0x00;                    // Byte8: reserved

        // --- 4. 准备用于CRC计算的数据 ---
        // 范围: CMD_ID, Length, Alive Counter, data[n]
        byte[] dataForCrc = new byte[1 + 1 + 1 + DATA_PAYLOAD_LENGTH]; // 总共11字节
        dataForCrc[0] = CMD_ID;
        dataForCrc[1] = LENGTH;
        dataForCrc[2] = ALIVE_COUNTER;
        System.arraycopy(dataPayload, 0, dataForCrc, 3, DATA_PAYLOAD_LENGTH);

        // --- 5. 计算CRC ---
        byte crc8 = calculateCrc8(dataForCrc);

        // --- 6. 组装最终的完整指令帧 ---
        // 顺序: CMD_ID, Length, Alive Counter, data[n], Crc8
        byte[] commandFrame = new byte[1 + 1 + 1 + DATA_PAYLOAD_LENGTH + 1]; // 总共12字节
        System.arraycopy(dataForCrc, 0, commandFrame, 0, dataForCrc.length);
        commandFrame[commandFrame.length - 1] = crc8; // 将CRC放在最后

        return commandFrame;
    }

    /**
     * 【全新】构建 "获取版本号" (0x81) 的指令帧。
     *
     * @return 用于获取版本号的完整指令字节数组。
     */
    public static byte[] buildGetVersionCommand_IncorrectLogic(byte pay) {
        // --- 1. 定义指令参数 ---
        final byte CMD_ID = (byte) 0x81;
        final byte ALIVE_COUNTER = 0x00;
        final byte[] dataPayload = { pay }; // 请求版本号

        // --- 2. 【按字面错误理解】计算Length字段 ---
        // 注释: Length = 包含Length+CMD ID+Data(n)+ALIVE_Count+CRC8
        // 我们计算这些字段的数量总和：
        // Length字段(1) + CMD_ID(1) + Alive Counter(1) + data(1) + Crc8字段(1) = 5
        final byte INCORRECT_LENGTH = 5; // 即 0x05

        // --- 3. 准备用于CRC计算的数据 ---
        // CRC计算范围依然是 CMD_ID, Length, Alive Counter, data[n]
        // 我们将使用上面计算出的 INCORRECT_LENGTH (0x05) 作为Length的值参与计算
        byte[] dataForCrc = new byte[] {
                CMD_ID,            // 0x81
                INCORRECT_LENGTH,  // 0x05
                ALIVE_COUNTER,     // 0x00
                dataPayload[0]     // 0x48
        };

        // --- 4. 计算CRC ---
        byte crc8 = calculateCrc8(dataForCrc);

        // --- 5. 组装最终的完整指令帧 ---
        // 顺序: CMD_ID, Length, Alive Counter, data[n], Crc8 (共5字节)
        byte[] commandFrame = new byte[] {
                CMD_ID,
                INCORRECT_LENGTH,
                ALIVE_COUNTER,
                dataPayload[0],
                crc8
        };

        return commandFrame;
    }
    /**
     * 根据提供的C代码翻译的CRC-8计算方法。
     * 算法参数: Poly=0x1D, Init=0xFF
     */
    public static byte calculateCrc8(byte[] data) {
        int crcValue = 0xFF;
        final int crcPoly = 0x1D;
        for (byte b : data) {
            crcValue ^= (b & 0xFF);
            for (int i = 0; i < 8; i++) {
                if ((crcValue & 0x80) != 0) {
                    crcValue = ((crcValue << 1) ^ crcPoly) & 0xFF;
                } else {
                    crcValue = (crcValue << 1) & 0xFF;
                }
            }
        }
        return (byte) crcValue;
    }

    // 辅助方法：将字节数组转换为十六进制字符串以便查看
    public static String bytesToHexString(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        for (byte b : bytes) {
            sb.append(String.format("0x%02X ", b));
        }
        return sb.toString().trim();
    }
}