package com.tyw.colortest.tools;

import android.graphics.Color;
import android.text.Html;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.text.style.AbsoluteSizeSpan;
import android.text.style.ForegroundColorSpan;
import android.text.style.StyleSpan;
import android.graphics.Typeface;
import android.widget.TextView;
import com.tyw.colortest.bean.InfoDataBean;
import java.util.List;

/**
 * HTML文本工具类
 * 用于将InfoDataBean数据转换为HTML格式并显示在TextView上
 */
public class HtmlTextUtils {

    /**
     * 将InfoDataBean列表转换为HTML字符串
     * @param dataList InfoDataBean列表
     * @return HTML格式的字符串
     */
    public static String convertToHtml(List<InfoDataBean> dataList) {
        if (dataList == null || dataList.isEmpty()) {
            return "";
        }

        StringBuilder htmlBuilder = new StringBuilder();
        
        for (InfoDataBean item : dataList) {
            // 开始构建HTML标签
            htmlBuilder.append("<font");
            
            // 添加颜色属性
            if (item.getColor() != null && !item.getColor().isEmpty()) {
                htmlBuilder.append(" color=\"").append(item.getColor()).append("\"");
            }
            
            htmlBuilder.append(">");
            
            // 处理字体大小 - 使用big/small标签来模拟大小变化
            int bigCount = 0;
            int smallCount = 0;
            
            if (item.getSize() > 0) {
                if (item.getSize() >= 24) {
                    // 大字体：使用多个big标签
                    bigCount = Math.min(3, (item.getSize() - 16) / 8); // 最多3个big标签
                } else if (item.getSize() <= 12) {
                    // 小字体：使用small标签
                    smallCount = Math.min(2, (16 - item.getSize()) / 4); // 最多2个small标签
                }
            }
            
            // 添加big标签
            for (int i = 0; i < bigCount; i++) {
                htmlBuilder.append("<big>");
            }
            
            // 添加small标签
            for (int i = 0; i < smallCount; i++) {
                htmlBuilder.append("<small>");
            }
            
            // 如果需要加粗，添加<b>标签
            if (item.isBold()) {
                htmlBuilder.append("<b>");
            }
            
            // 添加文本内容，处理换行符
            String text = item.getText();
            if (text != null) {
                // 将\n转换为<br>
                text = text.replace("\n", "<br>");
                htmlBuilder.append(text);
            }
            
            // 关闭加粗标签
            if (item.isBold()) {
                htmlBuilder.append("</b>");
            }
            
            // 关闭small标签
            for (int i = 0; i < smallCount; i++) {
                htmlBuilder.append("</small>");
            }
            
            // 关闭big标签
            for (int i = 0; i < bigCount; i++) {
                htmlBuilder.append("</big>");
            }
            
            // 关闭font标签
            htmlBuilder.append("</font>");
        }
        
        return htmlBuilder.toString();
    }

    /**
     * 将InfoDataBean列表转换为SpannableStringBuilder（更精确的样式控制）
     * @param dataList InfoDataBean列表
     * @return SpannableStringBuilder对象
     */
    public static SpannableStringBuilder convertToSpannable(List<InfoDataBean> dataList) {
        SpannableStringBuilder builder = new SpannableStringBuilder();
        
        if (dataList == null || dataList.isEmpty()) {
            return builder;
        }
        
        for (InfoDataBean item : dataList) {
            String text = item.getText();
            if (text == null) {
                continue;
            }
            
            int start = builder.length();
            builder.append(text);
            int end = builder.length();
            
            // 设置颜色
            if (item.getColor() != null && !item.getColor().isEmpty()) {
                try {
                    int color = Color.parseColor(item.getColor());
                    builder.setSpan(new ForegroundColorSpan(color), start, end, Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
                } catch (IllegalArgumentException e) {
                    // 颜色解析失败，忽略
                }
            }
            
            // 设置字体大小
            if (item.getSize() > 0) {
                builder.setSpan(new AbsoluteSizeSpan(item.getSize(), true), start, end, Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
            }
            
            // 设置加粗
            if (item.isBold()) {
                builder.setSpan(new StyleSpan(Typeface.BOLD), start, end, Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
            }
        }
        
        return builder;
    }

    /**
     * 将InfoDataBean列表转换为更现代的HTML字符串（使用span标签和CSS样式）
     * @param dataList InfoDataBean列表
     * @return HTML格式的字符串
     */
    public static String convertToModernHtml(List<InfoDataBean> dataList) {
        if (dataList == null || dataList.isEmpty()) {
            return "";
        }

        StringBuilder htmlBuilder = new StringBuilder();
        
        for (InfoDataBean item : dataList) {
            // 使用span标签和style属性
            htmlBuilder.append("<span style=\"");
            
            // 添加颜色样式
            if (item.getColor() != null && !item.getColor().isEmpty()) {
                htmlBuilder.append("color:").append(item.getColor()).append(";");
            }
            
            // 添加字体大小样式
            if (item.getSize() > 0) {
                htmlBuilder.append("font-size:").append(item.getSize()).append("px;");
            }
            
            // 添加加粗样式
            if (item.isBold()) {
                htmlBuilder.append("font-weight:bold;");
            }
            
            htmlBuilder.append("\">");
            
            // 添加文本内容，处理换行符
            String text = item.getText();
            if (text != null) {
                // 将\n转换为<br>
                text = text.replace("\n", "<br>");
                htmlBuilder.append(text);
            }
            
            htmlBuilder.append("</span>");
        }
        
        return htmlBuilder.toString();
    }

    /**
     * 直接将JSON字符串解析并设置到TextView
     * @param textView 目标TextView
     * @param jsonString JSON字符串
     */
    public static void setJsonToTextView(TextView textView, String jsonString) {
        try {
            List<InfoDataBean> dataList = InfoDataBean.parseFromJson(jsonString);
            setDataToTextView(textView, dataList);
        } catch (Exception e) {
            e.printStackTrace();
            textView.setText("JSON解析失败: " + e.getMessage());
        }
    }

    /**
     * 将InfoDataBean列表设置到TextView（使用SpannableStringBuilder，样式更精确）
     * @param textView 目标TextView
     * @param dataList InfoDataBean列表
     */
    public static void setDataToTextView(TextView textView, List<InfoDataBean> dataList) {
        if (textView == null) {
            return;
        }

        if (dataList == null || dataList.isEmpty()) {
            textView.setText("");
            return;
        }

        // 使用SpannableStringBuilder来获得更精确的样式控制
        SpannableStringBuilder spannableText = convertToSpannable(dataList);
        textView.setText(spannableText);
    }

    /**
     * 将InfoDataBean列表设置到TextView（使用HTML方式）
     * @param textView 目标TextView
     * @param dataList InfoDataBean列表
     */
    public static void setDataToTextViewWithHtml(TextView textView, List<InfoDataBean> dataList) {
        if (textView == null) {
            return;
        }

        if (dataList == null || dataList.isEmpty()) {
            textView.setText("");
            return;
        }

        // 转换为HTML并设置到TextView
        String htmlText = convertToHtml(dataList);
        
        // 使用Html.fromHtml()将HTML字符串转换为Spanned对象
        Spanned spannedText;
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
            spannedText = Html.fromHtml(htmlText, Html.FROM_HTML_MODE_LEGACY);
        } else {
            spannedText = Html.fromHtml(htmlText);
        }
        
        textView.setText(spannedText);
    }

    /**
     * 示例：解析你提供的JSON并显示到TextView
     * @param textView 目标TextView
     */
    public static void showExampleInTextView(TextView textView) {
        String jsonData = "[\n" +
                "  {\n" +
                "    \"text\": \"行人匆忙脚步，\",\n" +
                "    \"color\": \"#33aaff\",\n" +
                "    \"size\": 20,\n" +
                "    \"bold\": true\n" +
                "  },\n" +
                "  {\n" +
                "    \"text\": \"霓虹倒影闪烁。\\n\",\n" +
                "    \"color\": \"#ffbb33\",\n" +
                "    \"size\": 23,\n" +
                "    \"bold\": false\n" +
                "  }\n" +
                "]";

        setJsonToTextView(textView, jsonData);
    }
}