/**************************************************************************//**
  * \file     Sys_Tick.c
  * \brief    System tick timer driver file
  * \details
  * \author   Zhang Xuan
  * \version  V1.0.0
  * \date     19-Jul-2018
  * \par      History:
  *           V1.0.0 Initial release
  * \par      Copyright:
  *           (c) Heilongjiang TYW Electronics co., LTD
******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "BAT32A239.h"
#include "stdlib.h"
#include "common.h"
#include "Sys_Tick.h"

#include "gpio.h"
#include "Sys_Scheduler.h"
#include "eepromManage.h"
#include "Common_Interface.h"

/* Private typedef -----------------------------------------------------------*/
typedef struct
{
    uint32_t  u32MainCnt;
    uint32_t  u32DstVal1ms;
    uint32_t  u32DstVal100ms;
} Sys_Tick_Timer_st_t;

/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
#define   CLK_AWOTCLK_FREQ_Hz               (8000000UL)

/* Private variables ---------------------------------------------------------*/
void (*pfnSysTickCallBack[SYS_TICK_CALL_BACK_TYPE_NUM])(void);
Sys_Tick_Timer_st_t   g_stSysTickTimer;
volatile uint16_t     g_u16SysRollingCounter50us;
volatile uint16_t     g_u16SysRollingCounter1ms;

/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/**************************************************************************//**
  * \brief      Start tick timer
  * \retval     None
******************************************************************************/
void Sys_Tick_Timer_Start(void)
{
    uint32_t msCnt; 	// count value of 1ms			
    g_stSysTickTimer.u32MainCnt      = 0U;
    g_stSysTickTimer.u32DstVal1ms    =   1000UL / SYS_TICK_INT_INTERVAL;
    g_stSysTickTimer.u32DstVal100ms  = 100000UL / SYS_TICK_INT_INTERVAL;

    g_u16SysRollingCounter50us       = 0U;
    g_u16SysRollingCounter1ms        = 0U;

    //-----------------------------------------------------------------------
    // Systick setting 
    //-----------------------------------------------------------------------   
    SystemCoreClockUpdate();
    msCnt = (SystemCoreClock / 1000000)*SYS_TICK_INT_INTERVAL;

    SysTick->LOAD  = (uint32_t)(msCnt - 1UL);                         /* set reload register */
    NVIC_SetPriority (SysTick_IRQn, (1UL << __NVIC_PRIO_BITS) - 1UL); /* set Priority for Systick Interrupt */
    SysTick->VAL   = 0UL;                                             /* Load the SysTick Counter Value */
    SysTick->CTRL  = SysTick_CTRL_CLKSOURCE_Msk |
                     SysTick_CTRL_TICKINT_Msk   |
                     SysTick_CTRL_ENABLE_Msk;                         /* Enable SysTick IRQ and SysTick Timer */      

}

/**************************************************************************//**
  * \brief      GPIO initialization for sleep mode
  * \retval     None
******************************************************************************/
void Sys_Tick_Timer_Stop(void)
{
    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
}

void Sys_Tick_Timer_Call_Back_Reg(Sys_Tick_Call_Back_Type_en_t enType, void (*pfnIsr)(void))
{
    if (enType < SYS_TICK_CALL_BACK_TYPE_NUM)
    {
        pfnSysTickCallBack[enType] = pfnIsr;
    }
}

/**************************************************************************//**
  * \brief      Tick timer interrupt service routines
  * \retval     None
******************************************************************************/
void SysTick_Handler(void)
{
    uint16_t u16Counter;

	//*((volatile uint8_t*)(0x40040304)) |= (1 << PIN1);

  //  UDS_50us_Service();
    
    Sys_Exact_50us_Tasks();
    g_stSysTickTimer.u32MainCnt++;

    if (g_stSysTickTimer.u32MainCnt >= g_stSysTickTimer.u32DstVal1ms)
    {
        u16Counter                    = g_u16SysRollingCounter1ms;
        g_u16SysRollingCounter1ms     = u16Counter + 1U;
        eeprom_1ms_timeCount( );
        g_stSysTickTimer.u32DstVal1ms = g_stSysTickTimer.u32MainCnt + (1000UL / SYS_TICK_INT_INTERVAL);
    }

    if (g_stSysTickTimer.u32MainCnt >= g_stSysTickTimer.u32DstVal100ms)
    {
        Sys_Exact_100ms_Tasks();
        g_stSysTickTimer.u32DstVal100ms  = g_stSysTickTimer.u32MainCnt + (100000UL / SYS_TICK_INT_INTERVAL);
    }

    if (g_stSysTickTimer.u32MainCnt & 0xFFFF0000UL)
    {
        g_stSysTickTimer.u32MainCnt     &= 0x0000FFFFUL;
        g_stSysTickTimer.u32DstVal1ms   &= 0x0000FFFFUL;
        g_stSysTickTimer.u32DstVal100ms &= 0x0000FFFFUL;
    }

    g_u16SysRollingCounter50us = (uint16_t)g_stSysTickTimer.u32MainCnt;

		Sys_Scheduling_Tmr();
		
		Common_Input_Para();
	//*((volatile uint8_t*)(0x40040304)) &= ~(1 << PIN1);

}
