/***********************************************************************************************************************
* Copyright (C) All rights reserved.
***********************************************************************************************************************/

/***********************************************************************************************************************
* @file    gpio.c
* @brief   This file implements device driver for GPIO module.
* @version V1.0.0
* @date    2019/12/24
***********************************************************************************************************************/

/***********************************************************************************************************************
Includes
***********************************************************************************************************************/
#include "BAT32A239.h"
#include "gpio.h"

/***********************************************************************************************************************
Pragma directive
***********************************************************************************************************************/

/***********************************************************************************************************************
Global variables and functions
***********************************************************************************************************************/



/**
 * @brief Set specified GPIO as output function. 
 *
 * @param port port address, such as &P0, &P1, &P2...
 * @param pinMsk 
 *             e.g., bit0: 0x01, bit1: 0x02, bit0~3: 0x0F, bit0~7: 0xFF
 */
void GPIO_Output_Enable(__IO uint8_t *port, uint8_t pinMsk)
{
    *(port - 0x2A0) &= ~pinMsk;  /*!< PMC=0: Digital Function   */
    *(port + 0x020) &= ~pinMsk;  /*!< PM =0: Output Function    */
}

/**
 * @brief Set specified GPIO as input function. 
 *
 * @param port port address, such as &P0, &P1, &P2...
 * @param pinMsk 
 *             e.g., bit0: 0x01, bit1: 0x02, bit0~3: 0x0F, bit0~7: 0xFF
 */
void GPIO_Input_Enable(__IO uint8_t *port, uint8_t pinMsk)
{
    *(port - 0x2A0) &= ~pinMsk;  /*!< PMC=0: Digital Function   */
    *(port + 0x020) |=  pinMsk;  /*!< PM =1: Input Function     */
}

/**
 * @brief Enable pull up resister of input GPIO . 
 *
 * @param port port address, such as &P0, &P1, &P2...
 * @param pinMsk 
 *             e.g., bit0: 0x01, bit1: 0x02, bit0~3: 0x0F, bit0~7: 0xFF
 */
void GPIO_PullUp_Enable(__IO uint8_t *port, uint8_t pinMsk)
{
    *(port - 0x2D0) |=  pinMsk;  /*!< PU =1: Pull Up enable         */
}

/**
 * @brief Disable pull up resister of input GPIO . 
 *
 * @param port port address, such as &P0, &P1, &P2...
 * @param pinMsk 
 *             e.g., bit0: 0x01, bit1: 0x02, bit0~3: 0x0F, bit0~7: 0xFF
 */
void GPIO_PullUp_Disable(__IO uint8_t *port, uint8_t pinMsk)
{
    *(port - 0x2D0) &=  ~pinMsk;  /*!< PU =0: Pull Up disable        */
}

/**
 * @brief Nch Open Drain Output mode
 *
 * @param port address, such as &P0, &P1, &P3, &P5, &P7
 * @param pinMsk
 *             e.g., bit0: 0x01, bit1: 0x02, bit0~3: 0x0F, bit0~7: 0xFF
 */
void GPIO_Nch_OpenDrain(__IO uint8_t *port, uint8_t pinMsk)
{
    *(port - 0x2B0) |=  pinMsk;  /*!< POM =1: Nch OpenDrain Output */
}

void GPIO_Ttl_Input(__IO uint8_t *port, uint8_t pinMsk)
{
    *(port - 0x2C0) |=  pinMsk;  /*!< PIM =1: TTL input */
}

/**
 * @brief Set specified value to GPIO output
 *
 * @param port port address, such as &P0, &P1, &P2...
 * @param value 
 */
void GPIO_Set_Value(__IO uint8_t *port, uint8_t value)
{
    *port = value;           /*!< PL = value */
}

/**
 * @brief Get value from GPIO input 
 *
 * @param port port address, such as &P0, &P1, &P2...
 *
 * @return 
 */
uint8_t GPIO_Get_Value(__IO uint8_t *port)
{
    //PORT->PMS = 0x01;        /*!< Digital output level of the pin is read */ 
    return (*port);          /*!< PL = value                              */
}

/**
  * @brief  Initializes the PORTx
  * @param  PORTx: where x can be 0~14
  * @param  PINx: where x can be 0~7
  * @param  MODEx: such as INPUT,PULLUP_INPUT,TTL_INPUT,ANALOG_INPUT,OUTPUT,OPENDRAIN_OUTPUT      
	*
  * @retval None
  */
void PORT_Init(PORT_TypeDef PORTx,PIN_TypeDef PINx,PIN_ModeDef MODEx)
{
  	uint8_t mode = MODEx;
	uint8_t pos = 1<<PINx;
	
	switch(mode)
	{
		case INPUT:
			*((volatile uint8_t*)(&PORT->PMC0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->PM0+PORTx)) |= pos;
			*((volatile uint8_t*)(&PORT->PIM0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->POM0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->PU0+PORTx)) &= ~pos;
			break;
		case PULLUP_INPUT:
			*((volatile uint8_t*)(&PORT->PMC0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->PM0+PORTx)) |= pos;
			*((volatile uint8_t*)(&PORT->PIM0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->POM0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->PU0+PORTx)) |= pos;
			break;
		case TTL_INPUT:
			*((volatile uint8_t*)(&PORT->PMC0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->PM0+PORTx)) |= pos;
			*((volatile uint8_t*)(&PORT->PIM0+PORTx)) |= pos;
			*((volatile uint8_t*)(&PORT->POM0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->PU0+PORTx)) &= ~pos;
			break;
		case ANALOG_INPUT:
			*((volatile uint8_t*)(&PORT->PMC0+PORTx)) |= pos;
			break;
		case OUTPUT:
			*((volatile uint8_t*)(&PORT->PMC0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->PM0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->PIM0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->POM0+PORTx)) &= ~pos;
			break;
		case OPENDRAIN_OUTPUT:
			*((volatile uint8_t*)(&PORT->PMC0+PORTx)) &= ~pos;
			
			*((volatile uint8_t*)(&PORT->PM0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->PIM0+PORTx)) &= ~pos;
			*((volatile uint8_t*)(&PORT->POM0+PORTx)) |= pos;
			break;
	}
}


/**
  * @brief  Set the PORTx bit
  * @param  PORTx: where x can be 0~14
  * @param  PINx: where x can be 0~7    
	*
  * @retval None
  */
void PORT_SetBit(PORT_TypeDef PORTx,PIN_TypeDef PINx)
{
	uint8_t pos = 1<<PINx;
	*((volatile uint8_t*)(&PORT->P0+PORTx)) |= pos;
}

/**
  * @brief  Clear the PORTx bit
  * @param  PORTx: where x can be 0~14
  * @param  PINx: where x can be 0~7    
	*
  * @retval None
  */
void PORT_ClrBit(PORT_TypeDef PORTx,PIN_TypeDef PINx)
{
	uint8_t pos = 1<<PINx;
	*((volatile uint8_t*)(&PORT->P0+PORTx)) &= ~pos;
}

/**
  * @brief  Toggle the PORTx bit
  * @param  PORTx: where x can be 0~14
  * @param  PINx: where x can be 0~7    
	*
  * @retval None
 **/
void PORT_ToggleBit(PORT_TypeDef PORTx,PIN_TypeDef PINx)
{
	uint8_t pos = 1<<PINx;
	*((volatile uint8_t*)(&PORT->P0+PORTx)) ^= pos;
}

/**
  * @brief  Get the PORTx bit
  * @param  PORTx: where x can be 0~14
  * @param  PINx: where x can be 0~7    
	*
  * @retval None
  */
uint8_t PORT_GetBit(PORT_TypeDef PORTx,PIN_TypeDef PINx)
{
	uint8_t pos = 1<<PINx;
	//PORT->PMS = 0x01;        /*!< Digital output level of the pin is read */
	return *((volatile uint8_t*)(&PORT->P0+PORTx))&pos;
}


void GPIO_Init(GPIO_Config_Mode_en_t enGPIOConfigMode)
{
    if (enGPIOConfigMode == GPIO_WAKEUP_MODE)
    {
      //PORT0,    PIN0
      PORT_Init(VSPEED_PWM_IN, OUTPUT);
      PORT_ClrBit(VSPEED_PWM_IN);
      //PORT0,    PIN1
      PORT_Init(WAKEUP_KL15_In, INPUT);
      //PORT0,    PIN2
      PORT_Init(SEG1_SD, OUTPUT);
      PORT_ClrBit(SEG1_SD);
      //PORT0,    PIN3
      PORT_Init(SEG1_CSB, OUTPUT);
      PORT_ClrBit(SEG1_CSB);
      //PORT0,    PIN4
      PORT_Init(SEG1_SCL, OUTPUT);
      PORT_ClrBit(SEG1_SCL);
      //PORT0,    PIN5
      PORT_Init(HighBeam_P_In, INPUT);
      //PORT0,    PIN6
      PORT_Init(VSPEED_TEN_OUT, OUTPUT);
      PORT_SetBit(VSPEED_TEN_OUT);

      //PORT1,    PIN0
      PORT_Init(BACKLIGHT_PWM, OUTPUT);
      PORT_ClrBit(BACKLIGHT_PWM);
      //PORT1,    PIN1
      PORT_Init(LIN_RXD_ESP, OUTPUT);
      PORT_ClrBit(LIN_RXD_ESP);
      //PORT1,    PIN2
      PORT_Init(LIN_TXD_ESP, OUTPUT);
      PORT_ClrBit(LIN_TXD_ESP);
      //PORT1,    PIN3
      PORT_Init(EFI_FAULT_OUT, OUTPUT);
      PORT_SetBit(EFI_FAULT_OUT);
      //PORT1,    PIN4
      PORT_Init(AMBIENT_LIGHT_R, OUTPUT);
      PORT_ClrBit(AMBIENT_LIGHT_R);
      //PORT1,    PIN5
      PORT_Init(RESEVER1_OUT_LOW, OUTPUT);
      PORT_ClrBit(RESEVER1_OUT_LOW);
      //PORT1,    PIN6
      PORT_Init(AMBIENT_LIGHT_B, OUTPUT);
      PORT_ClrBit(AMBIENT_LIGHT_B);
      //PORT1,    PIN7
      PORT_Init(AMBIENT_LIGHT_G, OUTPUT);
      PORT_ClrBit(AMBIENT_LIGHT_G);

      //PORT2,    PIN0
      PORT_Init(AVDD_OUT_LOW, OUTPUT);
      PORT_ClrBit(AVDD_OUT_LOW);
      //PORT2,    PIN1
      PORT_Init(GND_OUT_LOW, OUTPUT);
      PORT_ClrBit(GND_OUT_LOW);
      //PORT2,    PIN2
      PORT_Init(POWER_KL30_In, INPUT);
      //PORT2,    PIN3
      PORT_Init(POWER_KL15_In, INPUT);
      //PORT2,    PIN4
      PORT_Init(RESEVER2_OUT_LOW, OUTPUT);
      PORT_ClrBit(RESEVER2_OUT_LOW);
      //PORT2,    PIN5
      PORT_Init(FUEL_MCU_IN, INPUT);
      //PORT2,    PIN6
      PORT_Init(VERSION_T_OUT_LOW, OUTPUT);
      PORT_ClrBit(VERSION_T_OUT_LOW);
      //PORT2,    PIN7  
      PORT_Init(FUEL_POWER_IN, INPUT);
    
      //PORT3,    PIN0
      PORT_Init(MAINTENANCE_LAMP_OUT, OUTPUT);
      PORT_ClrBit(MAINTENANCE_LAMP_OUT);
      //PORT3,    PIN1
      PORT_Init(RESEVER3_OUT_LOW, OUTPUT);
      PORT_ClrBit(RESEVER3_OUT_LOW);

      //PORT4,    PIN0
      PORT_Init(SWDIO, OUTPUT);
      PORT_ClrBit(SWDIO);
      //PORT4,    PIN1
      PORT_Init(RESEVER4_OUT_LOW, OUTPUT);
      PORT_ClrBit(RESEVER4_OUT_LOW);
      //PORT4,    PIN2
      PORT_Init(LeftTurn_P_In, INPUT);
      //PORT4,    PIN3
      PORT_Init(RightTurn_P_In, INPUT);

      //PORT5,    PIN0
      PORT_Init(CAN_MCU_RXD, INPUT);
      //PORT5 ,   PIN1
      PORT_Init(CAN_MCU_TXD, OUTPUT);
      PORT_ClrBit(CAN_MCU_TXD); 
      //PORT5,    PIN2
      PORT_Init(CAN_STB_MCU, OUTPUT);
      PORT_SetBit(CAN_STB_MCU); 
      //PORT5,    PIN3
      PORT_Init(MODE_P_IN, INPUT);
      //PORT5,    PIN4
      PORT_Init(SET_P_IN, INPUT);
      //PORT5,    PIN5
      PORT_Init(START_STOP_P_IN, INPUT);
 
      //PORT6,    PIN0
      PORT_Init(IIC_SCL_24G16, INPUT);
      //PORT6,    PIN1
      PORT_Init(IIC_SDA_24G16, INPUT);
      //PORT6,    PIN2
      PORT_Init(IIC_SCL_27030, INPUT);
      //PORT6,    PIN3
      PORT_Init(IIC_SDA_27030, INPUT);

      //PORT7,    PIN0
      PORT_Init(SEG2_SCL, OUTPUT);
      PORT_ClrBit(SEG2_SCL);
      //PORT7,    PIN1
      PORT_Init(SEG2_CSB, OUTPUT);
      PORT_ClrBit(SEG2_CSB);
      //PORT7,    PIN2
      PORT_Init(SEG2_SD, OUTPUT);
      PORT_ClrBit(SEG2_SD);
      //PORT7,    PIN3
      PORT_Init(SDB_LED_DRIVER, OUTPUT);
      PORT_ClrBit(SDB_LED_DRIVER);
      //PORT7,    PIN4
      PORT_Init(HighBeam_MUC_OUT, OUTPUT);
      PORT_SetBit(HighBeam_MUC_OUT);
      //PORT7,    PIN5
      PORT_Init(TPMS_MUC_OUT, OUTPUT);
      PORT_SetBit(TPMS_MUC_OUT);
      //PORT7,    PIN6
      PORT_Init(RightTurn_MUC_OUT, OUTPUT);
      PORT_SetBit(RightTurn_MUC_OUT);
      //PORT7,    PIN7
      PORT_Init(LeftTurn_MUC_OUT, OUTPUT);
      PORT_SetBit(LeftTurn_MUC_OUT);

      //PORT12,   PIN0
      PORT_Init(POWER_5V_MCU, OUTPUT);
      PORT_SetBit(POWER_5V_MCU);
      //PORT12,   PIN1
      PORT_Init(MCU_XI, OUTPUT);
      PORT_ClrBit(MCU_XI);
      //PORT12,   PIN2
      PORT_Init(MCU_XO, OUTPUT);
      PORT_ClrBit(MCU_XO);
      //PORT12,   PIN3
      PORT_Init(X32768_IN, OUTPUT);
      PORT_ClrBit(X32768_IN);
      //PORT12,   PIN4
      PORT_Init(X32768_OUT, OUTPUT);
      PORT_ClrBit(X32768_OUT);
 
      //PORT13,   PIN0
      PORT_Init(VSPEED_HUNDREDS_OUT, OUTPUT);
      PORT_ClrBit(VSPEED_HUNDREDS_OUT);
      //PORT13,   PIN6
      PORT_Init(RESEVER5_OUT_LOW, OUTPUT);
      PORT_ClrBit(RESEVER5_OUT_LOW);
      //PORT13,   PIN7
      PORT_Init(SWCLK, OUTPUT);
      PORT_ClrBit(SWCLK);

      //PORT14,   PIN0
      PORT_Init(L_BD433Ctrl_OUT, OUTPUT);
      PORT_SetBit(L_BD433Ctrl_OUT);
      //PORT14,   PIN1
      PORT_Init(FUEL_POWER_MCU_OUT, OUTPUT);
      PORT_SetBit(FUEL_POWER_MCU_OUT);
      //PORT14,   PIN6
      PORT_Init(VERSION_F_OUT_LOW, OUTPUT);
      PORT_ClrBit(VERSION_F_OUT_LOW);
      //PORT14,   PIN7
      PORT_Init(VERSION_S_OUT_LOW, OUTPUT);
      PORT_ClrBit(VERSION_S_OUT_LOW);

      PORT->PIOR0  =  0x00; /*外围IO重定向寄存器0*/
      PORT->PIOR1  =  0x00; /*外围IO重定向寄存器1*/
      PORT->PIOR2  =  0x00; /*外围IO重定向寄存器2*/
      PORT->PIOR3  =  0x00; /*外围IO重定向寄存器3*/
    }
    else
    {
      //PORT0,    PIN0
      PORT_Init(VSPEED_PWM_IN, OUTPUT);
      PORT_ClrBit(VSPEED_PWM_IN);
      //PORT0,    PIN1
      PORT_Init(WAKEUP_KL15_In, INPUT);
      //PORT0,    PIN2
      PORT_Init(SEG1_SD, OUTPUT);
      PORT_ClrBit(SEG1_SD);
      //PORT0,    PIN3
      PORT_Init(SEG1_CSB, OUTPUT);
      PORT_ClrBit(SEG1_CSB);
      //PORT0,    PIN4
      PORT_Init(SEG1_SCL, OUTPUT);
      PORT_ClrBit(SEG1_SCL);
      //PORT0,    PIN5
      PORT_Init(HighBeam_P_In, INPUT);
      //PORT0,    PIN6
      PORT_Init(VSPEED_TEN_OUT, OUTPUT);
      PORT_ClrBit(VSPEED_TEN_OUT);

      //PORT1,    PIN0
      PORT_Init(BACKLIGHT_PWM, OUTPUT);
      PORT_ClrBit(BACKLIGHT_PWM);
      //PORT1,    PIN1
      PORT_Init(LIN_RXD_ESP, OUTPUT);
      PORT_ClrBit(LIN_RXD_ESP);
      //PORT1,    PIN2
      PORT_Init(LIN_TXD_ESP, OUTPUT);
      PORT_ClrBit(LIN_TXD_ESP);
      //PORT1,    PIN3
      PORT_Init(EFI_FAULT_OUT, OUTPUT);
      PORT_ClrBit(EFI_FAULT_OUT);
      //PORT1,    PIN4
      PORT_Init(AMBIENT_LIGHT_R, OUTPUT);
      PORT_ClrBit(AMBIENT_LIGHT_R);
      //PORT1,    PIN5
      PORT_Init(RESEVER1_OUT_LOW, OUTPUT);
      PORT_ClrBit(RESEVER1_OUT_LOW);
      //PORT1,    PIN6
      PORT_Init(AMBIENT_LIGHT_B, OUTPUT);
      PORT_ClrBit(AMBIENT_LIGHT_B);
      //PORT1,    PIN7
      PORT_Init(AMBIENT_LIGHT_G, OUTPUT);
      PORT_ClrBit(AMBIENT_LIGHT_G);

      //PORT2,    PIN0
      PORT_Init(AVDD_OUT_LOW, OUTPUT);
      PORT_ClrBit(AVDD_OUT_LOW);
      //PORT2,    PIN1
      PORT_Init(GND_OUT_LOW, OUTPUT);
      PORT_ClrBit(GND_OUT_LOW);
      //PORT2,    PIN2
      PORT_Init(POWER_KL30_In, INPUT);
      //PORT2,    PIN3
      PORT_Init(POWER_KL15_In, INPUT);
      //PORT2,    PIN4
      PORT_Init(RESEVER2_OUT_LOW, OUTPUT);
      PORT_ClrBit(RESEVER2_OUT_LOW);
      //PORT2,    PIN5
      PORT_Init(FUEL_MCU_IN, INPUT);
      //PORT2,    PIN6
      PORT_Init(VERSION_T_OUT_LOW, OUTPUT);
      PORT_ClrBit(VERSION_T_OUT_LOW);
      //PORT2,    PIN7  
      PORT_Init(FUEL_POWER_IN, INPUT);
    
      //PORT3,    PIN0
      PORT_Init(MAINTENANCE_LAMP_OUT, OUTPUT);
      PORT_ClrBit(MAINTENANCE_LAMP_OUT);
      //PORT3,    PIN1
      PORT_Init(RESEVER3_OUT_LOW, OUTPUT);
      PORT_ClrBit(RESEVER3_OUT_LOW);

      //PORT4,    PIN0
      PORT_Init(SWDIO, OUTPUT);
      PORT_ClrBit(SWDIO);
      //PORT4,    PIN1
      PORT_Init(RESEVER4_OUT_LOW, OUTPUT);
      PORT_ClrBit(RESEVER4_OUT_LOW);
      //PORT4,    PIN2
      PORT_Init(LeftTurn_P_In, INPUT);
      //PORT4,    PIN3
      PORT_Init(RightTurn_P_In, INPUT);

      //PORT5,    PIN0
      PORT_Init(CAN_MCU_RXD, INPUT);
      //PORT5 ,   PIN1
      PORT_Init(CAN_MCU_TXD, OUTPUT);
      PORT_SetBit(CAN_MCU_TXD); 
      //PORT5,    PIN2
      PORT_Init(CAN_STB_MCU, OUTPUT);
      PORT_SetBit(CAN_STB_MCU); 
      //PORT5,    PIN3
      PORT_Init(MODE_P_IN, INPUT);
      //PORT5,    PIN4
      PORT_Init(SET_P_IN, INPUT);
      //PORT5,    PIN5
      PORT_Init(START_STOP_P_IN, INPUT);
 
      //PORT6,    PIN0
      PORT_Init(IIC_SCL_24G16, OUTPUT);
      PORT_SetBit(IIC_SCL_24G16); 
      //PORT6,    PIN1
      PORT_Init(IIC_SDA_24G16, OUTPUT);
      PORT_SetBit(IIC_SDA_24G16); 
      //PORT6,    PIN2
      PORT_Init(IIC_SCL_27030, OUTPUT);
      PORT_SetBit(IIC_SCL_27030); 
      //PORT6,    PIN3
      PORT_Init(IIC_SDA_27030, OUTPUT);
      PORT_SetBit(IIC_SDA_27030); 

      //PORT7,    PIN0
      PORT_Init(SEG2_SCL, OUTPUT);
      PORT_ClrBit(SEG2_SCL);
      //PORT7,    PIN1
      PORT_Init(SEG2_CSB, OUTPUT);
      PORT_ClrBit(SEG2_CSB);
      //PORT7,    PIN2
      PORT_Init(SEG2_SD, OUTPUT);
      PORT_ClrBit(SEG2_SD);
      //PORT7,    PIN3
      PORT_Init(SDB_LED_DRIVER, OUTPUT);
      PORT_ClrBit(SDB_LED_DRIVER);
      //PORT7,    PIN4
      PORT_Init(HighBeam_MUC_OUT, OUTPUT);
      PORT_ClrBit(HighBeam_MUC_OUT);
      //PORT7,    PIN5
      PORT_Init(TPMS_MUC_OUT, OUTPUT);
      PORT_ClrBit(TPMS_MUC_OUT);
      //PORT7,    PIN6
      PORT_Init(RightTurn_MUC_OUT, OUTPUT);
      PORT_ClrBit(RightTurn_MUC_OUT);
      //PORT7,    PIN7
      PORT_Init(LeftTurn_MUC_OUT, OUTPUT);
      PORT_ClrBit(LeftTurn_MUC_OUT);

      //PORT12,   PIN0
      PORT_Init(POWER_5V_MCU, OUTPUT);
      PORT_ClrBit(POWER_5V_MCU);
      //PORT12,   PIN1
      PORT_Init(MCU_XI, OUTPUT);
      PORT_ClrBit(MCU_XI);
      //PORT12,   PIN2
      PORT_Init(MCU_XO, OUTPUT);
      PORT_ClrBit(MCU_XO);
      //PORT12,   PIN3
      PORT_Init(X32768_IN, OUTPUT);
      PORT_ClrBit(X32768_IN);
      //PORT12,   PIN4
      PORT_Init(X32768_OUT, OUTPUT);
      PORT_ClrBit(X32768_OUT);
 
      //PORT13,   PIN0
      PORT_Init(VSPEED_HUNDREDS_OUT, OUTPUT);
      PORT_ClrBit(VSPEED_HUNDREDS_OUT);
      //PORT13,   PIN6
      PORT_Init(RESEVER5_OUT_LOW, OUTPUT);
      PORT_ClrBit(RESEVER5_OUT_LOW);
      //PORT13,   PIN7
      PORT_Init(SWCLK, OUTPUT);
      PORT_ClrBit(SWCLK);

      //PORT14,   PIN0
      PORT_Init(L_BD433Ctrl_OUT, OUTPUT);
      PORT_ClrBit(L_BD433Ctrl_OUT);
      //PORT14,   PIN1
      PORT_Init(FUEL_POWER_MCU_OUT, OUTPUT);
      PORT_ClrBit(FUEL_POWER_MCU_OUT);
      //PORT14,   PIN6
      PORT_Init(VERSION_F_OUT_LOW, OUTPUT);
      PORT_ClrBit(VERSION_F_OUT_LOW);
      //PORT14,   PIN7
      PORT_Init(VERSION_S_OUT_LOW, OUTPUT);
      PORT_ClrBit(VERSION_S_OUT_LOW);

      PORT->PIOR0  =  0x00; /*外围IO重定向寄存器0*/
      PORT->PIOR1  =  0x00; /*外围IO重定向寄存器1*/
      PORT->PIOR2  =  0x00; /*外围IO重定向寄存器2*/
      PORT->PIOR3  =  0x00; /*外围IO重定向寄存器3*/     
    }
}



