#ifndef UART_PROTOCOL_H
#define UART_PROTOCOL_H

typedef unsigned int uint32_t;
typedef int int32_t;
typedef unsigned short uint16_t;
typedef short int16_t;
typedef unsigned char uint8_t;

#define AMT_NULL 0

#define UART_UPDATE_ACK_NONE 0
#define UART_UPDATE_ACK_OK 1
#define UART_UPDATE_ACK_WAIT 2
#define UART_UPDATE_ACK_FAIL 3
#define UART_UPDATE_ACK_TIMEOUT 4

#define GROUP_PIC_DIR_LEFT      0x80
#define GROUP_PIC_DIR_RIGHT     0x00
#define GROUP_PIC_USE_PSD_Y_POS -1

typedef enum {
    ALIVE_COUNTER_COMM,
    ALIVE_COUNTER_LCD0,
    ALIVE_COUNTER_LCD1,
    ALIVE_COUNTER_TOTAL,
}ALIVE_COUNTER_TYPE;

/* Generate Function */
extern void ResetInitedState(void); /* Called every time the AMT630 is powered on or the MCU wake up */
extern uint8_t GetAliveCounter(ALIVE_COUNTER_TYPE n); /* 0:ALIVE_COUNTER_COMM, 1:ALIVE_COUNTER_LCD0, 2:ALIVE_COUNTER_LCD1 */
extern uint32_t GetCrcErrCnts(void);
extern uint32_t GetRecvLenErrCnts(void);
extern uint8_t GetLcdInitSt(void);
extern uint8_t GetCameraSignalSt(void);
extern uint8_t GetVideoCaptureSt(void);
extern uint8_t GetAmt630InBoot(void); 
extern uint8_t GetLcdInitStB(void);
extern uint8_t *GetUartRecvData(uint32_t *UartRecvBufLen, uint32_t *UartRecvPos);
extern void UartSendData(uint8_t *data, uint32_t len);
extern void AnalysisUartData(void);
/* Generate Function end */

/* Lcd Parameter */
enum {
    AUDIO_FREQ_NONE,
    AUDIO_FREQ_8K,
    AUDIO_FREQ_11K,
    AUDIO_FREQ_16K,
    AUDIO_FREQ_22K,
    AUDIO_FREQ_32K,
    AUDIO_FREQ_44K,
    AUDIO_FREQ_48K,
    AUDIO_FREQ_96K,
    AUDIO_FREQ_128K,
    AUDIO_FREQ_160K,
    AUDIO_FREQ_172K,
    AUDIO_FREQ_192K,
    AUDIO_FREQ_TOTAL,
};
typedef enum {
    AMT630_UART1 = 1,
    AMT630_UART2,
    AMT630_UART3,
} AMT630_UART_CH_TYPE;

#define LCD_WIRING_MODE_RGB 0
#define LCD_WIRING_MODE_RBG 1
#define LCD_WIRING_MODE_GRB 2
#define LCD_WIRING_MODE_GBR 3
#define LCD_WIRING_MODE_BRG 4
#define LCD_WIRING_MODE_BGR 5

#define LCD_INTERFACE_TTL 0
#define LCD_INTERFACE_LVDS 1
#define LCD_INTERFACE_MIPI 2

#define LCD_WIRING_BIT_ORDER_MSB 0
#define LCD_WIRING_BIT_ORDER_LSB 1

#define LVDS_PANEL_FORMAT_TI 0
#define LVDS_PANEL_FORMAT_NS 1

#define LVDS_PANEL_DATA_8BIT 0
#define LVDS_PANEL_DATA_6BIT 1

#define VIDEO_IN_NONE 0
#define VIDEO_IN_NTSC 1
#define VIDEO_IN_PAL 2

#pragma pack(1)
typedef struct {
    uint16_t LcdWidth;
    uint16_t LcdHeight;
    uint8_t LcdVideoInputValid : 2;
    uint8_t LcdBpp : 6;
    uint8_t Ark7116ResetPin;
    uint8_t LcdInterfaceType : 4;
    uint8_t LcdWiringMode : 4;
    uint8_t LcdWiringBitOrder : 2;
    uint8_t LcdDotClkReverse : 2;
    uint8_t LvdsPanelFormat : 2;
    uint8_t LvdsPanelData : 2;
    uint16_t LcdTimingVBP;
    uint16_t LcdTimingVFP;
    uint16_t LcdTimingVSW;
    uint16_t LcdTimingHBP;
    uint16_t LcdTimingHFP;
    uint16_t LcdTimingHSW;
    uint32_t LcdClkFreq;
    uint8_t SoundFreq;
    uint8_t SoundChannelNum;
}LCD_TFT_PARA_ST;

typedef struct {
    uint32_t MipiClkFreq;
    uint8_t MipiNpPolarityInverse;
    uint8_t MipiLane;
    uint8_t MipiDisplaymMode;
}LCD_TFT_MIPI_PARA_ST;

#define RN6752_INPUT_TYPE_NONE 0
#define RN6752_INPUT_TYPE_CVBS_NTSC 1
#define RN6752_INPUT_TYPE_CVBS_PAL 2
#define RN6752_INPUT_TYPE_AHD_720P_25 3
#define RN6752_INPUT_TYPE_AHD_720P_30 4
typedef struct {
    uint8_t Rn6752InputType;
    uint8_t Rn6752ResetPin;
    uint8_t Rn6752Address;
}RN6752_PARA_ST;
#pragma pack()
extern void SetLcdParameter(LCD_TFT_PARA_ST *pData, LCD_TFT_MIPI_PARA_ST *pDataMipi, AMT630_UART_CH_TYPE Amt630UartCh, RN6752_PARA_ST *pDataRn6752);
/* Lcd Parameter end */

/* Amt Software */
typedef struct {
    uint8_t AmtLdrVer[8];
    uint8_t StepLdrVer[8];
    uint8_t AmtAppVer[8];
    uint8_t UiVer[8];
    uint8_t SoundVer[8];
    uint8_t FontVer[8];
} SOFTWARE_VERSION_ST;
extern void SetSwVerBuf(uint8_t *buf);
extern void PackedTransfer_GetSwVer(void);
/* Amt Software end */

/* Pic Index */
extern void SetPicIndexData(uint8_t *data, uint32_t len, uint32_t crc);
/* Pic Index end */

/* Sequence Frame */
#define SE_FRAME_PIC_MAX 100
enum {
    SE_TYPE_NONE,
    SE_TYPE_ONE_SHOT,
    SE_TYPE_CONTINUOUS,
};
typedef struct {
    uint16_t Pic;
} SEQUENCE_FRAME_PIC_ST;
typedef struct {
    uint8_t SeFrameType;
    uint8_t SeFrameNumTotal;
    uint16_t SeFrameTotalTime;
    uint16_t SeFrameValidTime;
    SEQUENCE_FRAME_PIC_ST SeFramePicList[SE_FRAME_PIC_MAX];
} SEQUENCE_FRAME_ST;
extern SEQUENCE_FRAME_ST SeFrameData;
extern void PackedTransfer_SequenceFrame(uint8_t *SeData, uint16_t SeLen);
extern void PackedTransfer_SequenceFrameOff(void);
/* Sequence Frame end */

/* Pointer Move Type */
typedef enum {
    POINTER_TYPE_SPEED,
    POINTER_TYPE_ENGINE,
    POINTER_TYPE_COOLANT_FUEL,
    POINTER_TYPE_CUSTOMIZE,
    POINTER_TYPE_MAX,
} POINTER_MOVE_TYPE_ENUM;
typedef enum {
    POINTER_RENDER_TYPE_POINTER,
    POINTER_RENDER_TYPE_SEQUENCE_PIC,
} POINTER_RENDER_TYPE_ENUM;
typedef struct {
    uint32_t PointerAngleP;     /* 10:0.1, 15:0.15, 100:1 .... POINTER_TYPE_SPEED default:30 */
    uint32_t PointerAngleD;     /* 10:0.1, 15:0.15, 100:1 .... POINTER_TYPE_SPEED default:18 */
    uint32_t PointerAngleSlowP; /* 10:0.1, 15:0.15, 100:1 .... POINTER_TYPE_SPEED default:10 */
    uint32_t PointerAngleSlowD; /* 10:0.1, 15:0.15, 100:1 .... POINTER_TYPE_SPEED default:18 */
    uint32_t MaxInterval;       /* 10:0.1, 15:0.15, 100:1 .... POINTER_TYPE_SPEED default:340 */
} POINTER_MOVE_PARA;
extern void PackedTransfer_SetPointerMoveType(POINTER_MOVE_PARA *Data, POINTER_MOVE_TYPE_ENUM MoveType);
/* Pointer Move Type end */

/* Pointer */
typedef enum {
    POINTER_0,
    POINTER_1,
    POINTER_2,
    POINTER_3,
    POINTER_MAX,
} POINTER_ENUM;
extern void PackedTransfer_InitPointer(POINTER_ENUM PointerN, uint8_t *data, uint16_t len, POINTER_MOVE_TYPE_ENUM MoveType, POINTER_RENDER_TYPE_ENUM RenderType);
extern uint8_t GetPointerInitSt(POINTER_ENUM PointerN);
extern void PackedTransfer_SetPointerDestVal(POINTER_ENUM PointerN, uint16_t DestVal, uint8_t NoDamping); /* DestVal:0~1000, NoDamping: 0-have damping, 1:no damping */
extern uint16_t GetPointerCurVal(POINTER_ENUM PointerN);                               /* CurVal:0~1000 */

#define POINTER_NUM_LIST_MAX 20
extern void SetPointerNumPic(POINTER_ENUM PointerN, uint8_t n, uint16_t PicIndex, uint16_t PointerVal);
extern void SetPointerNumZoomPara(POINTER_ENUM PointerN, uint16_t ZoomRange, float ZoomMax);
extern void PackedTransfer_PointerNumData(POINTER_ENUM PointerN);
/* Pointer end */

/* Page Data */
typedef enum {
    GRAPHICS_PAGE_0,
    GRAPHICS_PAGE_1,
    GRAPHICS_PAGE_2,
    GRAPHICS_PAGE_MAX,
} GRAPHICS_PAGE_ENUM;

typedef enum {
    SWITCH_EFFECT_NONE,
    SWITCH_EFFECT_ONLY_ALPHA,
    SWITCH_EFFECT_ONLY_OFFSET,
    SWITCH_EFFECT_ALPHA_OFFSET,
} EFFECT_VALID_ENUM;

typedef enum {
    SWITCH_EFFECT_OFFSET_FADE_IN_RIGHT_TO_LEFT,
    SWITCH_EFFECT_OFFSET_FADE_IN_LEFT_TO_RIGHT,
    SWITCH_EFFECT_OFFSET_FADE_IN_UP_TO_DOWN,
    SWITCH_EFFECT_OFFSET_FADE_IN_DOWN_TO_UP,
} OFFSET_EFFECT_ENUM;

typedef enum {
    ALIGN_LEFT,
    ALIGN_CENTER,
    ALIGN_RIGHT,
} TEXT_ALIGN_TYPE;

typedef enum {
    TFT_ROT_0,
    TFT_ROT_90,
    TFT_ROT_180,
    TFT_ROT_270,
} TFT_ROT_TYPE;

#define TEXT_ROT_SCALE(r, s) ((((r) & 0xffff) << 16) | ((s) & 0xffff))
#define TEXT_COLOR(r, g, b, a) ((((r) & 0xff) << 24) | (((g) & 0xff) << 16) | (((b) & 0xff) << 8) | ((a) & 0xff))
#define TEXT_LINE_OFFSET(x, y) ((((x) & 0xffff) << 16) | ((y) & 0xffff))

#define GROUP_ALIGN_LEFT 0x0
#define GROUP_ALIGN_CENTER 0x40
extern void SetPageGuid(GRAPHICS_PAGE_ENUM page, uint8_t h, uint8_t m, uint8_t l);
extern void SetPageBackground(GRAPHICS_PAGE_ENUM page, uint16_t PicIndex);
extern void SetPagePic(GRAPHICS_PAGE_ENUM page, uint16_t PicIndex);
extern void SetPagePicPointerNum(GRAPHICS_PAGE_ENUM page, POINTER_ENUM PointN, uint16_t PicIndex);
extern void SetPagePicFree(GRAPHICS_PAGE_ENUM page, uint16_t PicIndex, int16_t PosX, int16_t PosY);
extern void SetPagePicFreeAppend(GRAPHICS_PAGE_ENUM page, uint16_t pic, int16_t PosX, int16_t PosY);
extern void SetPagePicGroupExtParaOriCoor(GRAPHICS_PAGE_ENUM page, uint8_t PicNum, uint16_t NumZeroPic);
extern void SetPagePicGroup(GRAPHICS_PAGE_ENUM page, int16_t Interval, int16_t PosX, int16_t PosY, uint16_t *PicList, uint32_t Align_Num, uint16_t Dot_Itv, TFT_ROT_TYPE TftRot);
extern void SetPageText(GRAPHICS_PAGE_ENUM page, uint16_t LabelNum, int16_t PosX, int16_t PosY, int16_t MaxX, TEXT_ALIGN_TYPE AlginType, uint32_t RotScale, uint32_t Color, uint32_t LineOffset);	/* TEXT_ROT_SCALE:use TEXT_ROT_SCALE macro, (r:0~3600); (s:-1000 ~ 1000, 0:no scale, 1 ~ 1000:zoom in, -1 ~ -1000:zoom out); Color:use TEXT_COLOR macro; LineOffset:use TEXT_LINE_OFFSET macro */
extern void SetPagePointerPic(GRAPHICS_PAGE_ENUM page, POINTER_ENUM PointN, uint16_t PicIndex);
extern void SetPagePointerSePic(GRAPHICS_PAGE_ENUM page, POINTER_ENUM PointN);
extern void SetPagePointerTrackPic(GRAPHICS_PAGE_ENUM page, POINTER_ENUM PointN, uint16_t PicIndex);
extern void SetPagePointerTrackActiceArea(GRAPHICS_PAGE_ENUM page, POINTER_ENUM PointN, int16_t ActivePosX, int16_t ActivePosY, uint16_t ActiveW, uint16_t ActiveH);
extern void SetPageSliderNumPic(GRAPHICS_PAGE_ENUM page, uint8_t SliderNum);
extern void SetPageSliderImagePic(GRAPHICS_PAGE_ENUM page, uint8_t SliderImageNum);
extern void PackedTransfer_PicDirectDraw(uint32_t PicIndex, uint8_t Valid);

extern void PackedTransfer_BgJpg(uint32_t PicIndex);
extern void PackedTransfer_Page(void);
extern void PackedTransfer_PageEffect(GRAPHICS_PAGE_ENUM page, EFFECT_VALID_ENUM EffectValid, OFFSET_EFFECT_ENUM OffsetEffect);
extern void PackedTransfer_ClearPageData(GRAPHICS_PAGE_ENUM page);
/* Page Data end */

/* Compose Pic */
extern void PackedTransfer_ComposeData(uint8_t *data, uint8_t len);
/* Compose Pic end */

/* Label List */
extern uint8_t GetLanguageInitSt(void);
extern void PackedTransfer_InitLanguageFont(uint8_t Language, uint8_t Font);
extern void PackedTransfer_LabelsList(uint16_t LabelNum, uint8_t *LabelBuf, uint32_t LabelLen);
/* Label List end */

/* Flash Check */
typedef enum {
    DISPLAY_CHECKING,
    DISPLAY_CHECK_PASS,
    DISPLAY_CHECK_FAIL,
}FLASH_CHECK_FORCE_DISPLAY_TYPE;
extern void PackedTransfer_CheckFlash(void);
extern uint8_t PackedTransfer_CheckFlashGetResult(void);	/*return value 0xff:checking, 0:fail, 1:success */
extern void PackedTransfer_CheckFlashForceDisplay(FLASH_CHECK_FORCE_DISPLAY_TYPE type, uint16_t timerMs);
/* Flash Check end */

/* Video Input */
extern void PackedTransfer_VedioInputOn(uint32_t PosX, uint32_t PosY, uint32_t Wdith, uint32_t Height, uint32_t CutX, uint32_t CutY);
extern void PackedTransfer_VedioInputOff(void);
extern void PackedTransfer_VedioInputArk7116OutputOff(void);
extern void PackedTransfer_VedioInputArk7116OutputOn(void);
/* Video Input end */

/* Get Mcu or Backup Data */
typedef enum {
	REQ_DATA,
	GET_STATUS,
	GET_APP_DATA_OFFSET,
    GET_BACKUP_DATA_OFFSET,
    GET_BACKUP_DATA_CRC,
    WRITE_BACKUP_DATA,
} READ_ARK630_CMD;

typedef enum {
    RECV_TYPE_DATA,
    RECV_TYPE_OFFSET_LEN,
    RECV_TYPE_WRITE_BACKUP_RESP,
    RECV_TYPE_BACKUP_CRC,
}RECV_MCU_DATA_TYPE;
extern void GetDataBuf(uint32_t *pAddr, uint32_t *pLen);
extern void SetRecvDataSt(RECV_MCU_DATA_TYPE type, uint32_t addr, uint32_t len, uint32_t crc);
extern void PackedTransfer_GetMcuData(uint32_t addr, uint16_t len);
extern void PackedTransfer_GetDataOffsetLen(uint8_t type);
extern void PackedTransfer_GetBackupDataCrc(uint32_t addr, uint32_t len);
extern void PackedTransfer_WriteBackupData(uint32_t addr, uint8_t *data, uint16_t len);
/* Get Mcu or Backup Data end */

/* Set tft io mode */
extern void PackedTransfer_SetTftIoMode(uint8_t st);	/*st 0:normal gpio, 1:TFT io*/
/* Set tft io mode end */

/* Slider Num */
#pragma pack(1)
#define SLIDER_NUM_MAX 300		/* The maximum value is 300 */
typedef struct {
    uint8_t SliderNum;
    uint8_t SliderValid;
    uint8_t ZoomOutMax;
    uint8_t AlphaMin;
    float DestVal;
    float CurVal;
    int16_t PosX;
    int16_t PosY;
    int16_t Interval;
    uint16_t PicNum;
    uint16_t DampingVal;
    uint16_t PicIndex[SLIDER_NUM_MAX];
} SLIDER_NUMBER_PARA_ST;
#pragma pack()
extern void PackedTransfer_SliderNumData(uint8_t n, uint8_t *data, uint32_t len);		/* PackedTransfer_SliderNumData(0, &SliderNumParaData, sizeof (SLIDER_NUMBER_PARA_ST)); */
extern void PackedTransfer_SliderNumDestVal(uint8_t n, uint16_t val, uint8_t ForceUpdate, uint8_t DisplayValid);
/* Slider Num end */

/* Slider Image */
#pragma pack(1)
#define SLIDER_IMAGE_PIC_MAX 5		/* The maximum value is 5 */
enum {
    SLIDER_IMAGE_TYPE_HORIZONTAL_LEFT_TO_RIGHT,
    SLIDER_IMAGE_TYPE_HORIZONTAL_RIGHT_TO_LEFT,
    SLIDER_IMAGE_TYPE_VERTICAL_UP_TO_DOWN,
    SLIDER_IMAGE_TYPE_VERTICAL_DOWN_TO_UP,
};
typedef struct {
    uint8_t SliderType;
    uint8_t SliderValid;
    uint8_t Reserve1;
    uint8_t Reserve2;
    float DestVal;
    float CurVal;
    int16_t PosX;
    int16_t PosY;
    int16_t Interval;
    uint16_t DampingVal;
    uint16_t PicIndex[SLIDER_IMAGE_PIC_MAX];
} SLIDER_IMAGE_PARA_ST;
#pragma pack()
extern void PackedTransfer_SliderImageData(uint8_t n, uint8_t *data, uint32_t len);		/* The setting range of parameter n is 0~9, PackedTransfer_SliderImageData(0, &SliderImageParaData, sizeof (SLIDER_IMAGE_PARA_ST)); */
extern void PackedTransfer_SliderImageDestVal(uint8_t n, uint16_t val, uint8_t ForceUpdate, uint8_t DisplayValid);
/* Slider Num end */

enum {NUM_0, NUM_1, NUM_2, NUM_3, NUM_4, NUM_5, NUM_6, NUM_7, NUM_8, NUM_9, NUM_POINT, NUM_LINE};
typedef struct {
    const uint16_t *NumList;
    uint16_t NumListLength;
    int16_t NumInterval;
    int16_t PointNumInterval;
    int16_t PointNumCoorX;
    int16_t LineCoorX;
    int16_t x;
    int16_t y;
    int16_t Number1TotalLattice;
    int16_t Number1UseLattice;
    int16_t Number7TotalLattice;
    int16_t Number7UseLattice;
    uint8_t DecimalNum;
    uint8_t MinDigits;
} DISPLAY_NUM_ST;


typedef struct {
    const uint16_t *CharacterList;
    uint16_t CharacterLength;
    int16_t CharacterInterval;
    int16_t PointCharacterInterval;
    int16_t PointCharacterCoorX;
    int16_t LineCoorX;
    int16_t x;
    int16_t y;
    int16_t Number1TotalLattice;
    int16_t Number1UseLattice;
    int16_t Number7TotalLattice;
    int16_t Number7UseLattice;
}DISPLAY_CHARACTER_ST;

typedef enum {
    CHAR_LEFT_JUSTIFYING = 0,
    CHAR_RIGHT_JUSTIFYING,

    TOTAL_JUSTIFYING_TYPE,
}CHAR_JUSTIFYING_TYPE;

extern void DisplayNum(uint8_t page, DISPLAY_NUM_ST *pNumData, int32_t NumVal);
extern void DisplayCharacter(uint8_t page, DISPLAY_CHARACTER_ST *pCharacterData, char* Character, uint8_t JustifyingType);

/*
note:
Update630HBegin UDS 34 Seriver call and power on 630H and lcd by your self
Update630HWrite UDS 36 Seriver call
Update630HEnd   UDS 37 Seriver call

return 0 is wait, return 1 is success, return 2 is fail.
*/
extern uint8_t Update630HBegin(uint32_t TimerCnts);
extern uint8_t Update630HWrite(uint32_t TimerCnts, uint8_t *data, uint32_t len);
extern uint8_t Update630HEnd(uint32_t TimerCnts);

/*Mobile screen projection*/
enum PhoneStatus
{
	Phone_Undefine = 0u,
	Phone_Idle,
	Phone_Connecting,
	Phone_EndCall,
	Phone_Outgoing,
	Phone_Incoming,
	Phone_Talking,
};

enum PhoneConTrol
{
	PhoneConTrol_Undefine = 0u,
	Phone_Pick_Up,
	Phone_Hangup_Out
};
enum BlueToothStatus
{
	BlueTooth_Undefine = 0u,
	BlueTooth_DisConnect,
	BlueTooth_Connection
};

enum ECNaviIcon
{
    EC_NAVI_ICON_NONE                          = 0,               ///< 收到此值，不显示导航图标
    EC_NAVI_ICON_DEFAULT                       = 1,               ///< 自车.请忽略这个元素，从左转图标开始
    EC_NAVI_ICON_LEFT                          = 2,               ///< 左转
    EC_NAVI_ICON_RIGHT                         = 3,               ///< 右转
    EC_NAVI_ICON_LEFT_FRONT                    = 4,               ///< 左前方
    EC_NAVI_ICON_RIGHT_FRONT                   = 5,               ///< 右前方
    EC_NAVI_ICON_LEFT_BACK                     = 6,               ///< 左后方
    EC_NAVI_ICON_RIGHT_BACK                    = 7,               ///< 右后方
    EC_NAVI_ICON_LEFT_TURN_AROUND              = 8,               ///< 左转掉头
    EC_NAVI_ICON_STRAIGHT                      = 9,               ///< 直行
    EC_NAVI_ICON_ARRIVED_WAYPOINT              = 10,              ///< 到达途经点
    EC_NAVI_ICON_ENTER_ROUNDABOUT              = 11,              ///< 进入环岛
    EC_NAVI_ICON_OUT_ROUNDABOUT                = 12,              ///< 驶出环岛
    EC_NAVI_ICON_ARRIVED_SERVICE_AREA          = 13,              ///< 到达服务区
    EC_NAVI_ICON_ARRIVED_TOLLGATE              = 14,              ///< 到达收费站
    EC_NAVI_ICON_ARRIVED_DESTINATION           = 15,              ///< 到达目的地
    EC_NAVI_ICON_ARRIVED_TUNNEL                = 16,              ///< 到达隧道
    EC_NAVI_ICON_CROSSWALK                     = 17,              ///< 通过人行横道
    EC_NAVI_ICON_OVERPASS                      = 18,              ///< 通过过街天桥
    EC_NAVI_ICON_UNDERPASS                     = 19,              ///< 通过地下通道
    EC_NAVI_ICON_SQUARE                        = 20,              ///< 通过广场
    EC_NAVI_ICON_PARK                          = 21,              ///< 通过公园
    EC_NAVI_ICON_STAIRCASE                     = 22,              ///< 通过扶梯
    EC_NAVI_ICON_LIFT                          = 23,              ///< 通过直梯
    EC_NAVI_ICON_CABLEWAY                      = 24,              ///< 通过索道
    EC_NAVI_ICON_SKY_CHANNEL                   = 25,              ///< 通过空中通道
    EC_NAVI_ICON_CHANNEL                       = 26,              ///< 通过通道、建筑物穿越通道
    EC_NAVI_ICON_WALK_ROAD                     = 27,              ///< 通过行人道路
    EC_NAVI_ICON_CRUISE_ROUTE                  = 28,              ///< 通过游船路线
    EC_NAVI_ICON_SIGHTSEEING_BUSLINE           = 29,              ///< 通过观光车路线
    EC_NAVI_ICON_SLIDEWAY                      = 30,              ///< 通过滑道
    EC_NAVI_ICON_LADDER                        = 31,              ///< 通过阶梯
    EC_NAVI_ICON_MERGE_LEFT                    = 51,              ///< 靠左行驶
    EC_NAVI_ICON_MERGE_RIGHT                   = 52,              ///< 靠右行驶
    EC_NAVI_ICON_SLOW                          = 53,              ///< 减速慢行
    EC_NAVI_ICON_ENTRY_RING_LEFT               = 54,              ///< 标准小环岛 绕环岛左转，右侧通行地区的逆时针环岛
    EC_NAVI_ICON_ENTRY_RING_RIGHT              = 55,              ///< 标准小环岛 绕环岛右转，右侧通行地区的逆时针环岛
    EC_NAVI_ICON_ENTRY_RING_CONTINUE           = 56,              ///< 标准小环岛 绕环岛直行，右侧通行地区的逆时针环岛
    EC_NAVI_ICON_ENTRY_RING_UTURN              = 57,              ///< 标准小环岛 绕环岛调头，右侧通行地区的逆时针环岛
    EC_NAVI_ICON_ENTRY_LEFT_RING               = 58,              ///< 进入环岛图标，左侧通行地区的顺时针环岛
    EC_NAVI_ICON_LEAVE_LEFT_RING               = 59,              ///< 驶出环岛图标，左侧通行地区的顺时针环岛
    EC_NAVI_ICON_UTURN_RIGHT                   = 60,              ///< 右转掉头图标，左侧通行地区的掉头
    EC_NAVI_ICON_SPECIAL_CONTINUE              = 61,              ///< 顺行图标(和直行有区别，顺行图标带有虚线)
    EC_NAVI_ICON_ENTRY_LEFT_RING_LEFT          = 62,              ///< 标准小环岛 绕环岛左转，左侧通行地区的顺时针环岛
    EC_NAVI_ICON_ENTRY_LEFT_RING_RIGHT         = 63,              ///< 标准小环岛 绕环岛右转，左侧通行地区的顺时针环岛
    EC_NAVI_ICON_ENTRY_LEFT_RING_CONTINUE      = 64,              ///< 标准小环岛 绕环岛直行，左侧通行地区的顺时针环岛
    EC_NAVI_ICON_ENTRY_LEFT_RING_UTURN         = 65,              ///< 标准小环岛 绕环岛调头，左侧通行地区的顺时针环岛
    EC_NAVI_ICON_SLOPE                         = 66,              ///< 通过斜坡图标
    EC_NAVI_ICON_BRIDGE                        = 67,              ///< 通过桥图标
    EC_NAVI_ICON_FERRYBOAT                     = 68,              ///< 通过渡轮图标
    EC_NAVI_ICON_SUBWAY                        = 69,              ///< 通过地铁图标
    EC_NAVI_ICON_ENTER_BUILDING                = 70,              ///< 进入建筑物图标
    EC_NAVI_ICON_LEAVE_BUILDING                = 71,              ///< 离开建筑物图标
    EC_NAVI_ICON_BY_ELEVATOR                   = 72,              ///< 电梯换层图标
    EC_NAVI_ICON_BY_STAIR                      = 73,              ///< 楼梯换层图标
    EC_NAVI_ICON_ESCALATOR                     = 74,              ///< 扶梯换层图标
    EC_NAVI_ICON_LOW_TRAFFIC_CROSS             = 75,              ///< 非导航段通过红绿灯路口图标
    EC_NAVI_ICON_LOW_CROSS                     = 76,              ///< 非导航段通过普通路口图标
    EC_NAVI_ICON_ROTARY_SHARP_LEFT             = 77,              ///< 环岛左后转，右侧通行地区的逆时针环岛
    EC_NAVI_ICON_ROTARY_SHARP_RIGHT            = 78,              ///< 环岛后右转，右侧通行地区的逆时针环岛
    EC_NAVI_ICON_ROTARY_SLIGHT_LEFT            = 79,              ///< 环岛左前转，右侧通行地区的逆时针环岛
    EC_NAVI_ICON_ROTARY_SLIGHT_RIGHT           = 80,              ///< 环岛右前转，右侧通行地区的逆时针环岛
    EC_NAVI_ICON_MAX
};

enum ScreenType
{
    Circular_Instrument = 1,
    Square_Instrument
};

typedef struct
{
    char ssid[20];
    char password[10];
}_update_info_stru;

typedef struct 
{
    uint8_t BlueToothSignalSate;
	uint8_t BlueToothPhoneSate;//incall¡¢outcall
	uint8_t  BlueToothPhoneName[100];
	uint8_t PhoneNumber[20];
    uint16_t PhoneBookTotalCount;
	uint8_t  BlueToothName[30];
    
}_BlueToothPhoneData;
typedef struct 
{
	unsigned char Data;
	unsigned char Moth;
	unsigned int  Year;
	unsigned char Hour;
    unsigned char Ninutes;
	unsigned char Second;
}_GPS_TIME;

typedef struct 
{
  uint8_t  navigation_state;
  uint8_t  EasyNavigationData;
  uint8_t  RoadName[100];
  uint32_t distance;
}_NavigationData_struct;

enum BLUE_HELMET_CONNECT_STATE
{
	BLUE_HELMET_UNSUPPORTED = 0u,
	BLUE_HELMET_STANDBY,
	BLUE_HELMET_CONNECTING,
	BLUE_HELMET_CONNECTED,
	BLUE_HELMET_PASUSED,
	BLUE_HELMET_STREAMING,
};

enum BLUE_HELMET_REQUIRE_STATE
{
    BLUE_HELMET_SCAN_END = 0u,
    BLUE_HELMET_SCAN_STATE,
    BLUE_HELMET_MAIN_REQUIRE_STATE,
    BLUE_HELMET_MAIN_REQUIRE_END,
    BLUE_HELMET_SECOND_REQUIRE_STATE,
    BLUE_HELMET_SECOND_REQUIRE_END,
};

typedef struct 
{
    uint8_t MainheadsetState;
    uint8_t SecondheadsetState;
    uint8_t MainMacAddressSuccess[20];
    uint8_t secondMacAddressSuccess[20];
    uint8_t Name[50];
    uint8_t ScanMacAddress[20];
}_BLUE_HELMET_DATA_struct;

typedef struct 
{
   uint16_t x;
   uint16_t y;
   uint16_t width;
   uint16_t height;
  
}_QRCODE_RECT;

typedef struct
{
   uint16_t qr_pic_id;
   uint16_t main_pic_id;
   uint16_t qr_logo_id ;
   uint16_t update_pic_id;
   uint16_t second_menu;
   uint16_t navigation_task_id;
}_PICID_Struct;

typedef struct{
	uint8_t blearn;
	uint8_t strmac[6];
    uint8_t temperature;
	uint16_t PressureVal;
}ec_stru_TirePressure;

extern ec_stru_TirePressure ec_stru_pressure;
extern _update_info_stru update_info;
extern uint8_t UUID_Received;
extern _BlueToothPhoneData BlueToothPhoneData;
extern _GPS_TIME GPS_Time;
extern _NavigationData_struct NavigationData_st;
extern _BLUE_HELMET_DATA_struct BlueHelmetDate;
extern void PackedTransfer_DisplayPhoneMirrorQRcode(uint8_t data,_PICID_Struct PICID_OBJ);
extern void PackedTransfer_BlueTooth_DataSet(uint8_t data);
extern void PackedTransfer_GetPhoneBookCmd(void);
extern uint8_t GetPhoneMirrorst(void) ;
extern void PackedTransfer_EasyNavigation(uint8_t data);

extern void GerProgramVer(uint8_t * bufdata);
extern uint8_t GetErrorInfo(void);
extern uint8_t GetBACKUUIDSTATE(void);
extern uint8_t GetBlueToothPowerSTATE(void);

extern void SetBluetoothTurnXX(uint8_t data);
extern void SetScreenType(uint8_t ntype,_QRCODE_RECT rect_st);
extern void SetUUIDData(char *data,uint8_t len);
extern void Set_UIFinishData(uint8_t data);
extern void SetQRCodeTypeData(uint8_t data);
extern void SetUpdateCtrl(uint8_t cmddata,_QRCODE_RECT qrcode_obj);
extern void SetBlueToothName(uint8_t *data,uint8_t len);
extern void SetTaskbarHeight(uint16_t height);
extern void Set_Pro_Unique_Identifier(uint8_t *data,uint8_t len);
extern uint8_t GetUpdate(void);
extern void SetBluetoothhelmet(uint8_t data, uint8_t *MacAddress);
extern uint8_t GetWifiConncetState(void);
#endif
