/**************************************************************************/ /**
                                                                              * \file     Analog_Circuits.h
                                                                              * \brief    Analog signal processing algorithm library header file
                                                                              * \details
                                                                              * \author   Zhang Xuan
                                                                              * \version  V1.0.0
                                                                              * \date     19-Sep-2018
                                                                              * \par      History:
                                                                              *           V1.0.0 Initial release
                                                                              * \par      Copyright:
                                                                              *           (c) Heilongjiang TYW Electronics co., LTD
                                                                              ******************************************************************************/

#ifndef _ANALOG_CIRCUITS_H_
#define _ANALOG_CIRCUITS_H_

/* Includes ------------------------------------------------------------------*/

/* C binding of definitions if building with C++ compiler */
#ifdef __cplusplus
extern
{
#endif

    /* Exported types ------------------------------------------------------------*/
    typedef struct
    {
        uint32_t u32Res1;
        uint32_t u32Res2;
        uint32_t u32Res4;
        uint32_t u32Res5;
    } ADC_Res_List_st_t;

    typedef uint16_t (*ADC_Circuit_Calc_Func)(uint16_t u16Voltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /* Exported constants --------------------------------------------------------*/
    /* Exported macro ------------------------------------------------------------*/
    /* Exported functions ------------------------------------------------------- */

    /*******************************************************************************

    *******************************************************************************/
    extern uint16_t ADC_Input_Voltage_Calc(uint16_t u16ADCSample, uint16_t u16ADCRes, uint16_t u16VRef);
    extern uint16_t ADC_VRef_Calibrate(uint16_t u16ADCSample, uint16_t u16ADCRes, uint16_t u16ActVoltage);
    extern uint16_t ADC_Data_Calibrate(uint16_t u16Value, uint8_t u8CalMode, uint16_t u16CalData);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |          ASig_Voltage_Calc_Circuit1           |
    |                                               |
    |                                 +----------   |
    |                                 |             |
    |                                 | MCU         |
    |                                 |             |
    |                                 |             |
    |    Vin >----------------------->| ADC In      |
    |                                 |             |
    |                                 |             |
    |                                 |             |
    |                                 |             |
    |                                 +----------   |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Voltage_Calc_Circuit101(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |          ASig_Voltage_Calc_Circuit2           |
    |                                               |
    |    Vin >------------+                         |
    |                     |                         |
    |                    +-+          +----------   |
    |                    | |          |             |
    |                    | | Res1     | MCU         |
    |                    +-+          |             |
    |                     |           |             |
    |                     +---------->| ADC In      |
    |                     |           |             |
    |                    +-+          |             |
    |                    | |          |             |
    |                    | | Res2     |             |
    |                    +-+          +----------   |
    |                     |                         |
    |                     |                         |
    |                    ---                        |
    |                    ///                        |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Voltage_Calc_Circuit102(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |            ASig_Res_Calc_Circuit1             |
    |                                               |
    |      VRes                                     |
    |       ---                                     |
    |        |                                      |
    |        |                                      |
    |       +-+                       +----------   |
    |       | |                       |             |
    |       | | Res4                  | MCU         |
    |       +-+                       |             |
    |        |                        |             |
    |        +----------------------->| ADC In      |
    |        |                        |             |
    |       +-+                       |             |
    |       | |                       |             |
    |       | | Res5                  |             |
    |       +-+                       +----------   |
    |        |                                      |
    |        |                                      |
    |        |                                      |
    |       +-+                                     |
    |       | | Res                                 |
    |       | | Sensor                              |
    |       +-+                                     |
    |        |                                      |
    |        |                                      |
    |       ---                                     |
    |       ///                                     |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Res_Calc_Circuit101(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |            ASig_Res_Calc_Circuit2             |
    |                                               |
    |      VRes                                     |
    |       ---                                     |
    |        |                                      |
    |        |                                      |
    |       +-+                       +----------   |
    |       | |                       |             |
    |       | | Res4                  | MCU         |
    |       +-+                       |             |
    |        |                        |             |
    |        +------------+---------->| ADC In      |
    |        |            |           |             |
    |       +-+           |           |             |
    |       | |           |           |             |
    |       | | Res5      |           |             |
    |       +-+           |           +----------   |
    |        |            |                         |
    |        |           +-+                        |
    |        |           | |                        |
    |       +-+          | | Res2                   |
    |       | | Res      +-+                        |
    |       | | Sensor    |                         |
    |       +-+           |                         |
    |        |            |                         |
    |        |            |                         |
    |       ---          ---                        |
    |       ///          ///                        |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Res_Calc_Circuit102(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |            ASig_Res_Calc_Circuit3             |
    |                                               |
    |      VRes                                     |
    |       ---                                     |
    |        |                                      |
    |        |                                      |
    |       +-+                                     |
    |       | |                                     |
    |       | | Res4                                |
    |       +-+                                     |
    |        |                                      |
    |        +------------+                         |
    |        |            |                         |
    |       +-+          +-+          +----------   |
    |       | |          | |          |             |
    |       | | Res5     | | Res1     | MCU         |
    |       +-+          +-+          |             |
    |        |            |           |             |
    |        |            +---------->| ADC In      |
    |        |            |           |             |
    |       +-+          +-+          |             |
    |       | | Res      | |          |             |
    |       | | Sensor   | | Res2     |             |
    |       +-+          +-+          +----------   |
    |        |            |                         |
    |        |            |                         |
    |       ---          ---                        |
    |       ///          ///                        |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Res_Calc_Circuit103(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |            ASig_Res_Calc_Circuit1             |
    |                                               |
    |      VRes                                     |
    |       ---                                     |
    |        |                                      |
    |        |                                      |
    |       +-+                                     |
    |       | | Res                                 |
    |       | | Sensor                              |
    |       +-+                                     |
    |        |                                      |
    |        |                                      |
    |        |                                      |
    |       +-+                       +----------   |
    |       | |                       |             |
    |       | | Res5                  | MCU         |
    |       +-+                       |             |
    |        |                        |             |
    |        +----------------------->| ADC In      |
    |        |                        |             |
    |       +-+                       |             |
    |       | |                       |             |
    |       | | Res4                  |             |
    |       +-+                       +----------   |
    |        |                                      |
    |        |                                      |
    |       ---                                     |
    |       ///                                     |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Res_Calc_Circuit201(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |            ASig_Res_Calc_Circuit1             |
    |                                               |
    |      VRes                                     |
    |       ---                                     |
    |        |                                      |
    |        |                                      |
    |       +-+                                     |
    |       | | Res                                 |
    |       | | Sensor                              |
    |       +-+                                     |
    |        |                                      |
    |        |                                      |
    |        |                                      |
    |       +-+                       +----------   |
    |       | |                       |             |
    |       | | Res5                  | MCU         |
    |       +-+                       |             |
    |        |                        |             |
    |        +------------+---------->| ADC In      |
    |        |            |           |             |
    |       +-+          +-+          |             |
    |       | |          | |          |             |
    |       | | Res4     | | Res2     |             |
    |       +-+          +-+          +----------   |
    |        |            |                         |
    |        |            |                         |
    |       ---          ---                        |
    |       ///          ///                        |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Res_Calc_Circuit202(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |            ASig_Res_Calc_Circuit1             |
    |                                               |
    |      VRes                                     |
    |       ---                                     |
    |        |                                      |
    |        |                                      |
    |       +-+                                     |
    |       | | Res                                 |
    |       | | Sensor                              |
    |       +-+                                     |
    |        |                                      |
    |        |       +----+                         |
    |        |       |    |                         |
    |       +-+      |   +-+          +----------   |
    |       | |      |   | |          |             |
    |       | | Res5 |   | | Res1     | MCU         |
    |       +-+      |   +-+          |             |
    |        |       |    |           |             |
    |        +-------+    +---------->| ADC In      |
    |        |            |           |             |
    |       +-+          +-+          |             |
    |       | |          | |          |             |
    |       | | Res4     | | Res2     |             |
    |       +-+          +-+          +----------   |
    |        |            |                         |
    |        |            |                         |
    |       ---          ---                        |
    |       ///          ///                        |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Res_Calc_Circuit203(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |            ASig_Res_Calc_Circuit1             |
    |                                               |
    |      VRes                                     |
    |       ---                                     |
    |        |                                      |
    |        |                                      |
    |       +-+                       +----------   |
    |      / | \  Current             |             |
    |      \ V /  Source              | MCU         |
    |       +-+                       |             |
    |        |                        |             |
    |        +----------------------->| ADC In      |
    |        |                        |             |
    |       +-+                       |             |
    |       | |                       |             |
    |       | | Res5                  |             |
    |       +-+                       +----------   |
    |        |                                      |
    |        |                                      |
    |        |                                      |
    |       +-+                                     |
    |       | | Res                                 |
    |       | | Sensor                              |
    |       +-+                                     |
    |        |                                      |
    |        |                                      |
    |       ---                                     |
    |       ///                                     |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Res_Calc_Circuit301(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |            ASig_Res_Calc_Circuit1             |
    |                                               |
    |      VRes                                     |
    |       ---                                     |
    |        |                                      |
    |        |                                      |
    |       +-+                       +----------   |
    |      / | \  Current             |             |
    |      \ V /  Source              | MCU         |
    |       +-+                       |             |
    |        |                        |             |
    |        +------------+---------->| ADC In      |
    |        |            |           |             |
    |       +-+           |           |             |
    |       | |           |           |             |
    |       | | Res5      |           |             |
    |       +-+           |           +----------   |
    |        |            |                         |
    |        |           +-+                        |
    |        |           | |                        |
    |       +-+          | | Res2                   |
    |       | | Res      +-+                        |
    |       | | Sensor    |                         |
    |       +-+           |                         |
    |        |            |                         |
    |        |            |                         |
    |       ---          ---                        |
    |       ///          ///                        |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Res_Calc_Circuit302(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

    /***
    +-----------------------------------------------+
    |        Equivalent Circuit for Function        |
    |            ASig_Res_Calc_Circuit1             |
    |                                               |
    |      VRes                                     |
    |       ---                                     |
    |        |                                      |
    |        |                                      |
    |       +-+                                     |
    |      / | \  Current                           |
    |      \ V /  Source                            |
    |       +-+                                     |
    |        |                                      |
    |        +------------+                         |
    |        |            |                         |
    |       +-+          +-+          +----------   |
    |       | |          | |          |             |
    |       | | Res5     | | Res1     | MCU         |
    |       +-+          +-+          |             |
    |        |            |           |             |
    |        |            +---------->| ADC In      |
    |        |            |           |             |
    |       +-+          +-+          |             |
    |       | | Res      | |          |             |
    |       | | Sensor   | | Res2     |             |
    |       +-+          +-+          +----------   |
    |        |            |                         |
    |        |            |                         |
    |       ---          ---                        |
    |       ///          ///                        |
    |                                               |
    +-----------------------------------------------+
    ***/
    extern uint16_t ADC_Res_Calc_Circuit303(uint16_t u16ADCVoltage, uint16_t u16Reference, uint16_t u16Resolution, const ADC_Res_List_st_t *pstResList);

#ifdef __cplusplus
}
#endif

#endif /* _ANALOG_CIRCUITS_H_ */
