/**************************************************************************//**
  * \file     Sys_Scheduler.c
  * \brief    Task Scheduler file
  * \details
  * \author   Zhang Xuan
  * \version  V1.0.0
  * \date     06-Sep-2018
  * \par      History:
  *           V1.0.0 Initial release
  * \par      Copyright:
  *           (c) Heilongjiang TYW Electronics co., LTD
******************************************************************************/

/* Includes ------------------------------------------------------------------*/

#include "Sys_Scheduler.h"

/* Private typedef -----------------------------------------------------------*/

typedef void (*Sys_Task_Group_t)(void);

/*** 调度更新标志联合体 ***/
typedef union
{
    uint8_t       CtrlByte;

    struct
    {
        uint8_t     Flag1ms     :   1;
        uint8_t     Flag2ms     :   1;
        uint8_t     Flag10ms    :   1;
        uint8_t     Flag20ms    :   1;
        uint8_t     Flag50ms    :   1;
        uint8_t     Flag100ms   :   1;
        uint8_t                 :   2;
    } Flag;
} Sys_Task_Update_un_t;

/*** 任务分组编号结构 ***/
typedef struct
{
    uint8_t     ID2ms;
    uint8_t     ID10ms;
    uint8_t     ID20ms;
    uint8_t     ID50ms;
    uint8_t     ID100ms;
} Sys_Task_Group_st_t;

/*** 调度控制结构 ***/
typedef struct
{
    uint16_t              msRocBak;
    uint8_t               RunMode;
    Sys_Task_Group_st_t   TaskGroup;
} Sys_Scheduling_st_t;

/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/




Sys_Scheduling_st_t   SysScheduling;

/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/**************************************************************************//**
  * \brief      系统初始化
  * \param      pmsRoc: 毫秒滚动计数器的地址
  * \retval     无
******************************************************************************/
void Sys_Init(void)
{
    SysScheduling.msRocBak            =  Sys_Get_ms_Rolling_Counter();
    SysScheduling.RunMode             =  1U;

    SysScheduling.TaskGroup.ID2ms     =  0U;
    SysScheduling.TaskGroup.ID10ms    =  0U;
    SysScheduling.TaskGroup.ID20ms    =  0U;
    SysScheduling.TaskGroup.ID50ms    =  0U;
    SysScheduling.TaskGroup.ID100ms   =  0U;

    Sys_Startup_Init();
}

/**************************************************************************//**
  * \brief      设置系统工作模式
  * \details    ==================================================================
  *               当前工作模式   |             可转换至以下工作模式
  *             -----------------+------------------------------------------------
  *             SYS_MODE_OFF     | SYS_MODE_ON,  SYS_MODE_SLEEP, SYS_MODE_STANDBY
  *             -----------------+------------------------------------------------
  *             SYS_MODE_ON      | SYS_MODE_OFF, SYS_MODE_STANDBY
  *             -----------------+------------------------------------------------
  *             SYS_MODE_STANDBY | SYS_MODE_OFF
  *             -----------------+------------------------------------------------
  *             SYS_MODE_SLEEP   | SYS_MODE_OFF
  *             ==================================================================
  *             系统工作模式的设置必须符合上表要求,错误的工作模式请求,将不会被
  *             响应,例如当前模式为SYS_MODE_ON时,请求转入SYS_MODE_SLEEP模式,该
  *             请求将被忽略, 系统保持SYS_MODE_ON模式
  * \param      Mode: 目标工作模式,可以是以下模式中的一个
  *             \arg  SYS_MODE_OFF
  *             \arg  SYS_MODE_ON
  *             \arg  SYS_MODE_SLEEP
  *             \arg  SYS_MODE_HALT
  * \retval     无
******************************************************************************/
void Sys_Enter_Sleep_Mode(void)
{
    /*if (SysScheduling.RunMode)
    {
        SysScheduling.RunMode =  0U;
    */
        Sys_Sleep_Init();
    /*}*/

    Clock_Enter_Sleep_Mode();
}

void Sys_Enter_Run_Mode(void)
{
    if (SysScheduling.RunMode == 0U)
    {
        SysScheduling.msRocBak            =  Sys_Get_ms_Rolling_Counter();
        SysScheduling.RunMode             =  1U;

        SysScheduling.TaskGroup.ID2ms     =  0U;
        SysScheduling.TaskGroup.ID10ms    =  0U;
        SysScheduling.TaskGroup.ID20ms    =  0U;
        SysScheduling.TaskGroup.ID50ms    =  0U;
        SysScheduling.TaskGroup.ID100ms   =  0U;

        Sys_Sleep_Init();
    }
}

/**************************************************************************//**
  * \brief      系统调度服务函数
  * \attention  该服务函数必须被实时调用
  * \retval     无
******************************************************************************/
void Sys_Scheduling_Service(void)
{
    uint16_t u16msROC;

    if (SysScheduling.RunMode)
    {
        Sys_Run_Mode_Pseudo_Real_Time_Tasks();

        u16msROC = Sys_Get_ms_Rolling_Counter();
        if (u16msROC != SysScheduling.msRocBak)
        {
            SysScheduling.msRocBak = u16msROC;

            Sys_Run_Mode_2ms_Tasks_Group0();

            SysScheduling.TaskGroup.ID10ms++;
            if (SysScheduling.TaskGroup.ID10ms >= 5u)
            {
                SysScheduling.TaskGroup.ID10ms = 0u;
                Sys_Run_Mode_10ms_Tasks_Group0();
            }
            SysScheduling.TaskGroup.ID20ms++;
            if (SysScheduling.TaskGroup.ID20ms >= 10u)
            {
                SysScheduling.TaskGroup.ID20ms = 0u;
                Sys_Run_Mode_20ms_Tasks_Group0();
            }
            SysScheduling.TaskGroup.ID50ms++;
            if (SysScheduling.TaskGroup.ID50ms >= 25u)
            {
                SysScheduling.TaskGroup.ID50ms = 0u;
                Sys_Run_Mode_50ms_Tasks_Group0();
            }
            SysScheduling.TaskGroup.ID100ms++;
            if (SysScheduling.TaskGroup.ID100ms >= 50u)
            {
                SysScheduling.TaskGroup.ID100ms = 0u;
                Sys_Run_Mode_100ms_Tasks_Group0();
            }
        }
    }
    else
    {
        Sys_Sleep_Mode_Tasks();

        if (SysScheduling.RunMode == 0U)
        {
            Clock_Enter_Sleep_Mode();
        }
    }
}

/**************************************************************************//**
  * \brief      精确的50us计时任务处理函数
  * \attention  该任务由中断调用
  * \retval     无
******************************************************************************/
void Sys_Exact_50us_Task_Handler(void)
{
    Sys_Exact_50us_Tasks();
}

/**************************************************************************//**
  * \brief      精确的100ms计时任务处理函数
  * \attention  该任务由中断调用
  * \retval     无
******************************************************************************/
void Sys_Exact_100ms_Task_Handler(void)
{
    Sys_Exact_100ms_Tasks();
}
