/**************************************************************************//**
  * \file     GPIO.h
  * \brief    GPIO driver header file
  * \details
  * \author   Zhang Xuan
  * \version  V1.0.0
  * \date     9-Jul-2018
  * \par      History:
  *           V1.0.0 Initial release
  * \par      Copyright:
  *           (c) Heilongjiang TYW Electronics co., LTD
******************************************************************************/

#ifndef _GPIO_H_
#define _GPIO_H_

/* Includes ------------------------------------------------------------------*/

#include "s6j3360drv.h"
#include "GPIO_Config.h"

/* C binding of definitions if building with C++ compiler */
#ifdef __cplusplus
extern  {
#endif

/* Exported types ------------------------------------------------------------*/
/**************************************************************************//**
  * \brief GPIO interrupt callback function
******************************************************************************/
typedef void (*GPIO_Int_Func_cb_t)(void); 

/**************************************************************************//**
  * \brief GPIO direction
  * Set pin to input or output
******************************************************************************/
typedef enum
{
  GPIO_DIR_IN = 0,
  GPIO_DIR_OUT,
}GPIO_Pin_Dir_en_t;

/**************************************************************************//**
  * \brief GPIO pin pull-up/down selection
  * This enumeration lists all the possible pull-up/down for input mode
******************************************************************************/
typedef enum
{
  GPIO_PULL_NONE = 0,
  GPIO_PULL_DOWN = 1,
  GPIO_PULL_UP   = 2,
}GPIO_Pin_Pull_Up_Dn_en_t;

/**************************************************************************//**
  * \brief GPIO pin configuration.
  * This structure is used to configure a GPIO pin
******************************************************************************/
typedef struct
{
  /*** Output Configuration ***/
  uint8_t                     OutputFuncID;     /* Output function(0 - 7) selection       */
  uint8_t                     OutputDrvCap;     /* Output drive capacity(0 - 3) selection */
  
  /*** Input Configuration ***/
  uint8_t                     InputEnable;      /* Input enable                           */
  uint8_t                     InputLevel;       /* Input level(0 - 3) selection           */
  GPIO_Pin_Pull_Up_Dn_en_t    PullUpDn;         /* Pull up / down selection               */
  uint8_t                     FilterEnable;     /* Noise filter enable                    */
}GPIO_Pin_Cfg_st_t;

/**************************************************************************//**
  * \brief GPIO interrupt mode
  * Select GPIO interrupt mode
******************************************************************************/
typedef enum
{
  GPIO_INT_DISABLE = 0,
  GPIO_INT_L_LEVEL,
  GPIO_INT_H_LEVEL,
  GPIO_INT_RISING_EDGE,
  GPIO_INT_FALLING_EDGE,
  GPIO_INT_BOTH_EDGES,
}GPIO_Int_Mode_en_t;

/* Exported constants --------------------------------------------------------*/
/* Exported macro ------------------------------------------------------------*/
/* Exported functions ------------------------------------------------------- */

extern void GPIO_Run_Mode_Init(void);
extern void GPIO_Sleep_Mode_Init(void);

extern void GPIO_Set_Pin_Direction(uint8_t PortNum, uint8_t PinNum, GPIO_Pin_Dir_en_t Dir);
extern void GPIO_Set_Pin_Pull_Up_Dn(uint8_t PortNum, uint8_t PinNum, GPIO_Pin_Pull_Up_Dn_en_t UpDn);
extern void GPIO_Set_Pin_Config(uint8_t PortNum, uint8_t PinNum, GPIO_Pin_Cfg_st_t *pCfg);
extern void GPIO_Set_Interrupt_Mode(uint8_t PortNum, uint8_t PinNum, GPIO_Int_Mode_en_t Mode, uint8_t NoiseFilter, GPIO_Int_Func_cb_t pFuncCb);

#ifdef __cplusplus
}
#endif

#endif /* _GPIO_H_ */