/**************************************************************************//**
  * \file     Sys_Scheduler.h
  * \brief    Task Scheduler header file
  * \details
  * \author   Zhang Xuan
  * \version  V1.0.0
  * \date     06-Sep-2018
  * \par      History:
  *           V1.0.0 Initial release
  * \par      Copyright:
  *           (c) Heilongjiang TYW Electronics co., LTD
******************************************************************************/

#ifndef _SYS_SCHEDULER_H_
#define _SYS_SCHEDULER_H_

/* Includes ------------------------------------------------------------------*/
#include "modules.h"

/* C binding of definitions if building with C++ compiler */
#ifdef __cplusplus
extern  {
#endif

/* Exported types ------------------------------------------------------------*/

/******************************************************************************

                          系统调度器的工作模式转换图
                          
                   +---------------------------------------------------+
                   |                                                   |
                   |   +---------------+                               |
                   |   |               |                               |
                   |   |  SLEEP  MODE  |                               |
                   |   |               |                               |
                   |   +---------------+                               |
                   |        ^     |                                    |
                   |        |     |                                    |
                   |       [5]   [6]                                   |
                   |        |     |                                    |
                   | +------|-----|----------------------------------+ |
                   | |      |     V                         RUN MODE | |
                   | | +---------------+           +---------------+ | |
     Power         | | |               |----[2]--->|               | | |
      On   ---[1]----->|   OFF  MODE   |           |   ON   MODE   | | |
                   | | |               |<---[3]----|               | | |
                   | | +---------------+           +---------------+ | |
                   | |      ^     |                        |         | |
                   | |      |     |                        |         | |
                   | |     [3]   [4]                       |         | |
                   | |      |     |                        |         | |
                   | |      |     V                        |         | |
                   | | +---------------+                   |         | |
                   | | |               |                   |         | |
                   | | | STANDBY  MODE |<-------[4]--------+         | |
                   | | |               |                             | |
                   | | +---------------+                             | |
                   | |                                               | |
                   | +-----------------------------------------------+ |
                   |                                                   |
                   +---------------------------------------------------+
     
       ===============================================================
            模式     |      名称      | 类型 |      用途
       --------------+----------------+------+------------------------      
          OFF MODE   | 主电源关闭模式 |      | 主电源关闭时任务调度
       --------------+----------------+      +------------------------
           ON MODE   | 主电源开启模式 | 运行 | 主电源开启时任务调度
       --------------+----------------+      +------------------------
        STANDBY MODE |  系统待机模式  |      | 工作环境异常时任务调度
       --------------+----------------+------+------------------------
         SLEEP MODE  |  系统休眠模式  | 休眠 | 系统休眠时任务调度
       ===============================================================
       ===============================================================
       状态转换过程中发生的初始化
       ---------------------------------------------------------------
       [1] Sys_Startup_Init           [4] Sys_Standby_Init
       [2] Sys_On_Init                [5] Sys_Sleep_Init
       [3] Sys_Off_Init               [6] Sys_Wakeup_Init
       ===============================================================                     
******************************************************************************/

typedef enum SysOprModeType
{
  SYS_MODE_OFF      = 0,
  SYS_MODE_ON       = 1,
  SYS_MODE_STANDBY  = 2,
  SYS_MODE_SLEEP    = 3,
}Sys_Opr_Mode_en_t;

/* Exported constants --------------------------------------------------------*/
/* Exported macro ------------------------------------------------------------*/

/*** 系统工作模式 ***/
extern Sys_Opr_Mode_en_t SysOprMode;
#define   SYS_OPR_MODE                      SysOprMode

/*** 中断相关任务使能 ***/
#define   SYS_INIT_BG_TASKS_ENABLE          (1)
#define   SYS_EXACT_50us_TASKS_ENABLE       (1)
#define   SYS_EXACT_100ms_TASKS_ENABLE      (1)

/*** 中断相关任务重定向 ***/
#if SYS_INIT_BG_TASKS_ENABLE
  #define SYS_INIT_MODE_TASK_HANDLER        Sys_Init_Mode_Task_Handler
#else
  #define SYS_INIT_MODE_TASK_HANDLER        NULL
#endif

#if SYS_EXACT_50us_TASKS_ENABLE
  #define SYS_EXACT_50us_TASK_HANDLER       Sys_Exact_50us_Task_Handler
#else
  #define SYS_EXACT_50us_TASK_HANDLER       NULL
#endif

#if SYS_EXACT_100ms_TASKS_ENABLE
  #define SYS_EXACT_100ms_TASK_HANDLER      Sys_Exact_100ms_Task_Handler
#else
  #define SYS_EXACT_100ms_TASK_HANDLER      NULL
#endif

/* Exported functions ------------------------------------------------------- */

extern void Sys_Init(void);
extern void Sys_Enter_Sleep_Mode(void);
extern void Sys_Enter_Run_Mode(void);
extern void Sys_Scheduling_Service(void);

extern void Sys_Exact_50us_Task_Handler(void);
extern void Sys_Exact_100ms_Task_Handler(void);

void Sys_Run_Mode_Pseudo_Real_Time_Tasks(void);



void Sys_Run_Mode_2ms_Tasks_Group0(void);


void Sys_Run_Mode_10ms_Tasks_Group0(void);


void Sys_Run_Mode_20ms_Tasks_Group0(void);

void Sys_Run_Mode_50ms_Tasks_Group0(void);


void Sys_Run_Mode_100ms_Tasks_Group0(void);


void Sys_Sleep_Mode_Tasks(void);
void Sys_Exact_1ms_Tasks(void);
void Sys_Exact_50us_Tasks(void);
void Sys_Exact_100ms_Tasks(void);

#ifdef __cplusplus
}
#endif

#endif /* _SYS_SCHEDULER_H_ */