
#include "Sound_Player.h"
#include "Sound_Scheduler.h"
#include "Sound_List.h"
#include "BuzzerCtrl_Interface.h"
#include "CAN_Communication_Matrix.h"
#include "Flash_synchronizer.h"
#include "GPIO.h"
#include "Buzzer.h"

/********************缺失的系统定义***************************************/
uint8_t AUDIO_EN; /*GPIO控制的不知道需不需要*/

/************************************************************************/
SoundPlayCtrlStruct SoundPlayCtrl;
volatile uint8_t    SoundPlaySeatbeltSync;

void Sound_Player_Init(void)
{
    SoundPlayCtrl.Playing = 11u; /*加载要播放的声音*/
    SoundPlayCtrl.Timer   = 5u;
    SoundPlayCtrl.Cycle   = 0u;
    SoundPlayCtrl.Status  = SND_PLAYER_PLAYING;
    AUDIO_EN              = 0u;
    Buzzer_Start_Up();
    BuzzerForceCtrlRelease( );
}

uint8_t Sound_Play(uint8_t SoundID)
{
    uint8_t m8 = 0u;

    if ( SoundID < SND_TOTAL_NUM )
    {
        if ( SoundPlayCtrl.Status != SND_PLAYER_IDLE ) /*如果有声音正在播放,则返回播放失败*/
        {
            m8 = 1u;
        }
        else
        {
            SoundPlayCtrl.Playing = SoundID; /*加载要播放的声音*/
            SoundPlayCtrl.Timer   = 0u;
            SoundPlayCtrl.Cycle   = SndAttributeTable [ SoundID ].Cycle;
            SoundPlayCtrl.Status  = SND_PLAYER_PLAY_REQ;

            m8 = 0u;
        }
        /*返回播放成功*/
    }
    else
    {
        m8 = 2u;
    }
    return m8; /*返回错误状态*/
}

uint8_t Sound_Stop(uint8_t SoundID)
{
    uint8_t m8 = 0u;

    if ( SoundID < SND_TOTAL_NUM )
    {
        if ( (SoundPlayCtrl.Status == SND_PLAYER_IDLE) || (SoundPlayCtrl.Playing != SoundID) ) /*如果播放器已停止播放或正在播放的声音不是请求停止播放的声音*/
        {
            m8 = 0u; /*返回停止成功*/
        }
        else if ( SoundPlayCtrl.Status == SND_PLAYER_PLAY_REQ ) /*请求停止播放的声音是刚请求播放的声音*/
        {
            AUDIO_EN              = 0u;
            SoundPlayCtrl.Playing = SND_SRC_NONE;
            SoundPlayCtrl.Timer   = 0u;
            SoundPlayCtrl.Cycle   = 0u;
            SoundPlayCtrl.Status  = SND_PLAYER_IDLE; /*取消播放请求*/

            m8 = 0u; /*返回停止成功*/
        }
        else
        {
            SoundPlayCtrl.Status = SND_PLAYER_STOP_REQ; /*请求停止播放声音*/
            m8                   = 1u;                  /*返回停止失败*/
        }
    }
    else
    {
        m8 = 2u; /*返回错误状态*/
    }

    return m8;
}

void Sound_Clear(void)
{
    if ( SoundPlayCtrl.Status == SND_PLAYER_PLAY_REQ )
    {
        AUDIO_EN              = 0u;
        SoundPlayCtrl.Playing = SND_SRC_NONE;
        SoundPlayCtrl.Timer   = 0u;
        SoundPlayCtrl.Cycle   = 0u;
        SoundPlayCtrl.Status  = SND_PLAYER_IDLE; /*取消播放请求*/
    }
    else
    {
        if ( SoundPlayCtrl.Playing < SND_TOTAL_NUM )
        {
            Buzzer_Stop_Play( );
        }

        AUDIO_EN              = 0u;
        SoundPlayCtrl.Playing = SND_SRC_NONE;
        SoundPlayCtrl.Timer   = 0u;
        SoundPlayCtrl.Cycle   = 0u;
        SoundPlayCtrl.Status  = SND_PLAYER_IDLE; /*取消播放请求*/
    }
}

uint8_t Sound_Priority_Query(uint8_t SoundID)
{
    uint8_t m8;
    if ( SoundID == SND_SRC_CURRENT )
    {
        if ( SoundPlayCtrl.Playing < SND_TOTAL_NUM )
        {
            m8 = SndAttributeTable [ SoundPlayCtrl.Playing ].Priority;
        }
        else
        {
            m8 = 0xFFu;
        }
    }
    else
    {
        if ( SoundID < SND_TOTAL_NUM )
        {
            m8 = SndAttributeTable [ SoundID ].Priority;
        }
        else
        {
            m8 = 0xFFu;
        }
    }
    return m8;
}

/*10ms*/
void Sound_Play_Service(void)
{
    uint8_t SoundName;
    SoundName = Get_SoundScheduling_Current( ); /*获取当前声音名*/

    if ( GetBuzzerForceCtrlSta( ) == 0u ) /*非诊断控制*/
    {
        switch ( SoundPlayCtrl.Status )
        {
            case SND_PLAYER_PLAY_REQ:
                if ( (SoundPlayCtrl.Cycle != 0u) && (SoundPlayCtrl.Playing < SND_TOTAL_NUM) )
                {

                    AUDIO_EN = 1u;
                    if ( SndAttributeTable [ SoundPlayCtrl.Playing ].Mode == SND_MODE_SINGLE )
                    {
                        Buzzer_Play_Track(SndAttributeTable [ SoundPlayCtrl.Playing ].Index);
                    }
                    else
                    {
                        if ( SoundPlayCtrl.Status == SND_PLAYER_PLAY_REQ )
                        {
                            Buzzer_Repeat_Play_Track(SndAttributeTable [ SoundPlayCtrl.Playing ].Index);
                        }
                    }

                    SoundPlayCtrl.Timer = (SndAttributeTable [ SoundPlayCtrl.Playing ].Period) / 10u;
                    if ( SoundPlayCtrl.Cycle != SND_CYCLE_NONSTOP )
                    {
                        SoundPlayCtrl.Cycle--;
                    }

                    SoundPlayCtrl.Status = SND_PLAYER_PLAYING;
                }
                else
                {
                    AUDIO_EN              = 0u;
                    SoundPlayCtrl.Playing = SND_SRC_NONE;
                    SoundPlayCtrl.Cycle   = 0u;
                    SoundPlayCtrl.Status  = SND_PLAYER_IDLE; /*取消播放请求*/
                }
                break;

            case SND_PLAYER_PLAYING:
                if ( SoundPlayCtrl.Timer )
                {
                    SoundPlayCtrl.Timer--;
                }
                if ( SoundPlayCtrl.Timer == 0u )
                {
                    if ( SoundPlayCtrl.Cycle == 0u )
                    {
                        if ( SndAttributeTable [ SoundPlayCtrl.Playing ].Mode != SND_MODE_SINGLE )
                        {
                            Buzzer_Stop_Play( );
                        }

                        AUDIO_EN = 0u;

                        SoundPlayCtrl.Playing = SND_SRC_NONE;
                        SoundPlayCtrl.Status  = SND_PLAYER_IDLE;
                    }
                    else
                    {
                        if ( SndAttributeTable [ SoundPlayCtrl.Playing ].Mode == SND_MODE_SINGLE )
                        {
                            Buzzer_Play_Track(SndAttributeTable [ SoundPlayCtrl.Playing ].Index);
                        }

                        SoundPlayCtrl.Timer = (SndAttributeTable [ SoundPlayCtrl.Playing ].Period) / 10u;
                        if ( SoundPlayCtrl.Cycle != SND_CYCLE_NONSTOP )
                        {
                            SoundPlayCtrl.Cycle--;
                        }
                    }
                }
                break;

            case SND_PLAYER_STOP_REQ:
                if ( SoundPlayCtrl.Playing < SND_TOTAL_NUM )
                {
                    if ( SndAttributeTable [ SoundPlayCtrl.Playing ].Mode != SND_MODE_SINGLE )
                    {
                         Buzzer_Stop_Play( );
                        AUDIO_EN = 0u;

                        SoundPlayCtrl.Playing = SND_SRC_NONE;
                        SoundPlayCtrl.Cycle   = 0u;
                        SoundPlayCtrl.Status  = SND_PLAYER_IDLE;
                    }
                    else
                    {
                        if ( SoundPlayCtrl.Timer )
                        {
                            SoundPlayCtrl.Timer--;
                        }

                        if ( SoundPlayCtrl.Timer == 0u )
                        {
                            AUDIO_EN              = 0u;
                            SoundPlayCtrl.Playing = SND_SRC_NONE;
                            SoundPlayCtrl.Cycle   = 0u;
                            SoundPlayCtrl.Status  = SND_PLAYER_IDLE;
                        }
                    }
                }
                break;

            default:
                break; /* case SND_PLAYER_IDLE      : break;*/
        }
    }
}

void DIAG_SoundPlay(void)
{
}

void DIAG_SoundStop(void)
{
}

uint8_t Get_SND_SEATBELT_SYNC(void)
{
    return SND_SEATBELT_SYNC;
}

void ReSet_SND_SEATBELT_SYNC(void)
{
    SND_SEATBELT_SYNC = 0u;
}

static uint8_t DiagCtrlSta = 0u;

void BuzzerForceCtrlRequest(void)
{
    DiagCtrlSta = 1u;
}

void BuzzerForceCtrlRelease(void)
{
    if ( DiagCtrlSta == 1u )
    {
        DiagCtrlSta = 0u;
        DIAG_SoundStop( );
    }
}

uint8_t GetBuzzerForceCtrlSta(void)
{
    return DiagCtrlSta;
}

void BuzzerCtrlPlayBack(void)
{
    DIAG_SoundPlay( );
}

void BuzzerCtrlStopPlaying(void)
{
    DIAG_SoundStop( );
}

uint8_t GetCurSndSrcIndx(void)
{
    return ( uint8_t )SndAttributeTable [ SoundPlayCtrl.Playing ].Index;
}
