/**************************************************************************//**
  * \file     ADC.c
  * \brief    ADC driver file
  * \details
  * \author   Zhang Xuan
  * \version  V1.0.0
  * \date     18-Sep-2018
  * \par      History:
  *           V1.0.0 Initial release
  * \par      Copyright:
  *           (c) Heilongjiang TYW Electronics co., LTD
******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "ADC.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/**************************************************************************//**
  * \brief      Initializes the ADC peripheral
  * \retval     None
******************************************************************************/
void ADC_Init(const uint8_t* pu8ChList, uint8_t u8ChNum)
{
    uint8_t  i;
    uint32_t u32ChCfg;
    volatile uint32_t *pu32ADCChctrlReg;
    
    if ((pu8ChList != NULL) && (u8ChNum > 0U) && (u8ChNum <= 64U))
    {
        /*** Step 1: Config global A/D convertet registers ***/
        ADC12B0_RT  = 120U;        /* Resumption time setting    do not modify */
        ADC12B0_ST0 = 40U;         /* Sampling time setting      do not modify */
        ADC12B0_ST1 = 40U;         /* Sampling time setting      do not modify */
        ADC12B0_ST2 = 40U;         /* Sampling time setting      do not modify */
        ADC12B0_ST3 = 40U;         /* Sampling time setting      do not modify */
        ADC12B0_CT  = 8U;          /* Comparison time setting    do not modify */
    
        ADC12B0_CTRL_RES     = 0U; /* Resolution setting 0 - 12bit 1 - 10bit 3 - 8bit */
        #ifdef __DEBUG
          ADC12B0_CTRL_DBGE  = 1U; /* Debug enable setting */
        #else
          ADC12B0_CTRL_DBGE  = 0U; /* Debug enable setting */
        #endif
        ADC12B0_CTRL_ACHMD   = 0U; /* ACH register mode setting */
        ADC12B0_CTRL_FSMD    = 1U; /* Force stop mode setting   */
        ADC12B0_CTRL_PDDMD   = 1U; /* Power down disable mode setting */
    
        /*** Step 2: Clear all ADC interrupt flags ***/
        ADC12B0_CDONEIRQC0   = 0xFFFFFFFFUL;
        ADC12B0_CDONEIRQC1   = 0xFFFFFFFFUL;
        ADC12B0_GRPIRQC0     = 0xFFFFFFFFUL;
        ADC12B0_GRPIRQC1     = 0xFFFFFFFFUL;
        ADC12B0_RCIRQC0      = 0xFFFFFFFFUL;
        ADC12B0_RCIRQC1      = 0xFFFFFFFFUL;
        ADC12B0_PCIRQC0      = 0xFFFFFFFFUL;
        ADC12B0_PCIRQC1      = 0xFFFFFFFFUL;
  
        /*** Step 3: Config ADC channels ***/
        pu32ADCChctrlReg = (volatile uint32_t*)(&ADC12B0.unCHCTRL0);
        u32ChCfg         = 0x00001000UL;    /* Configuration of ch0 is diffent from others */
        for (i = 0U; i < u8ChNum; i++)
        {
            u32ChCfg |= (uint32_t)(pu8ChList[i]);
            pu32ADCChctrlReg[i] = u32ChCfg;
            u32ChCfg  = 0x00001180UL;
        }
  
        /*** Step 4: Disable all ADC interrupts ***/
        ADC12B0_CDONEIRQE0   = 0x00000000UL;
        ADC12B0_CDONEIRQE1   = 0x00000000UL;
        ADC12B0_GRPIRQE0     = 0x00000000UL;
        ADC12B0_GRPIRQE1     = 0x00000000UL;
        ADC12B0_RCIRQE0      = 0x00000000UL;
        ADC12B0_RCIRQE1      = 0x00000000UL;
        ADC12B0_PCIRQE0      = 0x00000000UL;
        ADC12B0_PCIRQE1      = 0x00000000UL;
    }
}
  
/**************************************************************************//**
  * \brief      Deinitializes the ADC peripheral
  * \retval     None
******************************************************************************/
void ADC_DeInit(void)
{
    uint8_t  i;
    volatile uint32_t *pu32ADCChctrlReg;
    
    while (ADC12B0_STAT_BUSY)
    {
        if (ADC12B0_CTRL_FSMD)
        {
            ADC12B0_CTRL_FSTP = 1U;
        }
    }
    
    pu32ADCChctrlReg = (volatile uint32_t *)(&ADC12B0.unCHCTRL0);
    for (i = 0U; i < 64U; i++)
    {
      pu32ADCChctrlReg[i] = 0x00000000UL;
    }
    
    ADC12B0_CTRL   = 0x0000U;
}

/**************************************************************************//**
  * \brief      Start ADC conversion
  * \retval     None
******************************************************************************/
void ADC_Start_Conversion(void)
{
    ADC12B0_CHCTRL0_SWTRG = 1U;
}

/**************************************************************************//**
  * \brief      Stop ADC conversion
  * \retval     None
******************************************************************************/
void ADC_Stop_Conversion(void)
{
    if (ADC12B0_CTRL_FSMD)
    {
        ADC12B0_CTRL_FSTP = 1U;
    }
}

/**************************************************************************//**
  * \brief      Get ADC conversion status
  * \retval     0 - ADC is idle
  *             1 - ADC is busy
******************************************************************************/
uint8_t ADC_Get_Conversion_Status(void)
{
    uint8_t  Status;
    
    Status = ADC12B0_STAT_BUSY;
    return Status;
}

/**************************************************************************//**
  * \brief      Get ADC conversion data
  * \param      
  * \retval     ADC conversion data
******************************************************************************/
void ADC_Get_Conversion_Result(uint16_t* pu16Data, uint8_t u8ChNum)
{
    uint8_t i;
    volatile uint16_t *pu16ADCConvDataReg;
    
    if ((pu16Data != NULL) && (u8ChNum > 0U) && (u8ChNum <= 64U))
    {
        if (ADC12B0_STAT_BUSY == 0U)
        {
            pu16ADCConvDataReg = (volatile uint16_t*)(&ADC12B0.unCD0);
            for (i = 0U; i < u8ChNum; i++)
            {
                pu16Data[i] = pu16ADCConvDataReg[i];
            }
            
        }
    }
}
