

#include "Common_Interface.h"
#include "System_Status_Monitor.h"
#include "WDT.h"

#define API_ROLLING_COUNTER     (0U)


MonitorStruct SysFun_KL30Monitor;
MonitorStruct SysFun_KL15Monitor;

MonitorStruct SysNM_KL30Monitor;
MonitorStruct SysNM_KL15Monitor;
uint8_t SysWakeUpMode = 0u;

uint16_t System_ExportKL30Voltage;


static const uint16_t SysVol_Range[] =
{
    /**  0 --1st----  6500----2st-----  7000 ----3st-------    9000-----4st-----9500-----5st------16000         6st        16500    7st    18000     8st          18500    9st   65535 */
    Base_Vol, ENTER_LOW_N_RANGE, EXIT_LOW_N_RANGE, ENTER_LOW_Y_RANGE, EXIT_LOW_Y_RANGE, EXIT_HIGH_Y_RANGE, ENTER_HIGH_Y_RANGE, EXIT_HIGH_N_RANGE, ENTER_HIGH_N_RANGE, Top_Vol,
};


/*
     0 ----1st-------6500   不可运行   1
  6500 ----2st-------7000   回差段     2
  7000 ----3st-------9000   低压段     3
  9000 ----4st-------9500   回差段     4
  9500-----5st------16000   正常段     5
  16000----6st------16500   回差段     6
  16500----7st------18000   高压段     7
  18000----8st------18500   回差段     8
  18500----9st------65535   高压段     9

    0-----6.5   不可运行--ALL----------限制
  6.5-----9     除电机外均可运行--------正常
    9-----16.5  均可运行---------------正常
 16.5-----18.5  只有CAN可运行----------限制
 18.8-----65535 不可运行--ALL----------限制

*/

uint8_t Read_KL30_ConvertFinish(void)
{
    return SysWakeUpMode;
}


static uint8_t System_calc_range(uint16_t vol)
{
    uint8_t i;
    for (i = 0u; i < (sizeof(SysVol_Range) / sizeof(SysVol_Range[0])); i++)
    {
        if (vol < SysVol_Range[i])
        {
            break;
        }
    }
    return i;
}

static void System_Monitor_KL30Init(void)
{
    SysWakeUpMode = 0u;
    System_ExportKL30Voltage = 0u;
    SysFun_KL30Monitor.SumCnt = 0u;
    SysFun_KL30Monitor.DebounceTimer = 0u;
    SysFun_KL30Monitor.ActVoltage = 0u;
    SysFun_KL30Monitor.FinalSts = 0u;
    SysFun_KL30Monitor.ActSts = 0u;

    SysFun_KL15Monitor.SumCnt = 0u;
    SysFun_KL15Monitor.DebounceTimer = 0u;
    SysFun_KL15Monitor.ActVoltage = 0u;
    SysFun_KL15Monitor.FinalSts = 0u;
    SysFun_KL15Monitor.ActSts = 0u;

    SysNM_KL30Monitor.SumCnt = 0u;
    SysNM_KL30Monitor.DebounceTimer = 0u;
    SysNM_KL30Monitor.ActVoltage = 0u;
    SysNM_KL30Monitor.FinalSts = 0u;
    SysNM_KL30Monitor.ActSts = 0u;

    SysNM_KL15Monitor.SumCnt = 0u;
    SysNM_KL15Monitor.DebounceTimer = 0u;
    SysNM_KL15Monitor.ActVoltage = 0u;
    SysNM_KL15Monitor.FinalSts = 0u;
    SysNM_KL15Monitor.ActSts = 0u;

}

/**********************************************************************************************************
 *@Function:        Sys_KL30_Init
 *@Description:        电源状态KL30初始化
 *@Input:       none.
 *@Output:      none.
 *@Calls:       none.
 *@Return:      none.
 *@Note:        2ms.
 **********************************************************************************************************/
void Sys_KL30_Init ( void)
{
    System_Monitor_KL30Init();
    System_NM_WakeUp();
}
/**********************************************************************************************************
 *@Function:        Sys_Status_Update_Service
 *@Description:        电源状态更新
 *@Input:       none.
 *@Output:      none.
 *@Calls:       none.
 *@Return:      none.
 *@Note:        2ms.
 **********************************************************************************************************/
void Sys_Status_Update_Service ( void )
{
    System_Fun_Process();
    System_NM_Process();
    Sys_NM_Mode_Dec();
}


/**********************************************************************************************************
 *@Function:        Sys_NM_Mode_Dec
 *@Description:         网络管理正常状态转换
 *@Input:       none.
 *@Output:      none.
 *@Calls:       none.
 *@Return:      none.
 *@Note:        2ms.
 **********************************************************************************************************/
static void Sys_NM_Mode_Dec ( void )
{
    /*50402-2*/
    if ( SYS_OPR_STAT_RUN_NM )
    {
        /*网络管理唤醒状态处理*/
        if ( SYS_OPR_STAT_IGN_ON_NM )
        {
        }
        else
        {
        }
    }
    else
    {
    }
}

static void System_NM_WakeUp(void)
{
    uint16_t    KL30_Voltage;
    uint16_t    KL15_Voltage;
    uint16_t    Voltage;
    uint8_t  NormalKL30Cnt   = 0U;
    uint8_t  abnormalKL30Cnt = 0U;
    uint8_t  AttemptCnt = 0U;

    AttemptCnt = 0u;
    while (SysNM_KL30Monitor.FinalSts == 0u)
    {
        /*KL30 KL15电压采集*/
        KL30_Voltage = ADC_Conv_Single_Channel(ADC_CH_KL30_VOLTAGE);
        KL15_Voltage = ADC_Conv_Single_Channel(ADC_CH_KL15_V);
        System_ExportKL30Voltage = KL30_Voltage;
        if (KL30_Voltage > KL15_Voltage)
        {
            Voltage = KL30_Voltage;
        }
        else
        {
            Voltage = KL15_Voltage;
        }

        if ( ( Voltage <= 6500U ) || ( Voltage >= 18500U ) )
        {
            NormalKL30Cnt = 0U;
            abnormalKL30Cnt++;
        }
        else
        {
            NormalKL30Cnt++;
            abnormalKL30Cnt = 0U;
        }
        Common_BlockDelay(2U);
        AttemptCnt++;
        if (AttemptCnt >= 100u)
        {
            break;
        }
        /*消抖次数设置*/
        if ( NormalKL30Cnt >= 5U )
        {
            SysNM_KL30Monitor.FinalSts = 5u;
        }
        if ( abnormalKL30Cnt >= 5U )
        {
            /*设置成HALT模式*/
            SysNM_KL30Monitor.FinalSts = 9u;
        }
    }

    /*KL15 KL30采集结束，切换模式*/
    if ((SysNM_KL30Monitor.FinalSts != 0u))
    {
        /*切换模式*/
        SysWakeUpMode = 0x55U;
        SysNM_KL15Monitor.FinalSts = GPIO_KL15;
        SysNM_KL15Monitor.ActSts = GPIO_KL15;
        /*发送唤醒状态*/
        Sys_NM_Mode_Dec();
    }
}


static void System_NM_Process(void)
{
    uint8_t m8;
    static uint16_t DebpounceTimer;
    uint16_t Voltage;
    uint8_t  VoltageKL30Valid;
    uint8_t  VoltageKL15Valid;
    uint16_t Voltage_KL15;
    uint16_t Voltage_KL30;

    if ( SysNM_KL15Monitor.ActSts == GPIO_KL15 )
    {
        SysNM_KL15Monitor.DebounceTimer++;
        if (SysNM_KL15Monitor.DebounceTimer >= 2U)
        {
            SysNM_KL15Monitor.FinalSts = GPIO_KL15;
            SysNM_KL15Monitor.DebounceTimer = 0u;
        }

    }
    else
    {
        SysNM_KL15Monitor.DebounceTimer = 0u;
        SysNM_KL15Monitor.ActSts = GPIO_KL15;
    }


    VoltageKL30Valid =  ADC_Read_Signal_Valid(ADC_CH_KL30_VOLTAGE);
    VoltageKL15Valid =  ADC_Read_Signal_Valid(ADC_CH_KL15_V);

    if (( VoltageKL30Valid == 1U ) && ( VoltageKL15Valid == 1U ))
    {
        Voltage_KL15 = ADC_Read_Signal(ADC_CH_KL15_V);
        Voltage_KL30 = ADC_Read_Signal(ADC_CH_KL30_VOLTAGE);
        if (Voltage_KL30 > Voltage_KL15)
        {
            Voltage = Voltage_KL30;
        }
        else
        {
            Voltage = Voltage_KL15;
        }
        if (SysNM_KL30Monitor.SumCnt < 50u)
        {
            SysNM_KL30Monitor.ActVoltage += Voltage;
            SysNM_KL30Monitor.SumCnt += 1u;
            Voltage = (uint16_t)(SysNM_KL30Monitor.ActVoltage / SysNM_KL30Monitor.SumCnt);
        }
        else
        {
            SysNM_KL30Monitor.SumCnt = 0u;
            SysNM_KL30Monitor.ActVoltage = 0u;
            SysNM_KL30Monitor.ActVoltage += Voltage;
            SysNM_KL30Monitor.SumCnt += 1u;
            Voltage = (uint16_t) (SysNM_KL30Monitor.ActVoltage / SysNM_KL30Monitor.SumCnt);
        }

        m8 = System_calc_range(Voltage);
        if ( SysNM_KL30Monitor.ActSts == m8 )
        {
            SysNM_KL30Monitor.DebounceTimer++;

            /*计时累计时间1000*2ms*/
            if ( SysNM_KL30Monitor.DebounceTimer >= 1000U )
            {
                SysNM_KL30Monitor.DebounceTimer = 0U;
                DebpounceTimer = 0u;
                if (SysNM_KL30Monitor.FinalSts == 9u) /* 高压 */
                {
                    if ((m8 == 9u) || (m8 == 8u))
                    {
                        SysNM_KL30Monitor.FinalSts = 9u;
                    }
                    else
                    {
                        SysNM_KL30Monitor.FinalSts = m8;
                    }
                }
                if (SysNM_KL30Monitor.FinalSts == 1u) /* 低压 */
                {
                    if ((m8 == 1u) || (m8 == 2u))
                    {
                        SysNM_KL30Monitor.FinalSts = 1u;
                    }
                    else
                    {
                        SysNM_KL30Monitor.FinalSts = m8;
                    }
                }
                /*非高低压*/
                if ((SysNM_KL30Monitor.FinalSts != 1u) && (SysNM_KL30Monitor.FinalSts != 9u))
                {
                    SysNM_KL30Monitor.FinalSts = m8;
                }
            }
        }
        else
        {
            /*计时时间清除*/
            SysNM_KL30Monitor.DebounceTimer = 0U;
            DebpounceTimer++;
            if (DebpounceTimer >= 5u)
            {
                DebpounceTimer = 0u;
                SysNM_KL30Monitor.ActSts   = m8;
            }

        }
    }
}






static void System_Fun_Process(void)
{
    uint8_t m8;
    uint8_t  VoltageKL30Valid;
    uint8_t  VoltageKL15Valid;
    uint16_t Voltage;
    uint16_t Voltage_KL15;
    uint16_t Voltage_KL30;
    static uint16_t DebpounceTimer;

    if ( SysFun_KL15Monitor.ActSts == GPIO_KL15 )
    {
        SysFun_KL15Monitor.DebounceTimer++;
        if (SysFun_KL15Monitor.DebounceTimer >= t_KL15_Debounce)
        {
            SysFun_KL15Monitor.FinalSts = GPIO_KL15;
            SysFun_KL15Monitor.DebounceTimer = 0u;
        }

    }
    else
    {
        SysFun_KL15Monitor.DebounceTimer = 0u;
        SysFun_KL15Monitor.ActSts = GPIO_KL15;
    }


    VoltageKL30Valid =  ADC_Read_Signal_Valid(ADC_CH_KL30_VOLTAGE);
    VoltageKL15Valid =  ADC_Read_Signal_Valid(ADC_CH_KL15_V);

    if (( VoltageKL30Valid == 1U ) && ( VoltageKL15Valid == 1U ))
    {
        Voltage_KL15 = ADC_Read_Signal(ADC_CH_KL15_V);
        Voltage_KL30 = ADC_Read_Signal(ADC_CH_KL30_VOLTAGE);
        if (Voltage_KL30 > Voltage_KL15)
        {
            Voltage = Voltage_KL30;
        }
        else
        {
            Voltage = Voltage_KL15;
        }
        if (SysFun_KL30Monitor.SumCnt < 50u)
        {
            SysFun_KL30Monitor.ActVoltage += Voltage;
            SysFun_KL30Monitor.SumCnt += 1u;
            Voltage = (uint16_t)(SysFun_KL30Monitor.ActVoltage / SysFun_KL30Monitor.SumCnt);
        }
        else
        {
            SysFun_KL30Monitor.SumCnt = 0u;
            SysFun_KL30Monitor.ActVoltage = 0u;
            SysFun_KL30Monitor.ActVoltage += Voltage;
            SysFun_KL30Monitor.SumCnt += 1u;
            Voltage = (uint16_t) (SysFun_KL30Monitor.ActVoltage / SysFun_KL30Monitor.SumCnt);
        }

        /*System_CalcKL30(ADC_Read_Signal(ADC_CH_KL30_VOLTAGE));*/
        m8 = System_calc_range(Voltage);

        if ( SysFun_KL30Monitor.ActSts == m8 )
        {
            SysFun_KL30Monitor.DebounceTimer++;
            DebpounceTimer = 0u;
            /*计时累计时间1000*2ms*/
            if ( SysFun_KL30Monitor.DebounceTimer >= 1000U )/*40202-8*/
            {
                SysFun_KL30Monitor.DebounceTimer = 0U;

                if (SysFun_KL30Monitor.FinalSts >= 7u) /* 高压 */
                {
                    if ((m8 == 9u) || (m8 == 8u) || (m8 == 7u))
                    {
                        SysFun_KL30Monitor.FinalSts = 9u;
                        SysFun_KL15Monitor.FinalSts = 0u;
                    }
                    else
                    {
                        SysFun_KL30Monitor.FinalSts = m8;
                    }
                }
                if (SysFun_KL30Monitor.FinalSts == 1u) /* 低压 */
                {
                    if ((m8 == 1u) || (m8 == 2u))
                    {
                        SysFun_KL30Monitor.FinalSts = 1u;
                        SysFun_KL15Monitor.FinalSts = 0u;
                    }
                    else
                    {
                        SysFun_KL30Monitor.FinalSts = m8;
                    }
                }
                /*非高低压*/
                if ((SysFun_KL30Monitor.FinalSts != 1u) && (SysFun_KL30Monitor.FinalSts < 7u))
                {
                    SysFun_KL30Monitor.FinalSts = m8;
                }

            }
        }
        else
        {
            /*计时时间清除*/
            SysFun_KL30Monitor.DebounceTimer = 0U;
            DebpounceTimer++;
            if (DebpounceTimer >= 5u)
            {
                DebpounceTimer = 0u;
                SysFun_KL30Monitor.ActSts   = m8;
            }
        }
    }
    /*
    else
    {
        SysFun_KL30Monitor.FinalSts = 1u;
        SysFun_KL15Monitor.FinalSts = 0u;
    }*/
}

uint32_t Total_Sum;
uint32_t Total_Counter;
uint32_t Vol_Res;
void Analog_Voltage_KL30_Filter(void)
{
    uint8_t  VoltageKL30Valid;
    uint16_t Voltage_KL30;

    VoltageKL30Valid =  ADC_Read_Signal_Valid(ADC_CH_KL30_VOLTAGE);

    if (( VoltageKL30Valid == 1U ))
    {
        Voltage_KL30 = ADC_Read_Signal(ADC_CH_KL30_VOLTAGE);
        Total_Sum += Voltage_KL30;
        Total_Counter++;
        if (Total_Counter >= 50u)
        {
            Total_Sum /= 50u;

            Vol_Res = Total_Sum;
            Total_Sum = 0u;
            Total_Counter = 0u;
        }
    }
}


uint16_t System_Get_KL30Voltage(void)
{
    return System_ExportKL30Voltage;
}


uint16_t SampleBuffer[201];
uint16_t SampleCnt;
uint32_t SampleRes;
void System_CalcKL30(uint16_t Voltage)
{
    uint8_t i;
    if (SampleCnt >= 200u)
    {

        for (i = 0u; i < 200u; i++)
        {
            SampleRes += SampleBuffer[i];
            SampleBuffer[i] = 0u;
        }
        SampleRes = SampleRes / SampleCnt;
        SampleCnt = 0u;
    }
    else
    {
        SampleBuffer[SampleCnt] = Voltage;
        SampleCnt++;
    }
}

uint8_t System_KL30_Status_Get(void)
{
    return SysNM_KL30Monitor.FinalSts;
}