/******************************************************************************
 * $Revision: 423 $
 * $Date:: 2017-04-07 16:03:30 +0900#$
 *****************************************************************************/
/* __DISCLAIMER_START__                                                      */
/******************************************************************************
* (c)2017, Cypress Semiconductor Corporation
* or a subsidiary of Cypress Semiconductor Corporation. All rights
* reserved.
*
* This software, including source code, documentation and related
* materials ( "Software" ), is owned by Cypress Semiconductor
* Corporation or one of its subsidiaries ( "Cypress" ) and is protected by
* and subject to worldwide patent protection (United States and foreign),
* United States copyright laws and international treaty provisions.
* Therefore, you may use this Software only as provided in the license
* agreement accompanying the software package from which you
* obtained this Software ( "EULA" ).
*
* If no EULA applies, Cypress hereby grants you a personal, nonexclusive,
* non-transferable license to copy, modify, and compile the
* Software source code solely for use in connection with Cypress' s
* integrated circuit products. Any reproduction, modification, translation,
* compilation, or representation of this Software except as specified
* above is prohibited without the express written permission of Cypress.
*
* Disclaimer: THIS SOFTWARE IS PROVIDED AS-IS, WITH NO
* WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING,
* BUT NOT LIMITED TO, NONINFRINGEMENT, IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE. Cypress reserves the right to make
* changes to the Software without notice. Cypress does not assume any
* liability arising out of the application or use of the Software or any
* product or circuit described in the Software. Cypress does not
* authorize its products for use in any products where a malfunction or
* failure of the Cypress product may reasonably be expected to result in
* significant property damage, injury or death ( "High Risk Product" ). By
* including Cypress' s product in a High Risk Product, the manufacturer
* of such system or application assumes all risk of such use and in doing
* so agrees to indemnify Cypress against all liability.
******************************************************************************/
/* __DISCLAIMER_END__                                                        */
/*****************************************************************************
 ** \file tcflash.h
 **
 ** Headerfile for TCFlash functions
 **
 ** History:
 **   - 2014-12-01  0.01  HS  Initial version for Traveo
 *****************************************************************************/

#ifndef __TCFLASH_H__
#define __TCFLASH_H__

/*****************************************************************************/
/* Include files                                                             */
/*****************************************************************************/
/* C binding of definitions if building with C++ compiler */
#ifdef __cplusplus
extern "C" {
#endif
#include "s6j3360drv.h"
/**
 *****************************************************************************
 ** \defgroup TcflashGroup Tightly Coupled Flash Functions (TCFLASH)
 **
 ** \brief This section describes the interface for the TCFlash.
 **
 ** The TCFLASH module allows the user to modify the contents of the TCFLASH
 ** memories during run-time. This may be necessary if
 ** - the application shall be updated by a boot loader
 ** - data shall be stored in a non-volatile way to survive complete power-off
 **   phases (although EEFLASH may be more useful for this purpose
 **
 ** Following features are supported
 ** - Macro erase (formerly also known as chip erase)
 ** - Sector Erase (single and multiple sector)
 ** - 32-bit Writing
 ** - 64-bit Writing
 **
 ** Three functions are provided that can modify TCFLASH contents. The used
 ** Flash memory technology basically allows to change single bits from '1'
 ** to '0', whereas at least a sector erase with Tcflash_SectorErase()
 ** or Tcflash_MultipleSectorErase() are necessary to change any
 ** bit in this sector from '0' to '1'.
 ** When erasing is completed, all bits of the affected sector will be '1'.
 ** Furthermore a complete Flash macro can be erased at once by calling
 ** Tcflash_MacroErase(). Use this feature with care as it will erase
 ** the whole application. For MCU devices with multiple Flash macros
 ** the function must be called for every Flash macro that shall be deleted.
 ** When Writing to the tcflash, Tcflash_Write32() and Tcflash_Write64()
 ** can be used.
 **
 ** While a TCFLASH operation is in progress, no code can be executed from
 ** TCFLASH and no data can be read. Because of this the TCFLASH modifying
 ** functions themselves must be linked to and executed from RAM. This is 
 ** achieved by using certain toolchain specific mechanisms.
 **
 ** \pre Before using this module, the following must be ensured by the user:
 **      - Cache must be disabled (or at least TCFLASH AXI address area
 **        must be set as non-cacheable using the Cortex-R5 MPU)
 **      - Memory attribute of TCFLASH AXI IF is must be set
 **        as strongly ordered or device using the Cortex-R5 MPU.
 **      - All functions that are used while TCFLASH is modified must be placed in RAM.
 **      - All exception and IRQ handlers for those exceptions and IRQs that
 **        are enabled while TCFLASH is modified must be placed in RAM and
 **        the MCU must be configured accordingly.
 **
 ** \note It is strongly recommended to consult the HW manual for a better
 **       understanding of the TCFLASH features and operation.
 *****************************************************************************/
//@{

/*****************************************************************************/
/* Global pre-processor symbols/macros ('#define')                           */
/*****************************************************************************/
#define TCFLASH_AXI_PORT_LARGE_BASE                 0x01A00000  ///< Address is the same for all devices independent of nr. of integrated Flash macros
#define TCFLASH_TOTAL_LARGE_SECTOR_SIZE_PER_MACRO   0x00100000 ///< Total size of all large sectors of a single Flash macro (1024 KB)
#define TCFLASH_SMALL_SECTOR_ADDR_INC               0x00004000 ///< Address increment to get to the next interleaved small sector pair of the same Flash macro (16 KB)
#define MCU_FLASH_SIZE_BYTE                 (2*1024*1024 + 8*8*1024)
/** Conditional definition of various TCFLASH parameters depending on the number of integrated TCFLASH macros of the current device */
#define TCFLASH_AXI_PORT_SMALL_BASE             0x019f0000 ///< Base address of all small sectors (dependent on number of macros)
#define TCFLASH_TOTAL_SIZE_SMALL_SECTORS        (0x00010000) ///< Total size of all small sectors of all TCFLASH macros
#define TCFLASH_TOTAL_SIZE_LARGE_SECTORS        (MCU_FLASH_SIZE_BYTE - TCFLASH_TOTAL_SIZE_SMALL_SECTORS)
#define TCFLASH_LARGE_SECTOR_ADDR_INC           0x00020000 ///< Address increment to get to the next interleaved sector pair of the same Flash macro (128 KB for 1 macro, 256 KB for two macros)
/** Base address of Flash interface connected to the TCM port of Traveo. */
#define MCU_FLASH_TCM_INTERFACE_BASE_ADDRESS                (0x009F0000)

/** Base address of Flash interface connected to the private AXI bus
    This address area can only be accessed by the CPU core the Flash macro is assigned to. */
#define MCU_FLASH_AXI_PRIVATE_INTERFACE_BASE_ADDRESS        (0x019F0000)

/** Convert a flash TCM port address to a private AXI port address. The resulting type will be uint32_t. */
#define MCU_FLASH_ADDRESS_TCM2AXI_PRIVATE(addr)             ((MCU_FLASH_AXI_PRIVATE_INTERFACE_BASE_ADDRESS) + (((uint32_t)(addr)) - (MCU_FLASH_TCM_INTERFACE_BASE_ADDRESS)))

/*****************************************************************************/
/* Global type definitions ('typedef')                                       */
/*****************************************************************************/

/*****************************************************************************/
/* Global variable declarations ('extern', definition in C source)           */
/*****************************************************************************/

/*****************************************************************************/
/* Global function prototypes ('extern', definition in C source)             */
/*****************************************************************************/
extern en_result_t Tcflash_EnableWriteAccess(void);
extern en_result_t Tcflash_DisableWriteAccess(void);
extern en_result_t Tcflash_MacroErase(uint32_t u32Address, boolean_t bBlocking);
extern en_result_t Tcflash_SectorErase (uint32_t u32Address, boolean_t bBlocking);
extern en_result_t Tcflash_MultipleSectorErase(uint32_t * au32Address, uint8_t u8SectorCnt, boolean_t bBlocking);
extern en_result_t Tcflash_Write32(uint32_t u32Address, uint32_t u32Data, boolean_t bBlocking);
extern en_result_t Tcflash_Write64(uint32_t u32Address, uint64_t u64Data, boolean_t bBlocking);
extern en_result_t Tcflash_Reset(void);
extern un_tcfcfg0_fstat_t Tcflash_GetStatus(void);
uint32_t Tcflash_Read32(uint32_t u32Address, boolean_t bBlocking);


//@}
#ifdef __cplusplus
}
#endif

#endif /* __TCFLASH_H__ */
