/*
 * SPDX-FileCopyrightText: 2021-2024 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * @file
 * @brief ESP BSP: Wrover-kit
 */

#pragma once

#include "driver/gpio.h"
#include "driver/sdmmc_host.h"
#include "sdkconfig.h"
#include "lvgl.h"
// #include "iot_button.h"
#include "esp_lvgl_port.h"
#include "bsp/display.h"
/**************************************************************************************************
 *  BSP Capabilities
 **************************************************************************************************/

#define BSP_CAPS_DISPLAY        1
#define BSP_CAPS_TOUCH          0
#define BSP_CAPS_BUTTONS        1
#define BSP_CAPS_AUDIO          0
#define BSP_CAPS_AUDIO_SPEAKER  0
#define BSP_CAPS_AUDIO_MIC      0
#define BSP_CAPS_LED            1
#define BSP_CAPS_SDCARD         1
#define BSP_CAPS_IMU            0

/**************************************************************************************************
 * ESP-WROVER-KIT pinout
 *
 * @attention IO0 and IO2 are routed to switch button and red/green LEDs, so RGB LEDs and button CANNOT be used at the same time!
 * @attention IO2 is routed to uSD card DATA0 signal and green LED
 **************************************************************************************************/

// /* Buttons */
// #define BSP_BUTTON_BOOT_IO    (GPIO_NUM_0)

/* RGB LED */
typedef enum bsp_led_t {
    BSP_LED_RED = GPIO_NUM_0,
    BSP_LED_GREEN = GPIO_NUM_2,
    BSP_LED_BLUE = GPIO_NUM_4
} bsp_led_t;

/* Display */
#define BSP_LCD_SPI_MOSI      (GPIO_NUM_21)//
#define BSP_LCD_SPI_MISO      (GPIO_NUM_22)//
#define BSP_LCD_SPI_CLK       (GPIO_NUM_23) //
#define BSP_LCD_SPI_CS        (GPIO_NUM_14) //
#define BSP_LCD_DC            (GPIO_NUM_12) //
#define BSP_LCD_RST           (GPIO_NUM_25) //
#define BSP_LCD_BACKLIGHT     (-1)

/* uSD card */
#define BSP_SD_D0             -1//(GPIO_NUM_2)
#define BSP_SD_D1             -1//(GPIO_NUM_4)
#define BSP_SD_D2             -1//(GPIO_NUM_12)
#define BSP_SD_D3             -1//(GPIO_NUM_13)
#define BSP_SD_CMD            -1//(GPIO_NUM_15)
#define BSP_SD_CLK            -1//(GPIO_NUM_14)
#define BSP_SD_DET            -1//(GPIO_NUM_21)

#ifdef __cplusplus
extern "C" {
#endif

/* Button */
typedef enum {
    BSP_BUTTON_BOOT,
    BSP_BUTTON_NUM
} bsp_button_t;

/**
 * @brief BSP display configuration structure
 *
 */
typedef struct {
    lvgl_port_cfg_t lvgl_port_cfg;  /*!< LVGL port configuration */
    uint32_t        buffer_size;    /*!< Size of the buffer for the screen in pixels */
    bool            double_buffer;  /*!< True, if should be allocated two buffers */
    struct {
        unsigned int buff_dma: 1;    /*!< Allocated LVGL buffer will be DMA capable */
        unsigned int buff_spiram: 1; /*!< Allocated LVGL buffer will be in PSRAM */
    } flags;
} bsp_display_cfg_t;

/**************************************************************************************************
 *
 * SPIFFS
 *
 * After mounting the SPIFFS, it can be accessed with stdio functions ie.:
 * \code{.c}
 * FILE* f = fopen(BSP_SPIFFS_MOUNT_POINT"/hello.txt", "w");
 * fprintf(f, "Hello World!\n");
 * fclose(f);
 * \endcode
 **************************************************************************************************/
#define BSP_SPIFFS_MOUNT_POINT      CONFIG_BSP_SPIFFS_MOUNT_POINT

/**
 * @brief Mount SPIFFS to virtual file system
 *
 * @return
 *      - ESP_OK on success
 *      - ESP_ERR_INVALID_STATE if esp_vfs_spiffs_register was already called
 *      - ESP_ERR_NO_MEM if memory can not be allocated
 *      - ESP_FAIL if partition can not be mounted
 *      - other error codes
 */
esp_err_t bsp_spiffs_mount(void);

/**
 * @brief Unmount SPIFFS from virtual file system
 *
 * @return
 *      - ESP_OK on success
 *      - ESP_ERR_NOT_FOUND if the partition table does not contain SPIFFS partition with given label
 *      - ESP_ERR_INVALID_STATE if esp_vfs_spiffs_unregister was already called
 *      - ESP_ERR_NO_MEM if memory can not be allocated
 *      - ESP_FAIL if partition can not be mounted
 *      - other error codes
 */
esp_err_t bsp_spiffs_unmount(void);

/**************************************************************************************************
 *
 * uSD card
 *
 * After mounting the uSD card, it can be accessed with stdio functions ie.:
 * \code{.c}
 * FILE* f = fopen(BSP_MOUNT_POINT"/hello.txt", "w");
 * fprintf(f, "Hello %s!\n", bsp_sdcard->cid.name);
 * fclose(f);
 * \endcode
 *
 * @attention IO2 is also routed to RGB LED and push button
 **************************************************************************************************/
#define BSP_SD_MOUNT_POINT      CONFIG_BSP_SD_MOUNT_POINT
extern sdmmc_card_t *bsp_sdcard;

/**
 * @brief Mount microSD card to virtual file system
 *
 * @return
 *      - ESP_OK on success
 *      - ESP_ERR_INVALID_STATE if esp_vfs_fat_sdmmc_mount was already called
 *      - ESP_ERR_NO_MEM if memory can not be allocated
 *      - ESP_FAIL if partition can not be mounted
 *      - other error codes from SDMMC or SPI drivers, SDMMC protocol, or FATFS drivers
 */
esp_err_t bsp_sdcard_mount(void);

/**
 * @brief Unmount micorSD card from virtual file system
 *
 * @return
 *      - ESP_OK on success
 *      - ESP_ERR_NOT_FOUND if the partition table does not contain FATFS partition with given label
 *      - ESP_ERR_INVALID_STATE if esp_vfs_fat_spiflash_mount was already called
 *      - ESP_ERR_NO_MEM if memory can not be allocated
 *      - ESP_FAIL if partition can not be mounted
 *      - other error codes from wear levelling library, SPI flash driver, or FATFS drivers
 */
esp_err_t bsp_sdcard_unmount(void);

/**************************************************************************************************
 *
 * LEDs
 *
 **************************************************************************************************/

/**
 * @brief Set LED's GPIOs as output push-pull
 *
 * @return
 *     - ESP_OK Success
 *     - ESP_ERR_INVALID_ARG Parameter error
 */
esp_err_t bsp_leds_init(void);

/**
 * @brief Turn LED on/off
 *
 * @param led_io LED io
 * @param on Switch LED on/off
 * @return
 *     - ESP_OK Success
 *     - ESP_ERR_INVALID_ARG Parameter error
 */
esp_err_t bsp_led_set(const bsp_led_t led_io, const bool on);

/**************************************************************************************************
 *
 * Button
 *
 **************************************************************************************************/

// /**
//  * @brief Set button's GPIO as input
//  *
//  * @param[in] btn Button to be initialized
//  * @return
//  *     - ESP_OK Success
//  *     - ESP_ERR_INVALID_ARG Parameter error
//  */
// esp_err_t bsp_button_init(const bsp_button_t btn)
// __attribute__((deprecated("use espressif/button API instead")));

// /**
//  * @brief Get button's state
//  *
//  * @param[in] btn Button to read
//  * @return true  Button pressed
//  * @return false Button released
//  */
// bool bsp_button_get(const bsp_button_t btn)
// __attribute__((deprecated("use espressif/button API instead")));

// /**
//  * @brief Initialize all buttons
//  *
//  * Returned button handlers must be used with espressif/button component API
//  *
//  * @note For LCD panel button which is defined as BSP_BUTTON_MAIN, bsp_display_start should
//  *       be called before call this function.
//  *
//  * @param[out] btn_array      Output button array
//  * @param[out] btn_cnt        Number of button handlers saved to btn_array, can be NULL
//  * @param[in]  btn_array_size Size of output button array. Must be at least BSP_BUTTON_NUM
//  * @return
//  *     - ESP_OK               All buttons initialized
//  *     - ESP_ERR_INVALID_ARG  btn_array is too small or NULL
//  *     - ESP_FAIL             Underlaying iot_button_create failed
//  */
// esp_err_t bsp_iot_button_create(button_handle_t btn_array[], int *btn_cnt, int btn_array_size);

/**************************************************************************************************
 *
 * LCD interface
 *
 * ESP32-WROVER-KIT is shipped with 3.2inch ST7789 or ILI9341 display controller. It features 16-bit colors and 320x240 resolution.
 * If your colours on the display are distorted, try changing the display type in menuconfig.
 *
 * LVGL is used as graphics library. LVGL is NOT thread safe, therefore the user must take LVGL mutex
 * by calling bsp_display_lock() before calling any LVGL API (lv_...) and then give the mutex with
 * bsp_display_unlock().
 *
 * Display's backlight must be enabled explicitly by calling bsp_display_backlight_on()
 **************************************************************************************************/
#define BSP_LCD_PIXEL_CLOCK_HZ      (30 * 1000 * 1000)
#define BSP_LCD_SPI_NUM             (SPI2_HOST)

#define BSP_LCD_DRAW_BUFF_SIZE      (BSP_LCD_H_RES * 30)
#define BSP_LCD_DRAW_BUFF_DOUBLE    (1)

/**
 * @brief Initialize display
 *
 * This function initializes SPI, display controller and starts LVGL handling task.
 *
 * @return Pointer to LVGL display or NULL when error occurred
 */
lv_display_t *bsp_display_start(void);

/**
 * @brief Initialize display
 *
 * This function initializes SPI, display controller and starts LVGL handling task.
 * LCD backlight must be enabled separately by calling bsp_display_brightness_set()
 *
 * @param cfg display configuration
 *
 * @return Pointer to LVGL display or NULL when error occurred
 */
lv_display_t *bsp_display_start_with_config(const bsp_display_cfg_t *cfg);

/**
 * @brief Get pointer to input device (touch, buttons, ...)
 *
 * @note The LVGL input device is initialized in bsp_display_start() function.
 *
 * @return Pointer to LVGL input device or NULL when not initialized
 */
lv_indev_t *bsp_display_get_input_dev(void);

/**
 * @brief Take LVGL mutex
 *
 * @param timeout_ms Timeout in [ms]. 0 will block indefinitely.
 * @return true  Mutex was taken
 * @return false Mutex was NOT taken
 */
bool bsp_display_lock(uint32_t timeout_ms);

/**
 * @brief Give LVGL mutex
 *
 */
void bsp_display_unlock(void);

/**
 * @brief Rotate screen
 *
 * Display must be already initialized by calling bsp_display_start()
 *
 * @param[in] disp Pointer to LVGL display
 * @param[in] rotation Angle of the display rotation
 */
void bsp_display_rotate(lv_display_t *disp, lv_disp_rotation_t rotation);

#ifdef __cplusplus
}
#endif
